//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Detector/DetectorItem.h
//! @brief     Defines class DetectorItem
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_MODEL_DETECTOR_DETECTORITEM_H
#define BORNAGAIN_GUI_MODEL_DETECTOR_DETECTORITEM_H

#include "GUI/Model/Data/MaskItems.h"
#include "GUI/Model/Descriptor/SelectionProperty.h"
#include "GUI/Model/Detector/ResolutionFunctionItemCatalog.h"

class IDetector;
class IResolutionFunction2D;
class MaskContainerItem;
class ResolutionFunctionItem;

class DetectorItem {
public:
    virtual ~DetectorItem();

    std::unique_ptr<IDetector> createDetector() const;

    virtual void writeTo(QXmlStreamWriter* w) const;
    virtual void readFrom(QXmlStreamReader* r);

    //! Returns the size of x-axis of the detector
    virtual int xSize() const = 0;

    //! Returns the size of y-axis of the detector
    virtual int ySize() const = 0;

    //! sets the size of x-axis of the detector
    virtual void setXSize(size_t nx) = 0;

    //! sets the size of y-axis of the detector
    virtual void setYSize(size_t ny) = 0;

    void importMasks(const MaskContainerItem* maskContainer);
    MaskContainerItem& maskItems();

    SelectionProperty<ResolutionFunctionItemCatalog>& resolutionFunctionSelection();

    template <typename T>
    T* setResolutionFunctionType();

    //! Scales the values provided by axes (to perform deg->rad conversion on the way to domain).
    virtual double axesToCoreUnitsFactor() const { return 1.0; }

    bool isExpandResolutionFunc() const { return m_expandResolutionFunc; }
    void setExpandResolutionFunc(bool b) { m_expandResolutionFunc = b; }

protected:
    DetectorItem();

    virtual std::unique_ptr<IDetector> createDomainDetector() const = 0;
    std::unique_ptr<IResolutionFunction2D> createResolutionFunction() const;

    void addMasksToCore(IDetector* detector) const;

protected:
    //! for creation of domain detector; only filled and relevant in jobs
    MaskContainerItem m_maskContainerItem; // without serializtion
    SelectionProperty<ResolutionFunctionItemCatalog> m_resolutionFunction;
    bool m_expandResolutionFunc = true;
};

template <typename T>
T* DetectorItem::setResolutionFunctionType()
{
    m_resolutionFunction.setCurrentItem<T>();
    return dynamic_cast<T*>(m_resolutionFunction.currentItem());
}

#endif // BORNAGAIN_GUI_MODEL_DETECTOR_DETECTORITEM_H
