/*
 * Copyright (c) 2002 Robert Collins.
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     A copy of the GNU General Public License can be found at
 *     http://www.gnu.org/
 *
 * Written by Robert Collins <robertc@hotmail.com>
 *
 */

#include "Directory.h"
#include "FileSystemVisitor.h"
#include "File.h"
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <stdlib.h>
#include <sys/unistd.h>
#include <errno.h>

#include <iostream>
bool
Directory::Exists (string const &aDir)
{
    /* stat the path */
    struct stat statBuf;
    int statrv = stat(aDir.c_str(), &statBuf);
    if (statrv) {
	if (errno != ENOENT)
	    throw (string) "Could not stat '" + aDir + "'.";
	return false;
    }

    if (!S_ISDIR(statBuf.st_mode))
	return false;

    return true;
}

void
Directory::Create (string const &aDir)
{
    /* check it doesn't exist */
    if (Exists (aDir))
	throw (string) "Directory '" + aDir + "' already exists.";
    /* get a reverse list of path elements */
    string currentSection;
    string remainingSection = aDir;
    StripSlash (remainingSection);
    while (remainingSection.size()) {
	/* iterate until one of them doesn't exist, or access is denied
	 * or one is not a directory (or a link to a dir) 
	 */
    	unsigned int offset = remainingSection.rfind ("/");
    	if (offset != string::npos) {
    	    currentSection = remainingSection.substr (offset);
    	    remainingSection.erase(offset);
    	} else {
    	    currentSection = remainingSection;
    	    remainingSection.erase();
	}
	if (Exists(remainingSection)) {\
	    /* Now create each remaining element in the path */
	    CreateFrom (remainingSection + "/", aDir.substr (remainingSection.size()+1));
	    return;
	}
    }
}

string
Directory::CurrentDirectory()
{
    size_t size = 500;

    while (1) {
        char *buffer = new char[size];
	char *rv = getcwd (buffer, size);
	
	if (rv) {
	    string result(rv);
	    delete[] buffer;
	    return result;
	}
	delete[] buffer;
	
	if (errno != ERANGE)
	    throw (string) "Could not get current working directory";

	size = size * 2;
    }

}

void
Directory::CreateFrom (string const &basePath, string const &aPath)
{
    if (!Exists(basePath))
	throw (string) "Directory::CreateFrom: Directory '" + basePath + "' non existant.";
    string current;
    string remaining;
    string prefix = basePath;
    remaining = aPath;
    StripSlash (remaining);
    while (remaining.size()) {
	unsigned int offset = remaining.find ("/");
	if (offset != string::npos) {
	    current = remaining.substr (0, offset);
	    remaining.erase (0, offset + 1);
	} else {
	    current = remaining;
	    remaining.erase();
	}
	int mkdirResult = mkdir ((prefix + current).c_str(), 0777);
	if (mkdirResult)
	    throw (string) "Could not create path segment '" + current + "' after '" + prefix + "'.";
	prefix += current + "/";
    }
}

void
Directory::StripSlash (string &aString)
{
    if (aString[aString.size() - 1] == '/')
      aString.erase (aString.size() - 1);
}

Directory::Directory (string const &path, mode_t const &aMode) : _path (path), mode(aMode)
{
    theDir = opendir (path.c_str());
    if (!theDir)
	throw (string) "Directory::Directory: Could not open directory " + path;
}

Directory::~Directory()
{
    if (theDir)
	closedir (theDir);
}

void 
Directory::visitEntry (struct dirent *entry, FileSystemVisitor &theVisitor)
{
    if (entry->d_name == string(".") ||
	entry->d_name == string(".."))
	return;
    /* fixme: use a factory on the path */
    FileSystemComposite *aNode = FileSystemComposite::CreateFromPath (_path.fullName() + "/" + entry->d_name);
    aNode->visit (theVisitor);
    delete aNode;
}

#include <set>

void
Directory::visit (FileSystemVisitor &theVisitor)
{
    if (!theVisitor.visitDirectory( *this))
	return;
    rewinddir (theDir);
    struct dirent *entry;
    /* build in memory list */
    set<string> entries;
    while ((entry = readdir (theDir))) {
    if (!(entry->d_name == string(".") ||
	entry->d_name == string("..")))
	entries.insert (_path.fullName() + "/" + entry->d_name);
    }
    
    for (set<string>::iterator i = entries.begin(); i != entries.end(); ++i)
      {
	FileSystemComposite *aNode = FileSystemComposite::CreateFromPath (*i);
	aNode->visit (theVisitor);
	delete aNode;
      }
//	visitEntry (entry, theVisitor);
}

void
Directory::RemoveRecursive(Path const &aPath)
{
    string command (string("rm -rf ") + aPath.fullName());
    if (system(command.c_str())); // we don't care if the rm failed.
    // strings that have had cstr() called don't free their own buffer
    delete[] command.c_str();
}
