/*******************************************************************************

  Intel Data Center Bridging (DCB) Software
  Copyright(c) 2007-2009 Intel Corporation.

  This program is free software; you can redistribute it and/or modify it
  under the terms and conditions of the GNU General Public License,
  version 2, as published by the Free Software Foundation.

  This program is distributed in the hope it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

  The full GNU General Public License is included in this distribution in
  the file called "COPYING".

  Contact Information:
  e1000-eedc Mailing List <e1000-eedc@lists.sourceforge.net>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

*******************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <sys/socket.h>
#include <linux/rtnetlink.h>
#include <linux/if.h>
#include <syslog.h>
#include <unistd.h>
#include "dcbd.h"
#include "common.h"
#include "eloop.h"
#include "dcb_types.h"
#include "dcb_protocol.h"
#include "config.h"
#include "drv_cfg.h"
#include "bonding.h"
#include "event_iface.h"
#include "lldp/ports.h"
#include "lldp/l2_packet.h"

static void event_if_decode_rta(int type, struct rtattr *rta);

#define MAX_PAYLOAD 4096  /* maximum payload size*/
#define LINK_UP         1
#define LINK_DOWN       2
#define DEVICE_REMOVE   3
#define DEVICE_ADD      5

static char *device_name = NULL;
static int link_status = 0;

/* socket pair for the link event socket */
static int evs[2];

static char *decode_oper_state(int operstate)
{
	switch(operstate) {
	case IF_OPER_UNKNOWN:
		return "IF_OPER_UNKNOWN";
	case IF_OPER_NOTPRESENT:
		return "IF_OPER_NOTPRESENT";
	case IF_OPER_DOWN:
		link_status = LINK_DOWN;
		return "IF_OPER_DOWN";
	case IF_OPER_LOWERLAYERDOWN:
		return "IF_OPER_LOWERLAYERDOWN";
	case IF_OPER_TESTING:
		return "IF_OPER_TESTING";
	case IF_OPER_DORMANT:
		return "IF_OPER_DORMANT";
	case IF_OPER_UP:
		link_status = LINK_UP;
		return "IF_OPER_UP";
	default:
		return "UNKNOWN";
	}
}

static void event_if_decode_rta(int type, struct rtattr *rta)
{

	TRACE1("    rta_type  =", rta->rta_len);
	
	switch (type) {
	case IFLA_ADDRESS:
		TRACE(" IFLA_ADDRESS\n");
		break;
	case IFLA_BROADCAST:
		TRACE(" IFLA_BROADCAST\n");
		break;
	case IFLA_OPERSTATE:
		TRACE1(" IFLA_OPERSTATE ", type);
		decode_oper_state(*((int *)RTA_DATA(rta)));
		break;
	case IFLA_LINKMODE:
		TRACE1(" IFLA_LINKMODE  ", type)
		TRACE2("        LINKMODE = ", (*((int *)RTA_DATA(rta)))? 
			"IF_LINK_MODE_DORMANT": "IF_LINK_MODE_DEFAULT")
		break;
	case IFLA_IFNAME:
		device_name =  (char *)RTA_DATA(rta);
		TRACE(" IFLA_IFNAME\n")
		TRACE2(" device name is ", device_name)
		break;
	default:
		TRACE1(" unknown type : ", type)
		break;
	}
}

static void event_if_decode_nlmsg(int route_type, void *data, int len)
{
	struct rtattr *rta;
	int attrlen;
	int dcb_enable;
	int val;

	switch (route_type) {
	case RTM_NEWLINK:		
	case RTM_DELLINK:
	case RTM_SETLINK:
	case RTM_GETLINK:
		TRACE("  IFINFOMSG\n");
		TRACE1("  ifi_family = ",
			((struct ifinfomsg *)data)->ifi_family);
		TRACE1("  ifi_type   = ",
			((struct ifinfomsg *)data)->ifi_type);
		TRACE1("  ifi_index  = ",
			((struct ifinfomsg *)data)->ifi_index);
		TRACE1("  ifi_flags  = ",
			((struct ifinfomsg *)data)->ifi_flags);
		TRACE1("  ifi_change = ",
			((struct ifinfomsg *)data)->ifi_change);

		/* print attributes */
		rta = IFLA_RTA(data);

		attrlen = len - sizeof(struct ifinfomsg);
		while (RTA_OK(rta, attrlen)) {
			event_if_decode_rta(rta->rta_type, rta);
			rta = RTA_NEXT(rta, attrlen);
		}

		TRACE1("link status: ", link_status);
		TRACE2("device name: ", device_name);
		if (link_status) {
			switch (link_status) {
			case LINK_DOWN:
				printf("******* LINK DOWN: %s\n", device_name);
				/* Don't need to check the dcb status */
				if (is_bond(device_name)) {
					if (remove_bond_port(device_name) < 0)
						syslog(LOG_ERR, "failed to "
							"remove bond port %s",
							device_name);
				} else {
					
					val = get_port_hw_resetting(
						device_name);
					if (val != 1) {
						if (!init_cfg())
							break;
						remove_adapter(device_name); 
						destroy_cfg();
					} else {
						printf("******* IGNORING: %s\n",
							device_name);
						set_lldp_port_enable_state(
							device_name, 0);
					}
				}
				break;
			case LINK_UP:
				printf("******* LINK UP: %s\n", device_name);
				if (is_bond(device_name)) {
					if (add_bond_port(device_name) < 0)
						syslog(LOG_ERR, "failed to "
							"register bond port %s",
							device_name);
				}		
				else {
					if (get_port_hw_resetting(
						device_name) == 1) {
						printf("******* IGNORING: %s\n",
							device_name);
						set_port_hw_resetting(
							device_name, 0);
						set_lldp_port_enable_state(
							device_name, 1);
						break;
					}
					if (!init_cfg())
						break;
					if (get_dcb_enable_state(device_name,
						&dcb_enable))
						set_hw_state(device_name,
							dcb_enable);

					if (check_port_dcb_mode(device_name)) {
						printf("Device %s has been "
							"added.\n",
							device_name);
						add_adapter(device_name);
				 	}
					destroy_cfg();
				}
				break;
			default:
				break;
			}
		}
		break;
	case RTM_NEWADDR:
	case RTM_DELADDR:
	case RTM_GETADDR:
		TRACE("Address change.\n");
		break;
	default:
		TRACE("No decode for this type\n");
	}
}


static void event_if_process_recvmsg(struct nlmsghdr *nlmsg)
{
	

	/* print out details */
	event_if_decode_nlmsg(nlmsg->nlmsg_type, NLMSG_DATA(nlmsg),
		NLMSG_PAYLOAD(nlmsg, 0));
}

void sendto_event_iface(char *buf, int len)
{
	send(evs[EVENT_IF_WRITE], buf, len, 0);
}

static void dcbd_event_iface_receive(int sock, void *eloop_ctx, void *sock_ctx)
{
	struct nlmsghdr *nlh;
	struct sockaddr_nl dest_addr;
	char buf[MAX_MSG_SIZE];
	socklen_t fromlen = sizeof(dest_addr);
	int result;
	
	result = recvfrom(sock, buf, sizeof(buf), MSG_DONTWAIT,
		       (struct sockaddr *) &dest_addr, &fromlen);

	if (result < 0) {
		perror("recvfrom(Event interface)");
		return;
	}

	TRACE("PRINT BUF info.\n")

	if (sock_ctx) {
		/* message is from the RT netlink interface, then pass it
		 * on to the event interface socket
		*/
		sendto_event_iface(buf, result);
	} else {
		/* handle messages received on the link event socket
		*/
		device_name = NULL;
		link_status = 0;
		nlh = (struct nlmsghdr*)buf;
		event_if_process_recvmsg(nlh);
	}
}


int dcbd_event_iface_init()
{
	/* setup and register a socket for the event interface queue
	*/
	if (socketpair(AF_UNIX, SOCK_DGRAM, 0, evs) < 0) {
		printf("dcbd_event_iface_init: ERROR OPENING SOCKETPAIR\n");
		return -1;
	}

	eloop_register_read_sock(evs[EVENT_IF_READ], dcbd_event_iface_receive,
		NULL, NULL);

	/* The driver interface and event interface both use RT netlink.
	 * Get and register the socket for the RT netlink interface.
	 * Set the context pointer to non-NULL to allow
	 * dcbd_event_iface_receive to distinguish between the RT netlink
	 * and the link event interface sockets.  Messages
	 * received from RT netlink will be forwarded to the link event
	 * interface.
	*/
	eloop_register_read_sock(get_drv_if_socket(), dcbd_event_iface_receive,
		NULL, (void *) 1);

	return 0;
}

int dcbd_event_iface_deinit()
{
	return 0;
}
