/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "peoplewidget.h"

PeopleWidget::PeopleWidget(QString buttonText,
                           int type,
                           PumpController *pumpController,
                           QWidget *parent) : QWidget(parent)
{
    this->setMinimumSize(120, 120);

    this->pController = pumpController;
    connect(pController, SIGNAL(contactListReceived(QString,QVariantList,int)),
            this, SLOT(updateAllContactsList(QString,QVariantList,int)));



    this->searchLabel = new QLabel(tr("&Search:"));

    this->searchLineEdit = new QLineEdit();
    searchLineEdit->setPlaceholderText(tr("Enter a name here to search for it"));
    searchLabel->setBuddy(searchLineEdit);
    connect(searchLineEdit, SIGNAL(textChanged(QString)),
            this, SLOT(filterList(QString)));


    allContactsListWidget = new QListWidget();
    allContactsListWidget->setDragDropMode(QListView::DragDrop);
    allContactsListWidget->setDefaultDropAction(Qt::CopyAction);
    allContactsListWidget->setSortingEnabled(true);
    connect(allContactsListWidget, SIGNAL(activated(QModelIndex)),
            this, SLOT(returnClickedContact(QModelIndex)));


    addToButton = new QPushButton(QIcon::fromTheme("list-add"),
                                  buttonText);
    connect(addToButton, SIGNAL(clicked()),
            this, SLOT(returnContact()));



    // Layout
    mainLayout = new QVBoxLayout();
    mainLayout->addWidget(searchLabel);
    mainLayout->addWidget(searchLineEdit);
    mainLayout->addSpacing(2);
    mainLayout->addWidget(allContactsListWidget);
    mainLayout->addSpacing(4);
    mainLayout->addWidget(addToButton);
    this->setLayout(mainLayout);



    if (type == EmbeddedWidget)
    {
        //allContactsListWidget->setSelectionMode(QListView::ExtendedSelection);

        // FIXME 1.2: For now, Single selection mode in both cases
        allContactsListWidget->setSelectionMode(QListView::SingleSelection);
    }
    else     // StandaloneWidget
    {
        this->setWindowTitle("Dianara - "
                             + tr("Add a contact to a list"));
        this->setWindowIcon(QIcon::fromTheme("system-users"));
        this->setWindowFlags(Qt::Window);
        this->setWindowModality(Qt::WindowModal);

        allContactsListWidget->setSelectionMode(QListView::SingleSelection);

        cancelButton = new QPushButton(QIcon::fromTheme("dialog-cancel"),
                                       tr("&Cancel"));
        connect(cancelButton, SIGNAL(clicked()),
                this, SLOT(hide()));
        mainLayout->addWidget(cancelButton);
    }


    this->filterList(""); // Populate the list without filter, initially

    qDebug() << "PeopleWidget created";
}


PeopleWidget::~PeopleWidget()
{
    qDebug() << "PeopleWidget destroyed";
}


void PeopleWidget::resetWidget()
{
    this->allContactsListWidget->scrollToTop();

    this->searchLineEdit->clear(); // Might also trigger filterLists()
    this->searchLineEdit->setFocus();
}



//////////////////////////////////////////////////////////////////////////////
///////////////////////////////// SLOTS //////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


/*
 * Filter the list of all contacts based on what
 * the user entered in the search box
 *
 */
void PeopleWidget::filterList(QString searchTerms)
{
    allContactsListWidget->clear();

    allContactsListWidget->addItems(allContactsStringList.filter(searchTerms,
                                                                 Qt::CaseInsensitive));
}


/*
 * Update the list of all contacts (actually, 'following')
 * from the PumpController
 *
 */
void PeopleWidget::updateAllContactsList(QString listType,
                                         QVariantList contactsVariantList,
                                         int totalReceivedCount)
{
    if (listType != "following")
    {
        return;
    }
    qDebug() << "PeopleWidget: received list of Following; updating...";

    if (totalReceivedCount == 200)  // Only on first batch
    {
        allContactsStringList.clear();
    }

    QString singleContactString;
    foreach (QVariant contact, contactsVariantList)
    {
        singleContactString = contact.toMap().value("displayName").toString();
        singleContactString.append("  <");
        singleContactString.append(contact.toMap().value("id").toString().remove(0,5));
        singleContactString.append(">");

        allContactsStringList.append(singleContactString);
    }


    // Populate the list widget, using whatever filter is set
    this->filterList(searchLineEdit->text());
}



/*
 * Send current contact string in a SIGNAL
 *
 * Used when selecting a row and clicking the "add" button
 *
 */
void PeopleWidget::returnContact()
{
    QString currentContact;
    if (this->allContactsListWidget->currentRow() != -1)
    {
        currentContact = allContactsListWidget->currentItem()->data(0).toString();
    }

    emit addButtonPressed(currentContact);
}

/*
 * Used when clicking a contact
 *
 */
void PeopleWidget::returnClickedContact(QModelIndex modelIndex)
{
    emit contactSelected(modelIndex.data().toString());
}
