------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                    --
--  Copyright (C) 2000-2007 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-connections.adb,v 1.20 2008/07/03 06:22:48 vagul Exp $

with Ada.Exceptions;
with Ada.Strings.Fixed;
with Ada.Unchecked_Deallocation;

with OCI.Thread;

package body OCI.Thick.Connections is

   use type SWord;
   use type Ub4;
   use type OCIHandle;

   -----------
   -- Break --
   -----------

   procedure Break (It : Connection) is
   begin
      Check_Error (OCIBreak (Handle (It), Thread.Error));
   end Break;

   ------------
   -- Commit --
   ------------

   procedure Commit (Connect : in Connection) is
   begin
      Check_Error (OCITransCommit (OCISvcCtx (Connect.Handle), Thread.Error));
   end Commit;

   -------------
   -- Destroy --
   -------------

   procedure Destroy (Object : in out Connection) is
      H : OCIHandle := Object.Handle;

      procedure Free is
        new Ada.Unchecked_Deallocation (Type_Caches.Map, Type_Cache_Access);
   begin
      if H /= Empty_Handle then
         if Object.Sessn /= OCISession (Empty_Handle) then
            Ignore_Error (OCISessionEnd
                            (Svchp => OCISvcCtx (H),
                             Errhp => Thread.Error,
                             Usrhp => Object.Sessn));

            Free (OCIHandle (Object.Sessn), OCI_HTYPE_SESSION);
            Free (H, OCI_HTYPE_SVCCTX);
         else
            Ignore_Error
              (OCILogoff (Svchp => OCISvcCtx (H), Errhp => Thread.Error));
            --  ??? Check_Error -- 8i bug
         end if;

         Free (Object.Type_Cache);
      end if;
   end Destroy;

   --------------------
   -- Get_Connection --
   --------------------

   function Get_Connection
     (Context : in OCIExtProcContext) return Connection
   is
      Rc : SWord;
      H  : aliased OCISvcCtx;
      E  : aliased OCIEnv;
      Er : aliased OCIError;
      --  tr : OCI.Environments.Thread_Environment;
      Result : Connection;
   begin
      Rc := OCIExtProcGetEnv
        (octxp => Context,
         Envhp => E'Access,
         Errhp => Er'Access,
         Svchp => H'Access);

      --  tr := (RF.Controlled_Reference with Handle => E);
      --  Thread.Set_Environment (tr);

      Thread.Set_Error (E, Er);

      Check_Error (Rc);

      Result.Handle := OCIHandle (H);
      Result.Environment := Thread.Synch.Environment;

      return Result;
   end Get_Connection;

   --------------
   -- Get_Type --
   --------------

   function Get_Type
     (Connect : in Connection; Name : in String) return OCIType
   is
      CN  : Type_Caches.Cursor := Connect.Type_Cache.Find (Name);
      TDO : aliased OCIType;
      SS  : Boolean;
      Empty : Text (1 .. 0);
   begin
      if Type_Caches.Has_Element (CN) then
         return Type_Caches.Element (CN);

      else
         Check_Error
           (OCITypeByName
              (Env       => Thread.Environment,
               Err       => Thread.Error,
               Svc       => OCISvcCtx (Handle (Connect)),
               Schema    => Empty,
               S_Length  => 0,
               Type_Name => C.To_C (Name, False),
               T_Length  => Name'Length,
               Version   => Empty,
               V_Length  => 0,
               Duration  => OCI_DURATION_SESSION,
               Option    => OCI_TYPEGET_ALL,
               TDO       => TDO'Access));

         Connect.Type_Cache.Insert (Name, TDO, CN, SS);

         return TDO;
      end if;
   end Get_Type;

   -----------------
   -- Is_Blocking --
   -----------------

   function Is_Blocking (Connect : in Connection) return Boolean is
      function Get_Attrib is new Get_Attr_G (Ub1);
      use type Ub1;
   begin
      return Get_Attrib
               (Connect.Handle, OCI_HTYPE_SVCCTX, OCI_ATTR_NONBLOCKING_MODE)
                = 0;
   end Is_Blocking;

   ------------
   -- Logoff --
   ------------

   procedure Logoff (Connect : in out Connection) is
      Result : Connection;
   begin
      Connect := Result;
   end Logoff;

   -----------
   -- Logon --
   -----------

   function Logon
     (Server_Name : in String;
      User        : in String;
      Password    : in String) return Connection is
   begin
      return Logon (DB       => Attach (Server_Name),
                    User     => User,
                    Password => Password);
   end Logon;

   function Logon
     (DB       : Server;
      User     : String;
      Password : String) return Connection
   is
      Result : Connection;
   begin
      Result.Environment := Thread.Synch.Environment;
      Result.DB := DB;
      Result.Handle := Alloc_Handle (Thread.Environment, OCI_HTYPE_SVCCTX);
      Set_Attr (Result.Handle, OCI_HTYPE_SVCCTX, OCI_ATTR_SERVER, Handle (DB));
      Result.Sessn := OCISession
                        (Alloc_Handle
                           (Thread.Environment, OCI_HTYPE_SESSION));

      Set_Attr (OCIHandle (Result.Sessn),
                OCI_HTYPE_SESSION,
                OCI_ATTR_USERNAME,
                User);
      Set_Attr (OCIHandle (Result.Sessn),
                OCI_HTYPE_SESSION,
                OCI_ATTR_PASSWORD,
                Password);
      Set_Attr (Result.Handle,
                OCI_HTYPE_SVCCTX,
                OCI_ATTR_SESSION,
                OCIHandle (Result.Sessn));
      Check_Error
        (OCISessionBegin
           (OCISvcCtx (Result.Handle), Thread.Error, Result.Sessn));

      if OCI.Environments.Is_Objects then
         Result.Type_Cache := new Type_Caches.Map;
      end if;

      return Result;
   end Logon;

   function Logon (Connect : String) return Connection is
      User_First     : Positive;
      User_Last      : Positive;
      Password_First : Positive;
      Password_Last  : Natural;
      Server_First   : Positive;
      Server_Last    : Natural;
   begin
      Split_Connect_String
        (Connect        => Connect,
         User_First     => User_First,
         User_Last      => User_Last,
         Password_First => Password_First,
         Password_Last  => Password_Last,
         Server_First   => Server_First,
         Server_Last    => Server_Last);

      return Logon
        (Server_Name => Connect (Server_First .. Server_Last),
         User        => Connect (User_First .. User_Last),
         Password    => Connect (Password_First .. Password_Last));
   end Logon;

   -----------
   -- Reset --
   -----------

   procedure Reset (It : in Connection) is
   begin
      Check_Error (OCIReset (Handle (It), Thread.Error));
   end Reset;

   --------------
   -- Rollback --
   --------------

   procedure Rollback (Connect : in Connection) is
   begin
      Check_Error
        (OCITransRollback (OCISvcCtx (Connect.Handle), Thread.Error));
   end Rollback;

   function Rollback (Connect : in Connection) return Boolean is
   begin
      return OCITransRollback (OCISvcCtx (Connect.Handle), Thread.Error)
             = OCI_SUCCESS;
   end Rollback;

   --------------------
   -- Server_Version --
   --------------------

   function Server_Version (Connect : in Connection) return String is
      Buff : aliased Text := (0 .. 511 => C.nul);
   begin
      Check_Error
        (OCIServerVersion
           (Hndlp    => Connect.Handle,
            Errhp    => Thread.Error,
            Bufp     => CStr.To_Chars_Ptr (Buff'Unchecked_Access),
            Bufsz    => Buff'Length - 1,
            Hndltype => OCI_HTYPE_SVCCTX));
      return C.To_Ada (Buff);
   end Server_Version;

   ------------------
   -- Set_Blocking --
   ------------------

   procedure Set_Blocking (Connect : in out Connection; Mode : in Boolean) is
      B1 : aliased constant Ub1 := Boolean'Pos (not Mode);
   begin
      if Is_Blocking (Connect) = Mode then
         return;
      end if;

      Set_Attr
        (Connect.Handle,
         OCI_HTYPE_SVCCTX,
         OCI_ATTR_NONBLOCKING_MODE,
         Value => B1'Address);

      if Is_Blocking (Connect) /= Mode then
         raise Program_Error with "Set blocking mode error.";
      end if;
   end Set_Blocking;

   --------------------------
   -- Split_Connect_String --
   --------------------------

   procedure Split_Connect_String
     (Connect        : in     String;
      User_First     :    out Positive;
      User_Last      :    out Positive;
      Password_First :    out Positive;
      Password_Last  :    out Natural;
      Server_First   :    out Positive;
      Server_Last    :    out Natural)
   is
      DP, DS : Natural;
      use Ada.Strings.Fixed, Ada.Strings;
   begin
      DP := Index (Connect, "/");
      DS := Index (Connect, "@", Backward);

      User_First := Connect'First;

      if DS > 0 and then DP > DS then
         Ada.Exceptions.Raise_Exception
           (Constraint_Error'Identity,
            "Wrong connect string format """ & Connect & '"');
      elsif DS = 0 and then DP = 0 then
         User_Last  := Connect'Last;

         Password_First := Connect'First;
         Password_Last  := Connect'First - 1;
         Server_First   := Connect'First;
         Server_Last    := Connect'First - 1;
      else
         if DP /= 0 then
            User_Last      := DP - 1;
            Password_First := DP + 1;

            if DS /= 0 then
               Password_Last := DS - 1;
               Server_First  := DS + 1;
               Server_Last   := Connect'Last;
            else
               Password_Last := Connect'Last;
               Server_First  := Connect'First;
               Server_Last   := Connect'First - 1;
            end if;

         else -- DP = 0 and DS /= 0
            User_Last := DS - 1;
            Password_First := Connect'First;
            Password_Last  := Connect'First - 1;
            Server_First   := DS + 1;
            Server_Last    := Connect'Last;
         end if;
      end if;
   end Split_Connect_String;

   ---------------
   -- User_Name --
   ---------------

   function User_Name (Connect : in Connection) return String is
   begin
      return Get_Attr
               (OCIHandle (Connect.Sessn),
                OCI_HTYPE_SESSION,
                OCI_ATTR_USERNAME);
   end User_Name;

end OCI.Thick.Connections;
