/* 
 * Copyright (C) 2004, 2005 Jean-Yves Lefort <jylefort@brutele.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include <translate.h>
#include "gt-util.h"
#include "gt-conf.h"

typedef struct
{
  GtkWidget	*dialog;
  
  GtkWidget	*autodetect_language_check;
  GtkWidget	*services_view;
  GtkWidget	*services_move_up;
  GtkWidget	*services_move_down;
} Preferences;

enum {
  COLUMN_SERVICE,
  COLUMN_ENABLED,
  COLUMN_NAME,
  COLUMN_NICK,
  N_COLUMNS
};

static Preferences preferences = { NULL };
static gboolean services_block_notify = FALSE;

static void gt_preferences_services_sync_from_conf (void);
static void gt_preferences_services_append (TranslateService *service,
					    gboolean enabled,
					    const char *selected_name);
static void gt_preferences_services_sync_to_conf (void);

static void gt_preferences_services_move (int direction);
static void gt_preferences_services_notify_cb (GConfClient *client,
					       unsigned int cnxn_id,
					       GConfEntry *entry,
					       gpointer user_data);
static void gt_preferences_services_toggled_h (GtkCellRendererToggle *renderer,
					       const char *path,
					       gpointer user_data);

static void gt_preferences_update_sensitivity (void);

void
gt_preferences_display (GtkWindow *parent)
{
  GtkListStore *store;
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  GtkTreeSelection *selection;

  if (preferences.dialog)
    {
      gtk_window_present(GTK_WINDOW(preferences.dialog));
      return;
    }

#define W(name) #name, &preferences.name
  gt_create_interface("preferences",
		      W(dialog),
		      W(autodetect_language_check),
		      W(services_view),
		      W(services_move_up),
		      W(services_move_down),
		      NULL);
#undef W

  eel_add_weak_pointer(&preferences.dialog);

  if (parent)
    gtk_window_set_transient_for(GTK_WINDOW(preferences.dialog), parent);

  store = gtk_list_store_new(N_COLUMNS,
			     TRANSLATE_TYPE_SERVICE,
			     G_TYPE_BOOLEAN,
			     G_TYPE_STRING,
			     G_TYPE_STRING);

  /* listen to rows drag and drop */
  g_signal_connect(store, "row-deleted", G_CALLBACK(gt_preferences_services_sync_to_conf), NULL);

  gtk_tree_view_set_model(GTK_TREE_VIEW(preferences.services_view), GTK_TREE_MODEL(store));
  g_object_unref(store);

  gtk_tree_view_set_search_column(GTK_TREE_VIEW(preferences.services_view), COLUMN_NICK);
  gtk_tree_view_set_reorderable(GTK_TREE_VIEW(preferences.services_view), TRUE);

  renderer = gtk_cell_renderer_toggle_new();
  g_signal_connect(renderer, "toggled", G_CALLBACK(gt_preferences_services_toggled_h), NULL);

  column = gtk_tree_view_column_new_with_attributes(NULL, renderer,
						    "active", COLUMN_ENABLED,
						    NULL);

  gtk_tree_view_append_column(GTK_TREE_VIEW(preferences.services_view), column);

  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(NULL, renderer,
						    "text", COLUMN_NICK,
						    NULL);

  gtk_tree_view_append_column(GTK_TREE_VIEW(preferences.services_view), column);

  gt_preferences_services_sync_from_conf();

  gt_conf_notification_add(preferences.dialog, GT_CONF_SERVICES, gt_preferences_services_notify_cb, NULL);

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(preferences.services_view));
  g_signal_connect(selection, "changed", G_CALLBACK(gt_preferences_update_sensitivity), NULL);

  gt_conf_link(preferences.dialog, GT_CONF_PREFERENCES,
	       preferences.autodetect_language_check, GT_CONF_AUTODETECT_LANGUAGE, "active",
	       NULL);

#ifndef WITH_LANGUAGE_DETECTION
  gtk_widget_set_sensitive(preferences.autodetect_language_check, FALSE);
#endif

  gtk_widget_show(preferences.dialog);
}

static void
gt_preferences_services_sync_from_conf (void)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  char *selected_name = NULL;
  GSList *conf_services;
  GSList *all_services;
  GSList *l;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(preferences.services_view));
  if (gtk_tree_selection_get_selected(selection, &model, &iter))
    gtk_tree_model_get(model, &iter, COLUMN_NAME, &selected_name, -1);
  
  g_signal_handlers_block_by_func(model, gt_preferences_services_sync_to_conf, NULL);
  gtk_list_store_clear(GTK_LIST_STORE(model));
  g_signal_handlers_unblock_by_func(model, gt_preferences_services_sync_to_conf, NULL);

  conf_services = gt_conf_get_services();
  GT_LIST_FOREACH(l, conf_services)
    {
      const GTConfService *conf_service = l->data;
      gt_preferences_services_append(conf_service->service,
				     conf_service->enabled,
				     selected_name);
    }

  all_services = translate_get_services();
  GT_LIST_FOREACH(l, all_services)
    {
      TranslateService *service = l->data;

      if (! gt_conf_services_get_from_name(conf_services, translate_service_get_name(service)))
	gt_preferences_services_append(service,
				       TRUE,
				       selected_name);
    }

  g_free(selected_name);

  gt_conf_services_free(conf_services);
  gt_g_object_slist_free(all_services);

  gt_preferences_update_sensitivity();
}

static void
gt_preferences_services_append (TranslateService *service,
				gboolean enabled,
				const char *selected_name)
{
  GtkTreeModel *model;
  GtkTreeIter iter;

  g_return_if_fail(TRANSLATE_IS_SERVICE(service));

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(preferences.services_view));
  
  gtk_list_store_append(GTK_LIST_STORE(model), &iter);
  gtk_list_store_set(GTK_LIST_STORE(model), &iter,
		     COLUMN_SERVICE, service,
		     COLUMN_ENABLED, enabled,
		     COLUMN_NAME, translate_service_get_name(service),
		     COLUMN_NICK, translate_service_get_nick(service),
		     -1);

  if (selected_name && ! strcmp(selected_name, translate_service_get_name(service)))
    {
      GtkTreeSelection *selection;

      selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(preferences.services_view));
      gtk_tree_selection_select_iter(selection, &iter);
    }
}

static void
gt_preferences_services_sync_to_conf (void)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  gboolean valid;
  GSList *conf_services = NULL;

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(preferences.services_view));

  valid = gtk_tree_model_get_iter_first(model, &iter);
  while (valid)
    {
      TranslateService *service;
      gboolean enabled;

      gtk_tree_model_get(model, &iter,
			 COLUMN_SERVICE, &service,
			 COLUMN_ENABLED, &enabled,
			 -1);

      conf_services = g_slist_append(conf_services, gt_conf_service_new(service, enabled));
      g_object_unref(service);

      valid = gtk_tree_model_iter_next(model, &iter);
    }

  services_block_notify = TRUE;
  gt_conf_set_services(conf_services);
  gt_conf_services_free(conf_services);
}

static void
gt_preferences_services_move (int direction)
{
  GtkTreeSelection *selection;
  gboolean status;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreePath *path;
  GtkTreeIter other_iter;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(preferences.services_view));

  status = gtk_tree_selection_get_selected(selection, &model, &iter);
  g_return_if_fail(status == TRUE);

  path = gtk_tree_model_get_path(model, &iter);
  
  if (direction == -1)
    gtk_tree_path_prev(path);
  else if (direction == +1)
    gtk_tree_path_next(path);
  else
    g_return_if_reached();

  status = gtk_tree_model_get_iter(model, &other_iter, path);
  g_return_if_fail(status == TRUE);

  gtk_list_store_swap(GTK_LIST_STORE(model), &iter, &other_iter);
  gtk_tree_path_free(path);

  gt_preferences_services_sync_to_conf();
  gt_preferences_update_sensitivity();
}

static void
gt_preferences_services_notify_cb (GConfClient *client,
				   unsigned int cnxn_id,
				   GConfEntry *entry,
				   gpointer user_data)
{
  GDK_THREADS_ENTER();
  if (services_block_notify)
    services_block_notify = FALSE;
  else
    gt_preferences_services_sync_from_conf();
  GDK_THREADS_LEAVE();
}

static void
gt_preferences_services_toggled_h (GtkCellRendererToggle *renderer,
				   const char *path,
				   gpointer user_data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  gboolean status;
  gboolean enabled;

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(preferences.services_view));

  status = gtk_tree_model_get_iter_from_string(model, &iter, path);
  g_return_if_fail(status == TRUE);

  gtk_tree_model_get(model, &iter, COLUMN_ENABLED, &enabled, -1);
  gtk_list_store_set(GTK_LIST_STORE(model), &iter, COLUMN_ENABLED, ! enabled, -1);

  gt_preferences_services_sync_to_conf();
}

static void
gt_preferences_update_sensitivity (void)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gboolean has_previous = FALSE;
  gboolean has_next = FALSE;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(preferences.services_view));
  if (gtk_tree_selection_get_selected(selection, &model, &iter))
    {
      GtkTreePath *path;

      path = gtk_tree_model_get_path(model, &iter);

      has_previous = gtk_tree_path_prev(path);
      has_next = gtk_tree_model_iter_next(model, &iter);

      gtk_tree_path_free(path);
    }
  
  gtk_widget_set_sensitive(preferences.services_move_up, has_previous);
  gtk_widget_set_sensitive(preferences.services_move_down, has_next);
}

/* libglade callbacks */

void
gt_preferences_services_move_up_clicked_h (GtkButton *button,
					   gpointer user_data)
{
  gt_preferences_services_move(-1);
}

void
gt_preferences_services_move_down_clicked_h (GtkButton *button,
					     gpointer user_data)
{
  gt_preferences_services_move(+1);
}

void
gt_preferences_services_use_default_clicked_h (GtkButton *button,
					       gpointer user_data)
{
  gt_conf_set_services(NULL);
}

void
gt_preferences_response_h (GtkDialog *dialog, int response, gpointer user_data)
{
  switch (response)
    {
    case GTK_RESPONSE_HELP:
      gt_display_help(GTK_WINDOW(preferences.dialog), "preferences");
      break;

    case GTK_RESPONSE_CLOSE:
      gtk_widget_destroy(preferences.dialog);
      break;
    }
}
