// Copyright ©2016 The Gonum Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package quad

import (
	"fmt"
	"math"
	"testing"

	"gonum.org/v1/gonum/floats"
	"gonum.org/v1/gonum/floats/scalar"
)

func TestHermite(t *testing.T) {
	t.Parallel()
	for cas, test := range []struct {
		n   int
		f   func(float64) float64
		ev  float64
		tol float64
	}{
		{
			n:   2,
			f:   func(x float64) float64 { return 1 },
			ev:  math.SqrtPi,
			tol: 1e-14,
		},
		{
			n:   4,
			f:   func(x float64) float64 { return math.Exp(-2 * 0.5 * x) },
			ev:  math.SqrtPi * math.Exp(0.25),
			tol: 1e-4,
		},
		{
			n:   9,
			f:   func(x float64) float64 { return math.Exp(-2 * 0.5 * x) },
			ev:  math.SqrtPi * math.Exp(0.25),
			tol: 1e-12,
		},
		{
			n:   10,
			f:   func(x float64) float64 { return math.Exp(-2 * 0.5 * x) },
			ev:  math.SqrtPi * math.Exp(0.25),
			tol: 1e-12,
		},
		{
			n:   15,
			f:   func(x float64) float64 { return math.Exp(-2 * 2 * x) },
			ev:  math.SqrtPi * math.Exp(4),
			tol: 1e-4,
		},
		{
			n:   22,
			f:   func(x float64) float64 { return math.Exp(-2 * 2 * x) },
			ev:  math.SqrtPi * math.Exp(4),
			tol: 1e-12,
		},
		{
			n:   21,
			f:   func(x float64) float64 { return math.Exp(-2 * 2 * x) },
			ev:  math.SqrtPi * math.Exp(4),
			tol: 1e-12,
		},
	} {
		ev := Fixed(test.f, math.Inf(-1), math.Inf(1), test.n, Hermite{}, 0)
		if !scalar.EqualWithinAbsOrRel(test.ev, ev, test.tol, test.tol) {
			t.Errorf("Case %d: expected value mismatch.\nWant %v\ngot %v", cas, test.ev, ev)
		}
	}
}

func TestHermiteLocations(t *testing.T) {
	t.Parallel()
	// Test the hermite locations and weights against other precomputed table.
	// Data from http://dlmf.nist.gov/3.5.
	for _, test := range []struct {
		n int
		x []float64
		w []float64
	}{
		// Evaluated from Chebfun.
		{
			n: 5,
			x: []float64{-0.202018287045609e1, -0.958572464613819, 0, 0.958572464613819, 0.202018287045609e1},
			w: []float64{0.199532420590459e-1, 0.393619323152241, 0.945308720482942, 0.393619323152241, 0.199532420590459e-1},
		},
		{
			n: 10,
			x: []float64{-0.343615911883773760e1, -0.253273167423278980e1, -0.175668364929988177e1, -0.103661082978951365e1, -0.342901327223704609, 0.342901327223704609, 0.103661082978951365e1, 0.175668364929988177e1, 0.253273167423278980e1, 0.343615911883773760e1},
			w: []float64{0.764043285523262063e-5, 0.134364574678123269e-2, 0.338743944554810631e-1, 0.240138611082314686, 0.610862633735325799, 0.610862633735325799, 0.240138611082314686, 0.338743944554810631e-1, 0.134364574678123269e-2, 0.764043285523262063e-5},
		},
		{
			n: 215,
			x: []float64{-2.0083110126223993e+01, -1.9573161931385755e+01, -1.9153731166695263e+01, -1.8781558404513735e+01, -1.8440108632764016e+01, -1.8120857960601569e+01, -1.7818692349773503e+01, -1.7530242081974308e+01, -1.7253139752773148e+01, -1.6985642449377284e+01, -1.6726420346886542e+01, -1.6474429782282726e+01, -1.6228832750513551e+01, -1.5988943547145137e+01, -1.5754192102761674e+01, -1.5524098014731747e+01, -1.5298251677453143e+01, -1.5076300264808333e+01, -1.4857937115425768e+01, -1.4642893558208563e+01, -1.4430932522616979e+01, -1.4221843477251845e+01, -1.4015438372554696e+01, -1.3811548353270942e+01, -1.3610021068540190e+01, -1.3410718451340038e+01, -1.3213514870431910e+01, -1.3018295580800100e+01, -1.2824955415404188e+01, -1.2633397673617059e+01, -1.2443533171189967e+01, -1.2255279423804318e+01, -1.2068559941826503e+01, -1.1883303618198463e+01, -1.1699444194777641e+01, -1.1516919795109878e+01, -1.1335672513742256e+01, -1.1155648053883823e+01, -1.0976795406593546e+01, -1.0799066565787168e+01, -1.0622416274262394e+01, -1.0446801796686490e+01, -1.0272182716104577e+01, -1.0098520751035824e+01, -9.9257795906487338e+00, -9.7539247458611111e+00, -9.5829234145081710e+00, -9.4127443589734394e+00, -9.2433577948895387e+00, -9.0747352896969389e+00, -8.9068496700026394e+00, -8.7396749368126798e+00, -8.5731861878255415e+00, -8.4073595460710955e+00, -8.2421720942637187e+00, -8.0776018143113824e+00, -7.9136275314855800e+00, -7.7502288628124383e+00, -7.5873861692931381e+00, -7.4250805116042331e+00, -7.2632936089653120e+00, -7.1020078008938672e+00, -6.9412060115960434e+00, -6.7808717167671544e+00, -6.6209889125982082e+00, -6.4615420868046280e+00, -6.3025161915109100e+00, -6.1438966178406549e+00, -5.9856691720754620e+00, -5.8278200532585673e+00, -5.6703358321303936e+00, -5.5132034312931157e+00, -5.3564101065103049e+00, -5.1999434290559226e+00, -5.0437912690340267e+00, -4.8879417795972326e+00, -4.7323833819978187e+00, -4.5771047514106780e+00, -4.4220948034722349e+00, -4.2673426814837114e+00, -4.1128377442311503e+00, -3.9585695543782120e+00, -3.8045278673909002e+00, -3.6507026209565665e+00, -3.4970839248619634e+00, -3.3436620512978501e+00, -3.1904274255596032e+00, -3.0373706171156218e+00, -2.8844823310168755e+00, -2.7317533996228827e+00, -2.5791747746208049e+00, -2.4267375193157439e+00, -2.2744328011717010e+00, -2.1222518845837395e+00, -1.9701861238629792e+00, -1.8182269564169717e+00, -1.6663658961089640e+00, -1.5145945267802365e+00, -1.3629044959205205e+00, -1.2112875084719879e+00, -1.0597353207530273e+00, -9.0823973448851325e-01, -7.5679259093351370e-01, -6.0538576507805808e-01, -4.5401115992078167e-01, -3.0266070079942864e-01, -1.5132632976672361e-01, 7.9483151672010196e-15, 1.5132632976672361e-01, 3.0266070079942864e-01, 4.5401115992078167e-01, 6.0538576507805808e-01, 7.5679259093351370e-01, 9.0823973448851325e-01, 1.0597353207530273e+00, 1.2112875084719879e+00, 1.3629044959205205e+00, 1.5145945267802365e+00, 1.6663658961089640e+00, 1.8182269564169717e+00, 1.9701861238629792e+00, 2.1222518845837395e+00, 2.2744328011717010e+00, 2.4267375193157439e+00, 2.5791747746208049e+00, 2.7317533996228827e+00, 2.8844823310168755e+00, 3.0373706171156218e+00, 3.1904274255596032e+00, 3.3436620512978501e+00, 3.4970839248619634e+00, 3.6507026209565665e+00, 3.8045278673909002e+00, 3.9585695543782120e+00, 4.1128377442311503e+00, 4.2673426814837114e+00, 4.4220948034722349e+00, 4.5771047514106780e+00, 4.7323833819978187e+00, 4.8879417795972326e+00, 5.0437912690340267e+00, 5.1999434290559226e+00, 5.3564101065103049e+00, 5.5132034312931157e+00, 5.6703358321303936e+00, 5.8278200532585673e+00, 5.9856691720754620e+00, 6.1438966178406549e+00, 6.3025161915109100e+00, 6.4615420868046280e+00, 6.6209889125982082e+00, 6.7808717167671544e+00, 6.9412060115960434e+00, 7.1020078008938672e+00, 7.2632936089653120e+00, 7.4250805116042331e+00, 7.5873861692931381e+00, 7.7502288628124383e+00, 7.9136275314855800e+00, 8.0776018143113824e+00, 8.2421720942637187e+00, 8.4073595460710955e+00, 8.5731861878255415e+00, 8.7396749368126798e+00, 8.9068496700026394e+00, 9.0747352896969389e+00, 9.2433577948895387e+00, 9.4127443589734394e+00, 9.5829234145081710e+00, 9.7539247458611111e+00, 9.9257795906487338e+00, 1.0098520751035824e+01, 1.0272182716104577e+01, 1.0446801796686490e+01, 1.0622416274262394e+01, 1.0799066565787168e+01, 1.0976795406593546e+01, 1.1155648053883823e+01, 1.1335672513742256e+01, 1.1516919795109878e+01, 1.1699444194777641e+01, 1.1883303618198463e+01, 1.2068559941826503e+01, 1.2255279423804318e+01, 1.2443533171189967e+01, 1.2633397673617059e+01, 1.2824955415404188e+01, 1.3018295580800100e+01, 1.3213514870431910e+01, 1.3410718451340038e+01, 1.3610021068540190e+01, 1.3811548353270942e+01, 1.4015438372554696e+01, 1.4221843477251845e+01, 1.4430932522616979e+01, 1.4642893558208563e+01, 1.4857937115425768e+01, 1.5076300264808333e+01, 1.5298251677453143e+01, 1.5524098014731747e+01, 1.5754192102761674e+01, 1.5988943547145137e+01, 1.6228832750513551e+01, 1.6474429782282726e+01, 1.6726420346886542e+01, 1.6985642449377284e+01, 1.7253139752773148e+01, 1.7530242081974308e+01, 1.7818692349773503e+01, 1.8120857960601569e+01, 1.8440108632764016e+01, 1.8781558404513735e+01, 1.9153731166695263e+01, 1.9573161931385755e+01, 2.0083110126223993e+01},
			w: []float64{4.0474005560499094e-176, 1.8797130400856443e-167, 1.8432377040276421e-160, 2.2601442864626069e-154, 6.9363023753259492e-149, 7.6575370428605554e-144, 3.7894855074137878e-139, 9.7310398950542353e-135, 1.4382941114934084e-130, 1.3214094652141891e-126, 8.0053233705919762e-123, 3.3506484324551954e-119, 1.0061133084529732e-115, 2.2355880918528976e-112, 3.7723274691873339e-109, 4.9410774014014167e-106, 5.1189805080467357e-103, 4.2633269887655389e-100, 2.8951564896303124e-97, 1.6231768954981774e-94, 7.5966487579367170e-92, 2.9971587600601123e-89, 1.0056653752640396e-86, 2.8926565116618279e-84, 7.1838069610206879e-82, 1.5504440487214177e-79, 2.9253709118548054e-77, 4.8516529695481360e-75, 7.1080340648382430e-73, 9.2417798747589595e-71, 1.0709039437896534e-68, 1.1102986726619406e-66, 1.0337280935884043e-64, 8.6720569345151135e-63, 6.5759615285599325e-61, 4.5206307708839604e-59, 2.8251413443094276e-57, 1.6091856250751163e-55, 8.3743579179261028e-54, 3.9908716881614503e-52, 1.7453732998154445e-50, 7.0192904730947679e-49, 2.6008368230936308e-47, 8.8947358191690260e-46, 2.8125213532676820e-44, 8.2357837374304705e-43, 2.2368010157343200e-41, 5.6428177886700861e-40, 1.3240736407971940e-38, 2.8936578029605094e-37, 5.8971855560501346e-36, 1.1220768467833071e-34, 1.9956009474900652e-33, 3.3209946083724174e-32, 5.1767166839129889e-31, 7.5658951790722304e-30, 1.0377543674300410e-28, 1.3370496163293642e-27, 1.6195422525877890e-26, 1.8458039741402227e-25, 1.9809275961773415e-24, 2.0034006172471768e-23, 1.9107136033126365e-22, 1.7196934641222891e-21, 1.4615728740382834e-20, 1.1737567188678477e-19, 8.9122322093114715e-19, 6.4017277272147784e-18, 4.3526236743179514e-17, 2.8027154496363443e-16, 1.7100226833829111e-15, 9.8907988394494626e-15, 5.4258798698008971e-14, 2.8243097014577004e-13, 1.3955439320917938e-12, 6.5484943121277863e-12, 2.9192823809511808e-11, 1.2368298646403665e-10, 4.9819378483709125e-10, 1.9084879833147422e-09, 6.9554442384546823e-09, 2.4123461087320678e-08, 7.9645691304111948e-08, 2.5038904484412268e-07, 7.4974774628062337e-07, 2.1388110362272039e-06, 5.8142267442915587e-06, 1.5065139508846480e-05, 3.7214305501073257e-05, 8.7657861198914500e-05, 1.9692436368292282e-04, 4.2200152760283493e-04, 8.6279569789330843e-04, 1.6832545363270029e-03, 3.1340302956109622e-03, 5.5696401904272858e-03, 9.4487851093380676e-03, 1.5303805411902717e-02, 2.3666975130344652e-02, 3.4949960453868167e-02, 4.9288724246681882e-02, 6.6386134961517171e-02, 8.5401153510804823e-02, 1.0493731687783339e-01, 1.2316720474285285e-01, 1.3809371553328023e-01, 1.4790300421921079e-01, 1.5132498974070285e-01, 1.4790300421921079e-01, 1.3809371553328023e-01, 1.2316720474285285e-01, 1.0493731687783339e-01, 8.5401153510804823e-02, 6.6386134961517171e-02, 4.9288724246681882e-02, 3.4949960453868167e-02, 2.3666975130344652e-02, 1.5303805411902717e-02, 9.4487851093380676e-03, 5.5696401904272858e-03, 3.1340302956109622e-03, 1.6832545363270029e-03, 8.6279569789330843e-04, 4.2200152760283493e-04, 1.9692436368292282e-04, 8.7657861198914500e-05, 3.7214305501073257e-05, 1.5065139508846480e-05, 5.8142267442915587e-06, 2.1388110362272039e-06, 7.4974774628062337e-07, 2.5038904484412268e-07, 7.9645691304111948e-08, 2.4123461087320678e-08, 6.9554442384546823e-09, 1.9084879833147422e-09, 4.9819378483709125e-10, 1.2368298646403665e-10, 2.9192823809511808e-11, 6.5484943121277863e-12, 1.3955439320917938e-12, 2.8243097014577004e-13, 5.4258798698008971e-14, 9.8907988394494626e-15, 1.7100226833829111e-15, 2.8027154496363443e-16, 4.3526236743179514e-17, 6.4017277272147784e-18, 8.9122322093114715e-19, 1.1737567188678477e-19, 1.4615728740382834e-20, 1.7196934641222891e-21, 1.9107136033126365e-22, 2.0034006172471768e-23, 1.9809275961773415e-24, 1.8458039741402227e-25, 1.6195422525877890e-26, 1.3370496163293642e-27, 1.0377543674300410e-28, 7.5658951790722304e-30, 5.1767166839129889e-31, 3.3209946083724174e-32, 1.9956009474900652e-33, 1.1220768467833071e-34, 5.8971855560501346e-36, 2.8936578029605094e-37, 1.3240736407971940e-38, 5.6428177886700861e-40, 2.2368010157343200e-41, 8.2357837374304705e-43, 2.8125213532676820e-44, 8.8947358191690260e-46, 2.6008368230936308e-47, 7.0192904730947679e-49, 1.7453732998154445e-50, 3.9908716881614503e-52, 8.3743579179261028e-54, 1.6091856250751163e-55, 2.8251413443094276e-57, 4.5206307708839604e-59, 6.5759615285599325e-61, 8.6720569345151135e-63, 1.0337280935884043e-64, 1.1102986726619406e-66, 1.0709039437896534e-68, 9.2417798747589595e-71, 7.1080340648382430e-73, 4.8516529695481360e-75, 2.9253709118548054e-77, 1.5504440487214177e-79, 7.1838069610206879e-82, 2.8926565116618279e-84, 1.0056653752640396e-86, 2.9971587600601123e-89, 7.5966487579367170e-92, 1.6231768954981774e-94, 2.8951564896303124e-97, 4.2633269887655389e-100, 5.1189805080467357e-103, 4.9410774014014167e-106, 3.7723274691873339e-109, 2.2355880918528976e-112, 1.0061133084529732e-115, 3.3506484324551954e-119, 8.0053233705919762e-123, 1.3214094652141891e-126, 1.4382941114934084e-130, 9.7310398950542353e-135, 3.7894855074137878e-139, 7.6575370428605554e-144, 6.9363023753259492e-149, 2.2601442864626069e-154, 1.8432377040276421e-160, 1.8797130400856443e-167, 4.0474005560499094e-176},
		},
		{
			n: 216,
			x: []float64{-2.0131752068397390e+01, -1.9622211733828486e+01, -1.9203122650098184e+01, -1.8831257901839155e+01, -1.8490094811217439e+01, -1.8171115805943941e+01, -1.7869210618319627e+01, -1.7581012003064874e+01, -1.7304154286393636e+01, -1.7036895825521608e+01, -1.6777907762072694e+01, -1.6526147190139376e+01, -1.6280776711860355e+01, -1.6041111119395726e+01, -1.5806580756275796e+01, -1.5576705568204593e+01, -1.5351076247103551e+01, -1.5129340223792830e+01, -1.4911191060978103e+01, -1.4696360284715409e+01, -1.4484610999323769e+01, -1.4275732829628778e+01, -1.4069537866591046e+01, -1.3865857382137794e+01, -1.3664539141187824e+01, -1.3465445182689793e+01, -1.3268449972893279e+01, -1.3073438856897617e+01, -1.2880306751340244e+01, -1.2688957033629340e+01, -1.2499300592587632e+01, -1.2311255012587219e+01, -1.2124743868808109e+01, -1.1939696115566026e+01, -1.1756045553033841e+01, -1.1573730360348824e+01, -1.1392692685219812e+01, -1.1212878281848150e+01, -1.1034236190346626e+01, -1.0856718451952171e+01, -1.0680279855235202e+01, -1.0504877709252620e+01, -1.0330471640205081e+01, -1.0157023408668081e+01, -9.9844967448895385e+00, -9.8128572000013108e+00, -9.6420720112891818e+00, -9.4721099799172528e+00, -9.3029413597147883e+00, -9.1345377558145096e+00, -8.9668720320849413e+00, -8.7999182264315081e+00, -8.6336514731539875e+00, -8.4680479316454296e+00, -8.3030847208017082e+00, -8.1387398585838486e+00, -7.9749922062384426e+00, -7.8118214167366684e+00, -7.6492078870405447e+00, -7.4871327138470951e+00, -7.3255776524981444e+00, -7.1645250787759087e+00, -7.0039579533331269e+00, -6.8438597885318986e+00, -6.6842146174875827e+00, -6.5250069651340752e+00, -6.3662218211444577e+00, -6.2078446145563948e+00, -6.0498611899661094e+00, -5.8922577851667635e+00, -5.7350210101185848e+00, -5.5781378271479722e+00, -5.4215955322816969e+00, -5.2653817376305465e+00, -5.1094843547439881e+00, -4.9538915788638089e+00, -4.7985918740108131e+00, -4.6435739588438905e+00, -4.4888267932355790e+00, -4.3343395655126109e+00, -4.1801016803139959e+00, -4.0261027470226392e+00, -3.8723325687298416e+00, -3.7187811316950210e+00, -3.5654385952656149e+00, -3.4122952822245907e+00, -3.2593416695353823e+00, -3.1065683794558021e+00, -2.9539661709946619e+00, -2.8015259316863266e+00, -2.6492386696600243e+00, -2.4970955059821409e+00, -2.3450876672510903e+00, -2.1932064784252856e+00, -2.0414433558661624e+00, -1.8897898005787916e+00, -1.7382373916337388e+00, -1.5867777797546210e+00, -1.4354026810562848e+00, -1.2841038709195900e+00, -1.1328731779888348e+00, -9.8170247827895785e-01, -8.3058368937955573e-01, -6.7950876474356448e-01, -5.2846968804856509e-01, -3.7745846761911317e-01, -2.2646713089859094e-01, -7.5487718959414210e-02, 7.5487718959414210e-02, 2.2646713089859094e-01, 3.7745846761911317e-01, 5.2846968804856509e-01, 6.7950876474356448e-01, 8.3058368937955573e-01, 9.8170247827895785e-01, 1.1328731779888348e+00, 1.2841038709195900e+00, 1.4354026810562848e+00, 1.5867777797546210e+00, 1.7382373916337388e+00, 1.8897898005787916e+00, 2.0414433558661624e+00, 2.1932064784252856e+00, 2.3450876672510903e+00, 2.4970955059821409e+00, 2.6492386696600243e+00, 2.8015259316863266e+00, 2.9539661709946619e+00, 3.1065683794558021e+00, 3.2593416695353823e+00, 3.4122952822245907e+00, 3.5654385952656149e+00, 3.7187811316950210e+00, 3.8723325687298416e+00, 4.0261027470226392e+00, 4.1801016803139959e+00, 4.3343395655126109e+00, 4.4888267932355790e+00, 4.6435739588438905e+00, 4.7985918740108131e+00, 4.9538915788638089e+00, 5.1094843547439881e+00, 5.2653817376305465e+00, 5.4215955322816969e+00, 5.5781378271479722e+00, 5.7350210101185848e+00, 5.8922577851667635e+00, 6.0498611899661094e+00, 6.2078446145563948e+00, 6.3662218211444577e+00, 6.5250069651340752e+00, 6.6842146174875827e+00, 6.8438597885318986e+00, 7.0039579533331269e+00, 7.1645250787759087e+00, 7.3255776524981444e+00, 7.4871327138470951e+00, 7.6492078870405447e+00, 7.8118214167366684e+00, 7.9749922062384426e+00, 8.1387398585838486e+00, 8.3030847208017082e+00, 8.4680479316454296e+00, 8.6336514731539875e+00, 8.7999182264315081e+00, 8.9668720320849413e+00, 9.1345377558145096e+00, 9.3029413597147883e+00, 9.4721099799172528e+00, 9.6420720112891818e+00, 9.8128572000013108e+00, 9.9844967448895385e+00, 1.0157023408668081e+01, 1.0330471640205081e+01, 1.0504877709252620e+01, 1.0680279855235202e+01, 1.0856718451952171e+01, 1.1034236190346626e+01, 1.1212878281848150e+01, 1.1392692685219812e+01, 1.1573730360348824e+01, 1.1756045553033841e+01, 1.1939696115566026e+01, 1.2124743868808109e+01, 1.2311255012587219e+01, 1.2499300592587632e+01, 1.2688957033629340e+01, 1.2880306751340244e+01, 1.3073438856897617e+01, 1.3268449972893279e+01, 1.3465445182689793e+01, 1.3664539141187824e+01, 1.3865857382137794e+01, 1.4069537866591046e+01, 1.4275732829628778e+01, 1.4484610999323769e+01, 1.4696360284715409e+01, 1.4911191060978103e+01, 1.5129340223792830e+01, 1.5351076247103551e+01, 1.5576705568204593e+01, 1.5806580756275796e+01, 1.6041111119395726e+01, 1.6280776711860355e+01, 1.6526147190139376e+01, 1.6777907762072694e+01, 1.7036895825521608e+01, 1.7304154286393636e+01, 1.7581012003064874e+01, 1.7869210618319627e+01, 1.8171115805943941e+01, 1.8490094811217439e+01, 1.8831257901839155e+01, 1.9203122650098184e+01, 1.9622211733828486e+01, 2.0131752068397390e+01},
			w: []float64{5.7186844211118222e-177, 2.7466189949527236e-168, 2.7698274929964049e-161, 3.4828172461418377e-155, 1.0940903755394791e-149, 1.2347606047124676e-144, 6.2404773612052171e-140, 1.6353340891378212e-135, 2.4650950171742754e-131, 2.3085411036710099e-127, 1.4249650100034198e-123, 6.0746030482858092e-120, 1.8572077169453742e-116, 4.2005766103289114e-113, 7.2131362796894811e-110, 9.6125997143450870e-107, 1.0130358260780602e-103, 8.5810245129266594e-101, 5.9257988910096090e-98, 3.3780606107607827e-95, 1.6073077596553874e-92, 6.4463886647737192e-90, 2.1986159587286647e-87, 6.4275556938243753e-85, 1.6222763835321407e-82, 3.5581101139323407e-80, 6.8220148646876484e-78, 1.1496573365594917e-75, 1.7114200608857651e-73, 2.2608688324572314e-71, 2.6617645198762597e-69, 2.8038024736345674e-67, 2.6521234350652613e-65, 2.2603869398967457e-63, 1.7413601437152754e-61, 1.2161701083247541e-59, 7.7214787759599970e-58, 4.4681959655564470e-56, 2.3623580471674404e-54, 1.1437576644069702e-52, 5.0819691921164956e-51, 2.0764486091263794e-49, 7.8168818968667932e-48, 2.7161609526051418e-46, 8.7263192332605962e-45, 2.5963610791198993e-43, 7.1651564650842932e-42, 1.8367370768736730e-40, 4.3795689659428445e-39, 9.7263842188046952e-38, 2.0144288376257676e-36, 3.8953956971431042e-35, 7.0411654624671892e-34, 1.1909731529300876e-32, 1.8870059773886237e-31, 2.8034152482012231e-30, 3.9088957466494064e-29, 5.1199371899987404e-28, 6.3051173258996659e-27, 7.3062988441733997e-26, 7.9729602997232079e-25, 8.1994951837475763e-24, 7.9526775811935095e-23, 7.2794477168595113e-22, 6.2925924422563063e-21, 5.1402243977739270e-20, 3.9702645520078747e-19, 2.9013105636885269e-18, 2.0070043240260125e-17, 1.3149628048888219e-16, 8.1641639956286359e-16, 4.8056898389707300e-15, 2.6831763736632487e-14, 1.4216321504480736e-13, 7.1508314522575616e-13, 3.4161417919950439e-12, 1.5505876032631646e-11, 6.6896188793649314e-11, 2.7441476023994389e-10, 1.0706905991113023e-09, 3.9747846429602356e-09, 1.4044052119190546e-08, 4.7242215798796869e-08, 1.5133894631903387e-07, 4.6181858824280925e-07, 1.3427778873836719e-06, 3.7209613424878082e-06, 9.8293533546629111e-06, 2.4757612288389546e-05, 5.9469805842774389e-05, 1.3626145195379436e-04, 2.9786395909886146e-04, 6.2130683688573309e-04, 1.2368227283931328e-03, 2.3501040099770882e-03, 4.2629032262842719e-03, 7.3827553287034008e-03, 1.2208924001688722e-02, 1.9280976823883154e-02, 2.9081432020399377e-02, 4.1896299298471870e-02, 5.7655710815363609e-02, 7.5795804675644604e-02, 9.5193937891141694e-02, 1.1422353375366404e-01, 1.3094856577352781e-01, 1.4343569727181574e-01, 1.5011822522474477e-01, 1.5011822522474477e-01, 1.4343569727181574e-01, 1.3094856577352781e-01, 1.1422353375366404e-01, 9.5193937891141694e-02, 7.5795804675644604e-02, 5.7655710815363609e-02, 4.1896299298471870e-02, 2.9081432020399377e-02, 1.9280976823883154e-02, 1.2208924001688722e-02, 7.3827553287034008e-03, 4.2629032262842719e-03, 2.3501040099770882e-03, 1.2368227283931328e-03, 6.2130683688573309e-04, 2.9786395909886146e-04, 1.3626145195379436e-04, 5.9469805842774389e-05, 2.4757612288389546e-05, 9.8293533546629111e-06, 3.7209613424878082e-06, 1.3427778873836719e-06, 4.6181858824280925e-07, 1.5133894631903387e-07, 4.7242215798796869e-08, 1.4044052119190546e-08, 3.9747846429602356e-09, 1.0706905991113023e-09, 2.7441476023994389e-10, 6.6896188793649314e-11, 1.5505876032631646e-11, 3.4161417919950439e-12, 7.1508314522575616e-13, 1.4216321504480736e-13, 2.6831763736632487e-14, 4.8056898389707300e-15, 8.1641639956286359e-16, 1.3149628048888219e-16, 2.0070043240260125e-17, 2.9013105636885269e-18, 3.9702645520078747e-19, 5.1402243977739270e-20, 6.2925924422563063e-21, 7.2794477168595113e-22, 7.9526775811935095e-23, 8.1994951837475763e-24, 7.9729602997232079e-25, 7.3062988441733997e-26, 6.3051173258996659e-27, 5.1199371899987404e-28, 3.9088957466494064e-29, 2.8034152482012231e-30, 1.8870059773886237e-31, 1.1909731529300876e-32, 7.0411654624671892e-34, 3.8953956971431042e-35, 2.0144288376257676e-36, 9.7263842188046952e-38, 4.3795689659428445e-39, 1.8367370768736730e-40, 7.1651564650842932e-42, 2.5963610791198993e-43, 8.7263192332605962e-45, 2.7161609526051418e-46, 7.8168818968667932e-48, 2.0764486091263794e-49, 5.0819691921164956e-51, 1.1437576644069702e-52, 2.3623580471674404e-54, 4.4681959655564470e-56, 7.7214787759599970e-58, 1.2161701083247541e-59, 1.7413601437152754e-61, 2.2603869398967457e-63, 2.6521234350652613e-65, 2.8038024736345674e-67, 2.6617645198762597e-69, 2.2608688324572314e-71, 1.7114200608857651e-73, 1.1496573365594917e-75, 6.8220148646876484e-78, 3.5581101139323407e-80, 1.6222763835321407e-82, 6.4275556938243753e-85, 2.1986159587286647e-87, 6.4463886647737192e-90, 1.6073077596553874e-92, 3.3780606107607827e-95, 5.9257988910096090e-98, 8.5810245129266594e-101, 1.0130358260780602e-103, 9.6125997143450870e-107, 7.2131362796894811e-110, 4.2005766103289114e-113, 1.8572077169453742e-116, 6.0746030482858092e-120, 1.4249650100034198e-123, 2.3085411036710099e-127, 2.4650950171742754e-131, 1.6353340891378212e-135, 6.2404773612052171e-140, 1.2347606047124676e-144, 1.0940903755394791e-149, 3.4828172461418377e-155, 2.7698274929964049e-161, 2.7466189949527236e-168, 5.7186844211118222e-177},
		},
		{
			n: 217,
			x: []float64{-2.0180280152078982e+01, -1.9671145428267970e+01, -1.9252396116029669e+01, -1.8880837640701767e+01, -1.8539959593913657e+01, -1.8221250688860252e+01, -1.7919604408706316e+01, -1.7631655968298805e+01, -1.7355041415469405e+01, -1.7088020370809854e+01, -1.6829264937408070e+01, -1.6577732962488266e+01, -1.6332587652172350e+01, -1.6093144292573264e+01, -1.5858833637952957e+01, -1.5629175980464554e+01, -1.5403762307923412e+01, -1.5182240306661464e+01, -1.4964303762199654e+01, -1.4749684396618285e+01, -1.4538145488075983e+01, -1.4329476816694543e+01, -1.4123490613100632e+01, -1.3920018275613611e+01, -1.3718907684195138e+01, -1.3520020983073742e+01, -1.3323232735334253e+01, -1.3128428375570619e+01, -1.2935502903505391e+01, -1.2744359774012816e+01, -1.2554909948437734e+01, -1.2367071079310476e+01, -1.2180766806106346e+01, -1.1995926144008331e+01, -1.1812482951007951e+01, -1.1630375461345015e+01, -1.1449545875407471e+01, -1.1269939997911081e+01, -1.1091506917547994e+01, -1.0914198722404091e+01, -1.0737970246351363e+01, -1.0562778842365235e+01, -1.0388584179329960e+01, -1.0215348059403579e+01, -1.0043034253437085e+01, -9.8716083522966080e+00, -9.7010376322345699e+00, -9.5312909327068294e+00, -9.3623385452449437e+00, -9.1941521121732386e+00, -9.0267045341142556e+00, -8.8599698853577831e+00, -8.6939233362816957e+00, -8.5285410821102356e+00, -8.3638002773794167e+00, -8.1996789755519419e+00, -8.0361560732874953e+00, -7.8732112589291026e+00, -7.7108249648145373e+00, -7.5489783230637100e+00, -7.3876531245300106e+00, -7.2268317806359086e+00, -7.0664972878418659e+00, -6.9066331945227768e+00, -6.7472235700485381e+00, -6.5882529758852613e+00, -6.4297064385510465e+00, -6.2715694242761426e+00, -6.1138278152311072e+00, -5.9564678871991967e+00, -5.7994762885802258e+00, -5.6428400206233480e+00, -5.4865464187949282e+00, -5.3305831351959538e+00, -5.1749381219506319e+00, -5.0195996154942355e+00, -4.8645561216944264e+00, -4.7097964017452645e+00, -4.5553094587783480e+00, -4.4010845251394572e+00, -4.2471110502834843e+00, -4.0933786892435746e+00, -3.9398772916341032e+00, -3.7865968911497059e+00, -3.6335276955255318e+00, -3.4806600769262501e+00, -3.3279845627335654e+00, -3.1754918267040755e+00, -3.0231726804711898e+00, -2.8710180653663722e+00, -2.7190190445367941e+00, -2.5671667953376018e+00, -2.4154526019785036e+00, -2.2638678484054164e+00, -2.1124040113990956e+00, -1.9610526538735082e+00, -1.8098054183577674e+00, -1.6586540206460576e+00, -1.5075902436008417e+00, -1.3566059310952046e+00, -1.2056929820809235e+00, -1.0548433447690717e+00, -9.0404901091087708e-01, -7.5330201016646747e-01, -6.0259440454997382e-01, -4.5191828293935077e-01, -3.0126575563983427e-01, -1.5062894899007009e-01, 7.9851131193273576e-15, 1.5062894899007009e-01, 3.0126575563983427e-01, 4.5191828293935077e-01, 6.0259440454997382e-01, 7.5330201016646747e-01, 9.0404901091087708e-01, 1.0548433447690717e+00, 1.2056929820809235e+00, 1.3566059310952046e+00, 1.5075902436008417e+00, 1.6586540206460576e+00, 1.8098054183577674e+00, 1.9610526538735082e+00, 2.1124040113990956e+00, 2.2638678484054164e+00, 2.4154526019785036e+00, 2.5671667953376018e+00, 2.7190190445367941e+00, 2.8710180653663722e+00, 3.0231726804711898e+00, 3.1754918267040755e+00, 3.3279845627335654e+00, 3.4806600769262501e+00, 3.6335276955255318e+00, 3.7865968911497059e+00, 3.9398772916341032e+00, 4.0933786892435746e+00, 4.2471110502834843e+00, 4.4010845251394572e+00, 4.5553094587783480e+00, 4.7097964017452645e+00, 4.8645561216944264e+00, 5.0195996154942355e+00, 5.1749381219506319e+00, 5.3305831351959538e+00, 5.4865464187949282e+00, 5.6428400206233480e+00, 5.7994762885802258e+00, 5.9564678871991967e+00, 6.1138278152311072e+00, 6.2715694242761426e+00, 6.4297064385510465e+00, 6.5882529758852613e+00, 6.7472235700485381e+00, 6.9066331945227768e+00, 7.0664972878418659e+00, 7.2268317806359086e+00, 7.3876531245300106e+00, 7.5489783230637100e+00, 7.7108249648145373e+00, 7.8732112589291026e+00, 8.0361560732874953e+00, 8.1996789755519419e+00, 8.3638002773794167e+00, 8.5285410821102356e+00, 8.6939233362816957e+00, 8.8599698853577831e+00, 9.0267045341142556e+00, 9.1941521121732386e+00, 9.3623385452449437e+00, 9.5312909327068294e+00, 9.7010376322345699e+00, 9.8716083522966080e+00, 1.0043034253437085e+01, 1.0215348059403579e+01, 1.0388584179329960e+01, 1.0562778842365235e+01, 1.0737970246351363e+01, 1.0914198722404091e+01, 1.1091506917547994e+01, 1.1269939997911081e+01, 1.1449545875407471e+01, 1.1630375461345015e+01, 1.1812482951007951e+01, 1.1995926144008331e+01, 1.2180766806106346e+01, 1.2367071079310476e+01, 1.2554909948437734e+01, 1.2744359774012816e+01, 1.2935502903505391e+01, 1.3128428375570619e+01, 1.3323232735334253e+01, 1.3520020983073742e+01, 1.3718907684195138e+01, 1.3920018275613611e+01, 1.4123490613100632e+01, 1.4329476816694543e+01, 1.4538145488075983e+01, 1.4749684396618285e+01, 1.4964303762199654e+01, 1.5182240306661464e+01, 1.5403762307923412e+01, 1.5629175980464554e+01, 1.5858833637952957e+01, 1.6093144292573264e+01, 1.6332587652172350e+01, 1.6577732962488266e+01, 1.6829264937408070e+01, 1.7088020370809854e+01, 1.7355041415469405e+01, 1.7631655968298805e+01, 1.7919604408706316e+01, 1.8221250688860252e+01, 1.8539959593913657e+01, 1.8880837640701767e+01, 1.9252396116029669e+01, 1.9671145428267970e+01, 2.0180280152078982e+01},
			w: []float64{8.0790134837724204e-178, 4.0123708332001955e-169, 4.1608342695810454e-162, 5.3647010323313458e-156, 1.7249035730335862e-150, 1.9898962894711550e-145, 1.0270178951244021e-140, 2.7462859803070011e-136, 4.2216443703754674e-132, 4.0296773344706320e-128, 2.5341526783446447e-124, 1.1002243777181162e-120, 3.4246772430261828e-117, 7.8839348684832184e-114, 1.3776135348531781e-110, 1.8677560466742519e-107, 2.0021556471031417e-104, 1.7247770196475562e-101, 1.2111464936797437e-98, 7.0196501028640057e-96, 3.3954234881610162e-93, 1.3842435843570204e-90, 4.7985036069815231e-88, 1.4256947701704702e-85, 3.6567711491235357e-83, 8.1499750643585127e-81, 1.5877740343548737e-78, 2.7187040652791320e-76, 4.1119602314337377e-74, 5.5188731969772834e-72, 6.6010643664052121e-70, 7.0639875689587789e-68, 6.7880577161215916e-66, 5.8772891020073188e-64, 4.5996045356018730e-62, 3.2633290432137694e-60, 2.1047497049395128e-58, 1.2372745780617511e-56, 6.6453150452727337e-55, 3.2684621263244055e-53, 1.4753196845067616e-51, 6.1238771021510087e-50, 2.3420591906941631e-48, 8.2677634302135741e-47, 2.6986236644096577e-45, 8.1576808396884509e-44, 2.2873393442828659e-42, 5.9575720207549899e-41, 1.4433986536815081e-39, 3.2572816931648467e-38, 6.8552334632687487e-37, 1.3471217671214917e-35, 2.4745963539025701e-34, 4.2538970187949130e-33, 6.8502295544213803e-32, 1.0344007225282243e-30, 1.4660497224001333e-29, 1.9519881038552925e-28, 2.4437079996027598e-27, 2.8788818637419581e-26, 3.1940663724110344e-25, 3.3399348780894923e-24, 3.2939758359327470e-23, 3.0661414593245886e-22, 2.6955142404467904e-21, 2.2394681775190790e-20, 1.7594079537257698e-19, 1.3078564789171130e-18, 9.2038260248200122e-18, 6.1351365453804107e-17, 3.8757039748543155e-16, 2.3214598330581677e-15, 1.3190482037051148e-14, 7.1128747627508331e-14, 3.6416855631427732e-13, 1.7709744825377116e-12, 8.1836457113383607e-12, 3.5947660428421198e-11, 1.5015547963310099e-10, 5.9663534866912190e-10, 2.2558892777245505e-09, 8.1190427548544262e-09, 2.7822813440253852e-08, 9.0809483027322313e-08, 2.8236711509871372e-07, 8.3668655155751835e-07, 2.3631134387385760e-06, 6.3632839492173400e-06, 1.6339902252008705e-05, 4.0020290652701539e-05, 9.3510362001206373e-05, 2.0848221478522648e-04, 4.4359250162546185e-04, 9.0090002804434576e-04, 1.7466777765574378e-03, 3.2333646918818863e-03, 5.7155709629331409e-03, 9.6489576482962046e-03, 1.5558416546311961e-02, 2.3963959563684092e-02, 3.5261402555428784e-02, 4.9570551047913969e-02, 6.6582815385485522e-02, 8.5455825791019971e-02, 1.0480572103134121e-01, 1.2283174514792664e-01, 1.3757292020567533e-01, 1.4725235676872742e-01, 1.5062763955757250e-01, 1.4725235676872742e-01, 1.3757292020567533e-01, 1.2283174514792664e-01, 1.0480572103134121e-01, 8.5455825791019971e-02, 6.6582815385485522e-02, 4.9570551047913969e-02, 3.5261402555428784e-02, 2.3963959563684092e-02, 1.5558416546311961e-02, 9.6489576482962046e-03, 5.7155709629331409e-03, 3.2333646918818863e-03, 1.7466777765574378e-03, 9.0090002804434576e-04, 4.4359250162546185e-04, 2.0848221478522648e-04, 9.3510362001206373e-05, 4.0020290652701539e-05, 1.6339902252008705e-05, 6.3632839492173400e-06, 2.3631134387385760e-06, 8.3668655155751835e-07, 2.8236711509871372e-07, 9.0809483027322313e-08, 2.7822813440253852e-08, 8.1190427548544262e-09, 2.2558892777245505e-09, 5.9663534866912190e-10, 1.5015547963310099e-10, 3.5947660428421198e-11, 8.1836457113383607e-12, 1.7709744825377116e-12, 3.6416855631427732e-13, 7.1128747627508331e-14, 1.3190482037051148e-14, 2.3214598330581677e-15, 3.8757039748543155e-16, 6.1351365453804107e-17, 9.2038260248200122e-18, 1.3078564789171130e-18, 1.7594079537257698e-19, 2.2394681775190790e-20, 2.6955142404467904e-21, 3.0661414593245886e-22, 3.2939758359327470e-23, 3.3399348780894923e-24, 3.1940663724110344e-25, 2.8788818637419581e-26, 2.4437079996027598e-27, 1.9519881038552925e-28, 1.4660497224001333e-29, 1.0344007225282243e-30, 6.8502295544213803e-32, 4.2538970187949130e-33, 2.4745963539025701e-34, 1.3471217671214917e-35, 6.8552334632687487e-37, 3.2572816931648467e-38, 1.4433986536815081e-39, 5.9575720207549899e-41, 2.2873393442828659e-42, 8.1576808396884509e-44, 2.6986236644096577e-45, 8.2677634302135741e-47, 2.3420591906941631e-48, 6.1238771021510087e-50, 1.4753196845067616e-51, 3.2684621263244055e-53, 6.6453150452727337e-55, 1.2372745780617511e-56, 2.1047497049395128e-58, 3.2633290432137694e-60, 4.5996045356018730e-62, 5.8772891020073188e-64, 6.7880577161215916e-66, 7.0639875689587789e-68, 6.6010643664052121e-70, 5.5188731969772834e-72, 4.1119602314337377e-74, 2.7187040652791320e-76, 1.5877740343548737e-78, 8.1499750643585127e-81, 3.6567711491235357e-83, 1.4256947701704702e-85, 4.7985036069815231e-88, 1.3842435843570204e-90, 3.3954234881610162e-93, 7.0196501028640057e-96, 1.2111464936797437e-98, 1.7247770196475562e-101, 2.0021556471031417e-104, 1.8677560466742519e-107, 1.3776135348531781e-110, 7.8839348684832184e-114, 3.4246772430261828e-117, 1.1002243777181162e-120, 2.5341526783446447e-124, 4.0296773344706320e-128, 4.2216443703754674e-132, 2.7462859803070011e-136, 1.0270178951244021e-140, 1.9898962894711550e-145, 1.7249035730335862e-150, 5.3647010323313458e-156, 4.1608342695810454e-162, 4.0123708332001955e-169, 8.0790134837724204e-178},
		},

		{
			n: 218,
			x: []float64{-2.0228695170577915e+01, -1.9719963830858333e+01, -1.9301552400253843e+01, -1.8930298474919816e+01, -1.8589703851801165e+01, -1.8271263496827501e+01, -1.7969874624529179e+01, -1.7682174897106314e+01, -1.7405802075069708e+01, -1.7139017035820078e+01, -1.6880492838899503e+01, -1.6629188080724916e+01, -1.6384266568226192e+01, -1.6145044078855118e+01, -1.5910951775414047e+01, -1.5681510294638914e+01, -1.5456310918627016e+01, -1.5235001587812036e+01, -1.5017276309282583e+01, -1.4802867000027815e+01, -1.4591537111039028e+01, -1.4383076576817141e+01, -1.4177297766812716e+01, -1.3974032204957940e+01, -1.3773127885530926e+01, -1.3574447057358521e+01, -1.3377864379717227e+01, -1.3183265376084384e+01, -1.2990545128683925e+01, -1.2799607169295788e+01, -1.2610362531246457e+01, -1.2422728934700823e+01, -1.2236630082919943e+01, -1.2051995051456386e+01, -1.1868757755632400e+01, -1.1686856484310367e+01, -1.1506233490083723e+01, -1.1326834627713886e+01, -1.1148609034007171e+01, -1.0971508843435563e+01, -1.0795488934711024e+01, -1.0620506704266159e+01, -1.0446521863206758e+01, -1.0273496254809785e+01, -1.0101393690063306e+01, -9.9301797990985268e+00, -9.7598218966614141e+00, -9.5902888600219054e+00, -9.4215510179308293e+00, -9.2535800494152092e+00, -9.0863488913560904e+00, -8.9198316539249092e+00, -8.7540035430671104e+00, -8.5888407893192067e+00, -8.4243205823293916e+00, -8.2604210105245137e+00, -8.0971210054295799e+00, -7.9344002902008972e+00, -7.7722393319820444e+00, -7.6106192977338623e+00, -7.4495220132267015e+00, -7.2889299249154647e+00, -7.1288260644465895e+00, -6.9691940155714960e+00, -6.8100178832631793e+00, -6.6512822648525844e+00, -6.4929722230189020e+00, -6.3350732604836884e+00, -6.1775712962725331e+00, -6.0204526434206276e+00, -5.8637039880096422e+00, -5.7073123694332661e+00, -5.5512651617979705e+00, -5.3955500563732679e+00, -5.2401550450133421e+00, -5.0850684044781733e+00, -4.9302786815884012e+00, -4.7757746791533817e+00, -4.6215454426166902e+00, -4.4675802473678914e+00, -4.3138685866729931e+00, -4.1604001601800418e+00, -4.0071648629591818e+00, -3.8541527750397258e+00, -3.7013541514093888e+00, -3.5487594124432715e+00, -3.3963591347324886e+00, -3.2441440422843915e+00, -3.0921049980680149e+00, -2.9402329958803848e+00, -2.7885191525106028e+00, -2.6369547001801932e+00, -2.4855309792393636e+00, -2.3342394311001944e+00, -2.1830715913886820e+00, -2.0320190832985223e+00, -1.8810736111306099e+00, -1.7302269540027844e+00, -1.5794709597152470e+00, -1.4287975387577758e+00, -1.2781986584451195e+00, -1.1276663371680935e+00, -9.7719263874768358e-01, -8.2676966688043974e-01, -6.7638955966349490e-01, -5.2604448418800387e-01, -3.7572663118998539e-01, -2.2542820974780486e-01, -7.5141442015907556e-02, 7.5141442015907556e-02, 2.2542820974780486e-01, 3.7572663118998539e-01, 5.2604448418800387e-01, 6.7638955966349490e-01, 8.2676966688043974e-01, 9.7719263874768358e-01, 1.1276663371680935e+00, 1.2781986584451195e+00, 1.4287975387577758e+00, 1.5794709597152470e+00, 1.7302269540027844e+00, 1.8810736111306099e+00, 2.0320190832985223e+00, 2.1830715913886820e+00, 2.3342394311001944e+00, 2.4855309792393636e+00, 2.6369547001801932e+00, 2.7885191525106028e+00, 2.9402329958803848e+00, 3.0921049980680149e+00, 3.2441440422843915e+00, 3.3963591347324886e+00, 3.5487594124432715e+00, 3.7013541514093888e+00, 3.8541527750397258e+00, 4.0071648629591818e+00, 4.1604001601800418e+00, 4.3138685866729931e+00, 4.4675802473678914e+00, 4.6215454426166902e+00, 4.7757746791533817e+00, 4.9302786815884012e+00, 5.0850684044781733e+00, 5.2401550450133421e+00, 5.3955500563732679e+00, 5.5512651617979705e+00, 5.7073123694332661e+00, 5.8637039880096422e+00, 6.0204526434206276e+00, 6.1775712962725331e+00, 6.3350732604836884e+00, 6.4929722230189020e+00, 6.6512822648525844e+00, 6.8100178832631793e+00, 6.9691940155714960e+00, 7.1288260644465895e+00, 7.2889299249154647e+00, 7.4495220132267015e+00, 7.6106192977338623e+00, 7.7722393319820444e+00, 7.9344002902008972e+00, 8.0971210054295799e+00, 8.2604210105245137e+00, 8.4243205823293916e+00, 8.5888407893192067e+00, 8.7540035430671104e+00, 8.9198316539249092e+00, 9.0863488913560904e+00, 9.2535800494152092e+00, 9.4215510179308293e+00, 9.5902888600219054e+00, 9.7598218966614141e+00, 9.9301797990985268e+00, 1.0101393690063306e+01, 1.0273496254809785e+01, 1.0446521863206758e+01, 1.0620506704266159e+01, 1.0795488934711024e+01, 1.0971508843435563e+01, 1.1148609034007171e+01, 1.1326834627713886e+01, 1.1506233490083723e+01, 1.1686856484310367e+01, 1.1868757755632400e+01, 1.2051995051456386e+01, 1.2236630082919943e+01, 1.2422728934700823e+01, 1.2610362531246457e+01, 1.2799607169295788e+01, 1.2990545128683925e+01, 1.3183265376084384e+01, 1.3377864379717227e+01, 1.3574447057358521e+01, 1.3773127885530926e+01, 1.3974032204957940e+01, 1.4177297766812716e+01, 1.4383076576817141e+01, 1.4591537111039028e+01, 1.4802867000027815e+01, 1.5017276309282583e+01, 1.5235001587812036e+01, 1.5456310918627016e+01, 1.5681510294638914e+01, 1.5910951775414047e+01, 1.6145044078855118e+01, 1.6384266568226192e+01, 1.6629188080724916e+01, 1.6880492838899503e+01, 1.7139017035820078e+01, 1.7405802075069708e+01, 1.7682174897106314e+01, 1.7969874624529179e+01, 1.8271263496827501e+01, 1.8589703851801165e+01, 1.8930298474919816e+01, 1.9301552400253843e+01, 1.9719963830858333e+01, 2.0228695170577915e+01},
			w: []float64{1.1412038714903467e-178, 5.8600364276414301e-170, 6.2483530291970031e-163, 8.2600389517381484e-157, 2.7180946943728827e-151, 3.2050447674896954e-146, 1.6891299829881347e-141, 4.6087117363873922e-137, 7.2242769320344098e-133, 7.0281073750791237e-129, 4.5026465211632589e-125, 1.9907761957745900e-121, 6.3085296819235091e-118, 1.4780807064034936e-114, 2.6279883692096037e-111, 3.6246331337631131e-108, 3.9519127727199950e-105, 3.4620642776922925e-102, 2.4718758245223020e-99, 1.4565168188237643e-96, 7.1616419809360768e-94, 2.9675903948483656e-91, 1.0455125036940855e-88, 3.1567897170935801e-86, 8.2277370866559977e-84, 1.8632623782187992e-81, 3.6882177983197069e-79, 6.4162031293430147e-77, 9.8590364210588088e-75, 1.3442782637610740e-72, 1.6333988316711487e-70, 1.7756459769101764e-68, 1.7332892347041021e-66, 1.5244548820160086e-64, 1.2118948225793116e-62, 8.7339153184655864e-61, 5.7220382749093667e-59, 3.4167937984322514e-57, 1.8641137908561576e-55, 9.3133897167275065e-54, 4.2703424814876432e-52, 1.8006175119892686e-50, 6.9955013175656585e-49, 2.5086717527911961e-47, 8.3184624948029151e-46, 2.5546048751257637e-44, 7.2770684318833216e-43, 1.9256515369002052e-41, 4.7401499620934791e-40, 1.0868621062998897e-38, 2.3241907456347607e-37, 4.6409284690005433e-36, 8.6630347696542139e-35, 1.5133539937177412e-33, 2.4766678429636503e-32, 3.8008678166597364e-31, 5.4751646602962371e-30, 7.4097643192715769e-29, 9.4293221065635326e-28, 1.1292370018541933e-26, 1.2736861289852883e-25, 1.3540721404735189e-24, 1.3578087095789605e-23, 1.2851527881373697e-22, 1.1488929806753338e-21, 9.7071144039920359e-21, 7.7562537574525747e-20, 5.8643464026904450e-19, 4.1979458289914584e-18, 2.8466716026161652e-17, 1.8295539696950399e-16, 1.1150004956688747e-15, 6.4466274804790879e-15, 3.5376499268729578e-14, 1.8433622668090456e-13, 9.1243282637242511e-13, 4.2919888935314847e-12, 1.9193290587578772e-11, 8.1626646177788407e-11, 3.3026130822100178e-10, 1.2716614885102940e-09, 4.6613505196508951e-09, 1.6270858113762294e-08, 5.4099533071374670e-08, 1.7138819219313466e-07, 5.1747190361086424e-07, 1.4894285947831174e-06, 4.0877345728852175e-06, 1.0699746166171350e-05, 2.6716804837681635e-05, 6.3650699225936851e-05, 1.4471442394637491e-04, 3.1404212609163355e-04, 6.5058509971884274e-04, 1.2868513657693481e-03, 2.4306591608453860e-03, 4.3848013160776717e-03, 7.5554485133504786e-03, 1.2436693679604333e-02, 1.9558227968032915e-02, 2.9388347600235249e-02, 4.2196739277833864e-02, 5.7899315166709715e-02, 7.5925444911783990e-02, 9.5157950558542009e-02, 1.1399005091049424e-01, 1.3051762867943345e-01, 1.4284462872400217e-01, 1.4943738595687683e-01, 1.4943738595687683e-01, 1.4284462872400217e-01, 1.3051762867943345e-01, 1.1399005091049424e-01, 9.5157950558542009e-02, 7.5925444911783990e-02, 5.7899315166709715e-02, 4.2196739277833864e-02, 2.9388347600235249e-02, 1.9558227968032915e-02, 1.2436693679604333e-02, 7.5554485133504786e-03, 4.3848013160776717e-03, 2.4306591608453860e-03, 1.2868513657693481e-03, 6.5058509971884274e-04, 3.1404212609163355e-04, 1.4471442394637491e-04, 6.3650699225936851e-05, 2.6716804837681635e-05, 1.0699746166171350e-05, 4.0877345728852175e-06, 1.4894285947831174e-06, 5.1747190361086424e-07, 1.7138819219313466e-07, 5.4099533071374670e-08, 1.6270858113762294e-08, 4.6613505196508951e-09, 1.2716614885102940e-09, 3.3026130822100178e-10, 8.1626646177788407e-11, 1.9193290587578772e-11, 4.2919888935314847e-12, 9.1243282637242511e-13, 1.8433622668090456e-13, 3.5376499268729578e-14, 6.4466274804790879e-15, 1.1150004956688747e-15, 1.8295539696950399e-16, 2.8466716026161652e-17, 4.1979458289914584e-18, 5.8643464026904450e-19, 7.7562537574525747e-20, 9.7071144039920359e-21, 1.1488929806753338e-21, 1.2851527881373697e-22, 1.3578087095789605e-23, 1.3540721404735189e-24, 1.2736861289852883e-25, 1.1292370018541933e-26, 9.4293221065635326e-28, 7.4097643192715769e-29, 5.4751646602962371e-30, 3.8008678166597364e-31, 2.4766678429636503e-32, 1.5133539937177412e-33, 8.6630347696542139e-35, 4.6409284690005433e-36, 2.3241907456347607e-37, 1.0868621062998897e-38, 4.7401499620934791e-40, 1.9256515369002052e-41, 7.2770684318833216e-43, 2.5546048751257637e-44, 8.3184624948029151e-46, 2.5086717527911961e-47, 6.9955013175656585e-49, 1.8006175119892686e-50, 4.2703424814876432e-52, 9.3133897167275065e-54, 1.8641137908561576e-55, 3.4167937984322514e-57, 5.7220382749093667e-59, 8.7339153184655864e-61, 1.2118948225793116e-62, 1.5244548820160086e-64, 1.7332892347041021e-66, 1.7756459769101764e-68, 1.6333988316711487e-70, 1.3442782637610740e-72, 9.8590364210588088e-75, 6.4162031293430147e-77, 3.6882177983197069e-79, 1.8632623782187992e-81, 8.2277370866559977e-84, 3.1567897170935801e-86, 1.0455125036940855e-88, 2.9675903948483656e-91, 7.1616419809360768e-94, 1.4565168188237643e-96, 2.4718758245223020e-99, 3.4620642776922925e-102, 3.9519127727199950e-105, 3.6246331337631131e-108, 2.6279883692096037e-111, 1.4780807064034936e-114, 6.3085296819235091e-118, 1.9907761957745900e-121, 4.5026465211632589e-125, 7.0281073750791237e-129, 7.2242769320344098e-133, 4.6087117363873922e-137, 1.6891299829881347e-141, 3.2050447674896954e-146, 2.7180946943728827e-151, 8.2600389517381484e-157, 6.2483530291970031e-163, 5.8600364276414301e-170, 1.1412038714903467e-178},
		},
		{
			n: 219,
			x: []float64{-2.0276997908046905e+01, -1.9768667748260523e+01, -1.9350592328749663e+01, -1.8979641248255188e+01, -1.8639328445512675e+01, -1.8321155106754730e+01, -1.8020022158568153e+01, -1.7732569697856896e+01, -1.7456437188959409e+01, -1.7189886759582706e+01, -1.6931592420762687e+01, -1.6680513514209245e+01, -1.6435814444517028e+01, -1.6196811477889465e+01, -1.5962936183500821e+01, -1.5733709540823362e+01, -1.5508723124641774e+01, -1.5287625128097030e+01, -1.5070109778611977e+01, -1.4855909186982135e+01, -1.4644786976036578e+01, -1.4436533233749541e+01, -1.4230960467563948e+01, -1.4027900326255610e+01, -1.3827200917702472e+01, -1.3628724594657513e+01, -1.3432346111954516e+01, -1.3237951081351927e+01, -1.3045434667000123e+01, -1.2854700477033081e+01, -1.2665659616227074e+01, -1.2478229871866438e+01, -1.2292335010497055e+01, -1.2107904167551959e+01, -1.1924871315204850e+01, -1.1743174796469416e+01, -1.1562756915679721e+01, -1.1383563577182978e+01, -1.1205543965443459e+01, -1.1028650260865531e+01, -1.0852837386548828e+01, -1.0678062781931233e+01, -1.0504286199887655e+01, -1.0331469524360219e+01, -1.0159576606018058e+01, -9.9885731137985694e+00, -9.8184264004786943e+00, -9.6491053806754916e+00, -9.4805804198871506e+00, -9.3128232333657657e+00, -9.1458067937670240e+00, -8.9795052466532344e+00, -8.8138938330391774e+00, -8.6489488182673355e+00, -8.4846474265831215e+00, -8.3209677808533336e+00, -8.1578888469343305e+00, -7.9953903822513528e+00, -7.8334528881984324e+00, -7.6720575660104693e+00, -7.5111862757957555e+00, -7.3508214984498235e+00, -7.1909463001999656e+00, -7.0315442995549828e+00, -6.8725996364571813e+00, -6.7140969434532645e+00, -6.5560213187184688e+00, -6.3983583007839497e+00, -6.2410938448312283e+00, -6.0842143004302391e+00, -5.9277063906083614e+00, -5.7715571921480873e+00, -5.6157541170197165e+00, -5.4602848948636904e+00, -5.3051375564443513e+00, -5.1503004180034342e+00, -4.9957620664475568e+00, -4.8415113453092564e+00, -4.6875373414259709e+00, -4.5338293722856466e+00, -4.3803769739918419e+00, -4.2271698898044328e+00, -4.0741980592156279e+00, -3.9214516075238026e+00, -3.7689208358703388e+00, -3.6165962117072561e+00, -3.4644683596654966e+00, -3.3125280527959076e+00, -3.1607662041567641e+00, -3.0091738587233952e+00, -2.8577421855969432e+00, -2.7064624704909552e+00, -2.5553261084754681e+00, -2.4043245969596510e+00, -2.2534495288951528e+00, -2.1026925861830650e+00, -1.9520455332686681e+00, -1.8015002109084974e+00, -1.6510485300954487e+00, -1.5006824661279821e+00, -1.3503940528101779e+00, -1.2001753767700742e+00, -1.0500185718839765e+00, -8.9991581379505270e-01, -7.4985931451474896e-01, -5.9984131709602384e-01, -4.4985409036765389e-01, -2.9988992371893503e-01, -1.4994112192465620e-01, 8.0217422708380516e-15, 1.4994112192465620e-01, 2.9988992371893503e-01, 4.4985409036765389e-01, 5.9984131709602384e-01, 7.4985931451474896e-01, 8.9991581379505270e-01, 1.0500185718839765e+00, 1.2001753767700742e+00, 1.3503940528101779e+00, 1.5006824661279821e+00, 1.6510485300954487e+00, 1.8015002109084974e+00, 1.9520455332686681e+00, 2.1026925861830650e+00, 2.2534495288951528e+00, 2.4043245969596510e+00, 2.5553261084754681e+00, 2.7064624704909552e+00, 2.8577421855969432e+00, 3.0091738587233952e+00, 3.1607662041567641e+00, 3.3125280527959076e+00, 3.4644683596654966e+00, 3.6165962117072561e+00, 3.7689208358703388e+00, 3.9214516075238026e+00, 4.0741980592156279e+00, 4.2271698898044328e+00, 4.3803769739918419e+00, 4.5338293722856466e+00, 4.6875373414259709e+00, 4.8415113453092564e+00, 4.9957620664475568e+00, 5.1503004180034342e+00, 5.3051375564443513e+00, 5.4602848948636904e+00, 5.6157541170197165e+00, 5.7715571921480873e+00, 5.9277063906083614e+00, 6.0842143004302391e+00, 6.2410938448312283e+00, 6.3983583007839497e+00, 6.5560213187184688e+00, 6.7140969434532645e+00, 6.8725996364571813e+00, 7.0315442995549828e+00, 7.1909463001999656e+00, 7.3508214984498235e+00, 7.5111862757957555e+00, 7.6720575660104693e+00, 7.8334528881984324e+00, 7.9953903822513528e+00, 8.1578888469343305e+00, 8.3209677808533336e+00, 8.4846474265831215e+00, 8.6489488182673355e+00, 8.8138938330391774e+00, 8.9795052466532344e+00, 9.1458067937670240e+00, 9.3128232333657657e+00, 9.4805804198871506e+00, 9.6491053806754916e+00, 9.8184264004786943e+00, 9.9885731137985694e+00, 1.0159576606018058e+01, 1.0331469524360219e+01, 1.0504286199887655e+01, 1.0678062781931233e+01, 1.0852837386548828e+01, 1.1028650260865531e+01, 1.1205543965443459e+01, 1.1383563577182978e+01, 1.1562756915679721e+01, 1.1743174796469416e+01, 1.1924871315204850e+01, 1.2107904167551959e+01, 1.2292335010497055e+01, 1.2478229871866438e+01, 1.2665659616227074e+01, 1.2854700477033081e+01, 1.3045434667000123e+01, 1.3237951081351927e+01, 1.3432346111954516e+01, 1.3628724594657513e+01, 1.3827200917702472e+01, 1.4027900326255610e+01, 1.4230960467563948e+01, 1.4436533233749541e+01, 1.4644786976036578e+01, 1.4855909186982135e+01, 1.5070109778611977e+01, 1.5287625128097030e+01, 1.5508723124641774e+01, 1.5733709540823362e+01, 1.5962936183500821e+01, 1.6196811477889465e+01, 1.6435814444517028e+01, 1.6680513514209245e+01, 1.6931592420762687e+01, 1.7189886759582706e+01, 1.7456437188959409e+01, 1.7732569697856896e+01, 1.8020022158568153e+01, 1.8321155106754730e+01, 1.8639328445512675e+01, 1.8979641248255188e+01, 1.9350592328749663e+01, 1.9768667748260523e+01, 2.0276997908046905e+01},
			w: []float64{1.6118004761490936e-179, 8.5565157993831952e-171, 9.3801386072360495e-164, 1.2712817065467470e-157, 4.2810879764792902e-152, 5.1593581306426363e-147, 2.7763562197121041e-142, 7.7287684331971630e-138, 1.2353061004428027e-133, 1.2247428452374434e-129, 7.9930519467022852e-126, 3.5986937020445443e-122, 1.1608863141459933e-118, 2.7680773700442792e-115, 5.0074481042562811e-112, 7.0254966797767812e-109, 7.7903695739499256e-106, 6.9398586398625168e-103, 5.0378132240927281e-100, 3.0176790544586562e-97, 1.5082063803644767e-94, 6.3518001539538009e-92, 2.2741855202923794e-89, 6.9776584966048747e-87, 1.8479046829051634e-84, 4.2518674704681371e-82, 8.5507393331426323e-80, 1.5112093165489227e-77, 2.3589639789454550e-75, 3.2673822149378061e-73, 4.0328590535463278e-71, 4.4532356700297110e-69, 4.4154958570004513e-67, 3.9446173154369991e-65, 3.1851636643508064e-63, 2.3315719656364080e-61, 1.5515360955581329e-59, 9.4102483913550526e-58, 5.2146771930312127e-56, 2.6462964683019027e-54, 1.2324622103285675e-52, 5.2785960684684324e-51, 2.0830950016909702e-49, 7.5881492915547110e-48, 2.5559164938518617e-46, 7.9735166177827294e-45, 2.3073707359625722e-43, 6.2027848314701266e-42, 1.5511852212359039e-40, 3.6134675099711877e-39, 7.8508422681776689e-38, 1.5928021832222874e-36, 3.0210523716429586e-35, 5.3626533220153980e-34, 8.9182314112122594e-33, 1.3908724042317442e-31, 2.0361886950192076e-30, 2.8006915544325587e-29, 3.6224809761826824e-28, 4.4096199762595120e-27, 5.0558656605721755e-26, 5.4641111784906703e-25, 5.5704394297442813e-24, 5.3605363007899914e-23, 4.8726597293407410e-22, 4.1864031729275524e-21, 3.4017282707064429e-20, 2.6157557992877646e-19, 1.9044895750724533e-18, 1.3136464708861073e-17, 8.5885898614439566e-17, 5.3250576372585674e-16, 3.1325104388327040e-15, 1.7491400171437906e-14, 9.2749216025525341e-14, 4.6723068181067832e-13, 2.2369795792044991e-12, 1.0182840690359900e-11, 4.4087115371635113e-11, 1.8161128736541464e-10, 7.1204644180465281e-10, 2.6579545759517285e-09, 9.4491700732305002e-09, 3.2001745961210195e-08, 1.0327820095258369e-07, 3.1769761704760471e-07, 9.3174969312259845e-07, 2.6059663547935113e-06, 6.9521878670475772e-06, 1.7695019387908094e-05, 4.2977984770344540e-05, 9.9630005120895210e-05, 2.2047640877838706e-04, 4.6583912781443418e-04, 9.3989854709223869e-04, 1.8111811218635974e-03, 3.3337948402502641e-03, 5.8622951858897081e-03, 9.8491710526776490e-03, 1.5811821288832947e-02, 2.4258117817844916e-02, 3.5568359860607475e-02, 4.9846708381161745e-02, 6.6773674520725357e-02, 8.5505952447471292e-02, 1.0467230074567024e-01, 1.2249815849737687e-01, 1.3705778627617005e-01, 1.4661020932556390e-01, 1.4993984211667696e-01, 1.4661020932556390e-01, 1.3705778627617005e-01, 1.2249815849737687e-01, 1.0467230074567024e-01, 8.5505952447471292e-02, 6.6773674520725357e-02, 4.9846708381161745e-02, 3.5568359860607475e-02, 2.4258117817844916e-02, 1.5811821288832947e-02, 9.8491710526776490e-03, 5.8622951858897081e-03, 3.3337948402502641e-03, 1.8111811218635974e-03, 9.3989854709223869e-04, 4.6583912781443418e-04, 2.2047640877838706e-04, 9.9630005120895210e-05, 4.2977984770344540e-05, 1.7695019387908094e-05, 6.9521878670475772e-06, 2.6059663547935113e-06, 9.3174969312259845e-07, 3.1769761704760471e-07, 1.0327820095258369e-07, 3.2001745961210195e-08, 9.4491700732305002e-09, 2.6579545759517285e-09, 7.1204644180465281e-10, 1.8161128736541464e-10, 4.4087115371635113e-11, 1.0182840690359900e-11, 2.2369795792044991e-12, 4.6723068181067832e-13, 9.2749216025525341e-14, 1.7491400171437906e-14, 3.1325104388327040e-15, 5.3250576372585674e-16, 8.5885898614439566e-17, 1.3136464708861073e-17, 1.9044895750724533e-18, 2.6157557992877646e-19, 3.4017282707064429e-20, 4.1864031729275524e-21, 4.8726597293407410e-22, 5.3605363007899914e-23, 5.5704394297442813e-24, 5.4641111784906703e-25, 5.0558656605721755e-26, 4.4096199762595120e-27, 3.6224809761826824e-28, 2.8006915544325587e-29, 2.0361886950192076e-30, 1.3908724042317442e-31, 8.9182314112122594e-33, 5.3626533220153980e-34, 3.0210523716429586e-35, 1.5928021832222874e-36, 7.8508422681776689e-38, 3.6134675099711877e-39, 1.5511852212359039e-40, 6.2027848314701266e-42, 2.3073707359625722e-43, 7.9735166177827294e-45, 2.5559164938518617e-46, 7.5881492915547110e-48, 2.0830950016909702e-49, 5.2785960684684324e-51, 1.2324622103285675e-52, 2.6462964683019027e-54, 5.2146771930312127e-56, 9.4102483913550526e-58, 1.5515360955581329e-59, 2.3315719656364080e-61, 3.1851636643508064e-63, 3.9446173154369991e-65, 4.4154958570004513e-67, 4.4532356700297110e-69, 4.0328590535463278e-71, 3.2673822149378061e-73, 2.3589639789454550e-75, 1.5112093165489227e-77, 8.5507393331426323e-80, 4.2518674704681371e-82, 1.8479046829051634e-84, 6.9776584966048747e-87, 2.2741855202923794e-89, 6.3518001539538009e-92, 1.5082063803644767e-94, 3.0176790544586562e-97, 5.0378132240927281e-100, 6.9398586398625168e-103, 7.7903695739499256e-106, 7.0254966797767812e-109, 5.0074481042562811e-112, 2.7680773700442792e-115, 1.1608863141459933e-118, 3.5986937020445443e-122, 7.9930519467022852e-126, 1.2247428452374434e-129, 1.2353061004428027e-133, 7.7287684331971630e-138, 2.7763562197121041e-142, 5.1593581306426363e-147, 4.2810879764792902e-152, 1.2712817065467470e-157, 9.3801386072360495e-164, 8.5565157993831952e-171, 1.6118004761490936e-179},
		},
		{
			n: 220,
			x: []float64{-2.0325189139629405e+01, -1.9817257977795947e+01, -1.9399516717870224e+01, -1.9028866794578029e+01, -1.8688834225534919e+01, -1.8370926385157830e+01, -1.8070047892967271e+01, -1.7782841268044042e+01, -1.7506947669788651e+01, -1.7240630469775191e+01, -1.6982564625621777e+01, -1.6731710220470223e+01, -1.6487232253468321e+01, -1.6248447477010238e+01, -1.6014787864496149e+01, -1.5785774736307584e+01, -1.5560999958339213e+01, -1.5340111975060188e+01, -1.5122805233008011e+01, -1.4908812035694833e+01, -1.4697896176804390e+01, -1.4489847896889446e+01, -1.4284479840564213e+01, -1.4081623780688842e+01, -1.3881127938034011e+01, -1.3682854768615876e+01, -1.3486679122200039e+01, -1.3292486698232523e+01, -1.3100172742223814e+01, -1.2909640938118768e+01, -1.2720802461619986e+01, -1.2533575166624548e+01, -1.2347882882470566e+01, -1.2163654803990811e+01, -1.1980824959739557e+01, -1.1799331746419007e+01, -1.1619117519647554e+01, -1.1440128232906956e+01, -1.1262313117871999e+01, -1.1085624400434606e+01, -1.0910017047638791e+01, -1.0735448541485018e+01, -1.0561878676174262e+01, -1.0389269375869560e+01, -1.0217584530474927e+01, -1.0046789847284987e+01, -9.8768527166552929e+00, -9.7077420900935678e+00, -9.5394283693840993e+00, -9.3718833055374660e+00, -9.2050799065113704e+00, -9.0389923527798217e+00, -8.8735959199405876e+00, -8.7088669076480283e+00, -8.5447825742424133e+00, -8.3813210765192689e+00, -8.2184614141456525e+00, -8.0561833782850805e+00, -7.8944675040409047e+00, -7.7332950263697580e+00, -7.5726478391539116e+00, -7.4125084571532707e+00, -7.2528599805868588e+00, -7.0936860621182607e+00, -6.9349708760424020e+00, -6.7766990894903305e+00, -6.6188558354865279e+00, -6.4614266877089035e+00, -6.3043976368153372e+00, -6.1477550682134519e+00, -5.9914857411611600e+00, -5.8355767690955807e+00, -5.6800156010969900e+00, -5.5247900044023472e+00, -5.3698880478903845e+00, -5.2152980864665599e+00, -5.0610087462822682e+00, -4.9070089107279511e+00, -4.7532877071444766e+00, -4.5998344942017342e+00, -4.4466388498971172e+00, -4.2936905601303739e+00, -4.1409796078143595e+00, -3.9884961624844180e+00, -3.8362305703715940e+00, -3.6841733449075362e+00, -3.5323151576310980e+00, -3.3806468294686467e+00, -3.2291593223621100e+00, -3.0778437312202702e+00, -2.9266912761707076e+00, -2.7756932950907438e+00, -2.6248412363975424e+00, -2.4741266520783536e+00, -2.3235411909430654e+00, -2.1730765920823036e+00, -2.0227246785151167e+00, -1.8724773510110999e+00, -1.7223265820726865e+00, -1.5722644100637477e+00, -1.4222829334715719e+00, -1.2723743052896335e+00, -1.1225307275089995e+00, -9.7274444570696528e-01, -8.2300774372149477e-01, -6.7331293840074014e-01, -5.2365237441695056e-01, -3.7401841913451778e-01, -2.2440345752197671e-01, -7.4799887098051757e-02, 7.4799887098051757e-02, 2.2440345752197671e-01, 3.7401841913451778e-01, 5.2365237441695056e-01, 6.7331293840074014e-01, 8.2300774372149477e-01, 9.7274444570696528e-01, 1.1225307275089995e+00, 1.2723743052896335e+00, 1.4222829334715719e+00, 1.5722644100637477e+00, 1.7223265820726865e+00, 1.8724773510110999e+00, 2.0227246785151167e+00, 2.1730765920823036e+00, 2.3235411909430654e+00, 2.4741266520783536e+00, 2.6248412363975424e+00, 2.7756932950907438e+00, 2.9266912761707076e+00, 3.0778437312202702e+00, 3.2291593223621100e+00, 3.3806468294686467e+00, 3.5323151576310980e+00, 3.6841733449075362e+00, 3.8362305703715940e+00, 3.9884961624844180e+00, 4.1409796078143595e+00, 4.2936905601303739e+00, 4.4466388498971172e+00, 4.5998344942017342e+00, 4.7532877071444766e+00, 4.9070089107279511e+00, 5.0610087462822682e+00, 5.2152980864665599e+00, 5.3698880478903845e+00, 5.5247900044023472e+00, 5.6800156010969900e+00, 5.8355767690955807e+00, 5.9914857411611600e+00, 6.1477550682134519e+00, 6.3043976368153372e+00, 6.4614266877089035e+00, 6.6188558354865279e+00, 6.7766990894903305e+00, 6.9349708760424020e+00, 7.0936860621182607e+00, 7.2528599805868588e+00, 7.4125084571532707e+00, 7.5726478391539116e+00, 7.7332950263697580e+00, 7.8944675040409047e+00, 8.0561833782850805e+00, 8.2184614141456525e+00, 8.3813210765192689e+00, 8.5447825742424133e+00, 8.7088669076480283e+00, 8.8735959199405876e+00, 9.0389923527798217e+00, 9.2050799065113704e+00, 9.3718833055374660e+00, 9.5394283693840993e+00, 9.7077420900935678e+00, 9.8768527166552929e+00, 1.0046789847284987e+01, 1.0217584530474927e+01, 1.0389269375869560e+01, 1.0561878676174262e+01, 1.0735448541485018e+01, 1.0910017047638791e+01, 1.1085624400434606e+01, 1.1262313117871999e+01, 1.1440128232906956e+01, 1.1619117519647554e+01, 1.1799331746419007e+01, 1.1980824959739557e+01, 1.2163654803990811e+01, 1.2347882882470566e+01, 1.2533575166624548e+01, 1.2720802461619986e+01, 1.2909640938118768e+01, 1.3100172742223814e+01, 1.3292486698232523e+01, 1.3486679122200039e+01, 1.3682854768615876e+01, 1.3881127938034011e+01, 1.4081623780688842e+01, 1.4284479840564213e+01, 1.4489847896889446e+01, 1.4697896176804390e+01, 1.4908812035694833e+01, 1.5122805233008011e+01, 1.5340111975060188e+01, 1.5560999958339213e+01, 1.5785774736307584e+01, 1.6014787864496149e+01, 1.6248447477010238e+01, 1.6487232253468321e+01, 1.6731710220470223e+01, 1.6982564625621777e+01, 1.7240630469775191e+01, 1.7506947669788651e+01, 1.7782841268044042e+01, 1.8070047892967271e+01, 1.8370926385157830e+01, 1.8688834225534919e+01, 1.9028866794578029e+01, 1.9399516717870224e+01, 1.9817257977795947e+01, 2.0325189139629405e+01},
			w: []float64{2.2761605731502931e-180, 1.2490844229282551e-171, 1.4077080884426324e-164, 1.9558070725961945e-158, 6.7396164617778564e-153, 8.3007451278928549e-148, 4.5605427168506885e-143, 1.2952107195320918e-138, 2.1106922482765282e-134, 2.1325182525295938e-130, 1.4176541142180510e-126, 6.4990818384221324e-123, 2.1340663816447175e-119, 5.1782994410611593e-116, 9.5303885471678039e-113, 1.3600769560707746e-109, 1.5337501625076215e-106, 1.3892626373123839e-103, 1.0252931948997944e-100, 6.2430026550428978e-98, 3.1713496923820005e-95, 1.3573663268548106e-92, 4.9385791079870452e-90, 1.5396613562557895e-87, 4.1428768007295419e-85, 9.6845718261332366e-83, 1.9785969337019459e-80, 3.5522955000635345e-78, 5.6327088700629997e-76, 7.9248469784581938e-74, 9.9353884108905009e-72, 1.1143371044053519e-69, 1.1222249771693660e-67, 1.0182568638449594e-65, 8.3508496445242703e-64, 6.2085535684767686e-62, 4.1960840622737107e-60, 2.5847807937901128e-58, 1.4547608917709417e-56, 7.4980265458380947e-55, 3.5467498009174184e-53, 1.5428702374218767e-51, 6.1841637682672203e-50, 2.2881100073201874e-48, 7.8282784948541621e-47, 2.4806114249059706e-45, 7.2916745732054914e-44, 1.9911806078855746e-42, 5.0584298583405264e-41, 1.1970700137586669e-39, 2.6422337190194530e-38, 5.4462015155980312e-37, 1.0495069994748297e-35, 1.8928725655710039e-34, 3.1985659419085236e-33, 5.0689745622498271e-32, 7.5410079936029980e-31, 1.0540919389712685e-29, 1.3856239612378123e-28, 1.7143191474654609e-27, 1.9978524188114189e-26, 2.1947842513043781e-25, 2.2745432296820405e-24, 2.2252271383645215e-23, 2.0564749726923156e-22, 1.7964719327802204e-21, 1.4843379300503135e-20, 1.1606915413845120e-19, 8.5944540731309027e-19, 6.0293824390549243e-18, 4.0096487904356170e-17, 2.5289217277543671e-16, 1.5134484435681475e-15, 8.5980922774659504e-15, 4.6390554602844329e-14, 2.3781148094626393e-13, 1.1587426167580950e-12, 5.3685841373417416e-12, 2.3659835874748833e-11, 9.9219381603477156e-11, 3.9606009206879336e-10, 1.5053791970450186e-09, 5.4498580894419877e-09, 1.8797800781026250e-08, 6.1792198822459984e-08, 1.9363392840773747e-07, 5.7857719375034532e-07, 1.6488403637671232e-06, 4.4826356596959815e-06, 1.1628419918195159e-05, 2.8789242845201347e-05, 6.8037279449589797e-05, 1.5351500720814013e-04, 3.3076323494923371e-04, 6.8063888690970183e-04, 1.3378748383672085e-03, 2.5123201954271616e-03, 4.5076732111920138e-03, 7.7285957578851656e-03, 1.2663915587278220e-02, 1.9833481075447929e-02, 2.9691593353631068e-02, 4.2492045448586743e-02, 5.8137078835965875e-02, 7.6049788530935164e-02, 9.5118681388506904e-02, 1.1375654825578213e-01, 1.3009049092173738e-01, 1.4226074789149157e-01, 1.4876572633109977e-01, 1.4876572633109977e-01, 1.4226074789149157e-01, 1.3009049092173738e-01, 1.1375654825578213e-01, 9.5118681388506904e-02, 7.6049788530935164e-02, 5.8137078835965875e-02, 4.2492045448586743e-02, 2.9691593353631068e-02, 1.9833481075447929e-02, 1.2663915587278220e-02, 7.7285957578851656e-03, 4.5076732111920138e-03, 2.5123201954271616e-03, 1.3378748383672085e-03, 6.8063888690970183e-04, 3.3076323494923371e-04, 1.5351500720814013e-04, 6.8037279449589797e-05, 2.8789242845201347e-05, 1.1628419918195159e-05, 4.4826356596959815e-06, 1.6488403637671232e-06, 5.7857719375034532e-07, 1.9363392840773747e-07, 6.1792198822459984e-08, 1.8797800781026250e-08, 5.4498580894419877e-09, 1.5053791970450186e-09, 3.9606009206879336e-10, 9.9219381603477156e-11, 2.3659835874748833e-11, 5.3685841373417416e-12, 1.1587426167580950e-12, 2.3781148094626393e-13, 4.6390554602844329e-14, 8.5980922774659504e-15, 1.5134484435681475e-15, 2.5289217277543671e-16, 4.0096487904356170e-17, 6.0293824390549243e-18, 8.5944540731309027e-19, 1.1606915413845120e-19, 1.4843379300503135e-20, 1.7964719327802204e-21, 2.0564749726923156e-22, 2.2252271383645215e-23, 2.2745432296820405e-24, 2.1947842513043781e-25, 1.9978524188114189e-26, 1.7143191474654609e-27, 1.3856239612378123e-28, 1.0540919389712685e-29, 7.5410079936029980e-31, 5.0689745622498271e-32, 3.1985659419085236e-33, 1.8928725655710039e-34, 1.0495069994748297e-35, 5.4462015155980312e-37, 2.6422337190194530e-38, 1.1970700137586669e-39, 5.0584298583405264e-41, 1.9911806078855746e-42, 7.2916745732054914e-44, 2.4806114249059706e-45, 7.8282784948541621e-47, 2.2881100073201874e-48, 6.1841637682672203e-50, 1.5428702374218767e-51, 3.5467498009174184e-53, 7.4980265458380947e-55, 1.4547608917709417e-56, 2.5847807937901128e-58, 4.1960840622737107e-60, 6.2085535684767686e-62, 8.3508496445242703e-64, 1.0182568638449594e-65, 1.1222249771693660e-67, 1.1143371044053519e-69, 9.9353884108905009e-72, 7.9248469784581938e-74, 5.6327088700629997e-76, 3.5522955000635345e-78, 1.9785969337019459e-80, 9.6845718261332366e-83, 4.1428768007295419e-85, 1.5396613562557895e-87, 4.9385791079870452e-90, 1.3573663268548106e-92, 3.1713496923820005e-95, 6.2430026550428978e-98, 1.0252931948997944e-100, 1.3892626373123839e-103, 1.5337501625076215e-106, 1.3600769560707746e-109, 9.5303885471678039e-113, 5.1782994410611593e-116, 2.1340663816447175e-119, 6.4990818384221324e-123, 1.4176541142180510e-126, 2.1325182525295938e-130, 2.1106922482765282e-134, 1.2952107195320918e-138, 4.5605427168506885e-143, 8.3007451278928549e-148, 6.7396164617778564e-153, 1.9558070725961945e-158, 1.4077080884426324e-164, 1.2490844229282551e-171, 2.2761605731502931e-180},
		},
	} {
		h := Hermite{}
		x := make([]float64, test.n)
		weight := make([]float64, test.n)
		h.FixedLocations(x, weight, math.Inf(-1), math.Inf(1))
		if !floats.EqualApprox(x, test.x, 1e-14) {
			t.Errorf("n = %v, x mismatch", test.n)
			fmt.Println(x)
			fmt.Println()
			fmt.Println(test.x)
			//t.Errorf("x mismatch. \nWant %v\ngot %v", test.x, x)
		}
		if !floats.EqualApprox(weight, test.w, 1e-14) {
			t.Errorf("n = %v, w mismatch", test.n)
			//t.Errorf("w mismatch. Want %v, got %v", test.w, weight)
		}
	}
}

func TestHermiteInitialGuesses(t *testing.T) {
	t.Parallel()
	for _, test := range []struct {
		n    int
		want []float64
	}{
		// Evaluated from Chebfun.
		{
			n:    215,
			want: []float64{0, 0.151326329763712, 0.302660700793398, 0.454011159911720, 0.605385765065948, 0.756792590918330, 0.908239734470234, 1.059735320731617, 1.211287508447394, 1.362904495892717, 1.514594526749157, 1.666365896074552, 1.818226956379164, 1.970186123821706, 2.122251884538916, 2.274432801123240, 2.426737519263548, 2.579174774564779, 2.731753399562905, 2.884482330952826, 3.037370617047366, 3.190427425487010, 3.343662051220745, 3.497083924780193, 3.650702620869949, 3.804527867299248, 3.958569554281313, 4.112837744128784, 4.267342681375636, 4.422094803358200, 4.577104751290402, 4.732383381871004, 4.887941779463563, 5.043791268893151, 5.199943428907478, 5.356410106353889, 5.513203431128299, 5.670335831956713, 5.827820053075521, 5.985669171882509, 6.143896617637219, 6.302516191296349, 6.461542086578262, 6.620988912359293, 6.780871716514890, 6.941206011329554, 7.102007800612191, 7.263293608667420, 7.425080511288993, 7.587386168959320, 7.750228862458692, 7.913627531110426, 8.077601813913194, 8.242172093840729, 8.407359545621334, 8.573186187346858, 8.739674936302684, 8.906849669458699, 9.074735289116129, 9.243357794268617, 9.412744358308792, 9.582923413795772, 9.753924745096450, 9.925779589826769, 10.098520750150884, 10.272182715150265, 10.446801795655594, 10.622416273146731, 10.799066564577450, 10.976795405279171, 11.155648052452676, 11.335672512180423, 11.516919793401330, 11.699444192903831, 11.883303616137852, 12.068559939553955, 12.255279421290370, 12.443533168399910, 12.633397670509774, 12.824955411930718, 13.018295576901773, 13.213514866037960, 13.410718446364541, 13.610021062878001, 13.811548346792373, 14.015438365098394, 14.221843468615116, 14.430932512542578, 14.642893546366283, 14.857937101386424, 15.076300248006383, 15.298251657131900, 15.524097989861344, 15.754192071913305, 15.988943508292977, 16.228832700713156, 16.474429717130270, 16.726420259572109, 16.985642328947012, 17.253139580754688, 17.530241825363181, 17.818691945179868, 18.120857274390474, 18.440107345774539, 18.781555607447057, 19.153723464992801, 19.573128956064245, 20.082675660407357},
		},
		{
			n:    216,
			want: []float64{7.5487718957947841e-02, 2.2646713089416223e-01, 3.7745846761172613e-01, 5.2846968803819716e-01, 6.7950876473020116e-01, 8.3058368936317251e-01, 9.8170247825952706e-01, 1.1328731779663146e+00, 1.2841038708939367e+00, 1.4354026810274640e+00, 1.5867777797225648e+00, 1.7382373915984026e+00, 1.8897898005400910e+00, 2.0414433558240375e+00, 2.1932064783796412e+00, 2.3450876672018457e+00, 2.4970955059292010e+00, 2.6492386696032790e+00, 2.8015259316256671e+00, 2.9539661709299665e+00, 3.1065683793869288e+00, 3.2593416694621893e+00, 3.4122952821469283e+00, 3.5654385951833136e+00, 3.7187811316079062e+00, 3.8723325686377130e+00, 4.0261027469252912e+00, 4.1801016802112088e+00, 4.3343395654041350e+00, 4.4888267931211692e+00, 4.6435739587232687e+00, 4.7985918738836766e+00, 4.9538915787298405e+00, 5.1094843546028494e+00, 5.2653817374818566e+00, 5.4215955321250595e+00, 5.5781378269829656e+00, 5.7350210099447425e+00, 5.8922577849835758e+00, 6.0498611897730488e+00, 6.2078446143528714e+00, 6.3662218209298418e+00, 6.5250069649076838e+00, 6.6842146172486627e+00, 6.8438597882796577e+00, 7.0039579530666876e+00, 7.1645250784943162e+00, 7.3255776522003675e+00, 7.4871327135320032e+00, 7.6492078867069049e+00, 7.8118214163831334e+00, 7.9749922058635363e+00, 8.1387398581859554e+00, 8.3030847203790454e+00, 8.4680479311960379e+00, 8.6336514726757141e+00, 8.7999182259219690e+00, 8.9668720315415094e+00, 9.1345377552342626e+00, 9.3029413590944845e+00, 9.4721099792532808e+00, 9.6420720105775217e+00, 9.8128571992374614e+00, 9.9844967440684584e+00, 1.0157023407784097e+01, 1.0330471639251822e+01, 1.0504877708222864e+01, 1.0680279854120787e+01, 1.0856718450743809e+01, 1.1034236189033734e+01, 1.1212878280418622e+01, 1.1392692683659751e+01, 1.1573730358642214e+01, 1.1756045551162156e+01, 1.1939696113507749e+01, 1.2124743866538131e+01, 1.2311255010076101e+01, 1.2499300589800702e+01, 1.2688957030525529e+01, 1.2880306747870632e+01, 1.3073438853003600e+01, 1.3268449968504161e+01, 1.3465445177719724e+01, 1.3664539135531767e+01, 1.3865857375666192e+01, 1.4069537859142688e+01, 1.4275732821001174e+01, 1.4484610989259913e+01, 1.4696360272885402e+01, 1.4911191046953165e+01, 1.5129340207007946e+01, 1.5351076226802732e+01, 1.5576705543358910e+01, 1.5806580725457744e+01, 1.6041111080581306e+01, 1.6280776662107755e+01, 1.6526147125048691e+01, 1.6777907674840002e+01, 1.7036895705202610e+01, 1.7304154114531990e+01, 1.7581011746684457e+01, 1.7869210214084568e+01, 1.8171115120332015e+01, 1.8490093525334313e+01, 1.8831255107137672e+01, 1.9203114954794152e+01, 1.9622178785373663e+01, 2.0131752068311702e+01},
		},
	} {
		got := make([]float64, test.n/2+test.n%2)
		for i := range got {
			got[i] = Hermite{}.hermiteInitialGuess(i, test.n)
		}
		if !floats.EqualApprox(got, test.want, 1e-14) {
			t.Errorf("n = %v, hermite initial guesses mismatch", test.n)
		}
	}
}

func TestHermiteLocationsAsy0(t *testing.T) {
	t.Parallel()
	for _, test := range []struct {
		n int
		x []float64
		w []float64
	}{
		// Evaluated from Chebfun.
		{
			n: 215,
			x: []float64{-7.9483151672010196e-15, 1.5132632976672361e-01, 3.0266070079942864e-01, 4.5401115992078167e-01, 6.0538576507805808e-01, 7.5679259093351370e-01, 9.0823973448851325e-01, 1.0597353207530273e+00, 1.2112875084719879e+00, 1.3629044959205205e+00, 1.5145945267802365e+00, 1.6663658961089640e+00, 1.8182269564169717e+00, 1.9701861238629792e+00, 2.1222518845837395e+00, 2.2744328011717010e+00, 2.4267375193157439e+00, 2.5791747746208049e+00, 2.7317533996228827e+00, 2.8844823310168755e+00, 3.0373706171156218e+00, 3.1904274255596032e+00, 3.3436620512978501e+00, 3.4970839248619634e+00, 3.6507026209565665e+00, 3.8045278673909002e+00, 3.9585695543782120e+00, 4.1128377442311503e+00, 4.2673426814837114e+00, 4.4220948034722349e+00, 4.5771047514106780e+00, 4.7323833819978187e+00, 4.8879417795972326e+00, 5.0437912690340267e+00, 5.1999434290559226e+00, 5.3564101065103049e+00, 5.5132034312931157e+00, 5.6703358321303936e+00, 5.8278200532585673e+00, 5.9856691720754620e+00, 6.1438966178406549e+00, 6.3025161915109100e+00, 6.4615420868046280e+00, 6.6209889125982082e+00, 6.7808717167671544e+00, 6.9412060115960434e+00, 7.1020078008938672e+00, 7.2632936089653120e+00, 7.4250805116042331e+00, 7.5873861692931381e+00, 7.7502288628124383e+00, 7.9136275314855800e+00, 8.0776018143113824e+00, 8.2421720942637187e+00, 8.4073595460710955e+00, 8.5731861878255415e+00, 8.7396749368126798e+00, 8.9068496700026394e+00, 9.0747352896969389e+00, 9.2433577948895387e+00, 9.4127443589734394e+00, 9.5829234145081710e+00, 9.7539247458611111e+00, 9.9257795906487338e+00, 1.0098520751035824e+01, 1.0272182716104577e+01, 1.0446801796686490e+01, 1.0622416274262394e+01, 1.0799066565787168e+01, 1.0976795406593546e+01, 1.1155648053883823e+01, 1.1335672513742256e+01, 1.1516919795109878e+01, 1.1699444194777641e+01, 1.1883303618198463e+01, 1.2068559941826503e+01, 1.2255279423804318e+01, 1.2443533171189967e+01, 1.2633397673617059e+01, 1.2824955415404188e+01, 1.3018295580800100e+01, 1.3213514870431910e+01, 1.3410718451340038e+01, 1.3610021068540190e+01, 1.3811548353270942e+01, 1.4015438372554696e+01, 1.4221843477251845e+01, 1.4430932522616979e+01, 1.4642893558208563e+01, 1.4857937115425768e+01, 1.5076300264808333e+01, 1.5298251677453143e+01, 1.5524098014731747e+01, 1.5754192102761674e+01, 1.5988943547145137e+01, 1.6228832750513551e+01, 1.6474429782282726e+01, 1.6726420346886542e+01, 1.6985642449377284e+01, 1.7253139752773148e+01, 1.7530242081974308e+01, 1.7818692349773503e+01, 1.8120857960601569e+01, 1.8440108632764016e+01, 1.8781558404513735e+01, 1.9153731166695263e+01, 1.9573161931385755e+01, 2.0083110126223993e+01},
			w: []float64{5.8004562850365771e-04, 5.6692877486338696e-04, 5.2932867305102921e-04, 4.7211368597165944e-04, 4.0223648470876573e-04, 3.2735218319190782e-04, 2.5446548811048362e-04, 1.8892919855999243e-04, 1.3396711152850057e-04, 9.0718165504482523e-05, 5.8661199606591556e-05, 3.6218251240150691e-05, 2.1349054444552587e-05, 1.2013089019084848e-05, 6.4521031002773646e-06, 3.3071925113778075e-06, 1.6175791039359969e-06, 7.5483313427524376e-07, 3.3600239642860852e-07, 1.4264659961771696e-07, 5.7746366478382785e-08, 2.2286582090203475e-08, 8.1983021699506761e-09, 2.8738670556378732e-09, 9.5976924324201262e-10, 3.0529085215378481e-10, 9.2467927287192434e-11, 2.6660996519674056e-11, 7.3154481204350507e-12, 1.9096325566425540e-12, 4.7409073505752031e-13, 1.1189936218333804e-13, 2.5101111888653927e-14, 5.3492761412487281e-15, 1.0825895304391782e-15, 2.0798005039728529e-16, 3.7912539356931888e-17, 6.5547084049896133e-18, 1.0743122119399353e-18, 1.6684093877262400e-19, 2.4538539136228336e-20, 3.4161583900178126e-21, 4.4991409209589655e-22, 5.6023724685401444e-23, 6.5917775903350454e-24, 7.3239791710744789e-25, 7.6792588730186891e-26, 7.5931172671070367e-27, 7.0751732949679160e-28, 6.2078867832748634e-29, 5.1250608797215456e-30, 3.9778286806416675e-31, 2.9000923323091595e-32, 1.9842934651116758e-33, 1.2729744163019100e-34, 7.6493618655638258e-36, 4.3010461850159373e-37, 2.2604572504006732e-38, 1.1091714341886240e-39, 5.0753225126752125e-41, 2.1629552371814077e-42, 8.5739087326714758e-44, 3.1568681170134092e-45, 1.0780709245918455e-46, 3.4094518277810594e-48, 9.9692987408874913e-50, 2.6905726285477170e-51, 6.6902112757430557e-53, 1.5297458011420645e-54, 3.2099851519180199e-56, 6.1681886704578063e-58, 1.0829083084555836e-59, 1.7328083889009238e-61, 2.5206396804555329e-63, 3.3240965181095399e-65, 3.9623955222122382e-67, 4.2558991248868728e-69, 4.1048947183601511e-71, 3.5424776999043975e-73, 2.7245890408644923e-75, 1.8596929038788086e-77, 1.1213274238978696e-79, 5.9430256314002078e-82, 2.7536336403486887e-84, 1.1087876280230634e-86, 3.8548279808836764e-89, 1.1488445098745636e-91, 2.9118805234147757e-94, 6.2218187764965726e-97, 1.1097458975689325e-99, 1.6341809683598844e-102, 1.9621625424695376e-105, 1.8939702898325753e-108, 1.4459753551186440e-111, 8.5692594596311645e-115, 3.8565449589488849e-118, 1.2843410590864927e-121, 3.0685300780965873e-125, 5.0651104293759753e-129, 5.5131423653412833e-133, 3.7300165435945031e-137, 1.4525522232777015e-141, 2.9352196847507026e-146, 2.6587623615117695e-151, 8.6633861029598960e-157, 7.0653364942986233e-163, 7.2051505413036788e-170, 1.5514139491189097e-178},
		},
		{
			n: 216,
			x: []float64{7.5487718959414210e-02, 2.2646713089859094e-01, 3.7745846761911317e-01, 5.2846968804856509e-01, 6.7950876474356448e-01, 8.3058368937955573e-01, 9.8170247827895785e-01, 1.1328731779888348e+00, 1.2841038709195900e+00, 1.4354026810562848e+00, 1.5867777797546210e+00, 1.7382373916337388e+00, 1.8897898005787916e+00, 2.0414433558661624e+00, 2.1932064784252856e+00, 2.3450876672510903e+00, 2.4970955059821409e+00, 2.6492386696600243e+00, 2.8015259316863266e+00, 2.9539661709946619e+00, 3.1065683794558021e+00, 3.2593416695353823e+00, 3.4122952822245907e+00, 3.5654385952656149e+00, 3.7187811316950210e+00, 3.8723325687298416e+00, 4.0261027470226392e+00, 4.1801016803139959e+00, 4.3343395655126109e+00, 4.4888267932355790e+00, 4.6435739588438905e+00, 4.7985918740108131e+00, 4.9538915788638089e+00, 5.1094843547439881e+00, 5.2653817376305465e+00, 5.4215955322816969e+00, 5.5781378271479722e+00, 5.7350210101185848e+00, 5.8922577851667635e+00, 6.0498611899661094e+00, 6.2078446145563948e+00, 6.3662218211444577e+00, 6.5250069651340752e+00, 6.6842146174875827e+00, 6.8438597885318986e+00, 7.0039579533331269e+00, 7.1645250787759087e+00, 7.3255776524981444e+00, 7.4871327138470951e+00, 7.6492078870405447e+00, 7.8118214167366684e+00, 7.9749922062384426e+00, 8.1387398585838486e+00, 8.3030847208017082e+00, 8.4680479316454296e+00, 8.6336514731539875e+00, 8.7999182264315081e+00, 8.9668720320849413e+00, 9.1345377558145096e+00, 9.3029413597147883e+00, 9.4721099799172528e+00, 9.6420720112891818e+00, 9.8128572000013108e+00, 9.9844967448895385e+00, 1.0157023408668081e+01, 1.0330471640205081e+01, 1.0504877709252620e+01, 1.0680279855235202e+01, 1.0856718451952171e+01, 1.1034236190346626e+01, 1.1212878281848150e+01, 1.1392692685219812e+01, 1.1573730360348824e+01, 1.1756045553033841e+01, 1.1939696115566026e+01, 1.2124743868808109e+01, 1.2311255012587219e+01, 1.2499300592587632e+01, 1.2688957033629340e+01, 1.2880306751340244e+01, 1.3073438856897617e+01, 1.3268449972893279e+01, 1.3465445182689793e+01, 1.3664539141187824e+01, 1.3865857382137794e+01, 1.4069537866591046e+01, 1.4275732829628778e+01, 1.4484610999323769e+01, 1.4696360284715409e+01, 1.4911191060978103e+01, 1.5129340223792830e+01, 1.5351076247103551e+01, 1.5576705568204593e+01, 1.5806580756275796e+01, 1.6041111119395726e+01, 1.6280776711860355e+01, 1.6526147190139376e+01, 1.6777907762072694e+01, 1.7036895825521608e+01, 1.7304154286393636e+01, 1.7581012003064874e+01, 1.7869210618319627e+01, 1.8171115805943941e+01, 1.8490094811217439e+01, 1.8831257901839155e+01, 1.9203122650098184e+01, 1.9622211733828486e+01, 2.0131752068397390e+01},
			w: []float64{5.7408951238509774e-04, 5.4853385977693967e-04, 5.0077995633045094e-04, 4.3681926493182888e-04, 3.6404534695321319e-04, 2.8986205027358900e-04, 2.2048980967270541e-04, 1.6022189177224014e-04, 1.1121464501100074e-04, 7.3735261435177744e-05, 4.6689968632275442e-05, 2.8233496634858870e-05, 1.6302404527221912e-05, 8.9873835313609794e-06, 4.7299184092204348e-06, 2.3760322139117827e-06, 1.1391060264678187e-06, 5.2109775739704647e-07, 2.2742736124678848e-07, 9.4679280581569520e-08, 3.7589897335857270e-08, 1.4229883676778969e-08, 5.1351173480468011e-09, 1.7661093963625318e-09, 5.7875785413194057e-10, 1.8066600901602130e-10, 5.3707955985666741e-11, 1.5200567246892033e-11, 4.0945877360252466e-12, 1.0494304449814065e-12, 2.5582770078364328e-13, 5.9298335011290506e-14, 1.3064177735039720e-14, 2.7346561920969734e-15, 5.4366757055637182e-16, 1.0261135273171614e-16, 1.8378155831501011e-17, 3.1221798154524404e-18, 5.0287455392773255e-19, 7.6752848097550855e-20, 1.1095334788910946e-20, 1.5183281292392555e-21, 1.9657499371912346e-22, 2.4064442018314010e-23, 2.7838422576260478e-24, 3.0413021389695441e-25, 3.1356913424697775e-26, 3.0490587561112177e-27, 2.7941107980138270e-28, 2.4112340295373517e-29, 1.9579916032501855e-30, 1.4948591683644822e-31, 1.0720958700672810e-32, 7.2163812209007653e-34, 4.5545782039839515e-35, 2.6927171840186886e-36, 1.4896964129251313e-37, 7.7036780004272946e-39, 3.7196117693798106e-40, 1.6748563396289038e-41, 7.0241404150854973e-43, 2.7401344340734534e-44, 9.9291319468776600e-46, 3.3371619908503302e-47, 1.0387276524925886e-48, 2.9893704843803614e-50, 7.9408570659652511e-52, 1.9434716944530867e-53, 4.3740144067322860e-55, 9.0342460240718143e-57, 1.7087495134365174e-58, 2.9528859528856670e-60, 4.6509376421165100e-62, 6.6593952486160953e-64, 8.6442830921042970e-66, 1.0142381095581560e-67, 1.0722439547251424e-69, 1.0179251007077803e-71, 8.6461259693742430e-74, 6.5448969088799971e-76, 4.3965762236216239e-78, 2.6089085327841006e-80, 1.3607099985773063e-82, 6.2039892663366641e-85, 2.4580575133840935e-87, 8.4080554628125066e-90, 2.4652597109139967e-92, 6.1467455174872763e-95, 1.2918545805718045e-97, 2.2661732049781124e-100, 3.2815976681148591e-103, 3.8741015126651295e-106, 3.6760977386321105e-109, 2.7584831111443303e-112, 1.6064052012003602e-115, 7.1024252453209961e-119, 2.3230796238780657e-122, 5.4494213912665180e-126, 8.8284366174946482e-130, 9.4271378060446301e-134, 6.2539251873936211e-138, 2.3865140957941506e-142, 4.7220323342515999e-147, 4.1840743138172546e-152, 1.3319161291512504e-157, 1.0592510752538756e-163, 1.0503755670967946e-170, 2.1869674690632971e-179},
		},
	} {
		n := test.n
		x := make([]float64, n/2+n%2)
		w := make([]float64, n/2+n%2)
		for i := range x {
			x[i], w[i] = Hermite{}.locationsAsy0(i, n)
		}
		if !floats.EqualApprox(x, test.x, 1e-13) {
			t.Errorf("n = %v, x mismatch", n)
			fmt.Println(x[107])
			fmt.Println(test.x[107])
		}
		if !floats.EqualApprox(w, test.w, 1e-13) {
			t.Errorf("n = %v, w mismatch", n)
		}
	}
}

func TestHermiteAsyAiry(t *testing.T) {
	t.Parallel()
	for _, test := range []struct {
		n     int
		theta []float64
		val   []float64
		dval  []float64
	}{
		{
			n:     215,
			theta: []float64{1.5707963267948966e+00, 1.5635071294267224e+00, 1.5562171573546757e+00, 1.5489256352159648e+00, 1.5416317863284521e+00, 1.5343348320272066e+00, 1.5270339909965103e+00, 1.5197284785957685e+00, 1.5124175061777518e+00, 1.5051002803975593e+00, 1.4977760025106577e+00, 1.4904438676582881e+00, 1.4831030641384946e+00, 1.4757527726609410e+00, 1.4683921655836265e+00, 1.4610204061295120e+00, 1.4536366475809854e+00, 1.4462400324499760e+00, 1.4388296916214274e+00, 1.4314047434676882e+00, 1.4239642929312548e+00, 1.4165074305731233e+00, 1.4090332315838545e+00, 1.4015407547542300e+00, 1.3940290414022056e+00, 1.3864971142525955e+00, 1.3789439762656834e+00, 1.3713686094106747e+00, 1.3637699733795743e+00, 1.3561470042367485e+00, 1.3484986129990375e+00, 1.3408236841408716e+00, 1.3331210740183812e+00, 1.3253896092059803e+00, 1.3176280847383373e+00, 1.3098352622500289e+00, 1.3020098680044709e+00, 1.2941505908029671e+00, 1.2862560797638511e+00, 1.2783249419607658e+00, 1.2703557399080556e+00, 1.2623469888800907e+00, 1.2542971540500127e+00, 1.2462046474319508e+00, 1.2380678246090893e+00, 1.2298849812281600e+00, 1.2216543492388388e+00, 1.2133740928542220e+00, 1.2050423042059359e+00, 1.1966569986644591e+00, 1.1882161097919153e+00, 1.1797174838907749e+00, 1.1711588741076131e+00, 1.1625379340461441e+00, 1.1538522108381697e+00, 1.1450991376146524e+00, 1.1362760253118058e+00, 1.1273800537386238e+00, 1.1184082618225748e+00, 1.1093575369389341e+00, 1.1002246032162142e+00, 1.0910060086950395e+00, 1.0816981112001869e+00, 1.0722970627649082e+00, 1.0627987924225111e+00, 1.0531989871517369e+00, 1.0434930707289145e+00, 1.0336761802000907e+00, 1.0237431396389916e+00, 1.0136884308001370e+00, 1.0035061602085740e+00, 9.9319002214595331e-01, 9.8273325689369295e-01, 9.7212860347355579e-01, 9.6136824597870241e-01, 9.5044375240720391e-01, 9.3934600468605900e-01, 9.2806511829511895e-01, 9.1659034955139063e-01, 9.0490998817422319e-01, 8.9301123219309975e-01, 8.8088004154453947e-01, 8.6850096578161351e-01, 8.5585694011780433e-01, 8.4292904244735056e-01, 8.2969620188675874e-01, 8.1613484656631152e-01, 8.0221847457494289e-01, 7.8791712669745728e-01, 7.7319673223616503e-01, 7.5801828879746735e-01, 7.4233682191908168e-01, 7.2610004838413222e-01, 7.0924663405346811e-01, 6.9170388642137270e-01, 6.7338464244284535e-01, 6.5418298314185364e-01, 6.3396819041411068e-01, 6.1257598542667102e-01, 5.8979540387821172e-01, 5.6534835113739590e-01, 5.3885619819698471e-01, 5.0978185232293227e-01, 4.7732127920035572e-01, 4.4017830784988765e-01, 3.9602307074600884e-01, 3.3985069280702440e-01, 2.5624520569356829e-01},
			val:   []float64{-2.7524590607067879e-13, -1.2484792737707130e-10, 2.5043777255407545e-10, -3.7621487726711831e-10, 5.0266571286479591e-10, -6.3022107051014496e-10, 7.5847898514904004e-10, -8.8850597267422281e-10, 1.0201287099545048e-09, -1.1532700149758681e-09, 1.2885402426941860e-09, -1.4262514564750722e-09, 1.5667774694950304e-09, -1.7098460348004283e-09, 1.8563663242006209e-09, -2.0060430373954962e-09, 2.1597272205910619e-09, -2.3174687646872261e-09, 2.4795270439737100e-09, -2.6462815453642877e-09, 2.8188431716836157e-09, -2.9962948915230151e-09, 3.1805879826447352e-09, -3.3709772507788239e-09, 3.5681573894354513e-09, -3.7732681484183285e-09, 3.9863813736176932e-09, -4.2080297462347532e-09, 4.4391671460302399e-09, -4.6802436355831232e-09, 4.9322546933533433e-09, -5.1957881449299223e-09, 5.4715433664057677e-09, -5.7608265327475837e-09, 6.0644153161437157e-09, -6.3835515563504339e-09, 6.7192819880764795e-09, -7.0729371214636086e-09, 7.4458702579674617e-09, -7.8396410076078555e-09, 8.2556088536449877e-09, -8.6960989460562939e-09, 9.1626310723815202e-09, -9.6575986146054327e-09, 1.0182896152806950e-08, -1.0741758184989815e-08, 1.1337214303287172e-08, -1.1971481080324528e-08, 1.2648762056240369e-08, -1.3372488830525768e-08, 1.4147041859097387e-08, -1.4977276046958432e-08, 1.5868475891599271e-08, -1.6825823997480545e-08, 1.7856900524878266e-08, -1.8967833880184117e-08, 2.0167708166421882e-08, -2.1465280681285451e-08, 2.2870631970634630e-08, -2.4395881051717265e-08, 2.6054060647767539e-08, -2.7859818527100110e-08, 2.9830454146699167e-08, -3.1984835487787522e-08, 3.4345470368342559e-08, -3.6937900081988413e-08, 3.9790883510012329e-08, -4.2938418365980210e-08, 4.6419377177480922e-08, -5.0279213559974237e-08, 5.4570799296449321e-08, -5.9356303125081130e-08, 6.4708755363671345e-08, -7.0714511032393380e-08, 7.7475892426711618e-08, -8.5115456468135130e-08, 9.3779863576446978e-08, -1.0364644483987255e-07, 1.1493006182865740e-07, -1.2789317815470753e-07, 1.4285900893044798e-07, -1.6022759919461249e-07, 1.8049814080062288e-07, -2.0429953256541455e-07, 2.3243044535889061e-07, -2.6591504027157263e-07, 3.0608147425737922e-07, -3.5467098673335754e-07, 4.1399606056878556e-07, -4.8716979156405424e-07, 5.7844824292016844e-07, -6.9374768843585542e-07, 8.4144962449118331e-07, -1.0336795693201387e-06, 1.2884020459910342e-06, -1.6329629222918408e-06, 2.1103031286712709e-06, -2.7903321871954755e-06, 3.7918023218017778e-06, -5.3269144947324421e-06, 7.7989206456257246e-06, -1.2034979469568085e-05, 1.9905658561076196e-05, -3.6224899321604820e-05, 7.5828952282754516e-05, -1.9870421394114517e-04, 7.9136577883937422e-04, -9.1266148556594703e-03},
			dval:  []float64{-2.9359856130945122e+01, 2.9359466144908318e+01, -2.9358296010652477e+01, 2.9356345199427643e+01, -2.9353612829040124e+01, 2.9350097662433203e+01, -2.9345798105693621e+01, 2.9340712205476308e+01, -2.9334837645838910e+01, 2.9328171744474840e+01, -2.9320711448331572e+01, 2.9312453328598885e+01, -2.9303393575049100e+01, 2.9293527989709109e+01, -2.9282851979841201e+01, 2.9271360550207458e+01, -2.9259048294588819e+01, 2.9245909386527540e+01, -2.9231937569258221e+01, 2.9217126144788914e+01, -2.9201467962090302e+01, 2.9184955404346876e+01, -2.9167580375219231e+01, 2.9149334284062498e+01, -2.9130208030040425e+01, 2.9110191985068752e+01, -2.9089275975516287e+01, 2.9067449262584436e+01, -2.9044700521279459e+01, 2.9021017817883425e+01, -2.8996388585821325e+01, 2.8970799599811933e+01, -2.8944236948180023e+01, 2.8916686003195348e+01, -2.8888131389291605e+01, 2.8858556949003951e+01, -2.8827945706448649e+01, 2.8796279828150279e+01, -2.8763540581003511e+01, 2.8729708287134507e+01, -2.8694762275403512e+01, 2.8658680829263385e+01, -2.8621441130659093e+01, 2.8583019199620328e+01, -2.8543389829161509e+01, 2.8502526515061927e+01, -2.8460401380051497e+01, 2.8416985091874277e+01, -2.8372246774642264e+01, 2.8326153912823031e+01, -2.8278672247129006e+01, 2.8229765661486788e+01, -2.8179396060166667e+01, 2.8127523234037287e+01, -2.8074104714780372e+01, 2.8019095615750665e+01, -2.7962448457994284e+01, 2.7904112979739956e+01, -2.7844035927448619e+01, 2.7782160826241125e+01, -2.7718427727214667e+01, 2.7652772928798932e+01, -2.7585128668881747e+01, 2.7515422783940370e+01, -2.7443578330833645e+01, 2.7369513166223737e+01, -2.7293139477783004e+01, 2.7214363260373997e+01, -2.7133083729234610e+01, 2.7049192660814331e+01, -2.6962573650237655e+01, 2.6873101272350414e+01, -2.6780640130847800e+01, 2.6685043776981303e+01, -2.6586153475654008e+01, 2.6483796792158788e+01, -2.6377785967154090e+01, 2.6267916040395331e+01, -2.6153962674834343e+01, 2.6035679621411951e+01, -2.5912795750458177e+01, 2.5785011557068390e+01, -2.5651995023754676e+01, 2.5513376692151432e+01, -2.5368743753874771e+01, 2.5217632914945654e+01, -2.5059521712941240e+01, 2.4893817863127197e+01, -2.4719846067215368e+01, 2.4536831517922174e+01, -2.4343879046260469e+01, 2.4139946442705973e+01, -2.3923809867871942e+01, 2.3694018337866666e+01, -2.3448832829666316e+01, 2.3186143264519330e+01, -2.2903352884396824e+01, 2.2597213200674400e+01, -2.2263581543467414e+01, 2.1897052703993687e+01, -2.1490376258364485e+01, 2.1033488463000687e+01, -2.0511802124454178e+01, 1.9902938182242686e+01, -1.9169784832381577e+01, 1.8243381598901596e+01, -1.6969663168974311e+01, 1.4853826851969115e+01},
		},
		{
			n:     215,
			theta: []float64{1.5707963267948968e+00, 1.5635071294265777e+00, 1.5562171573543850e+00, 1.5489256352155283e+00, 1.5416317863278686e+00, 1.5343348320264747e+00, 1.5270339909956292e+00, 1.5197284785947358e+00, 1.5124175061765655e+00, 1.5051002803962170e+00, 1.4977760025091569e+00, 1.4904438676566254e+00, 1.4831030641366665e+00, 1.4757527726589439e+00, 1.4683921655814560e+00, 1.4610204061271637e+00, 1.4536366475784539e+00, 1.4462400324472560e+00, 1.4388296916185130e+00, 1.4314047434645731e+00, 1.4239642929279313e+00, 1.4165074305695844e+00, 1.4090332315800913e+00, 1.4015407547502341e+00, 1.3940290413979675e+00, 1.3864971142481046e+00, 1.3789439762609286e+00, 1.3713686094056443e+00, 1.3637699733742550e+00, 1.3561470042311266e+00, 1.3484986129930978e+00, 1.3408236841345980e+00, 1.3331210740117563e+00, 1.3253896091989852e+00, 1.3176280847309516e+00, 1.3098352622422307e+00, 1.3020098679962364e+00, 1.2941505907942705e+00, 1.2862560797546647e+00, 1.2783249419510594e+00, 1.2703557398977969e+00, 1.2623469888692436e+00, 1.2542971540385390e+00, 1.2462046474198085e+00, 1.2380678245962331e+00, 1.2298849812145400e+00, 1.2216543492243996e+00, 1.2133740928389052e+00, 1.2050423041896758e+00, 1.1966569986471847e+00, 1.1882161097735480e+00, 1.1797174838712285e+00, 1.1711588740867924e+00, 1.1625379340239450e+00, 1.1538522108144755e+00, 1.1450991375893356e+00, 1.1362760252847235e+00, 1.1273800537096179e+00, 1.1184082617914692e+00, 1.1093575369055320e+00, 1.1002246031802951e+00, 1.0910060086563567e+00, 1.0816981111584625e+00, 1.0722970627198296e+00, 1.0627987923737243e+00, 1.0531989870988405e+00, 1.0434930706714527e+00, 1.0336761801375436e+00, 1.0237431395707643e+00, 1.0136884307255469e+00, 1.0035061601268345e+00, 9.9319002205615470e-01, 9.8273325679477930e-01, 9.7212860336429585e-01, 9.6136824585765468e-01, 9.5044375227267230e-01, 9.3934600453603823e-01, 9.2806511812722514e-01, 9.1659034936277473e-01, 9.0490998796145916e-01, 8.9301123195204113e-01, 8.8088004127013353e-01, 8.6850096546765831e-01, 8.5585693975662547e-01, 8.4292904202937069e-01, 8.2969620139991496e-01, 8.1613484599525510e-01, 8.0221847389992995e-01, 7.8791712589278218e-01, 7.7319673126791522e-01, 7.5801828762024448e-01, 7.4233682047111893e-01, 7.2610004657985505e-01, 7.0924663177187774e-01, 6.9170388348738876e-01, 6.7338463859636577e-01, 6.5418297798453895e-01, 6.3396818331389482e-01, 6.1257597533770680e-01, 5.8979538898074346e-01, 5.6534832806423485e-01, 5.3885616021897675e-01, 5.0978178458826806e-01, 4.7732114425935024e-01, 4.4017799165717791e-01, 3.9602210904603868e-01, 3.3984592789666196e-01, 2.5616263514373872e-01},
			val:   []float64{-2.7524577037687927e-13, -1.3157296144588393e-13, -1.4217705189589319e-13, -9.3051784272423659e-14, 6.2263913433904649e-14, -7.2338165854311426e-14, 1.6489823942055246e-13, 8.2086154015730068e-14, 1.7030857773598118e-13, 1.9200825273544764e-13, 3.4144423192398587e-13, -1.3350292964848512e-13, 7.7406393378692183e-16, 3.0802286839980475e-15, 1.0626665274130983e-13, 9.7508116816654928e-14, -1.2286283979350783e-13, 1.6602061523803380e-13, 1.2145800400612989e-14, -2.0125760394077007e-13, 1.1752853222876857e-13, 1.5246361311903834e-13, -1.7100196342840500e-13, -9.0852635256292980e-14, -7.8101776477326909e-14, -8.6215405399587178e-14, -1.8424041061507494e-13, -1.7748959370373156e-13, 1.6659488983250428e-14, -2.3029664495695825e-14, 4.9882044749360307e-14, -1.6935708059066857e-13, -1.6889419531778083e-14, -1.5488691966202885e-13, 4.7496471682336465e-15, -6.0236075531674274e-14, -4.7790439973748918e-14, 1.0528243973915236e-14, -1.4481370116381506e-13, 4.8870674172048881e-14, 5.9145722182844837e-14, -3.6252017525118028e-14, -1.4799821290649442e-13, 7.8483748114372226e-14, -3.1698431394015554e-14, -2.1250000650523357e-13, -1.2207160366356720e-13, -4.4799710774576449e-14, -6.6076225108040054e-14, 4.9432907671096125e-14, 1.1714372287574581e-13, 7.1071624668919607e-14, -1.4699620399669987e-13, 4.9028118279404591e-14, 5.6995970859340297e-15, 1.8655926516135062e-14, 3.2409175615798361e-14, 7.1510069979380908e-14, -7.2558225742592175e-14, 4.1294505314113302e-14, -1.8304081875904709e-14, 1.3492357431194347e-13, -2.0526864648283952e-13, 1.1949578558177085e-14, 5.3441698191945163e-14, 4.9207462229933007e-14, -1.9962117626180037e-13, 8.2375100050737861e-15, 5.8811770905000782e-14, -3.8856969371752553e-15, 4.8345877891596704e-14, -1.2020353849039574e-13, 1.0795889278969854e-14, 2.7165003590955340e-14, 7.9664181703224841e-15, -3.1357633506603503e-14, 3.0114465704049404e-14, 5.4841545026050259e-14, 8.3356466046602926e-14, 3.4875109219720919e-14, 5.3887688890067731e-14, -8.5090430827758076e-15, 1.4753806570541977e-14, 4.1314792687095403e-15, 2.8114340045724569e-14, -8.9380771455156681e-15, -3.2451925819727029e-14, 5.0739464388639648e-14, 5.2751575373088884e-14, 3.8348847777085630e-14, -3.1825724566756715e-14, 6.6426259272244758e-14, -6.4295491511138863e-14, 8.9401622475124014e-15, 2.5866873030708164e-14, 9.6302605421508054e-14, 4.6623114840341440e-14, -1.7497072577375209e-14, 3.3453526916664849e-14, -5.5592329550244261e-14, 1.4115683421722400e-13, -3.9306913138151204e-13, 1.2007938611150981e-12, -4.7530501604406474e-12, 2.5399685303214922e-11, -2.2827587443604869e-10, 5.3187001008179367e-09, -1.4222383864770318e-06},
			dval:  []float64{-2.9359856130945126e+01, 2.9359466144908307e+01, -2.9358296010652474e+01, 2.9356345199427643e+01, -2.9353612829040117e+01, 2.9350097662433203e+01, -2.9345798105693628e+01, 2.9340712205476311e+01, -2.9334837645838906e+01, 2.9328171744474847e+01, -2.9320711448331569e+01, 2.9312453328598881e+01, -2.9303393575049114e+01, 2.9293527989709105e+01, -2.9282851979841208e+01, 2.9271360550207465e+01, -2.9259048294588812e+01, 2.9245909386527540e+01, -2.9231937569258211e+01, 2.9217126144788910e+01, -2.9201467962090309e+01, 2.9184955404346905e+01, -2.9167580375219234e+01, 2.9149334284062515e+01, -2.9130208030040418e+01, 2.9110191985068738e+01, -2.9089275975516284e+01, 2.9067449262584432e+01, -2.9044700521279456e+01, 2.9021017817883422e+01, -2.8996388585821315e+01, 2.8970799599811926e+01, -2.8944236948180027e+01, 2.8916686003195355e+01, -2.8888131389291594e+01, 2.8858556949003940e+01, -2.8827945706448645e+01, 2.8796279828150276e+01, -2.8763540581003515e+01, 2.8729708287134510e+01, -2.8694762275403530e+01, 2.8658680829263389e+01, -2.8621441130659079e+01, 2.8583019199620328e+01, -2.8543389829161509e+01, 2.8502526515061930e+01, -2.8460401380051490e+01, 2.8416985091874288e+01, -2.8372246774642264e+01, 2.8326153912823038e+01, -2.8278672247129016e+01, 2.8229765661486780e+01, -2.8179396060166667e+01, 2.8127523234037294e+01, -2.8074104714780368e+01, 2.8019095615750661e+01, -2.7962448457994284e+01, 2.7904112979739949e+01, -2.7844035927448619e+01, 2.7782160826241114e+01, -2.7718427727214678e+01, 2.7652772928798946e+01, -2.7585128668881751e+01, 2.7515422783940377e+01, -2.7443578330833653e+01, 2.7369513166223744e+01, -2.7293139477783015e+01, 2.7214363260374000e+01, -2.7133083729234624e+01, 2.7049192660814349e+01, -2.6962573650237655e+01, 2.6873101272350418e+01, -2.6780640130847814e+01, 2.6685043776981328e+01, -2.6586153475654026e+01, 2.6483796792158806e+01, -2.6377785967154097e+01, 2.6267916040395356e+01, -2.6153962674834389e+01, 2.6035679621411997e+01, -2.5912795750458240e+01, 2.5785011557068451e+01, -2.5651995023754750e+01, 2.5513376692151542e+01, -2.5368743753874902e+01, 2.5217632914945817e+01, -2.5059521712941454e+01, 2.4893817863127474e+01, -2.4719846067215741e+01, 2.4536831517922678e+01, -2.4343879046261161e+01, 2.4139946442706947e+01, -2.3923809867873366e+01, 2.3694018337868730e+01, -2.3448832829669414e+01, 2.3186143264524166e+01, -2.2903352884404576e+01, 2.2597213200687420e+01, -2.2263581543490410e+01, 2.1897052704036881e+01, -2.1490376258452002e+01, 2.1033488463196051e+01, -2.0511802124949831e+01, 1.9902938183741984e+01, -1.9169784838249630e+01, 1.8243381633602635e+01, -1.6969663610835688e+01, 1.4853865657323199e+01},
		},
		{
			n:     215,
			theta: []float64{1.5707963267948970e+00, 1.5635071294265774e+00, 1.5562171573543853e+00, 1.5489256352155283e+00, 1.5416317863278686e+00, 1.5343348320264747e+00, 1.5270339909956290e+00, 1.5197284785947358e+00, 1.5124175061765652e+00, 1.5051002803962172e+00, 1.4977760025091564e+00, 1.4904438676566252e+00, 1.4831030641366665e+00, 1.4757527726589439e+00, 1.4683921655814558e+00, 1.4610204061271639e+00, 1.4536366475784541e+00, 1.4462400324472562e+00, 1.4388296916185130e+00, 1.4314047434645729e+00, 1.4239642929279310e+00, 1.4165074305695846e+00, 1.4090332315800915e+00, 1.4015407547502341e+00, 1.3940290413979675e+00, 1.3864971142481046e+00, 1.3789439762609288e+00, 1.3713686094056441e+00, 1.3637699733742550e+00, 1.3561470042311266e+00, 1.3484986129930978e+00, 1.3408236841345977e+00, 1.3331210740117563e+00, 1.3253896091989850e+00, 1.3176280847309516e+00, 1.3098352622422307e+00, 1.3020098679962364e+00, 1.2941505907942705e+00, 1.2862560797546649e+00, 1.2783249419510594e+00, 1.2703557398977969e+00, 1.2623469888692436e+00, 1.2542971540385393e+00, 1.2462046474198085e+00, 1.2380678245962331e+00, 1.2298849812145398e+00, 1.2216543492243999e+00, 1.2133740928389052e+00, 1.2050423041896758e+00, 1.1966569986471847e+00, 1.1882161097735477e+00, 1.1797174838712285e+00, 1.1711588740867926e+00, 1.1625379340239450e+00, 1.1538522108144755e+00, 1.1450991375893356e+00, 1.1362760252847235e+00, 1.1273800537096179e+00, 1.1184082617914692e+00, 1.1093575369055320e+00, 1.1002246031802951e+00, 1.0910060086563569e+00, 1.0816981111584627e+00, 1.0722970627198296e+00, 1.0627987923737243e+00, 1.0531989870988405e+00, 1.0434930706714529e+00, 1.0336761801375436e+00, 1.0237431395707643e+00, 1.0136884307255469e+00, 1.0035061601268345e+00, 9.9319002205615448e-01, 9.8273325679477930e-01, 9.7212860336429585e-01, 9.6136824585765468e-01, 9.5044375227267230e-01, 9.3934600453603823e-01, 9.2806511812722525e-01, 9.1659034936277461e-01, 9.0490998796145927e-01, 8.9301123195204102e-01, 8.8088004127013353e-01, 8.6850096546765831e-01, 8.5585693975662547e-01, 8.4292904202937069e-01, 8.2969620139991496e-01, 8.1613484599525521e-01, 8.0221847389993006e-01, 7.8791712589278207e-01, 7.7319673126791533e-01, 7.5801828762024459e-01, 7.4233682047111904e-01, 7.2610004657985516e-01, 7.0924663177187774e-01, 6.9170388348738865e-01, 6.7338463859636599e-01, 6.5418297798453884e-01, 6.3396818331389482e-01, 6.1257597533770669e-01, 5.8979538898074335e-01, 5.6534832806423441e-01, 5.3885616021897553e-01, 5.0978178458826395e-01, 4.7732114425933253e-01, 4.4017799165707200e-01, 3.9602210904493385e-01, 3.3984592786463697e-01, 2.5616262227240338e-01},
			val:   []float64{-4.8194502666337268e-14, -1.8219292956257052e-14, 8.5393034645390052e-14, -9.3051784272423659e-14, 6.2263913433904649e-14, -7.2338165854311426e-14, -6.2580913476718581e-14, 8.2086154015730068e-14, -5.7165830534287467e-14, -3.5936383892293965e-14, -1.1380084519138131e-13, 9.4176333524136458e-14, 7.7406393378692183e-16, 3.0802286839980475e-15, -2.3573275472589349e-13, -1.3018327378045775e-13, 1.0520206501483371e-13, -6.2213644291038904e-14, 1.2145800400612989e-14, 2.7271297683652584e-14, -2.2535922871268711e-13, -7.6298928186738198e-14, 1.7255690675064885e-13, -9.0852635256292980e-14, -7.8101776477326909e-14, -8.6215405399587178e-14, 4.5189259409156241e-14, 1.6773935900994068e-13, 1.6659488983250428e-14, -2.3029664495695825e-14, 4.9882044749360307e-14, 1.1850055135335312e-13, -1.6889419531778083e-14, 7.5802404690310745e-14, 4.7496471682336465e-15, -6.0236075531674274e-14, -4.7790439973748918e-14, 1.0528243973915236e-14, 2.0302122716337677e-13, 4.8870674172048881e-14, 5.9145722182844837e-14, -3.6252017525118028e-14, 2.6839385139099395e-14, 7.8483748114372226e-14, -3.1698431394015554e-14, -3.7141479062747567e-14, 1.1275835568784933e-13, -4.4799710774576449e-14, -6.6076225108040054e-14, 4.9432907671096125e-14, -1.1915508175098585e-13, 7.1071624668919607e-14, 1.4874359999295257e-13, 4.9028118279404591e-14, 5.6995970859340297e-15, 1.8655926516135062e-14, 3.2409175615798361e-14, 7.1510069979380908e-14, -7.2558225742592175e-14, 4.1294505314113302e-14, -1.8304081875904709e-14, -4.6448090671875563e-14, 9.6949756857760571e-14, 1.1949578558177085e-14, 5.3441698191945163e-14, 4.9207462229933007e-14, -1.6005168277506475e-14, 8.2375100050737861e-15, 5.8811770905000782e-14, -3.8856969371752553e-15, 4.8345877891596704e-14, 9.6875037765842389e-14, 1.0795889278969854e-14, 2.7165003590955340e-14, 7.9664181703224841e-15, -3.1357633506603503e-14, 3.0114465704049404e-14, 5.4841606599486569e-14, 1.9665914005907120e-14, -2.8849555274550242e-14, -4.2752529356173103e-14, -8.5090430827758076e-15, 1.4753806570541977e-14, 4.1314792687095403e-15, 2.8114340045724569e-14, -8.9380771455156681e-15, 3.6308713608736668e-16, -4.9851972317585696e-14, -9.9115770904832069e-14, -1.2903366624947848e-14, 5.3938844852967152e-14, -3.7521135648997941e-14, 4.0086325460559029e-14, 8.9401622475124014e-15, -9.8967441436620103e-14, -6.5207531930586132e-14, 4.6622484011593159e-14, -1.7497072577375209e-14, 3.3453526916664849e-14, 4.8715471045751314e-14, 3.3943255654417215e-14, 5.7122398702729805e-15, -9.4574045519538209e-16, 7.2999684957756045e-15, 2.7503783126358897e-14, -7.8728766754343666e-15, 3.5695346037194369e-14, -6.2144904069338325e-14},
			dval:  []float64{-2.9359856130945122e+01, 2.9359466144908321e+01, -2.9358296010652467e+01, 2.9356345199427643e+01, -2.9353612829040117e+01, 2.9350097662433203e+01, -2.9345798105693628e+01, 2.9340712205476311e+01, -2.9334837645838913e+01, 2.9328171744474844e+01, -2.9320711448331572e+01, 2.9312453328598878e+01, -2.9303393575049114e+01, 2.9293527989709105e+01, -2.9282851979841219e+01, 2.9271360550207469e+01, -2.9259048294588819e+01, 2.9245909386527547e+01, -2.9231937569258211e+01, 2.9217126144788914e+01, -2.9201467962090305e+01, 2.9184955404346880e+01, -2.9167580375219234e+01, 2.9149334284062515e+01, -2.9130208030040418e+01, 2.9110191985068738e+01, -2.9089275975516287e+01, 2.9067449262584439e+01, -2.9044700521279456e+01, 2.9021017817883422e+01, -2.8996388585821315e+01, 2.8970799599811929e+01, -2.8944236948180027e+01, 2.8916686003195352e+01, -2.8888131389291594e+01, 2.8858556949003940e+01, -2.8827945706448645e+01, 2.8796279828150276e+01, -2.8763540581003500e+01, 2.8729708287134510e+01, -2.8694762275403530e+01, 2.8658680829263389e+01, -2.8621441130659086e+01, 2.8583019199620328e+01, -2.8543389829161509e+01, 2.8502526515061934e+01, -2.8460401380051490e+01, 2.8416985091874288e+01, -2.8372246774642264e+01, 2.8326153912823038e+01, -2.8278672247129023e+01, 2.8229765661486780e+01, -2.8179396060166667e+01, 2.8127523234037294e+01, -2.8074104714780368e+01, 2.8019095615750661e+01, -2.7962448457994284e+01, 2.7904112979739949e+01, -2.7844035927448619e+01, 2.7782160826241114e+01, -2.7718427727214678e+01, 2.7652772928798939e+01, -2.7585128668881751e+01, 2.7515422783940377e+01, -2.7443578330833653e+01, 2.7369513166223744e+01, -2.7293139477783015e+01, 2.7214363260374000e+01, -2.7133083729234624e+01, 2.7049192660814349e+01, -2.6962573650237655e+01, 2.6873101272350418e+01, -2.6780640130847814e+01, 2.6685043776981328e+01, -2.6586153475654026e+01, 2.6483796792158806e+01, -2.6377785967154097e+01, 2.6267916040395356e+01, -2.6153962674834375e+01, 2.6035679621411987e+01, -2.5912795750458244e+01, 2.5785011557068451e+01, -2.5651995023754750e+01, 2.5513376692151542e+01, -2.5368743753874902e+01, 2.5217632914945817e+01, -2.5059521712941446e+01, 2.4893817863127481e+01, -2.4719846067215748e+01, 2.4536831517922678e+01, -2.4343879046261172e+01, 2.4139946442706954e+01, -2.3923809867873342e+01, 2.3694018337868730e+01, -2.3448832829669424e+01, 2.3186143264524159e+01, -2.2903352884404576e+01, 2.2597213200687420e+01, -2.2263581543490410e+01, 2.1897052704036888e+01, -2.1490376258451999e+01, 2.1033488463196083e+01, -2.0511802124949803e+01, 1.9902938183742044e+01, -1.9169784838249644e+01, 1.8243381633602628e+01, -1.6969663610835667e+01, 1.4853865657324123e+01},
		},
		{
			n:     216,
			theta: []float64{1.5671686105755729e+00, 1.5599127961686350e+00, 1.5526558355349958e+00, 1.5453969634500344e+00, 1.5381354130763418e+00, 1.5308704153134880e+00, 1.5236011981426352e+00, 1.5163269859644990e+00, 1.5090469989291080e+00, 1.5017604522557937e+00, 1.4944665555417909e+00, 1.4871645120577743e+00, 1.4798535180286077e+00, 1.4725327618975033e+00, 1.4652014235717277e+00, 1.4578586736478822e+00, 1.4505036726147191e+00, 1.4431355700313173e+00, 1.4357535036783504e+00, 1.4283565986800277e+00, 1.4209439665941588e+00, 1.4135147044676193e+00, 1.4060678938543345e+00, 1.3986025997926903e+00, 1.3911178697390723e+00, 1.3836127324540022e+00, 1.3760861968370792e+00, 1.3685372507066480e+00, 1.3609648595198076e+00, 1.3533679650280226e+00, 1.3457454838632317e+00, 1.3380963060489126e+00, 1.3304192934301153e+00, 1.3227132780159603e+00, 1.3149770602275335e+00, 1.3072094070434859e+00, 1.2994090500349611e+00, 1.2915746832807011e+00, 1.2837049611523335e+00, 1.2757984959588993e+00, 1.2678538554386305e+00, 1.2598695600848058e+00, 1.2518440802912205e+00, 1.2437758333013242e+00, 1.2356631799434625e+00, 1.2275044211328083e+00, 1.2192977941185159e+00, 1.2110414684522999e+00, 1.2027335416520410e+00, 1.1943720345310531e+00, 1.1859548861603062e+00, 1.1774799484271210e+00, 1.1689449801495315e+00, 1.1603476407006212e+00, 1.1516854830915373e+00, 1.1429559464555037e+00, 1.1341563478678038e+00, 1.1252838734282915e+00, 1.1163355685232694e+00, 1.1073083271723616e+00, 1.0981988803530085e+00, 1.0890037831801158e+00, 1.0797194008007944e+00, 1.0703418928435628e+00, 1.0608671962372638e+00, 1.0512910061865695e+00, 1.0416087550574320e+00, 1.0318155888861182e+00, 1.0219063411782010e+00, 1.0118755036074250e+00, 1.0017171931566209e+00, 9.9142511516121901e-01, 9.8099252161709072e-01, 9.7041216399420693e-01, 9.5967623965057169e-01, 9.4877633076008561e-01, 9.3770333444439402e-01, 9.2644738252057113e-01, 9.1499774892808761e-01, 9.0334274245922008e-01, 8.9146958185913860e-01, 8.7936424964779292e-01, 8.6701132009412707e-01, 8.5439375557323693e-01, 8.4149266395991695e-01, 8.2828700761781482e-01, 8.1475325173188318e-01, 8.0086493591225205e-01, 7.8659214774134534e-01, 7.7190086960046345e-01, 7.5675215971653753e-01, 7.4110111338801099e-01, 7.2489552835323978e-01, 7.0807416530075917e-01, 6.9056444397251160e-01, 6.7227933577676646e-01, 6.5311308498739518e-01, 6.3293517484306561e-01, 6.1158157942775537e-01, 5.8884165912926645e-01, 5.6443774724725360e-01, 5.3799179737419422e-01, 5.0896754324534177e-01, 4.7656218794974137e-01, 4.3948155571863207e-01, 3.9539936631707273e-01, 3.3931845258954985e-01, 2.5576413756997579e-01},
			val:   []float64{6.1063360924047490e-11, -1.8435203658490556e-10, 3.0743162265786900e-10, -4.3161999307057153e-10, 5.5600184481941261e-10, -6.8154103751235407e-10, 8.0806640731641332e-10, -9.3648670703302450e-10, 1.0665511846836497e-09, -1.1984227187029326e-09, 1.3320294714997634e-09, -1.4683350272554789e-09, 1.6073141540400390e-09, -1.7488108299475996e-09, 1.8942373170611600e-09, -2.0425637634202854e-09, 2.1952560444756934e-09, -2.3517907718469464e-09, 2.5131243047743498e-09, -2.6788312866718815e-09, 2.8501736508238295e-09, -3.0272882762354559e-09, 3.2101466593235099e-09, -3.3995591281295402e-09, 3.5962230906180571e-09, -3.8005765479605162e-09, 4.0130477356611703e-09, -4.2339156669326267e-09, 4.4646349512119503e-09, -4.7049927091960580e-09, 4.9561002253847192e-09, -5.2191334055490069e-09, 5.4945583626542044e-09, -5.7832433193687254e-09, 6.0863314853460021e-09, -6.4051174909280434e-09, 6.7403553891566650e-09, -7.0933989938263408e-09, 7.4659234299589640e-09, -7.8591408458401506e-09, 8.2751055799843569e-09, -8.7148794606644850e-09, 9.1813301061361439e-09, -9.6760514978089342e-09, 1.0201632305616356e-08, -1.0760129925294149e-08, 1.1355163586751651e-08, -1.1989527720947381e-08, 1.2666360398983932e-08, -1.3390280463762196e-08, 1.4164961106968852e-08, -1.4995012747245716e-08, 1.5885976564229918e-08, -1.6843921766929025e-08, 1.7874947115661739e-08, -1.8986267082409428e-08, 2.0186191955347422e-08, -2.1484132370733845e-08, 2.2890126360750540e-08, -2.4415809556224372e-08, 2.6074391376958674e-08, -2.7880720950296676e-08, 2.9852038341333996e-08, -3.2007470796983141e-08, 3.4369209413828328e-08, -3.6962720963996261e-08, 3.9816961736227733e-08, -4.2965941232291533e-08, 4.6448638367911141e-08, -5.0310249303133202e-08, 5.4604204622873211e-08, -5.9392177920227513e-08, 6.4747522871256546e-08, -7.0756457006013807e-08, 7.7521575966439530e-08, -8.5165469547851754e-08, 9.3834858505459719e-08, -1.0370708662191418e-07, 1.1499734508496046e-07, -1.2796816261852297e-07, 1.4294288769073078e-07, -1.6032201969864769e-07, 1.8060502833216690e-07, -2.0442120587020166e-07, 2.3256958638562142e-07, -2.6607532013847684e-07, 3.0626738095928866e-07, -3.5488829260119791e-07, 4.1425182777935264e-07, -4.8747349724306658e-07, 5.7881217418665015e-07, -6.9418829303134906e-07, 8.4198941248161877e-07, -1.0343491900465793e-06, 1.2892453275257475e-06, -1.6340426463170451e-06, 2.1117130712431397e-06, -2.7922162634453614e-06, 3.7943899196860205e-06, -5.3305886421188924e-06, 7.8043580879824044e-06, -1.2043461755474721e-05, 1.9919842840142057e-05, -3.6251001189673213e-05, 7.5884216611786810e-05, -1.9885074842226311e-04, 7.9195669340038259e-04, -1.8028552886227383e-09},
			dval:  []float64{-2.9427800603424885e+01, 2.9427025985916121e+01, -2.9425476404186753e+01, 2.9423151164251298e+01, -2.9420049223750460e+01, 2.9416169190288226e+01, -2.9411509319206207e+01, 2.9406067510787029e+01, -2.9399841306877359e+01, 2.9392827886918830e+01, -2.9385024063373031e+01, 2.9376426276524445e+01, -2.9367030588643040e+01, 2.9356832677485638e+01, -2.9345827829112856e+01, 2.9334010929995458e+01, -2.9321376458381245e+01, 2.9307918474890496e+01, -2.9293630612304742e+01, 2.9278506064510260e+01, -2.9262537574553473e+01, 2.9245717421762187e+01, -2.9228037407881232e+01, 2.9209488842167364e+01, -2.9190062525382210e+01, 2.9169748732617208e+01, -2.9148537194877900e+01, 2.9126417079348521e+01, -2.9103376968250728e+01, 2.9079404836201942e+01, -2.9054488025970496e+01, 2.9028613222515041e+01, -2.9001766425185139e+01, 2.8973932917948382e+01, -2.8945097237496739e+01, 2.8915243139070405e+01, -2.8884353559822184e+01, 2.8852410579527451e+01, -2.8819395378426641e+01, 2.8785288191964121e+01, -2.8750068262165129e+01, 2.8713713785364632e+01, -2.8676201855972838e+01, 2.8637508405928582e+01, -2.8597608139454216e+01, 2.8556474462684317e+01, -2.8514079407692357e+01, 2.8470393550387154e+01, -2.8425385921689980e+01, 2.8379023911335469e+01, -2.8331273163562241e+01, 2.8282097463870919e+01, -2.8231458615927721e+01, 2.8179316307577007e+01, -2.8125627964795978e+01, 2.8070348592274335e+01, -2.8013430599129791e+01, 2.7954823608071283e+01, -2.7894474246092045e+01, 2.7832325914509102e+01, -2.7768318535855649e+01, 2.7702388274772812e+01, -2.7634467229625290e+01, 2.7564483091071313e+01, -2.7492358763234872e+01, 2.7418011942441147e+01, -2.7341354647661582e+01, 2.7262292695845463e+01, -2.7180725114157759e+01, 2.7096543479754125e+01, -2.7009631176051933e+01, 2.6919862552432274e+01, -2.6827101971847423e+01, 2.6731202727801637e+01, -2.6632005808479860e+01, 2.6529338481236699e+01, -2.6423012664989045e+01, 2.6312823050968596e+01, -2.6198544923370154e+01, 2.6079931620126878e+01, -2.5956711559610614e+01, 2.5828584740479602e+01, -2.5695218597789456e+01, 2.5556243066913400e+01, -2.5411244665072719e+01, 2.5259759344501671e+01, -2.5101263795907457e+01, 2.4935164777812641e+01, -2.4760785904535670e+01, 2.4577351124781824e+01, -2.4383963836126313e+01, 2.4179580164227872e+01, -2.3962974319140965e+01, 2.3732693009179656e+01, -2.3486994450692443e+01, 2.3223765221215142e+01, -2.2940404454619134e+01, 2.2633658531147908e+01, -2.2299378247064869e+01, 2.1932149880547183e+01, -2.1524711604262862e+01, 2.1066983867982440e+01, -2.0544356598262947e+01, 1.9934415682618429e+01, -1.9199991275984207e+01, 1.8272015227078569e+01, -1.6996182114611788e+01, 1.4876958810870860e+01},
		},
		{
			n:     216,
			theta: []float64{1.5671686105755023e+00, 1.5599127961684220e+00, 1.5526558355346407e+00, 1.5453969634495357e+00, 1.5381354130756992e+00, 1.5308704153127000e+00, 1.5236011981417006e+00, 1.5163269859634152e+00, 1.5090469989278730e+00, 1.5017604522544048e+00, 1.4944665555402459e+00, 1.4871645120560699e+00, 1.4798535180267400e+00, 1.4725327618954691e+00, 1.4652014235695219e+00, 1.4578586736455008e+00, 1.4505036726121565e+00, 1.4431355700285682e+00, 1.4357535036754083e+00, 1.4283565986768869e+00, 1.4209439665908115e+00, 1.4135147044640579e+00, 1.4060678938505511e+00, 1.3986025997886760e+00, 1.3911178697348172e+00, 1.3836127324494960e+00, 1.3760861968323106e+00, 1.3685372507016056e+00, 1.3609648595144777e+00, 1.3533679650223920e+00, 1.3457454838572851e+00, 1.3380963060426339e+00, 1.3304192934234869e+00, 1.3227132780089634e+00, 1.3149770602201478e+00, 1.3072094070356892e+00, 1.2994090500267301e+00, 1.2915746832720101e+00, 1.2837049611431546e+00, 1.2757984959492026e+00, 1.2678538554283829e+00, 1.2598695600739727e+00, 1.2518440802797628e+00, 1.2437758332892002e+00, 1.2356631799306261e+00, 1.2275044211192108e+00, 1.2192977941041021e+00, 1.2110414684370108e+00, 1.2027335416358120e+00, 1.1943720345138122e+00, 1.1859548861419755e+00, 1.1774799484076148e+00, 1.1689449801287550e+00, 1.1603476406784692e+00, 1.1516854830678944e+00, 1.1429559464302423e+00, 1.1341563478407819e+00, 1.1252838733993509e+00, 1.1163355684922343e+00, 1.1073083271390356e+00, 1.0981988803171721e+00, 1.0890037831415225e+00, 1.0797194007591668e+00, 1.0703418927985888e+00, 1.0608671961885900e+00, 1.0512910061337959e+00, 1.0416087550001036e+00, 1.0318155888237164e+00, 1.0219063411101317e+00, 1.0118755035330074e+00, 1.0017171930750695e+00, 9.9142511507162645e-01, 9.8099252151840288e-01, 9.7041216388519547e-01, 9.5967623952979786e-01, 9.4877633062585665e-01, 9.3770333429470865e-01, 9.2644738235305046e-01, 9.1499774873988793e-01, 9.0334274224692235e-01, 8.9146958161860457e-01, 8.7936424937397950e-01, 8.6701131978084423e-01, 8.5439375521282512e-01, 8.4149266354281749e-01, 8.2828700713198744e-01, 8.1475325116200825e-01, 8.0086493523862246e-01, 7.8659214693830359e-01, 7.7190086863415974e-01, 7.5675215854165734e-01, 7.4110111194290051e-01, 7.2489552655247935e-01, 7.0807416302356885e-01, 6.9056444104412407e-01, 6.7227933193754241e-01, 6.5311307983969791e-01, 6.3293516775593595e-01, 6.1158156935716357e-01, 5.8884164425859431e-01, 5.6443772421506910e-01, 5.3799175946275157e-01, 5.0896747562778555e-01, 4.7656205323872197e-01, 4.3948124005672007e-01, 3.9539840620540745e-01, 3.3931369540774287e-01, 2.5576413755369809e-01},
			val:   []float64{-2.1420664007171084e-13, 5.3170379037842784e-14, -3.4734236933038639e-15, 2.3242110868118431e-13, 3.7943143720204046e-14, -5.7160511614569708e-14, 2.1177269961331864e-13, -3.4948583926844979e-14, 1.7369271883827313e-13, 3.5623756291990570e-13, 6.5520118362261941e-14, 1.1255590127344777e-13, 9.4867994932892441e-14, -6.1601712969494469e-14, 4.7023788875015558e-14, -1.4272998866826663e-13, -2.6605445153274162e-14, -2.5223571223200907e-13, -1.3724084095111965e-13, -7.0111736669803626e-14, 5.7347666545680944e-14, -5.0924962889888544e-14, -5.2683776576685045e-15, -2.1544335394504838e-13, 4.0230445264824984e-15, 6.9175538975130938e-15, -1.1339463388095042e-13, -3.9896959286779273e-14, -8.8360560296236634e-14, -1.0474371284611670e-13, 1.2687327963040444e-14, -1.7782734520430910e-13, 2.0086781777635686e-13, -7.5400883703521135e-14, 1.2278140016266356e-13, -1.1244970732922019e-13, 1.6072100253117342e-14, 5.2340169298568757e-15, 1.1013085876614924e-13, -9.6621831158318670e-15, -1.8366177977891496e-13, 1.0601073284832238e-14, -1.3784763212103856e-13, -1.5863525424166753e-13, -1.6638681463670250e-13, 8.6360908147481066e-14, -7.8672683965135368e-14, 6.8378486764803313e-14, 9.0534733729868808e-14, 6.6505185887739604e-15, 3.3141885340142080e-15, -2.0537396911778824e-13, 7.3393589371575821e-14, 9.8483281662822096e-14, -2.3812738806298458e-13, 6.5975425886450090e-15, 8.7013139463913052e-14, 7.5278742714453987e-14, -1.0708940266003440e-13, 2.0958580081069137e-13, -1.5480716019053119e-14, 4.8469624575186930e-15, -6.1362291991375536e-14, -3.5942923464396180e-14, -7.6176865685553763e-14, 1.0095757892256693e-13, -4.2683956504767779e-14, -4.8938621882647329e-14, 1.6489272569372713e-15, -1.1390338364880240e-13, 2.1513601964632049e-14, -1.0152923724696736e-14, -3.9597826958407387e-14, -8.5123156147931521e-14, -6.5276802085245163e-14, 6.3966825794140229e-14, -2.8207572680644779e-14, -5.9767990455898208e-15, -1.1189271498293956e-13, -4.5600777141900301e-14, 1.5226732724861721e-14, 3.9473540473700922e-14, -5.6017378761616795e-15, -2.4079255903843183e-14, -3.1139558336514997e-14, -5.6034014603771978e-14, 6.9309294575728887e-14, -3.7871955014469560e-14, 4.2988763678275364e-14, 1.0661412180664073e-14, 6.2069599395465107e-14, -4.8913422170368503e-14, 3.3884911719768557e-14, 2.8191908465098299e-15, 2.1228960171536080e-14, -5.6191478942893769e-14, -1.4786950368470112e-14, -2.1665036441200809e-14, 1.4936420829725019e-14, -1.1615745673091571e-13, 1.4661728300981146e-13, -3.6088669931521093e-13, 1.1825080499589470e-12, -4.7300739521811106e-12, 2.5459387908526307e-11, -2.2842779135337983e-10, 5.3230524570865730e-09, 1.8504694403187737e-14},
			dval:  []float64{-2.9427800603424888e+01, 2.9427025985916128e+01, -2.9425476404186732e+01, 2.9423151164251305e+01, -2.9420049223750453e+01, 2.9416169190288219e+01, -2.9411509319206210e+01, 2.9406067510787015e+01, -2.9399841306877356e+01, 2.9392827886918813e+01, -2.9385024063373031e+01, 2.9376426276524445e+01, -2.9367030588643043e+01, 2.9356832677485624e+01, -2.9345827829112846e+01, 2.9334010929995465e+01, -2.9321376458381234e+01, 2.9307918474890485e+01, -2.9293630612304742e+01, 2.9278506064510257e+01, -2.9262537574553480e+01, 2.9245717421762187e+01, -2.9228037407881235e+01, 2.9209488842167364e+01, -2.9190062525382206e+01, 2.9169748732617215e+01, -2.9148537194877896e+01, 2.9126417079348521e+01, -2.9103376968250736e+01, 2.9079404836201938e+01, -2.9054488025970500e+01, 2.9028613222515034e+01, -2.9001766425185131e+01, 2.8973932917948375e+01, -2.8945097237496743e+01, 2.8915243139070423e+01, -2.8884353559822181e+01, 2.8852410579527444e+01, -2.8819395378426645e+01, 2.8785288191964142e+01, -2.8750068262165122e+01, 2.8713713785364636e+01, -2.8676201855972852e+01, 2.8637508405928578e+01, -2.8597608139454227e+01, 2.8556474462684314e+01, -2.8514079407692350e+01, 2.8470393550387161e+01, -2.8425385921689987e+01, 2.8379023911335462e+01, -2.8331273163562233e+01, 2.8282097463870926e+01, -2.8231458615927728e+01, 2.8179316307577018e+01, -2.8125627964795974e+01, 2.8070348592274335e+01, -2.8013430599129794e+01, 2.7954823608071287e+01, -2.7894474246092045e+01, 2.7832325914509109e+01, -2.7768318535855656e+01, 2.7702388274772812e+01, -2.7634467229625294e+01, 2.7564483091071331e+01, -2.7492358763234883e+01, 2.7418011942441158e+01, -2.7341354647661582e+01, 2.7262292695845463e+01, -2.7180725114157763e+01, 2.7096543479754128e+01, -2.7009631176051940e+01, 2.6919862552432292e+01, -2.6827101971847430e+01, 2.6731202727801641e+01, -2.6632005808479885e+01, 2.6529338481236724e+01, -2.6423012664989066e+01, 2.6312823050968632e+01, -2.6198544923370196e+01, 2.6079931620126928e+01, -2.5956711559610671e+01, 2.5828584740479652e+01, -2.5695218597789530e+01, 2.5556243066913513e+01, -2.5411244665072843e+01, 2.5259759344501834e+01, -2.5101263795907666e+01, 2.4935164777812918e+01, -2.4760785904536036e+01, 2.4577351124782325e+01, -2.4383963836127013e+01, 2.4179580164228856e+01, -2.3962974319142369e+01, 2.3732693009181720e+01, -2.3486994450695558e+01, 2.3223765221219963e+01, -2.2940404454626893e+01, 2.2633658531160954e+01, -2.2299378247087898e+01, 2.1932149880590437e+01, -2.1524711604350518e+01, 2.1066983868178077e+01, -2.0544356598759389e+01, 1.9934415684120079e+01, -1.9199991281861426e+01, 1.8272015261833435e+01, -1.6996182557157578e+01, 1.4876958810870862e+01},
		},
	} {
		got := make([]float64, len(test.theta))
		gotD := make([]float64, len(test.theta))
		for i := range got {
			got[i], gotD[i] = Hermite{}.hermpolyAsyAiry(i, test.n, test.theta[i])
		}
		if !floats.EqualApprox(test.val, got, 1e-12) {
			t.Errorf("n = %v, hermpolyAsyAiry mismatch", test.n)
		}
		if !floats.EqualApprox(test.dval, gotD, 1e-12) {
			t.Errorf("n = %v, hermpolyAsyAiry deriv mismatch", test.n)
		}
	}
}
