//*****************************************************************************
//                                  ChoUnits.cpp                              *
//                                 --------------                             *
// Started     : 27/03/2004                                                   *
// Last Update : 11/04/2011                                                   *
// Copyright   : (C) 2004 by MSWaters                                         *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "utility/ChoUnits.hpp"

//*****************************************************************************
// Constructor.

ChoUnits::ChoUnits( void )
{
  bClear( );
}

//*****************************************************************************
// Destructor.

ChoUnits::~ChoUnits( )
{
}

//*****************************************************************************
// Create an instance of this object.
//
// Argument List :
//   poWin  - The parent window
//   oWinID - The window identifier
//   iWidth - The width of the choice control in pixels
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  ChoUnits::bCreate( wxWindow * poWin, wxWindowID oWinID, int iWidth )
{
  wxSize  oSize=wxDefaultSize;

  // Check if the object has already been created
  if( bIsCreated( ) )                                       return( TRUE );

  // Create the object
  if( iWidth > 0 ) oSize.SetWidth( iWidth );
  if( ! Create( poWin, oWinID, wxDefaultPosition, oSize ) ) return( FALSE );

  // Set the units
  if( ! bSetUnitsType( m_eUnitsType ) )                     return( FALSE );
  if( ! bSetUnits( m_osDefUnits ) )                         return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Clear object attributes.
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  ChoUnits::bClear( void )
{
  m_eUnitsType = eUNITS_NONE;
  m_eDefUType  = eUNITS_NONE;
  m_osDefUnits = wxT("Units");

  bSetUnitsType( eUNITS_NONE );

  if( bIsCreated( ) ) wxChoice::Clear( );

  return( TRUE );
}

//*****************************************************************************
// Set default object attributes.
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  ChoUnits::bSetDefaults( void )
{
  // Set the default units type
  if( ! bSetUnitsType( m_eDefUType ) ) return( FALSE );

  // Set the default units
  if( ! bSetUnits( m_osDefUnits ) )    return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Set the type of units to be displayed.
//
// Argument List :
//   eUType - The type of units to be displayed
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  ChoUnits::bSetUnitsType( eUnitsType eUType )
{
  if( ! bIsCreated( ) )                      return( FALSE );

  if( ! IsEmpty( ) && m_eUnitsType==eUType ) return( TRUE );

  Clear( );              // Delete existing choice items from the list

  m_eUnitsType = eUType; // Set the objects units type attribute

  // Load the new choice items into the control
  switch( m_eUnitsType )
  {
    case eUNITS_CAP :  // Capacitance
      Append( wxT("mF"), new wxStringClientData( wxT("-3")  ) );
      Append( wxT("uF"), new wxStringClientData( wxT("-6")  ) );
      Append( wxT("nF"), new wxStringClientData( wxT("-9")  ) );
      Append( wxT("pF"), new wxStringClientData( wxT("-12") ) );
      SetStringSelection( wxT("uF") );
      break;

    case eUNITS_IND :  // Inductance
      Append( wxT("H") , new wxStringClientData( wxT("0")  ) );
      Append( wxT("mH"), new wxStringClientData( wxT("-3") ) );
      Append( wxT("uH"), new wxStringClientData( wxT("-6") ) );
      SetStringSelection( wxT("mH") );
      break;

    case eUNITS_RES :  // Resistance
      Append( wxT("MOhm"), new wxStringClientData( wxT("6") ) );
      Append( wxT("KOhm"), new wxStringClientData( wxT("3") ) );
      Append( wxT("Ohm") , new wxStringClientData( wxT("0") ) );
      SetStringSelection( wxT("KOhm") );
      break;

    case eUNITS_VOLT : // Voltage
      Append( wxT("KV"), new wxStringClientData(  wxT("3") ) );
      Append( wxT("V") , new wxStringClientData(  wxT("0") ) );
      Append( wxT("mV"), new wxStringClientData( wxT("-3") ) );
      Append( wxT("uV"), new wxStringClientData( wxT("-6") ) );
      SetStringSelection( wxT("mV") );
      break;

    case eUNITS_CURR : // Current
      Append( wxT("KA"), new wxStringClientData(  wxT("3") ) );
      Append( wxT("A") , new wxStringClientData(  wxT("0") ) );
      Append( wxT("mA"), new wxStringClientData( wxT("-3") ) );
      Append( wxT("uA"), new wxStringClientData( wxT("-6") ) );
      SetStringSelection( wxT("mA") );
      break;

    case eUNITS_TIME : // Time
      Append( wxT("sec") , new wxStringClientData(   wxT("0") ) );
      Append( wxT("msec"), new wxStringClientData(  wxT("-3") ) );
      Append( wxT("usec"), new wxStringClientData(  wxT("-6") ) );
      Append( wxT("nsec"), new wxStringClientData(  wxT("-9") ) );
      Append( wxT("psec"), new wxStringClientData( wxT("-12") ) );
      SetStringSelection( wxT("msec") );
      break;

    case eUNITS_FREQ : // Frequency
      Append( wxT("THz"), new wxStringClientData( wxT("12") ) );
      Append( wxT("GHz"), new wxStringClientData(  wxT("9") ) );
      Append( wxT("MHz"), new wxStringClientData(  wxT("6") ) );
      Append( wxT("KHz"), new wxStringClientData(  wxT("3") ) );
      Append( wxT("Hz") , new wxStringClientData(  wxT("0") ) );
      SetStringSelection( wxT("KHz") );
      break;

    case eUNITS_PHASE : // Phase
      Append( wxT("Degree"), new wxStringClientData( wxT("0") ) );
      Append( wxT("Radian"), new wxStringClientData( wxT("0") ) );
      Append( wxT("Grad")  , new wxStringClientData( wxT("0") ) );
      SetStringSelection( wxT("Degree") );
      break;

    case eUNITS_TEMP : // Temperature
      Append( wxT("Deg C"), new wxStringClientData( wxT("0") ) );
      Append( wxT("Deg F"), new wxStringClientData( wxT("0") ) );
      Append( wxT("Deg K"), new wxStringClientData( wxT("0") ) );
      SetStringSelection( wxT("Deg C") );
      break;

    case eUNITS_SCLR : // Scalar
      Append( wxT("Tera") , new wxStringClientData(  wxT("12") ) );
      Append( wxT("Giga") , new wxStringClientData(   wxT("9") ) );
      Append( wxT("Mega") , new wxStringClientData(   wxT("6") ) );
      Append( wxT("Kilo") , new wxStringClientData(   wxT("3") ) );
      Append( wxT("None") , new wxStringClientData(   wxT("0") ) );
      Append( wxT("milli"), new wxStringClientData(  wxT("-3") ) );
      Append( wxT("micro"), new wxStringClientData(  wxT("-6") ) );
      Append( wxT("nano") , new wxStringClientData(  wxT("-9") ) );
      Append( wxT("pico") , new wxStringClientData( wxT("-12") ) );
      Append( wxT("femto"), new wxStringClientData( wxT("-15") ) );
      SetStringSelection( wxT("None") );
      break;

    case eUNITS_NONE :  // No units specified
      Append( wxT("Units"), new wxStringClientData( wxT("0") ) );
      SetStringSelection( wxT("Units") );
      break;
  }

  m_osDefUnits = GetStringSelection( );

  return( TRUE );
}

//*****************************************************************************
// Set the units to be displayed.
//
// Argument List:
//   rosUnits - The units to be displayed as a string
//
// Return Values:
//   Success - TRUE
//   Failure - FALSE

bool  ChoUnits::bSetUnits( const wxString rosUnits )
{
  int  i1;

  if( ! bIsCreated( ) )   return( FALSE );

  i1 = FindString( rosUnits );
  if( i1 == wxNOT_FOUND ) return( FALSE );

  SetSelection( i1 );

  return( TRUE );
}

//*****************************************************************************
// Set the units to be displayed.
//
// Argument List :
//   rosUnits - The exponent associated with the units to be displayed
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  ChoUnits::bSetUnits( int iExponent )
{
  wxString  os1;
  long      li1;
  uint      ui1;

  if( ! bIsCreated( ) ) return( FALSE );

  for( ui1=0; ui1<(uint)GetCount( ); ui1++ )
  {
    li1 = 0;
    os1 = ( (wxStringClientData *) GetClientObject( ui1 ) )->GetData( );
    os1.ToLong( &li1 );
    if( (int) li1 == iExponent )
    {
      SetSelection( ui1 );
      return( TRUE );
    }
  }

  if( ! IsShown( ) )    return( TRUE );

  return( FALSE );
}

//*****************************************************************************
// Set the default units.
//
// Argument List :
//   rosUnits - The default units to be used
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE
/*
bool  ChoUnits::bSetDefUnits( const wxString rosUnits )
{
  eUnitsType  eUType;
  int         i1;

  // Don't proceed unless the display object has been created
  if( ! bIsCreated( ) )   return( FALSE );

  // Temporarily set the units type to the default
  if( m_eUnitsType != m_eDefUType )
  {
    eUType = m_eUnitsType;
    bSetUnitsType( m_eDefUType );
  }

  // Attempt to find the specified units
  i1 = FindString( rosUnits );

  // Return the units type to the original value
  if( eUType != m_eDefUType )
    bSetUnitsType( eUType );

  // If the units couldn't be found return
  if( i1 == wxNOT_FOUND ) return( FALSE );

  // Set the default units
  m_osDefUnits = rosUnits;

  return( TRUE );
}
*/
//*****************************************************************************
// Get the currently selected units.
//
// Return Values :
//   Success - The selected units
//   Failure - An empty string

const wxString & ChoUnits::rosGetUnits( void )
{
  static  wxString  os1;

  os1.Empty( );

  if( ! bIsCreated( ) ) return( os1 );

  os1 = GetStringSelection( );

  return( os1 );
}

//*****************************************************************************
// Get the currently selected units as an exponent eg. if the units are mV
// return -3 or if the units are MOhm return 6.
//
// Return Values :
//   Success - The units exponent
//   Failure - 0

int  ChoUnits::iGetUnits( void )
{
  wxString  os1;
  long int  li1;

  if( ! bIsCreated( ) ) return( 0 );

  li1 = (long) GetSelection( );
  os1 = ( (wxStringClientData *) GetClientObject( (int) li1 ) )->GetData( );
  os1.ToLong( &li1 );

  return( (int) li1 );
}

//*****************************************************************************
// Combine a value with it's units to produce the equivalent long value.
//
// Argument List :
//   fValue - The float value
//
// Return Values :
//   The value & units as a long

long  ChoUnits::liGetValue( float fValue )
{
  double  df1;
  long    li1;

  df1 = dfGetValue( fValue );

  // Round the value
  if( df1 >= 0.0 ) li1 = (long) ( df1 + 0.5 );
  else             li1 = (long) ( df1 - 0.5 );

  return( li1 );
}

//*****************************************************************************
// Combine a value with it's units to produce a single value.
//
// Argument List :
//   fValue  - The float value
//
// Return Values :
//   Success - The value combined with it's units as a double
//   Failure - 0.0

double  ChoUnits::dfGetValue( float fValue )
{
  wxString  os1;
  double    df1;
  long      li1;
  int       i1;

  if( ! bIsCreated( ) || fValue==0.0 ) return( 0.0 );

  i1 = GetSelection( );
  if( i1 == wxNOT_FOUND )              return( 0.0 );

  os1 = ( (wxStringClientData *) GetClientObject( i1 ) )->GetData( );
  os1.ToLong( &li1 );

  switch( m_eUnitsType )
  {
    case eUNITS_CAP :   // Capacitance

    case eUNITS_IND :   // Inductance

    case eUNITS_RES :   // Resistance

    case eUNITS_VOLT :  // Voltage

    case eUNITS_CURR :  // Current

    case eUNITS_TIME :  // Time

    case eUNITS_FREQ :  // Frequency

    case eUNITS_SCLR :  // Scalar
      if( fValue == 0.0 ) df1 = 0.0;
      else if( li1 == 0 ) df1 = (double) fValue;
      else                df1 = (double) fValue * pow( (float) 10.0, (float) li1 );
      break;

    case eUNITS_PHASE : // Phase (convert phase to Degree)
      df1 = (double) fValue;
      switch( i1 )
      {
        case 0:                       break; // Degree (360 degree per cycle)
        case 1:  df1 *= 180.0/3.1416; break; // Radian (2Pi radian per cycle)
        case 2:  df1 *= 9.0/10.0;     break; // Grad   (400 grad   per cycle)
        default: df1  = 0.0;
      }
      break;

    case eUNITS_TEMP :  // Temperature (convert temperature to Deg. C)
      df1 = (double) fValue;
      switch( i1 )
      {
        case 0:                             break; // Celcius
        case 1:  df1  = (df1 - 32.0) / 1.8; break; // Fahrenheit
        case 2:  df1 -= 273.15;             break; // Kelvin
        default: df1  = 0.0;
      }
      break;

    default :
      df1 = 0.0;
  }

  return( df1 );
}

//*****************************************************************************
// Combine a value with it's units to produce the equivalent string value.
//
// Argument List :
//   fValue - The float value
//
// Return Values :
//   Success - The string containing the engineering format
//   Failure - An empty string

const wxString & ChoUnits::rosGetValue( float fValue )
{
  static  wxString  osValue;
  wxString  osUnits;

  osValue.Empty( );

  if( ! bIsCreated( ) )    return( osValue );

  osValue.Printf( wxT("%#.2f"), fValue );

  osUnits = rosGetUnits( );
  if( osUnits.IsEmpty( ) ) return( osValue );

  if( osUnits != wxT("None") )
  {
    if( osUnits == wxT("micro") )
      osValue << wxT('u');
    else
    {
      switch( (char) osUnits.GetChar( 0 ) )
      {
        case wxT('T'): osValue << wxT("Tera"); break;
        case wxT('G'): osValue << wxT("Giga"); break;
        case wxT('M'): osValue << wxT("Meg");  break;
        case wxT('K'): osValue << wxT('K');    break;
        case wxT('k'): osValue << wxT('k');    break;
        case wxT('m'): osValue << wxT('m');    break;
        case wxT('u'): osValue << wxT('u');    break;
        case wxT('n'): osValue << wxT('n');    break;
        case wxT('p'): osValue << wxT('p');    break;
        case wxT('f'): osValue << wxT('f');    break;
        default:                               break;
      }
    }
  }

  return( osValue );
}

//*****************************************************************************
