/**
* @license Apache-2.0
*
* Copyright (c) 2020 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var trunc = require( './../../../../base/special/trunc' );
var randu = require( '@stdlib/random/base/randu' );
var PINF = require( '@stdlib/constants/float64/pinf' );
var NINF = require( '@stdlib/constants/float64/ninf' );
var isCoprime = require( './../lib' );


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof isCoprime, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function returns `true` if provided coprime integers', function test( t ) {
	t.equal( isCoprime( 14.0, 15.0 ), true, 'returns expected value' );
	t.equal( isCoprime( 5.0, 7.0 ), true, 'returns expected value' );
	t.equal( isCoprime( 13.0, 19.0 ), true, 'returns expected value' );

	t.equal( isCoprime( -14.0, 15.0 ), true, 'returns expected value' );
	t.equal( isCoprime( 5.0, -7.0 ), true, 'returns expected value' );
	t.equal( isCoprime( -13.0, -19.0 ), true, 'returns expected value' );

	t.end();
});

tape( 'the function returns `false` if provided integers which are not coprime', function test( t ) {
	t.equal( isCoprime( 14.0, 21.0 ), false, 'returns expected value' );
	t.equal( isCoprime( 15.0, 45.0 ), false, 'returns expected value' );
	t.equal( isCoprime( 20.0, 90.0 ), false, 'returns expected value' );
	t.end();
});

tape( 'the function returns `false` if not provided an integer', function test( t ) {
	var v;
	var i;
	for ( i = 0; i < 100; i++ ) {
		v = ( randu()*100.0 ) - 50.0;
		if ( trunc(v) !== v ) {
			t.equal( isCoprime( v, 5.0 ), false, 'returns false when provided '+v );
			t.equal( isCoprime( 4.0, v ), false, 'returns false when provided '+v );
			t.equal( isCoprime( v, v ), false, 'returns false when provided '+v );
		}
	}
	t.end();
});

tape( 'the function returns `false` if provided `NaN`', function test( t ) {
	t.equal( isCoprime( NaN, NaN ), false, 'returns expected value' );
	t.equal( isCoprime( 4.0, NaN ), false, 'returns expected value' );
	t.equal( isCoprime( NaN, 5.0 ), false, 'returns expected value' );
	t.end();
});

tape( 'the function returns `false` if provided `+infinity`', function test( t ) {
	t.equal( isCoprime( PINF, PINF ), false, 'returns expected value' );
	t.equal( isCoprime( 4.0, PINF ), false, 'returns expected value' );
	t.equal( isCoprime( PINF, 5.0 ), false, 'returns expected value' );
	t.end();
});

tape( 'the function returns `false` if provided `-infinity`', function test( t ) {
	t.equal( isCoprime( NINF, NINF ), false, 'returns expected value' );
	t.equal( isCoprime( 4.0, NINF ), false, 'returns expected value' );
	t.equal( isCoprime( NINF, 5.0 ), false, 'returns expected value' );
	t.end();
});

tape( 'the function returns `true` if provided `1`', function test( t ) {
	t.equal( isCoprime( 1, 1 ), true, 'returns expected value' );
	t.equal( isCoprime( 1, 4 ), true, 'returns expected value' );
	t.equal( isCoprime( 4, 1 ), true, 'returns expected value' );
	t.end();
});
