/*
 * Copyright 2013 Canonical Ltd.
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <errno.h>
#include <glib.h>
#include "powerd-internal.h"
#include "log.h"

struct powerd_client {   
    const char *dbus_name;
    const char *name;
    gboolean ack_pending;
};

static GHashTable *client_hash;
static int acks_needed;
static int ack_state;

static void client_ack(struct powerd_client *client)
{
    if (!client->ack_pending)
        return;
    client->ack_pending = FALSE;

    if (acks_needed <= 0) {
        powerd_error("Client needs ack while total acks needed is %d",
                     acks_needed);
        return;
    }
    if (--acks_needed == 0)
        power_request_transition_acked();
}

/* Must be called from main loop */
int powerd_client_register(const char *dbus_name, const char *name)
{
    struct powerd_client *client;
   
    if (!dbus_name || !name)
        return -EINVAL;

    client = g_new0(struct powerd_client, 1);
    if (!client)
        return -ENOMEM;

    client->dbus_name = g_strdup(dbus_name);
    client->name = g_strdup(name);
    g_hash_table_insert(client_hash, (gpointer)client->dbus_name, client);
    powerd_dbus_name_watch_add(dbus_name);
    return 0;
}

/* Must be called from main loop */
void powerd_client_unregister(const char *dbus_name)
{
    struct powerd_client *client;

    client = g_hash_table_lookup(client_hash, dbus_name);
    if (!client)
        return;

    client_ack(client);
    powerd_dbus_name_watch_remove(client->dbus_name);
    g_hash_table_remove(client_hash, client->dbus_name);
}

/*
 * Must be called from main loop. Returns TRUE if caller should
 * wait for acks.
 */
gboolean powerd_client_transition_start(int state)
{
    GHashTableIter iter;
    gpointer key, value;

    acks_needed = 0;
    ack_state = state;

    g_hash_table_iter_init(&iter, client_hash);
    while (g_hash_table_iter_next(&iter, &key, &value)) {
        struct powerd_client *client = value;
        client->ack_pending = TRUE;
        acks_needed++;
    }
    return acks_needed != 0;
}

/* Must be called from main loop */
void powerd_client_transition_finish(int state)
{
    GHashTableIter iter;
    gpointer key, value;

    if (state != ack_state) {
        powerd_warn("Attempt to finish client transition with wrong state");
        return;
    }

    if (acks_needed)
        return;
    acks_needed = 0;

    g_hash_table_iter_init(&iter, client_hash);
    while (g_hash_table_iter_next(&iter, &key, &value)) {
        struct powerd_client *client = value;
        if (client->ack_pending) {
            powerd_warn("Client %s (%s) failed to acknowledge state change",
                        client->name, client->dbus_name);
            client->ack_pending = FALSE;
        }
    }
}

/* Must be called from main loop */
int powerd_client_ack(const char *dbus_name, int state)
{
    struct powerd_client *client;

    if (state != ack_state)
        return -EINVAL;

    client = g_hash_table_lookup(client_hash, dbus_name);
    if (!client)
        return -ENODEV;
    client_ack(client);
    return 0;
}

static void client_hash_destroy(gpointer data)
{
    struct powerd_client *client = data;
    g_free((gpointer)client->dbus_name);
    g_free((gpointer)client->name);
    g_free(client);
}

int powerd_client_init(void)
{
    client_hash = g_hash_table_new_full(g_str_hash, g_str_equal, NULL,
                                        client_hash_destroy);
    return 0;
}

void powerd_client_deinit(void)
{
    g_hash_table_destroy(client_hash);
}
