// Copyright 2012-2015 The Rust Project Developers. See the COPYRIGHT
// file at the top-level directory of this distribution and at
// http://rust-lang.org/COPYRIGHT.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

// NOTE: The following code was generated by "scripts/unicode.py", do not edit directly

#![allow(missing_docs, non_upper_case_globals, non_snake_case)]

/// The version of [Unicode](http://www.unicode.org/)
/// that this version of unicode-security is based on.
pub const UNICODE_VERSION: (u64, u64, u64) = (15, 1, 0);


#[allow(dead_code)]
pub mod util {
    use core::result::Result::{Ok, Err};

    pub fn bsearch_range_value_table<T: Copy>(c: char, r: &'static [(char, char, T)]) -> Option<T> {
        use core::cmp::Ordering::{Equal, Less, Greater};
        match r.binary_search_by(|&(lo, hi, _)| {
            if lo <= c && c <= hi { Equal }
            else if hi < c { Less }
            else { Greater }
        }) {
            Ok(idx) => {
                let (_, _, cat) = r[idx];
                Some(cat)
            }
            Err(_) => None
        }
    }

}

#[cfg(feature = "general-category")]
pub mod general_category {

    #[derive(Copy, Clone, Hash, Eq, PartialEq, Ord, PartialOrd, Debug)]
    /// The most general classification of a character.
    pub enum GeneralCategory {
        /// `Lu`, an uppercase letter
        UppercaseLetter,
        /// `Ll`, a lowercase letter
        LowercaseLetter,
        /// `Lt`, a digraphic character, with first part uppercase
        TitlecaseLetter,
        /// `Lm`, a modifier letter
        ModifierLetter,
        /// `Lo`, other letters, including syllables and ideographs
        OtherLetter,
        /// `Mn`, a nonspacing combining mark (zero advance width)
        NonspacingMark,
        /// `Mc`, a spacing combining mark (positive advance width)
        SpacingMark,
        /// `Me`, an enclosing combining mark
        EnclosingMark,
        /// `Nd`, a decimal digit
        DecimalNumber,
        /// `Nl`, a letterlike numeric character
        LetterNumber,
        /// `No`, a numeric character of other type
        OtherNumber,
        /// `Pc`, a connecting punctuation mark, like a tie
        ConnectorPunctuation,
        /// `Pd`, a dash or hyphen punctuation mark
        DashPunctuation,
        /// `Ps`, an opening punctuation mark (of a pair)
        OpenPunctuation,
        /// `Pe`, a closing punctuation mark (of a pair)
        ClosePunctuation,
        /// `Pi`, an initial quotation mark
        InitialPunctuation,
        /// `Pf`, a final quotation mark
        FinalPunctuation,
        /// `Po`, a punctuation mark of other type
        OtherPunctuation,
        /// `Sm`, a symbol of mathematical use
        MathSymbol,
        /// `Sc`, a currency sign
        CurrencySymbol,
        /// `Sk`, a non-letterlike modifier symbol
        ModifierSymbol,
        /// `So`, a symbol of other type
        OtherSymbol,
        /// `Zs`, a space character (of various non-zero widths)
        SpaceSeparator,
        /// `Zl`, U+2028 LINE SEPARATOR only
        LineSeparator,
        /// `Zp`, U+2029 PARAGRAPH SEPARATOR only
        ParagraphSeparator,
        /// `Cc`, a C0 or C1 control code
        Control,
        /// `Cf`, a format control character
        Format,
        /// `Cs`, a surrogate code point
        Surrogate,
        /// `Co`, a private-use character
        PrivateUse,
        /// `Cn`, a reserved unassigned code point or a noncharacter
        Unassigned,
    }

    #[derive(Copy, Clone, Hash, Eq, PartialEq, Ord, PartialOrd, Debug)]
    /// Groupings of the most general classification of a character.
    pub enum GeneralCategoryGroup {
        /// Lu | Ll | Lt | Lm | Lo
        Letter,
        /// Mn | Mc | Me
        Mark,
        /// Nd | Nl | No
        Number,
        /// Pc | Pd | Ps | Pe | Pi | Pf | Po
        Punctuation,
        /// Sm | Sc | Sk | So
        Symbol,        
        /// Zs | Zl | Zp
        Separator,
        /// Cc | Cf | Cs | Co | Cn
        Other,
    }

    #[inline]
    pub(crate) fn general_category_of_char(c: char) -> GeneralCategory {
        super::util::bsearch_range_value_table(c, GENERAL_CATEGORY).unwrap_or(GeneralCategory::Unassigned)
    }

    #[inline]
    pub(crate) fn general_category_is_letter_cased(gc: GeneralCategory) -> bool {
        matches!(gc, GeneralCategory::UppercaseLetter | GeneralCategory::LowercaseLetter | GeneralCategory::TitlecaseLetter)
    }

    #[inline]
    pub(crate) fn general_category_group(gc: GeneralCategory) -> GeneralCategoryGroup {
        match gc {
            GeneralCategory::UppercaseLetter |
            GeneralCategory::LowercaseLetter |
            GeneralCategory::TitlecaseLetter |
            GeneralCategory::ModifierLetter |
            GeneralCategory::OtherLetter => GeneralCategoryGroup::Letter,
            GeneralCategory::NonspacingMark |
            GeneralCategory::SpacingMark |
            GeneralCategory::EnclosingMark => GeneralCategoryGroup::Mark,
            GeneralCategory::DecimalNumber |
            GeneralCategory::LetterNumber |
            GeneralCategory::OtherNumber => GeneralCategoryGroup::Number,
            GeneralCategory::ConnectorPunctuation |
            GeneralCategory::DashPunctuation |
            GeneralCategory::OpenPunctuation |
            GeneralCategory::ClosePunctuation |
            GeneralCategory::InitialPunctuation |
            GeneralCategory::FinalPunctuation |
            GeneralCategory::OtherPunctuation => GeneralCategoryGroup::Punctuation,
            GeneralCategory::MathSymbol |
            GeneralCategory::CurrencySymbol |
            GeneralCategory::ModifierSymbol |
            GeneralCategory::OtherSymbol => GeneralCategoryGroup::Symbol,
            GeneralCategory::SpaceSeparator |
            GeneralCategory::LineSeparator |
            GeneralCategory::ParagraphSeparator => GeneralCategoryGroup::Separator,
            GeneralCategory::Control |
            GeneralCategory::Format |
            GeneralCategory::Surrogate |
            GeneralCategory::PrivateUse |
            GeneralCategory::Unassigned => GeneralCategoryGroup::Other,
        }
    }
    // General category table:
    const GENERAL_CATEGORY: &[(char, char, GeneralCategory)] = &[
        ('\u{0}', '\u{1f}', GeneralCategory::Control), ('\u{20}', '\u{20}',
        GeneralCategory::SpaceSeparator), ('\u{21}', '\u{23}', GeneralCategory::OtherPunctuation),
        ('\u{24}', '\u{24}', GeneralCategory::CurrencySymbol), ('\u{25}', '\u{27}',
        GeneralCategory::OtherPunctuation), ('\u{28}', '\u{28}', GeneralCategory::OpenPunctuation),
        ('\u{29}', '\u{29}', GeneralCategory::ClosePunctuation), ('\u{2a}', '\u{2a}',
        GeneralCategory::OtherPunctuation), ('\u{2b}', '\u{2b}', GeneralCategory::MathSymbol),
        ('\u{2c}', '\u{2c}', GeneralCategory::OtherPunctuation), ('\u{2d}', '\u{2d}',
        GeneralCategory::DashPunctuation), ('\u{2e}', '\u{2f}', GeneralCategory::OtherPunctuation),
        ('\u{30}', '\u{39}', GeneralCategory::DecimalNumber), ('\u{3a}', '\u{3b}',
        GeneralCategory::OtherPunctuation), ('\u{3c}', '\u{3e}', GeneralCategory::MathSymbol),
        ('\u{3f}', '\u{40}', GeneralCategory::OtherPunctuation), ('\u{41}', '\u{5a}',
        GeneralCategory::UppercaseLetter), ('\u{5b}', '\u{5b}', GeneralCategory::OpenPunctuation),
        ('\u{5c}', '\u{5c}', GeneralCategory::OtherPunctuation), ('\u{5d}', '\u{5d}',
        GeneralCategory::ClosePunctuation), ('\u{5e}', '\u{5e}', GeneralCategory::ModifierSymbol),
        ('\u{5f}', '\u{5f}', GeneralCategory::ConnectorPunctuation), ('\u{60}', '\u{60}',
        GeneralCategory::ModifierSymbol), ('\u{61}', '\u{7a}', GeneralCategory::LowercaseLetter),
        ('\u{7b}', '\u{7b}', GeneralCategory::OpenPunctuation), ('\u{7c}', '\u{7c}',
        GeneralCategory::MathSymbol), ('\u{7d}', '\u{7d}', GeneralCategory::ClosePunctuation),
        ('\u{7e}', '\u{7e}', GeneralCategory::MathSymbol), ('\u{7f}', '\u{9f}',
        GeneralCategory::Control), ('\u{a0}', '\u{a0}', GeneralCategory::SpaceSeparator), ('\u{a1}',
        '\u{a1}', GeneralCategory::OtherPunctuation), ('\u{a2}', '\u{a5}',
        GeneralCategory::CurrencySymbol), ('\u{a6}', '\u{a6}', GeneralCategory::OtherSymbol),
        ('\u{a7}', '\u{a7}', GeneralCategory::OtherPunctuation), ('\u{a8}', '\u{a8}',
        GeneralCategory::ModifierSymbol), ('\u{a9}', '\u{a9}', GeneralCategory::OtherSymbol),
        ('\u{aa}', '\u{aa}', GeneralCategory::OtherLetter), ('\u{ab}', '\u{ab}',
        GeneralCategory::InitialPunctuation), ('\u{ac}', '\u{ac}', GeneralCategory::MathSymbol),
        ('\u{ad}', '\u{ad}', GeneralCategory::Format), ('\u{ae}', '\u{ae}',
        GeneralCategory::OtherSymbol), ('\u{af}', '\u{af}', GeneralCategory::ModifierSymbol),
        ('\u{b0}', '\u{b0}', GeneralCategory::OtherSymbol), ('\u{b1}', '\u{b1}',
        GeneralCategory::MathSymbol), ('\u{b2}', '\u{b3}', GeneralCategory::OtherNumber), ('\u{b4}',
        '\u{b4}', GeneralCategory::ModifierSymbol), ('\u{b5}', '\u{b5}',
        GeneralCategory::LowercaseLetter), ('\u{b6}', '\u{b7}', GeneralCategory::OtherPunctuation),
        ('\u{b8}', '\u{b8}', GeneralCategory::ModifierSymbol), ('\u{b9}', '\u{b9}',
        GeneralCategory::OtherNumber), ('\u{ba}', '\u{ba}', GeneralCategory::OtherLetter),
        ('\u{bb}', '\u{bb}', GeneralCategory::FinalPunctuation), ('\u{bc}', '\u{be}',
        GeneralCategory::OtherNumber), ('\u{bf}', '\u{bf}', GeneralCategory::OtherPunctuation),
        ('\u{c0}', '\u{d6}', GeneralCategory::UppercaseLetter), ('\u{d7}', '\u{d7}',
        GeneralCategory::MathSymbol), ('\u{d8}', '\u{de}', GeneralCategory::UppercaseLetter),
        ('\u{df}', '\u{f6}', GeneralCategory::LowercaseLetter), ('\u{f7}', '\u{f7}',
        GeneralCategory::MathSymbol), ('\u{f8}', '\u{ff}', GeneralCategory::LowercaseLetter),
        ('\u{100}', '\u{100}', GeneralCategory::UppercaseLetter), ('\u{101}', '\u{101}',
        GeneralCategory::LowercaseLetter), ('\u{102}', '\u{102}', GeneralCategory::UppercaseLetter),
        ('\u{103}', '\u{103}', GeneralCategory::LowercaseLetter), ('\u{104}', '\u{104}',
        GeneralCategory::UppercaseLetter), ('\u{105}', '\u{105}', GeneralCategory::LowercaseLetter),
        ('\u{106}', '\u{106}', GeneralCategory::UppercaseLetter), ('\u{107}', '\u{107}',
        GeneralCategory::LowercaseLetter), ('\u{108}', '\u{108}', GeneralCategory::UppercaseLetter),
        ('\u{109}', '\u{109}', GeneralCategory::LowercaseLetter), ('\u{10a}', '\u{10a}',
        GeneralCategory::UppercaseLetter), ('\u{10b}', '\u{10b}', GeneralCategory::LowercaseLetter),
        ('\u{10c}', '\u{10c}', GeneralCategory::UppercaseLetter), ('\u{10d}', '\u{10d}',
        GeneralCategory::LowercaseLetter), ('\u{10e}', '\u{10e}', GeneralCategory::UppercaseLetter),
        ('\u{10f}', '\u{10f}', GeneralCategory::LowercaseLetter), ('\u{110}', '\u{110}',
        GeneralCategory::UppercaseLetter), ('\u{111}', '\u{111}', GeneralCategory::LowercaseLetter),
        ('\u{112}', '\u{112}', GeneralCategory::UppercaseLetter), ('\u{113}', '\u{113}',
        GeneralCategory::LowercaseLetter), ('\u{114}', '\u{114}', GeneralCategory::UppercaseLetter),
        ('\u{115}', '\u{115}', GeneralCategory::LowercaseLetter), ('\u{116}', '\u{116}',
        GeneralCategory::UppercaseLetter), ('\u{117}', '\u{117}', GeneralCategory::LowercaseLetter),
        ('\u{118}', '\u{118}', GeneralCategory::UppercaseLetter), ('\u{119}', '\u{119}',
        GeneralCategory::LowercaseLetter), ('\u{11a}', '\u{11a}', GeneralCategory::UppercaseLetter),
        ('\u{11b}', '\u{11b}', GeneralCategory::LowercaseLetter), ('\u{11c}', '\u{11c}',
        GeneralCategory::UppercaseLetter), ('\u{11d}', '\u{11d}', GeneralCategory::LowercaseLetter),
        ('\u{11e}', '\u{11e}', GeneralCategory::UppercaseLetter), ('\u{11f}', '\u{11f}',
        GeneralCategory::LowercaseLetter), ('\u{120}', '\u{120}', GeneralCategory::UppercaseLetter),
        ('\u{121}', '\u{121}', GeneralCategory::LowercaseLetter), ('\u{122}', '\u{122}',
        GeneralCategory::UppercaseLetter), ('\u{123}', '\u{123}', GeneralCategory::LowercaseLetter),
        ('\u{124}', '\u{124}', GeneralCategory::UppercaseLetter), ('\u{125}', '\u{125}',
        GeneralCategory::LowercaseLetter), ('\u{126}', '\u{126}', GeneralCategory::UppercaseLetter),
        ('\u{127}', '\u{127}', GeneralCategory::LowercaseLetter), ('\u{128}', '\u{128}',
        GeneralCategory::UppercaseLetter), ('\u{129}', '\u{129}', GeneralCategory::LowercaseLetter),
        ('\u{12a}', '\u{12a}', GeneralCategory::UppercaseLetter), ('\u{12b}', '\u{12b}',
        GeneralCategory::LowercaseLetter), ('\u{12c}', '\u{12c}', GeneralCategory::UppercaseLetter),
        ('\u{12d}', '\u{12d}', GeneralCategory::LowercaseLetter), ('\u{12e}', '\u{12e}',
        GeneralCategory::UppercaseLetter), ('\u{12f}', '\u{12f}', GeneralCategory::LowercaseLetter),
        ('\u{130}', '\u{130}', GeneralCategory::UppercaseLetter), ('\u{131}', '\u{131}',
        GeneralCategory::LowercaseLetter), ('\u{132}', '\u{132}', GeneralCategory::UppercaseLetter),
        ('\u{133}', '\u{133}', GeneralCategory::LowercaseLetter), ('\u{134}', '\u{134}',
        GeneralCategory::UppercaseLetter), ('\u{135}', '\u{135}', GeneralCategory::LowercaseLetter),
        ('\u{136}', '\u{136}', GeneralCategory::UppercaseLetter), ('\u{137}', '\u{138}',
        GeneralCategory::LowercaseLetter), ('\u{139}', '\u{139}', GeneralCategory::UppercaseLetter),
        ('\u{13a}', '\u{13a}', GeneralCategory::LowercaseLetter), ('\u{13b}', '\u{13b}',
        GeneralCategory::UppercaseLetter), ('\u{13c}', '\u{13c}', GeneralCategory::LowercaseLetter),
        ('\u{13d}', '\u{13d}', GeneralCategory::UppercaseLetter), ('\u{13e}', '\u{13e}',
        GeneralCategory::LowercaseLetter), ('\u{13f}', '\u{13f}', GeneralCategory::UppercaseLetter),
        ('\u{140}', '\u{140}', GeneralCategory::LowercaseLetter), ('\u{141}', '\u{141}',
        GeneralCategory::UppercaseLetter), ('\u{142}', '\u{142}', GeneralCategory::LowercaseLetter),
        ('\u{143}', '\u{143}', GeneralCategory::UppercaseLetter), ('\u{144}', '\u{144}',
        GeneralCategory::LowercaseLetter), ('\u{145}', '\u{145}', GeneralCategory::UppercaseLetter),
        ('\u{146}', '\u{146}', GeneralCategory::LowercaseLetter), ('\u{147}', '\u{147}',
        GeneralCategory::UppercaseLetter), ('\u{148}', '\u{149}', GeneralCategory::LowercaseLetter),
        ('\u{14a}', '\u{14a}', GeneralCategory::UppercaseLetter), ('\u{14b}', '\u{14b}',
        GeneralCategory::LowercaseLetter), ('\u{14c}', '\u{14c}', GeneralCategory::UppercaseLetter),
        ('\u{14d}', '\u{14d}', GeneralCategory::LowercaseLetter), ('\u{14e}', '\u{14e}',
        GeneralCategory::UppercaseLetter), ('\u{14f}', '\u{14f}', GeneralCategory::LowercaseLetter),
        ('\u{150}', '\u{150}', GeneralCategory::UppercaseLetter), ('\u{151}', '\u{151}',
        GeneralCategory::LowercaseLetter), ('\u{152}', '\u{152}', GeneralCategory::UppercaseLetter),
        ('\u{153}', '\u{153}', GeneralCategory::LowercaseLetter), ('\u{154}', '\u{154}',
        GeneralCategory::UppercaseLetter), ('\u{155}', '\u{155}', GeneralCategory::LowercaseLetter),
        ('\u{156}', '\u{156}', GeneralCategory::UppercaseLetter), ('\u{157}', '\u{157}',
        GeneralCategory::LowercaseLetter), ('\u{158}', '\u{158}', GeneralCategory::UppercaseLetter),
        ('\u{159}', '\u{159}', GeneralCategory::LowercaseLetter), ('\u{15a}', '\u{15a}',
        GeneralCategory::UppercaseLetter), ('\u{15b}', '\u{15b}', GeneralCategory::LowercaseLetter),
        ('\u{15c}', '\u{15c}', GeneralCategory::UppercaseLetter), ('\u{15d}', '\u{15d}',
        GeneralCategory::LowercaseLetter), ('\u{15e}', '\u{15e}', GeneralCategory::UppercaseLetter),
        ('\u{15f}', '\u{15f}', GeneralCategory::LowercaseLetter), ('\u{160}', '\u{160}',
        GeneralCategory::UppercaseLetter), ('\u{161}', '\u{161}', GeneralCategory::LowercaseLetter),
        ('\u{162}', '\u{162}', GeneralCategory::UppercaseLetter), ('\u{163}', '\u{163}',
        GeneralCategory::LowercaseLetter), ('\u{164}', '\u{164}', GeneralCategory::UppercaseLetter),
        ('\u{165}', '\u{165}', GeneralCategory::LowercaseLetter), ('\u{166}', '\u{166}',
        GeneralCategory::UppercaseLetter), ('\u{167}', '\u{167}', GeneralCategory::LowercaseLetter),
        ('\u{168}', '\u{168}', GeneralCategory::UppercaseLetter), ('\u{169}', '\u{169}',
        GeneralCategory::LowercaseLetter), ('\u{16a}', '\u{16a}', GeneralCategory::UppercaseLetter),
        ('\u{16b}', '\u{16b}', GeneralCategory::LowercaseLetter), ('\u{16c}', '\u{16c}',
        GeneralCategory::UppercaseLetter), ('\u{16d}', '\u{16d}', GeneralCategory::LowercaseLetter),
        ('\u{16e}', '\u{16e}', GeneralCategory::UppercaseLetter), ('\u{16f}', '\u{16f}',
        GeneralCategory::LowercaseLetter), ('\u{170}', '\u{170}', GeneralCategory::UppercaseLetter),
        ('\u{171}', '\u{171}', GeneralCategory::LowercaseLetter), ('\u{172}', '\u{172}',
        GeneralCategory::UppercaseLetter), ('\u{173}', '\u{173}', GeneralCategory::LowercaseLetter),
        ('\u{174}', '\u{174}', GeneralCategory::UppercaseLetter), ('\u{175}', '\u{175}',
        GeneralCategory::LowercaseLetter), ('\u{176}', '\u{176}', GeneralCategory::UppercaseLetter),
        ('\u{177}', '\u{177}', GeneralCategory::LowercaseLetter), ('\u{178}', '\u{179}',
        GeneralCategory::UppercaseLetter), ('\u{17a}', '\u{17a}', GeneralCategory::LowercaseLetter),
        ('\u{17b}', '\u{17b}', GeneralCategory::UppercaseLetter), ('\u{17c}', '\u{17c}',
        GeneralCategory::LowercaseLetter), ('\u{17d}', '\u{17d}', GeneralCategory::UppercaseLetter),
        ('\u{17e}', '\u{180}', GeneralCategory::LowercaseLetter), ('\u{181}', '\u{182}',
        GeneralCategory::UppercaseLetter), ('\u{183}', '\u{183}', GeneralCategory::LowercaseLetter),
        ('\u{184}', '\u{184}', GeneralCategory::UppercaseLetter), ('\u{185}', '\u{185}',
        GeneralCategory::LowercaseLetter), ('\u{186}', '\u{187}', GeneralCategory::UppercaseLetter),
        ('\u{188}', '\u{188}', GeneralCategory::LowercaseLetter), ('\u{189}', '\u{18b}',
        GeneralCategory::UppercaseLetter), ('\u{18c}', '\u{18d}', GeneralCategory::LowercaseLetter),
        ('\u{18e}', '\u{191}', GeneralCategory::UppercaseLetter), ('\u{192}', '\u{192}',
        GeneralCategory::LowercaseLetter), ('\u{193}', '\u{194}', GeneralCategory::UppercaseLetter),
        ('\u{195}', '\u{195}', GeneralCategory::LowercaseLetter), ('\u{196}', '\u{198}',
        GeneralCategory::UppercaseLetter), ('\u{199}', '\u{19b}', GeneralCategory::LowercaseLetter),
        ('\u{19c}', '\u{19d}', GeneralCategory::UppercaseLetter), ('\u{19e}', '\u{19e}',
        GeneralCategory::LowercaseLetter), ('\u{19f}', '\u{1a0}', GeneralCategory::UppercaseLetter),
        ('\u{1a1}', '\u{1a1}', GeneralCategory::LowercaseLetter), ('\u{1a2}', '\u{1a2}',
        GeneralCategory::UppercaseLetter), ('\u{1a3}', '\u{1a3}', GeneralCategory::LowercaseLetter),
        ('\u{1a4}', '\u{1a4}', GeneralCategory::UppercaseLetter), ('\u{1a5}', '\u{1a5}',
        GeneralCategory::LowercaseLetter), ('\u{1a6}', '\u{1a7}', GeneralCategory::UppercaseLetter),
        ('\u{1a8}', '\u{1a8}', GeneralCategory::LowercaseLetter), ('\u{1a9}', '\u{1a9}',
        GeneralCategory::UppercaseLetter), ('\u{1aa}', '\u{1ab}', GeneralCategory::LowercaseLetter),
        ('\u{1ac}', '\u{1ac}', GeneralCategory::UppercaseLetter), ('\u{1ad}', '\u{1ad}',
        GeneralCategory::LowercaseLetter), ('\u{1ae}', '\u{1af}', GeneralCategory::UppercaseLetter),
        ('\u{1b0}', '\u{1b0}', GeneralCategory::LowercaseLetter), ('\u{1b1}', '\u{1b3}',
        GeneralCategory::UppercaseLetter), ('\u{1b4}', '\u{1b4}', GeneralCategory::LowercaseLetter),
        ('\u{1b5}', '\u{1b5}', GeneralCategory::UppercaseLetter), ('\u{1b6}', '\u{1b6}',
        GeneralCategory::LowercaseLetter), ('\u{1b7}', '\u{1b8}', GeneralCategory::UppercaseLetter),
        ('\u{1b9}', '\u{1ba}', GeneralCategory::LowercaseLetter), ('\u{1bb}', '\u{1bb}',
        GeneralCategory::OtherLetter), ('\u{1bc}', '\u{1bc}', GeneralCategory::UppercaseLetter),
        ('\u{1bd}', '\u{1bf}', GeneralCategory::LowercaseLetter), ('\u{1c0}', '\u{1c3}',
        GeneralCategory::OtherLetter), ('\u{1c4}', '\u{1c4}', GeneralCategory::UppercaseLetter),
        ('\u{1c5}', '\u{1c5}', GeneralCategory::TitlecaseLetter), ('\u{1c6}', '\u{1c6}',
        GeneralCategory::LowercaseLetter), ('\u{1c7}', '\u{1c7}', GeneralCategory::UppercaseLetter),
        ('\u{1c8}', '\u{1c8}', GeneralCategory::TitlecaseLetter), ('\u{1c9}', '\u{1c9}',
        GeneralCategory::LowercaseLetter), ('\u{1ca}', '\u{1ca}', GeneralCategory::UppercaseLetter),
        ('\u{1cb}', '\u{1cb}', GeneralCategory::TitlecaseLetter), ('\u{1cc}', '\u{1cc}',
        GeneralCategory::LowercaseLetter), ('\u{1cd}', '\u{1cd}', GeneralCategory::UppercaseLetter),
        ('\u{1ce}', '\u{1ce}', GeneralCategory::LowercaseLetter), ('\u{1cf}', '\u{1cf}',
        GeneralCategory::UppercaseLetter), ('\u{1d0}', '\u{1d0}', GeneralCategory::LowercaseLetter),
        ('\u{1d1}', '\u{1d1}', GeneralCategory::UppercaseLetter), ('\u{1d2}', '\u{1d2}',
        GeneralCategory::LowercaseLetter), ('\u{1d3}', '\u{1d3}', GeneralCategory::UppercaseLetter),
        ('\u{1d4}', '\u{1d4}', GeneralCategory::LowercaseLetter), ('\u{1d5}', '\u{1d5}',
        GeneralCategory::UppercaseLetter), ('\u{1d6}', '\u{1d6}', GeneralCategory::LowercaseLetter),
        ('\u{1d7}', '\u{1d7}', GeneralCategory::UppercaseLetter), ('\u{1d8}', '\u{1d8}',
        GeneralCategory::LowercaseLetter), ('\u{1d9}', '\u{1d9}', GeneralCategory::UppercaseLetter),
        ('\u{1da}', '\u{1da}', GeneralCategory::LowercaseLetter), ('\u{1db}', '\u{1db}',
        GeneralCategory::UppercaseLetter), ('\u{1dc}', '\u{1dd}', GeneralCategory::LowercaseLetter),
        ('\u{1de}', '\u{1de}', GeneralCategory::UppercaseLetter), ('\u{1df}', '\u{1df}',
        GeneralCategory::LowercaseLetter), ('\u{1e0}', '\u{1e0}', GeneralCategory::UppercaseLetter),
        ('\u{1e1}', '\u{1e1}', GeneralCategory::LowercaseLetter), ('\u{1e2}', '\u{1e2}',
        GeneralCategory::UppercaseLetter), ('\u{1e3}', '\u{1e3}', GeneralCategory::LowercaseLetter),
        ('\u{1e4}', '\u{1e4}', GeneralCategory::UppercaseLetter), ('\u{1e5}', '\u{1e5}',
        GeneralCategory::LowercaseLetter), ('\u{1e6}', '\u{1e6}', GeneralCategory::UppercaseLetter),
        ('\u{1e7}', '\u{1e7}', GeneralCategory::LowercaseLetter), ('\u{1e8}', '\u{1e8}',
        GeneralCategory::UppercaseLetter), ('\u{1e9}', '\u{1e9}', GeneralCategory::LowercaseLetter),
        ('\u{1ea}', '\u{1ea}', GeneralCategory::UppercaseLetter), ('\u{1eb}', '\u{1eb}',
        GeneralCategory::LowercaseLetter), ('\u{1ec}', '\u{1ec}', GeneralCategory::UppercaseLetter),
        ('\u{1ed}', '\u{1ed}', GeneralCategory::LowercaseLetter), ('\u{1ee}', '\u{1ee}',
        GeneralCategory::UppercaseLetter), ('\u{1ef}', '\u{1f0}', GeneralCategory::LowercaseLetter),
        ('\u{1f1}', '\u{1f1}', GeneralCategory::UppercaseLetter), ('\u{1f2}', '\u{1f2}',
        GeneralCategory::TitlecaseLetter), ('\u{1f3}', '\u{1f3}', GeneralCategory::LowercaseLetter),
        ('\u{1f4}', '\u{1f4}', GeneralCategory::UppercaseLetter), ('\u{1f5}', '\u{1f5}',
        GeneralCategory::LowercaseLetter), ('\u{1f6}', '\u{1f8}', GeneralCategory::UppercaseLetter),
        ('\u{1f9}', '\u{1f9}', GeneralCategory::LowercaseLetter), ('\u{1fa}', '\u{1fa}',
        GeneralCategory::UppercaseLetter), ('\u{1fb}', '\u{1fb}', GeneralCategory::LowercaseLetter),
        ('\u{1fc}', '\u{1fc}', GeneralCategory::UppercaseLetter), ('\u{1fd}', '\u{1fd}',
        GeneralCategory::LowercaseLetter), ('\u{1fe}', '\u{1fe}', GeneralCategory::UppercaseLetter),
        ('\u{1ff}', '\u{1ff}', GeneralCategory::LowercaseLetter), ('\u{200}', '\u{200}',
        GeneralCategory::UppercaseLetter), ('\u{201}', '\u{201}', GeneralCategory::LowercaseLetter),
        ('\u{202}', '\u{202}', GeneralCategory::UppercaseLetter), ('\u{203}', '\u{203}',
        GeneralCategory::LowercaseLetter), ('\u{204}', '\u{204}', GeneralCategory::UppercaseLetter),
        ('\u{205}', '\u{205}', GeneralCategory::LowercaseLetter), ('\u{206}', '\u{206}',
        GeneralCategory::UppercaseLetter), ('\u{207}', '\u{207}', GeneralCategory::LowercaseLetter),
        ('\u{208}', '\u{208}', GeneralCategory::UppercaseLetter), ('\u{209}', '\u{209}',
        GeneralCategory::LowercaseLetter), ('\u{20a}', '\u{20a}', GeneralCategory::UppercaseLetter),
        ('\u{20b}', '\u{20b}', GeneralCategory::LowercaseLetter), ('\u{20c}', '\u{20c}',
        GeneralCategory::UppercaseLetter), ('\u{20d}', '\u{20d}', GeneralCategory::LowercaseLetter),
        ('\u{20e}', '\u{20e}', GeneralCategory::UppercaseLetter), ('\u{20f}', '\u{20f}',
        GeneralCategory::LowercaseLetter), ('\u{210}', '\u{210}', GeneralCategory::UppercaseLetter),
        ('\u{211}', '\u{211}', GeneralCategory::LowercaseLetter), ('\u{212}', '\u{212}',
        GeneralCategory::UppercaseLetter), ('\u{213}', '\u{213}', GeneralCategory::LowercaseLetter),
        ('\u{214}', '\u{214}', GeneralCategory::UppercaseLetter), ('\u{215}', '\u{215}',
        GeneralCategory::LowercaseLetter), ('\u{216}', '\u{216}', GeneralCategory::UppercaseLetter),
        ('\u{217}', '\u{217}', GeneralCategory::LowercaseLetter), ('\u{218}', '\u{218}',
        GeneralCategory::UppercaseLetter), ('\u{219}', '\u{219}', GeneralCategory::LowercaseLetter),
        ('\u{21a}', '\u{21a}', GeneralCategory::UppercaseLetter), ('\u{21b}', '\u{21b}',
        GeneralCategory::LowercaseLetter), ('\u{21c}', '\u{21c}', GeneralCategory::UppercaseLetter),
        ('\u{21d}', '\u{21d}', GeneralCategory::LowercaseLetter), ('\u{21e}', '\u{21e}',
        GeneralCategory::UppercaseLetter), ('\u{21f}', '\u{21f}', GeneralCategory::LowercaseLetter),
        ('\u{220}', '\u{220}', GeneralCategory::UppercaseLetter), ('\u{221}', '\u{221}',
        GeneralCategory::LowercaseLetter), ('\u{222}', '\u{222}', GeneralCategory::UppercaseLetter),
        ('\u{223}', '\u{223}', GeneralCategory::LowercaseLetter), ('\u{224}', '\u{224}',
        GeneralCategory::UppercaseLetter), ('\u{225}', '\u{225}', GeneralCategory::LowercaseLetter),
        ('\u{226}', '\u{226}', GeneralCategory::UppercaseLetter), ('\u{227}', '\u{227}',
        GeneralCategory::LowercaseLetter), ('\u{228}', '\u{228}', GeneralCategory::UppercaseLetter),
        ('\u{229}', '\u{229}', GeneralCategory::LowercaseLetter), ('\u{22a}', '\u{22a}',
        GeneralCategory::UppercaseLetter), ('\u{22b}', '\u{22b}', GeneralCategory::LowercaseLetter),
        ('\u{22c}', '\u{22c}', GeneralCategory::UppercaseLetter), ('\u{22d}', '\u{22d}',
        GeneralCategory::LowercaseLetter), ('\u{22e}', '\u{22e}', GeneralCategory::UppercaseLetter),
        ('\u{22f}', '\u{22f}', GeneralCategory::LowercaseLetter), ('\u{230}', '\u{230}',
        GeneralCategory::UppercaseLetter), ('\u{231}', '\u{231}', GeneralCategory::LowercaseLetter),
        ('\u{232}', '\u{232}', GeneralCategory::UppercaseLetter), ('\u{233}', '\u{239}',
        GeneralCategory::LowercaseLetter), ('\u{23a}', '\u{23b}', GeneralCategory::UppercaseLetter),
        ('\u{23c}', '\u{23c}', GeneralCategory::LowercaseLetter), ('\u{23d}', '\u{23e}',
        GeneralCategory::UppercaseLetter), ('\u{23f}', '\u{240}', GeneralCategory::LowercaseLetter),
        ('\u{241}', '\u{241}', GeneralCategory::UppercaseLetter), ('\u{242}', '\u{242}',
        GeneralCategory::LowercaseLetter), ('\u{243}', '\u{246}', GeneralCategory::UppercaseLetter),
        ('\u{247}', '\u{247}', GeneralCategory::LowercaseLetter), ('\u{248}', '\u{248}',
        GeneralCategory::UppercaseLetter), ('\u{249}', '\u{249}', GeneralCategory::LowercaseLetter),
        ('\u{24a}', '\u{24a}', GeneralCategory::UppercaseLetter), ('\u{24b}', '\u{24b}',
        GeneralCategory::LowercaseLetter), ('\u{24c}', '\u{24c}', GeneralCategory::UppercaseLetter),
        ('\u{24d}', '\u{24d}', GeneralCategory::LowercaseLetter), ('\u{24e}', '\u{24e}',
        GeneralCategory::UppercaseLetter), ('\u{24f}', '\u{293}', GeneralCategory::LowercaseLetter),
        ('\u{294}', '\u{294}', GeneralCategory::OtherLetter), ('\u{295}', '\u{2af}',
        GeneralCategory::LowercaseLetter), ('\u{2b0}', '\u{2c1}', GeneralCategory::ModifierLetter),
        ('\u{2c2}', '\u{2c5}', GeneralCategory::ModifierSymbol), ('\u{2c6}', '\u{2d1}',
        GeneralCategory::ModifierLetter), ('\u{2d2}', '\u{2df}', GeneralCategory::ModifierSymbol),
        ('\u{2e0}', '\u{2e4}', GeneralCategory::ModifierLetter), ('\u{2e5}', '\u{2eb}',
        GeneralCategory::ModifierSymbol), ('\u{2ec}', '\u{2ec}', GeneralCategory::ModifierLetter),
        ('\u{2ed}', '\u{2ed}', GeneralCategory::ModifierSymbol), ('\u{2ee}', '\u{2ee}',
        GeneralCategory::ModifierLetter), ('\u{2ef}', '\u{2ff}', GeneralCategory::ModifierSymbol),
        ('\u{300}', '\u{36f}', GeneralCategory::NonspacingMark), ('\u{370}', '\u{370}',
        GeneralCategory::UppercaseLetter), ('\u{371}', '\u{371}', GeneralCategory::LowercaseLetter),
        ('\u{372}', '\u{372}', GeneralCategory::UppercaseLetter), ('\u{373}', '\u{373}',
        GeneralCategory::LowercaseLetter), ('\u{374}', '\u{374}', GeneralCategory::ModifierLetter),
        ('\u{375}', '\u{375}', GeneralCategory::ModifierSymbol), ('\u{376}', '\u{376}',
        GeneralCategory::UppercaseLetter), ('\u{377}', '\u{377}', GeneralCategory::LowercaseLetter),
        ('\u{37a}', '\u{37a}', GeneralCategory::ModifierLetter), ('\u{37b}', '\u{37d}',
        GeneralCategory::LowercaseLetter), ('\u{37e}', '\u{37e}',
        GeneralCategory::OtherPunctuation), ('\u{37f}', '\u{37f}',
        GeneralCategory::UppercaseLetter), ('\u{384}', '\u{385}', GeneralCategory::ModifierSymbol),
        ('\u{386}', '\u{386}', GeneralCategory::UppercaseLetter), ('\u{387}', '\u{387}',
        GeneralCategory::OtherPunctuation), ('\u{388}', '\u{38a}',
        GeneralCategory::UppercaseLetter), ('\u{38c}', '\u{38c}', GeneralCategory::UppercaseLetter),
        ('\u{38e}', '\u{38f}', GeneralCategory::UppercaseLetter), ('\u{390}', '\u{390}',
        GeneralCategory::LowercaseLetter), ('\u{391}', '\u{3a1}', GeneralCategory::UppercaseLetter),
        ('\u{3a3}', '\u{3ab}', GeneralCategory::UppercaseLetter), ('\u{3ac}', '\u{3ce}',
        GeneralCategory::LowercaseLetter), ('\u{3cf}', '\u{3cf}', GeneralCategory::UppercaseLetter),
        ('\u{3d0}', '\u{3d1}', GeneralCategory::LowercaseLetter), ('\u{3d2}', '\u{3d4}',
        GeneralCategory::UppercaseLetter), ('\u{3d5}', '\u{3d7}', GeneralCategory::LowercaseLetter),
        ('\u{3d8}', '\u{3d8}', GeneralCategory::UppercaseLetter), ('\u{3d9}', '\u{3d9}',
        GeneralCategory::LowercaseLetter), ('\u{3da}', '\u{3da}', GeneralCategory::UppercaseLetter),
        ('\u{3db}', '\u{3db}', GeneralCategory::LowercaseLetter), ('\u{3dc}', '\u{3dc}',
        GeneralCategory::UppercaseLetter), ('\u{3dd}', '\u{3dd}', GeneralCategory::LowercaseLetter),
        ('\u{3de}', '\u{3de}', GeneralCategory::UppercaseLetter), ('\u{3df}', '\u{3df}',
        GeneralCategory::LowercaseLetter), ('\u{3e0}', '\u{3e0}', GeneralCategory::UppercaseLetter),
        ('\u{3e1}', '\u{3e1}', GeneralCategory::LowercaseLetter), ('\u{3e2}', '\u{3e2}',
        GeneralCategory::UppercaseLetter), ('\u{3e3}', '\u{3e3}', GeneralCategory::LowercaseLetter),
        ('\u{3e4}', '\u{3e4}', GeneralCategory::UppercaseLetter), ('\u{3e5}', '\u{3e5}',
        GeneralCategory::LowercaseLetter), ('\u{3e6}', '\u{3e6}', GeneralCategory::UppercaseLetter),
        ('\u{3e7}', '\u{3e7}', GeneralCategory::LowercaseLetter), ('\u{3e8}', '\u{3e8}',
        GeneralCategory::UppercaseLetter), ('\u{3e9}', '\u{3e9}', GeneralCategory::LowercaseLetter),
        ('\u{3ea}', '\u{3ea}', GeneralCategory::UppercaseLetter), ('\u{3eb}', '\u{3eb}',
        GeneralCategory::LowercaseLetter), ('\u{3ec}', '\u{3ec}', GeneralCategory::UppercaseLetter),
        ('\u{3ed}', '\u{3ed}', GeneralCategory::LowercaseLetter), ('\u{3ee}', '\u{3ee}',
        GeneralCategory::UppercaseLetter), ('\u{3ef}', '\u{3f3}', GeneralCategory::LowercaseLetter),
        ('\u{3f4}', '\u{3f4}', GeneralCategory::UppercaseLetter), ('\u{3f5}', '\u{3f5}',
        GeneralCategory::LowercaseLetter), ('\u{3f6}', '\u{3f6}', GeneralCategory::MathSymbol),
        ('\u{3f7}', '\u{3f7}', GeneralCategory::UppercaseLetter), ('\u{3f8}', '\u{3f8}',
        GeneralCategory::LowercaseLetter), ('\u{3f9}', '\u{3fa}', GeneralCategory::UppercaseLetter),
        ('\u{3fb}', '\u{3fc}', GeneralCategory::LowercaseLetter), ('\u{3fd}', '\u{42f}',
        GeneralCategory::UppercaseLetter), ('\u{430}', '\u{45f}', GeneralCategory::LowercaseLetter),
        ('\u{460}', '\u{460}', GeneralCategory::UppercaseLetter), ('\u{461}', '\u{461}',
        GeneralCategory::LowercaseLetter), ('\u{462}', '\u{462}', GeneralCategory::UppercaseLetter),
        ('\u{463}', '\u{463}', GeneralCategory::LowercaseLetter), ('\u{464}', '\u{464}',
        GeneralCategory::UppercaseLetter), ('\u{465}', '\u{465}', GeneralCategory::LowercaseLetter),
        ('\u{466}', '\u{466}', GeneralCategory::UppercaseLetter), ('\u{467}', '\u{467}',
        GeneralCategory::LowercaseLetter), ('\u{468}', '\u{468}', GeneralCategory::UppercaseLetter),
        ('\u{469}', '\u{469}', GeneralCategory::LowercaseLetter), ('\u{46a}', '\u{46a}',
        GeneralCategory::UppercaseLetter), ('\u{46b}', '\u{46b}', GeneralCategory::LowercaseLetter),
        ('\u{46c}', '\u{46c}', GeneralCategory::UppercaseLetter), ('\u{46d}', '\u{46d}',
        GeneralCategory::LowercaseLetter), ('\u{46e}', '\u{46e}', GeneralCategory::UppercaseLetter),
        ('\u{46f}', '\u{46f}', GeneralCategory::LowercaseLetter), ('\u{470}', '\u{470}',
        GeneralCategory::UppercaseLetter), ('\u{471}', '\u{471}', GeneralCategory::LowercaseLetter),
        ('\u{472}', '\u{472}', GeneralCategory::UppercaseLetter), ('\u{473}', '\u{473}',
        GeneralCategory::LowercaseLetter), ('\u{474}', '\u{474}', GeneralCategory::UppercaseLetter),
        ('\u{475}', '\u{475}', GeneralCategory::LowercaseLetter), ('\u{476}', '\u{476}',
        GeneralCategory::UppercaseLetter), ('\u{477}', '\u{477}', GeneralCategory::LowercaseLetter),
        ('\u{478}', '\u{478}', GeneralCategory::UppercaseLetter), ('\u{479}', '\u{479}',
        GeneralCategory::LowercaseLetter), ('\u{47a}', '\u{47a}', GeneralCategory::UppercaseLetter),
        ('\u{47b}', '\u{47b}', GeneralCategory::LowercaseLetter), ('\u{47c}', '\u{47c}',
        GeneralCategory::UppercaseLetter), ('\u{47d}', '\u{47d}', GeneralCategory::LowercaseLetter),
        ('\u{47e}', '\u{47e}', GeneralCategory::UppercaseLetter), ('\u{47f}', '\u{47f}',
        GeneralCategory::LowercaseLetter), ('\u{480}', '\u{480}', GeneralCategory::UppercaseLetter),
        ('\u{481}', '\u{481}', GeneralCategory::LowercaseLetter), ('\u{482}', '\u{482}',
        GeneralCategory::OtherSymbol), ('\u{483}', '\u{487}', GeneralCategory::NonspacingMark),
        ('\u{488}', '\u{489}', GeneralCategory::EnclosingMark), ('\u{48a}', '\u{48a}',
        GeneralCategory::UppercaseLetter), ('\u{48b}', '\u{48b}', GeneralCategory::LowercaseLetter),
        ('\u{48c}', '\u{48c}', GeneralCategory::UppercaseLetter), ('\u{48d}', '\u{48d}',
        GeneralCategory::LowercaseLetter), ('\u{48e}', '\u{48e}', GeneralCategory::UppercaseLetter),
        ('\u{48f}', '\u{48f}', GeneralCategory::LowercaseLetter), ('\u{490}', '\u{490}',
        GeneralCategory::UppercaseLetter), ('\u{491}', '\u{491}', GeneralCategory::LowercaseLetter),
        ('\u{492}', '\u{492}', GeneralCategory::UppercaseLetter), ('\u{493}', '\u{493}',
        GeneralCategory::LowercaseLetter), ('\u{494}', '\u{494}', GeneralCategory::UppercaseLetter),
        ('\u{495}', '\u{495}', GeneralCategory::LowercaseLetter), ('\u{496}', '\u{496}',
        GeneralCategory::UppercaseLetter), ('\u{497}', '\u{497}', GeneralCategory::LowercaseLetter),
        ('\u{498}', '\u{498}', GeneralCategory::UppercaseLetter), ('\u{499}', '\u{499}',
        GeneralCategory::LowercaseLetter), ('\u{49a}', '\u{49a}', GeneralCategory::UppercaseLetter),
        ('\u{49b}', '\u{49b}', GeneralCategory::LowercaseLetter), ('\u{49c}', '\u{49c}',
        GeneralCategory::UppercaseLetter), ('\u{49d}', '\u{49d}', GeneralCategory::LowercaseLetter),
        ('\u{49e}', '\u{49e}', GeneralCategory::UppercaseLetter), ('\u{49f}', '\u{49f}',
        GeneralCategory::LowercaseLetter), ('\u{4a0}', '\u{4a0}', GeneralCategory::UppercaseLetter),
        ('\u{4a1}', '\u{4a1}', GeneralCategory::LowercaseLetter), ('\u{4a2}', '\u{4a2}',
        GeneralCategory::UppercaseLetter), ('\u{4a3}', '\u{4a3}', GeneralCategory::LowercaseLetter),
        ('\u{4a4}', '\u{4a4}', GeneralCategory::UppercaseLetter), ('\u{4a5}', '\u{4a5}',
        GeneralCategory::LowercaseLetter), ('\u{4a6}', '\u{4a6}', GeneralCategory::UppercaseLetter),
        ('\u{4a7}', '\u{4a7}', GeneralCategory::LowercaseLetter), ('\u{4a8}', '\u{4a8}',
        GeneralCategory::UppercaseLetter), ('\u{4a9}', '\u{4a9}', GeneralCategory::LowercaseLetter),
        ('\u{4aa}', '\u{4aa}', GeneralCategory::UppercaseLetter), ('\u{4ab}', '\u{4ab}',
        GeneralCategory::LowercaseLetter), ('\u{4ac}', '\u{4ac}', GeneralCategory::UppercaseLetter),
        ('\u{4ad}', '\u{4ad}', GeneralCategory::LowercaseLetter), ('\u{4ae}', '\u{4ae}',
        GeneralCategory::UppercaseLetter), ('\u{4af}', '\u{4af}', GeneralCategory::LowercaseLetter),
        ('\u{4b0}', '\u{4b0}', GeneralCategory::UppercaseLetter), ('\u{4b1}', '\u{4b1}',
        GeneralCategory::LowercaseLetter), ('\u{4b2}', '\u{4b2}', GeneralCategory::UppercaseLetter),
        ('\u{4b3}', '\u{4b3}', GeneralCategory::LowercaseLetter), ('\u{4b4}', '\u{4b4}',
        GeneralCategory::UppercaseLetter), ('\u{4b5}', '\u{4b5}', GeneralCategory::LowercaseLetter),
        ('\u{4b6}', '\u{4b6}', GeneralCategory::UppercaseLetter), ('\u{4b7}', '\u{4b7}',
        GeneralCategory::LowercaseLetter), ('\u{4b8}', '\u{4b8}', GeneralCategory::UppercaseLetter),
        ('\u{4b9}', '\u{4b9}', GeneralCategory::LowercaseLetter), ('\u{4ba}', '\u{4ba}',
        GeneralCategory::UppercaseLetter), ('\u{4bb}', '\u{4bb}', GeneralCategory::LowercaseLetter),
        ('\u{4bc}', '\u{4bc}', GeneralCategory::UppercaseLetter), ('\u{4bd}', '\u{4bd}',
        GeneralCategory::LowercaseLetter), ('\u{4be}', '\u{4be}', GeneralCategory::UppercaseLetter),
        ('\u{4bf}', '\u{4bf}', GeneralCategory::LowercaseLetter), ('\u{4c0}', '\u{4c1}',
        GeneralCategory::UppercaseLetter), ('\u{4c2}', '\u{4c2}', GeneralCategory::LowercaseLetter),
        ('\u{4c3}', '\u{4c3}', GeneralCategory::UppercaseLetter), ('\u{4c4}', '\u{4c4}',
        GeneralCategory::LowercaseLetter), ('\u{4c5}', '\u{4c5}', GeneralCategory::UppercaseLetter),
        ('\u{4c6}', '\u{4c6}', GeneralCategory::LowercaseLetter), ('\u{4c7}', '\u{4c7}',
        GeneralCategory::UppercaseLetter), ('\u{4c8}', '\u{4c8}', GeneralCategory::LowercaseLetter),
        ('\u{4c9}', '\u{4c9}', GeneralCategory::UppercaseLetter), ('\u{4ca}', '\u{4ca}',
        GeneralCategory::LowercaseLetter), ('\u{4cb}', '\u{4cb}', GeneralCategory::UppercaseLetter),
        ('\u{4cc}', '\u{4cc}', GeneralCategory::LowercaseLetter), ('\u{4cd}', '\u{4cd}',
        GeneralCategory::UppercaseLetter), ('\u{4ce}', '\u{4cf}', GeneralCategory::LowercaseLetter),
        ('\u{4d0}', '\u{4d0}', GeneralCategory::UppercaseLetter), ('\u{4d1}', '\u{4d1}',
        GeneralCategory::LowercaseLetter), ('\u{4d2}', '\u{4d2}', GeneralCategory::UppercaseLetter),
        ('\u{4d3}', '\u{4d3}', GeneralCategory::LowercaseLetter), ('\u{4d4}', '\u{4d4}',
        GeneralCategory::UppercaseLetter), ('\u{4d5}', '\u{4d5}', GeneralCategory::LowercaseLetter),
        ('\u{4d6}', '\u{4d6}', GeneralCategory::UppercaseLetter), ('\u{4d7}', '\u{4d7}',
        GeneralCategory::LowercaseLetter), ('\u{4d8}', '\u{4d8}', GeneralCategory::UppercaseLetter),
        ('\u{4d9}', '\u{4d9}', GeneralCategory::LowercaseLetter), ('\u{4da}', '\u{4da}',
        GeneralCategory::UppercaseLetter), ('\u{4db}', '\u{4db}', GeneralCategory::LowercaseLetter),
        ('\u{4dc}', '\u{4dc}', GeneralCategory::UppercaseLetter), ('\u{4dd}', '\u{4dd}',
        GeneralCategory::LowercaseLetter), ('\u{4de}', '\u{4de}', GeneralCategory::UppercaseLetter),
        ('\u{4df}', '\u{4df}', GeneralCategory::LowercaseLetter), ('\u{4e0}', '\u{4e0}',
        GeneralCategory::UppercaseLetter), ('\u{4e1}', '\u{4e1}', GeneralCategory::LowercaseLetter),
        ('\u{4e2}', '\u{4e2}', GeneralCategory::UppercaseLetter), ('\u{4e3}', '\u{4e3}',
        GeneralCategory::LowercaseLetter), ('\u{4e4}', '\u{4e4}', GeneralCategory::UppercaseLetter),
        ('\u{4e5}', '\u{4e5}', GeneralCategory::LowercaseLetter), ('\u{4e6}', '\u{4e6}',
        GeneralCategory::UppercaseLetter), ('\u{4e7}', '\u{4e7}', GeneralCategory::LowercaseLetter),
        ('\u{4e8}', '\u{4e8}', GeneralCategory::UppercaseLetter), ('\u{4e9}', '\u{4e9}',
        GeneralCategory::LowercaseLetter), ('\u{4ea}', '\u{4ea}', GeneralCategory::UppercaseLetter),
        ('\u{4eb}', '\u{4eb}', GeneralCategory::LowercaseLetter), ('\u{4ec}', '\u{4ec}',
        GeneralCategory::UppercaseLetter), ('\u{4ed}', '\u{4ed}', GeneralCategory::LowercaseLetter),
        ('\u{4ee}', '\u{4ee}', GeneralCategory::UppercaseLetter), ('\u{4ef}', '\u{4ef}',
        GeneralCategory::LowercaseLetter), ('\u{4f0}', '\u{4f0}', GeneralCategory::UppercaseLetter),
        ('\u{4f1}', '\u{4f1}', GeneralCategory::LowercaseLetter), ('\u{4f2}', '\u{4f2}',
        GeneralCategory::UppercaseLetter), ('\u{4f3}', '\u{4f3}', GeneralCategory::LowercaseLetter),
        ('\u{4f4}', '\u{4f4}', GeneralCategory::UppercaseLetter), ('\u{4f5}', '\u{4f5}',
        GeneralCategory::LowercaseLetter), ('\u{4f6}', '\u{4f6}', GeneralCategory::UppercaseLetter),
        ('\u{4f7}', '\u{4f7}', GeneralCategory::LowercaseLetter), ('\u{4f8}', '\u{4f8}',
        GeneralCategory::UppercaseLetter), ('\u{4f9}', '\u{4f9}', GeneralCategory::LowercaseLetter),
        ('\u{4fa}', '\u{4fa}', GeneralCategory::UppercaseLetter), ('\u{4fb}', '\u{4fb}',
        GeneralCategory::LowercaseLetter), ('\u{4fc}', '\u{4fc}', GeneralCategory::UppercaseLetter),
        ('\u{4fd}', '\u{4fd}', GeneralCategory::LowercaseLetter), ('\u{4fe}', '\u{4fe}',
        GeneralCategory::UppercaseLetter), ('\u{4ff}', '\u{4ff}', GeneralCategory::LowercaseLetter),
        ('\u{500}', '\u{500}', GeneralCategory::UppercaseLetter), ('\u{501}', '\u{501}',
        GeneralCategory::LowercaseLetter), ('\u{502}', '\u{502}', GeneralCategory::UppercaseLetter),
        ('\u{503}', '\u{503}', GeneralCategory::LowercaseLetter), ('\u{504}', '\u{504}',
        GeneralCategory::UppercaseLetter), ('\u{505}', '\u{505}', GeneralCategory::LowercaseLetter),
        ('\u{506}', '\u{506}', GeneralCategory::UppercaseLetter), ('\u{507}', '\u{507}',
        GeneralCategory::LowercaseLetter), ('\u{508}', '\u{508}', GeneralCategory::UppercaseLetter),
        ('\u{509}', '\u{509}', GeneralCategory::LowercaseLetter), ('\u{50a}', '\u{50a}',
        GeneralCategory::UppercaseLetter), ('\u{50b}', '\u{50b}', GeneralCategory::LowercaseLetter),
        ('\u{50c}', '\u{50c}', GeneralCategory::UppercaseLetter), ('\u{50d}', '\u{50d}',
        GeneralCategory::LowercaseLetter), ('\u{50e}', '\u{50e}', GeneralCategory::UppercaseLetter),
        ('\u{50f}', '\u{50f}', GeneralCategory::LowercaseLetter), ('\u{510}', '\u{510}',
        GeneralCategory::UppercaseLetter), ('\u{511}', '\u{511}', GeneralCategory::LowercaseLetter),
        ('\u{512}', '\u{512}', GeneralCategory::UppercaseLetter), ('\u{513}', '\u{513}',
        GeneralCategory::LowercaseLetter), ('\u{514}', '\u{514}', GeneralCategory::UppercaseLetter),
        ('\u{515}', '\u{515}', GeneralCategory::LowercaseLetter), ('\u{516}', '\u{516}',
        GeneralCategory::UppercaseLetter), ('\u{517}', '\u{517}', GeneralCategory::LowercaseLetter),
        ('\u{518}', '\u{518}', GeneralCategory::UppercaseLetter), ('\u{519}', '\u{519}',
        GeneralCategory::LowercaseLetter), ('\u{51a}', '\u{51a}', GeneralCategory::UppercaseLetter),
        ('\u{51b}', '\u{51b}', GeneralCategory::LowercaseLetter), ('\u{51c}', '\u{51c}',
        GeneralCategory::UppercaseLetter), ('\u{51d}', '\u{51d}', GeneralCategory::LowercaseLetter),
        ('\u{51e}', '\u{51e}', GeneralCategory::UppercaseLetter), ('\u{51f}', '\u{51f}',
        GeneralCategory::LowercaseLetter), ('\u{520}', '\u{520}', GeneralCategory::UppercaseLetter),
        ('\u{521}', '\u{521}', GeneralCategory::LowercaseLetter), ('\u{522}', '\u{522}',
        GeneralCategory::UppercaseLetter), ('\u{523}', '\u{523}', GeneralCategory::LowercaseLetter),
        ('\u{524}', '\u{524}', GeneralCategory::UppercaseLetter), ('\u{525}', '\u{525}',
        GeneralCategory::LowercaseLetter), ('\u{526}', '\u{526}', GeneralCategory::UppercaseLetter),
        ('\u{527}', '\u{527}', GeneralCategory::LowercaseLetter), ('\u{528}', '\u{528}',
        GeneralCategory::UppercaseLetter), ('\u{529}', '\u{529}', GeneralCategory::LowercaseLetter),
        ('\u{52a}', '\u{52a}', GeneralCategory::UppercaseLetter), ('\u{52b}', '\u{52b}',
        GeneralCategory::LowercaseLetter), ('\u{52c}', '\u{52c}', GeneralCategory::UppercaseLetter),
        ('\u{52d}', '\u{52d}', GeneralCategory::LowercaseLetter), ('\u{52e}', '\u{52e}',
        GeneralCategory::UppercaseLetter), ('\u{52f}', '\u{52f}', GeneralCategory::LowercaseLetter),
        ('\u{531}', '\u{556}', GeneralCategory::UppercaseLetter), ('\u{559}', '\u{559}',
        GeneralCategory::ModifierLetter), ('\u{55a}', '\u{55f}', GeneralCategory::OtherPunctuation),
        ('\u{560}', '\u{588}', GeneralCategory::LowercaseLetter), ('\u{589}', '\u{589}',
        GeneralCategory::OtherPunctuation), ('\u{58a}', '\u{58a}',
        GeneralCategory::DashPunctuation), ('\u{58d}', '\u{58e}', GeneralCategory::OtherSymbol),
        ('\u{58f}', '\u{58f}', GeneralCategory::CurrencySymbol), ('\u{591}', '\u{5bd}',
        GeneralCategory::NonspacingMark), ('\u{5be}', '\u{5be}', GeneralCategory::DashPunctuation),
        ('\u{5bf}', '\u{5bf}', GeneralCategory::NonspacingMark), ('\u{5c0}', '\u{5c0}',
        GeneralCategory::OtherPunctuation), ('\u{5c1}', '\u{5c2}', GeneralCategory::NonspacingMark),
        ('\u{5c3}', '\u{5c3}', GeneralCategory::OtherPunctuation), ('\u{5c4}', '\u{5c5}',
        GeneralCategory::NonspacingMark), ('\u{5c6}', '\u{5c6}', GeneralCategory::OtherPunctuation),
        ('\u{5c7}', '\u{5c7}', GeneralCategory::NonspacingMark), ('\u{5d0}', '\u{5ea}',
        GeneralCategory::OtherLetter), ('\u{5ef}', '\u{5f2}', GeneralCategory::OtherLetter),
        ('\u{5f3}', '\u{5f4}', GeneralCategory::OtherPunctuation), ('\u{600}', '\u{605}',
        GeneralCategory::Format), ('\u{606}', '\u{608}', GeneralCategory::MathSymbol), ('\u{609}',
        '\u{60a}', GeneralCategory::OtherPunctuation), ('\u{60b}', '\u{60b}',
        GeneralCategory::CurrencySymbol), ('\u{60c}', '\u{60d}', GeneralCategory::OtherPunctuation),
        ('\u{60e}', '\u{60f}', GeneralCategory::OtherSymbol), ('\u{610}', '\u{61a}',
        GeneralCategory::NonspacingMark), ('\u{61b}', '\u{61b}', GeneralCategory::OtherPunctuation),
        ('\u{61c}', '\u{61c}', GeneralCategory::Format), ('\u{61d}', '\u{61f}',
        GeneralCategory::OtherPunctuation), ('\u{620}', '\u{63f}', GeneralCategory::OtherLetter),
        ('\u{640}', '\u{640}', GeneralCategory::ModifierLetter), ('\u{641}', '\u{64a}',
        GeneralCategory::OtherLetter), ('\u{64b}', '\u{65f}', GeneralCategory::NonspacingMark),
        ('\u{660}', '\u{669}', GeneralCategory::DecimalNumber), ('\u{66a}', '\u{66d}',
        GeneralCategory::OtherPunctuation), ('\u{66e}', '\u{66f}', GeneralCategory::OtherLetter),
        ('\u{670}', '\u{670}', GeneralCategory::NonspacingMark), ('\u{671}', '\u{6d3}',
        GeneralCategory::OtherLetter), ('\u{6d4}', '\u{6d4}', GeneralCategory::OtherPunctuation),
        ('\u{6d5}', '\u{6d5}', GeneralCategory::OtherLetter), ('\u{6d6}', '\u{6dc}',
        GeneralCategory::NonspacingMark), ('\u{6dd}', '\u{6dd}', GeneralCategory::Format),
        ('\u{6de}', '\u{6de}', GeneralCategory::OtherSymbol), ('\u{6df}', '\u{6e4}',
        GeneralCategory::NonspacingMark), ('\u{6e5}', '\u{6e6}', GeneralCategory::ModifierLetter),
        ('\u{6e7}', '\u{6e8}', GeneralCategory::NonspacingMark), ('\u{6e9}', '\u{6e9}',
        GeneralCategory::OtherSymbol), ('\u{6ea}', '\u{6ed}', GeneralCategory::NonspacingMark),
        ('\u{6ee}', '\u{6ef}', GeneralCategory::OtherLetter), ('\u{6f0}', '\u{6f9}',
        GeneralCategory::DecimalNumber), ('\u{6fa}', '\u{6fc}', GeneralCategory::OtherLetter),
        ('\u{6fd}', '\u{6fe}', GeneralCategory::OtherSymbol), ('\u{6ff}', '\u{6ff}',
        GeneralCategory::OtherLetter), ('\u{700}', '\u{70d}', GeneralCategory::OtherPunctuation),
        ('\u{70f}', '\u{70f}', GeneralCategory::Format), ('\u{710}', '\u{710}',
        GeneralCategory::OtherLetter), ('\u{711}', '\u{711}', GeneralCategory::NonspacingMark),
        ('\u{712}', '\u{72f}', GeneralCategory::OtherLetter), ('\u{730}', '\u{74a}',
        GeneralCategory::NonspacingMark), ('\u{74d}', '\u{7a5}', GeneralCategory::OtherLetter),
        ('\u{7a6}', '\u{7b0}', GeneralCategory::NonspacingMark), ('\u{7b1}', '\u{7b1}',
        GeneralCategory::OtherLetter), ('\u{7c0}', '\u{7c9}', GeneralCategory::DecimalNumber),
        ('\u{7ca}', '\u{7ea}', GeneralCategory::OtherLetter), ('\u{7eb}', '\u{7f3}',
        GeneralCategory::NonspacingMark), ('\u{7f4}', '\u{7f5}', GeneralCategory::ModifierLetter),
        ('\u{7f6}', '\u{7f6}', GeneralCategory::OtherSymbol), ('\u{7f7}', '\u{7f9}',
        GeneralCategory::OtherPunctuation), ('\u{7fa}', '\u{7fa}', GeneralCategory::ModifierLetter),
        ('\u{7fd}', '\u{7fd}', GeneralCategory::NonspacingMark), ('\u{7fe}', '\u{7ff}',
        GeneralCategory::CurrencySymbol), ('\u{800}', '\u{815}', GeneralCategory::OtherLetter),
        ('\u{816}', '\u{819}', GeneralCategory::NonspacingMark), ('\u{81a}', '\u{81a}',
        GeneralCategory::ModifierLetter), ('\u{81b}', '\u{823}', GeneralCategory::NonspacingMark),
        ('\u{824}', '\u{824}', GeneralCategory::ModifierLetter), ('\u{825}', '\u{827}',
        GeneralCategory::NonspacingMark), ('\u{828}', '\u{828}', GeneralCategory::ModifierLetter),
        ('\u{829}', '\u{82d}', GeneralCategory::NonspacingMark), ('\u{830}', '\u{83e}',
        GeneralCategory::OtherPunctuation), ('\u{840}', '\u{858}', GeneralCategory::OtherLetter),
        ('\u{859}', '\u{85b}', GeneralCategory::NonspacingMark), ('\u{85e}', '\u{85e}',
        GeneralCategory::OtherPunctuation), ('\u{860}', '\u{86a}', GeneralCategory::OtherLetter),
        ('\u{870}', '\u{887}', GeneralCategory::OtherLetter), ('\u{888}', '\u{888}',
        GeneralCategory::ModifierSymbol), ('\u{889}', '\u{88e}', GeneralCategory::OtherLetter),
        ('\u{890}', '\u{891}', GeneralCategory::Format), ('\u{898}', '\u{89f}',
        GeneralCategory::NonspacingMark), ('\u{8a0}', '\u{8c8}', GeneralCategory::OtherLetter),
        ('\u{8c9}', '\u{8c9}', GeneralCategory::ModifierLetter), ('\u{8ca}', '\u{8e1}',
        GeneralCategory::NonspacingMark), ('\u{8e2}', '\u{8e2}', GeneralCategory::Format),
        ('\u{8e3}', '\u{902}', GeneralCategory::NonspacingMark), ('\u{903}', '\u{903}',
        GeneralCategory::SpacingMark), ('\u{904}', '\u{939}', GeneralCategory::OtherLetter),
        ('\u{93a}', '\u{93a}', GeneralCategory::NonspacingMark), ('\u{93b}', '\u{93b}',
        GeneralCategory::SpacingMark), ('\u{93c}', '\u{93c}', GeneralCategory::NonspacingMark),
        ('\u{93d}', '\u{93d}', GeneralCategory::OtherLetter), ('\u{93e}', '\u{940}',
        GeneralCategory::SpacingMark), ('\u{941}', '\u{948}', GeneralCategory::NonspacingMark),
        ('\u{949}', '\u{94c}', GeneralCategory::SpacingMark), ('\u{94d}', '\u{94d}',
        GeneralCategory::NonspacingMark), ('\u{94e}', '\u{94f}', GeneralCategory::SpacingMark),
        ('\u{950}', '\u{950}', GeneralCategory::OtherLetter), ('\u{951}', '\u{957}',
        GeneralCategory::NonspacingMark), ('\u{958}', '\u{961}', GeneralCategory::OtherLetter),
        ('\u{962}', '\u{963}', GeneralCategory::NonspacingMark), ('\u{964}', '\u{965}',
        GeneralCategory::OtherPunctuation), ('\u{966}', '\u{96f}', GeneralCategory::DecimalNumber),
        ('\u{970}', '\u{970}', GeneralCategory::OtherPunctuation), ('\u{971}', '\u{971}',
        GeneralCategory::ModifierLetter), ('\u{972}', '\u{980}', GeneralCategory::OtherLetter),
        ('\u{981}', '\u{981}', GeneralCategory::NonspacingMark), ('\u{982}', '\u{983}',
        GeneralCategory::SpacingMark), ('\u{985}', '\u{98c}', GeneralCategory::OtherLetter),
        ('\u{98f}', '\u{990}', GeneralCategory::OtherLetter), ('\u{993}', '\u{9a8}',
        GeneralCategory::OtherLetter), ('\u{9aa}', '\u{9b0}', GeneralCategory::OtherLetter),
        ('\u{9b2}', '\u{9b2}', GeneralCategory::OtherLetter), ('\u{9b6}', '\u{9b9}',
        GeneralCategory::OtherLetter), ('\u{9bc}', '\u{9bc}', GeneralCategory::NonspacingMark),
        ('\u{9bd}', '\u{9bd}', GeneralCategory::OtherLetter), ('\u{9be}', '\u{9c0}',
        GeneralCategory::SpacingMark), ('\u{9c1}', '\u{9c4}', GeneralCategory::NonspacingMark),
        ('\u{9c7}', '\u{9c8}', GeneralCategory::SpacingMark), ('\u{9cb}', '\u{9cc}',
        GeneralCategory::SpacingMark), ('\u{9cd}', '\u{9cd}', GeneralCategory::NonspacingMark),
        ('\u{9ce}', '\u{9ce}', GeneralCategory::OtherLetter), ('\u{9d7}', '\u{9d7}',
        GeneralCategory::SpacingMark), ('\u{9dc}', '\u{9dd}', GeneralCategory::OtherLetter),
        ('\u{9df}', '\u{9e1}', GeneralCategory::OtherLetter), ('\u{9e2}', '\u{9e3}',
        GeneralCategory::NonspacingMark), ('\u{9e6}', '\u{9ef}', GeneralCategory::DecimalNumber),
        ('\u{9f0}', '\u{9f1}', GeneralCategory::OtherLetter), ('\u{9f2}', '\u{9f3}',
        GeneralCategory::CurrencySymbol), ('\u{9f4}', '\u{9f9}', GeneralCategory::OtherNumber),
        ('\u{9fa}', '\u{9fa}', GeneralCategory::OtherSymbol), ('\u{9fb}', '\u{9fb}',
        GeneralCategory::CurrencySymbol), ('\u{9fc}', '\u{9fc}', GeneralCategory::OtherLetter),
        ('\u{9fd}', '\u{9fd}', GeneralCategory::OtherPunctuation), ('\u{9fe}', '\u{9fe}',
        GeneralCategory::NonspacingMark), ('\u{a01}', '\u{a02}', GeneralCategory::NonspacingMark),
        ('\u{a03}', '\u{a03}', GeneralCategory::SpacingMark), ('\u{a05}', '\u{a0a}',
        GeneralCategory::OtherLetter), ('\u{a0f}', '\u{a10}', GeneralCategory::OtherLetter),
        ('\u{a13}', '\u{a28}', GeneralCategory::OtherLetter), ('\u{a2a}', '\u{a30}',
        GeneralCategory::OtherLetter), ('\u{a32}', '\u{a33}', GeneralCategory::OtherLetter),
        ('\u{a35}', '\u{a36}', GeneralCategory::OtherLetter), ('\u{a38}', '\u{a39}',
        GeneralCategory::OtherLetter), ('\u{a3c}', '\u{a3c}', GeneralCategory::NonspacingMark),
        ('\u{a3e}', '\u{a40}', GeneralCategory::SpacingMark), ('\u{a41}', '\u{a42}',
        GeneralCategory::NonspacingMark), ('\u{a47}', '\u{a48}', GeneralCategory::NonspacingMark),
        ('\u{a4b}', '\u{a4d}', GeneralCategory::NonspacingMark), ('\u{a51}', '\u{a51}',
        GeneralCategory::NonspacingMark), ('\u{a59}', '\u{a5c}', GeneralCategory::OtherLetter),
        ('\u{a5e}', '\u{a5e}', GeneralCategory::OtherLetter), ('\u{a66}', '\u{a6f}',
        GeneralCategory::DecimalNumber), ('\u{a70}', '\u{a71}', GeneralCategory::NonspacingMark),
        ('\u{a72}', '\u{a74}', GeneralCategory::OtherLetter), ('\u{a75}', '\u{a75}',
        GeneralCategory::NonspacingMark), ('\u{a76}', '\u{a76}', GeneralCategory::OtherPunctuation),
        ('\u{a81}', '\u{a82}', GeneralCategory::NonspacingMark), ('\u{a83}', '\u{a83}',
        GeneralCategory::SpacingMark), ('\u{a85}', '\u{a8d}', GeneralCategory::OtherLetter),
        ('\u{a8f}', '\u{a91}', GeneralCategory::OtherLetter), ('\u{a93}', '\u{aa8}',
        GeneralCategory::OtherLetter), ('\u{aaa}', '\u{ab0}', GeneralCategory::OtherLetter),
        ('\u{ab2}', '\u{ab3}', GeneralCategory::OtherLetter), ('\u{ab5}', '\u{ab9}',
        GeneralCategory::OtherLetter), ('\u{abc}', '\u{abc}', GeneralCategory::NonspacingMark),
        ('\u{abd}', '\u{abd}', GeneralCategory::OtherLetter), ('\u{abe}', '\u{ac0}',
        GeneralCategory::SpacingMark), ('\u{ac1}', '\u{ac5}', GeneralCategory::NonspacingMark),
        ('\u{ac7}', '\u{ac8}', GeneralCategory::NonspacingMark), ('\u{ac9}', '\u{ac9}',
        GeneralCategory::SpacingMark), ('\u{acb}', '\u{acc}', GeneralCategory::SpacingMark),
        ('\u{acd}', '\u{acd}', GeneralCategory::NonspacingMark), ('\u{ad0}', '\u{ad0}',
        GeneralCategory::OtherLetter), ('\u{ae0}', '\u{ae1}', GeneralCategory::OtherLetter),
        ('\u{ae2}', '\u{ae3}', GeneralCategory::NonspacingMark), ('\u{ae6}', '\u{aef}',
        GeneralCategory::DecimalNumber), ('\u{af0}', '\u{af0}', GeneralCategory::OtherPunctuation),
        ('\u{af1}', '\u{af1}', GeneralCategory::CurrencySymbol), ('\u{af9}', '\u{af9}',
        GeneralCategory::OtherLetter), ('\u{afa}', '\u{aff}', GeneralCategory::NonspacingMark),
        ('\u{b01}', '\u{b01}', GeneralCategory::NonspacingMark), ('\u{b02}', '\u{b03}',
        GeneralCategory::SpacingMark), ('\u{b05}', '\u{b0c}', GeneralCategory::OtherLetter),
        ('\u{b0f}', '\u{b10}', GeneralCategory::OtherLetter), ('\u{b13}', '\u{b28}',
        GeneralCategory::OtherLetter), ('\u{b2a}', '\u{b30}', GeneralCategory::OtherLetter),
        ('\u{b32}', '\u{b33}', GeneralCategory::OtherLetter), ('\u{b35}', '\u{b39}',
        GeneralCategory::OtherLetter), ('\u{b3c}', '\u{b3c}', GeneralCategory::NonspacingMark),
        ('\u{b3d}', '\u{b3d}', GeneralCategory::OtherLetter), ('\u{b3e}', '\u{b3e}',
        GeneralCategory::SpacingMark), ('\u{b3f}', '\u{b3f}', GeneralCategory::NonspacingMark),
        ('\u{b40}', '\u{b40}', GeneralCategory::SpacingMark), ('\u{b41}', '\u{b44}',
        GeneralCategory::NonspacingMark), ('\u{b47}', '\u{b48}', GeneralCategory::SpacingMark),
        ('\u{b4b}', '\u{b4c}', GeneralCategory::SpacingMark), ('\u{b4d}', '\u{b4d}',
        GeneralCategory::NonspacingMark), ('\u{b55}', '\u{b56}', GeneralCategory::NonspacingMark),
        ('\u{b57}', '\u{b57}', GeneralCategory::SpacingMark), ('\u{b5c}', '\u{b5d}',
        GeneralCategory::OtherLetter), ('\u{b5f}', '\u{b61}', GeneralCategory::OtherLetter),
        ('\u{b62}', '\u{b63}', GeneralCategory::NonspacingMark), ('\u{b66}', '\u{b6f}',
        GeneralCategory::DecimalNumber), ('\u{b70}', '\u{b70}', GeneralCategory::OtherSymbol),
        ('\u{b71}', '\u{b71}', GeneralCategory::OtherLetter), ('\u{b72}', '\u{b77}',
        GeneralCategory::OtherNumber), ('\u{b82}', '\u{b82}', GeneralCategory::NonspacingMark),
        ('\u{b83}', '\u{b83}', GeneralCategory::OtherLetter), ('\u{b85}', '\u{b8a}',
        GeneralCategory::OtherLetter), ('\u{b8e}', '\u{b90}', GeneralCategory::OtherLetter),
        ('\u{b92}', '\u{b95}', GeneralCategory::OtherLetter), ('\u{b99}', '\u{b9a}',
        GeneralCategory::OtherLetter), ('\u{b9c}', '\u{b9c}', GeneralCategory::OtherLetter),
        ('\u{b9e}', '\u{b9f}', GeneralCategory::OtherLetter), ('\u{ba3}', '\u{ba4}',
        GeneralCategory::OtherLetter), ('\u{ba8}', '\u{baa}', GeneralCategory::OtherLetter),
        ('\u{bae}', '\u{bb9}', GeneralCategory::OtherLetter), ('\u{bbe}', '\u{bbf}',
        GeneralCategory::SpacingMark), ('\u{bc0}', '\u{bc0}', GeneralCategory::NonspacingMark),
        ('\u{bc1}', '\u{bc2}', GeneralCategory::SpacingMark), ('\u{bc6}', '\u{bc8}',
        GeneralCategory::SpacingMark), ('\u{bca}', '\u{bcc}', GeneralCategory::SpacingMark),
        ('\u{bcd}', '\u{bcd}', GeneralCategory::NonspacingMark), ('\u{bd0}', '\u{bd0}',
        GeneralCategory::OtherLetter), ('\u{bd7}', '\u{bd7}', GeneralCategory::SpacingMark),
        ('\u{be6}', '\u{bef}', GeneralCategory::DecimalNumber), ('\u{bf0}', '\u{bf2}',
        GeneralCategory::OtherNumber), ('\u{bf3}', '\u{bf8}', GeneralCategory::OtherSymbol),
        ('\u{bf9}', '\u{bf9}', GeneralCategory::CurrencySymbol), ('\u{bfa}', '\u{bfa}',
        GeneralCategory::OtherSymbol), ('\u{c00}', '\u{c00}', GeneralCategory::NonspacingMark),
        ('\u{c01}', '\u{c03}', GeneralCategory::SpacingMark), ('\u{c04}', '\u{c04}',
        GeneralCategory::NonspacingMark), ('\u{c05}', '\u{c0c}', GeneralCategory::OtherLetter),
        ('\u{c0e}', '\u{c10}', GeneralCategory::OtherLetter), ('\u{c12}', '\u{c28}',
        GeneralCategory::OtherLetter), ('\u{c2a}', '\u{c39}', GeneralCategory::OtherLetter),
        ('\u{c3c}', '\u{c3c}', GeneralCategory::NonspacingMark), ('\u{c3d}', '\u{c3d}',
        GeneralCategory::OtherLetter), ('\u{c3e}', '\u{c40}', GeneralCategory::NonspacingMark),
        ('\u{c41}', '\u{c44}', GeneralCategory::SpacingMark), ('\u{c46}', '\u{c48}',
        GeneralCategory::NonspacingMark), ('\u{c4a}', '\u{c4d}', GeneralCategory::NonspacingMark),
        ('\u{c55}', '\u{c56}', GeneralCategory::NonspacingMark), ('\u{c58}', '\u{c5a}',
        GeneralCategory::OtherLetter), ('\u{c5d}', '\u{c5d}', GeneralCategory::OtherLetter),
        ('\u{c60}', '\u{c61}', GeneralCategory::OtherLetter), ('\u{c62}', '\u{c63}',
        GeneralCategory::NonspacingMark), ('\u{c66}', '\u{c6f}', GeneralCategory::DecimalNumber),
        ('\u{c77}', '\u{c77}', GeneralCategory::OtherPunctuation), ('\u{c78}', '\u{c7e}',
        GeneralCategory::OtherNumber), ('\u{c7f}', '\u{c7f}', GeneralCategory::OtherSymbol),
        ('\u{c80}', '\u{c80}', GeneralCategory::OtherLetter), ('\u{c81}', '\u{c81}',
        GeneralCategory::NonspacingMark), ('\u{c82}', '\u{c83}', GeneralCategory::SpacingMark),
        ('\u{c84}', '\u{c84}', GeneralCategory::OtherPunctuation), ('\u{c85}', '\u{c8c}',
        GeneralCategory::OtherLetter), ('\u{c8e}', '\u{c90}', GeneralCategory::OtherLetter),
        ('\u{c92}', '\u{ca8}', GeneralCategory::OtherLetter), ('\u{caa}', '\u{cb3}',
        GeneralCategory::OtherLetter), ('\u{cb5}', '\u{cb9}', GeneralCategory::OtherLetter),
        ('\u{cbc}', '\u{cbc}', GeneralCategory::NonspacingMark), ('\u{cbd}', '\u{cbd}',
        GeneralCategory::OtherLetter), ('\u{cbe}', '\u{cbe}', GeneralCategory::SpacingMark),
        ('\u{cbf}', '\u{cbf}', GeneralCategory::NonspacingMark), ('\u{cc0}', '\u{cc4}',
        GeneralCategory::SpacingMark), ('\u{cc6}', '\u{cc6}', GeneralCategory::NonspacingMark),
        ('\u{cc7}', '\u{cc8}', GeneralCategory::SpacingMark), ('\u{cca}', '\u{ccb}',
        GeneralCategory::SpacingMark), ('\u{ccc}', '\u{ccd}', GeneralCategory::NonspacingMark),
        ('\u{cd5}', '\u{cd6}', GeneralCategory::SpacingMark), ('\u{cdd}', '\u{cde}',
        GeneralCategory::OtherLetter), ('\u{ce0}', '\u{ce1}', GeneralCategory::OtherLetter),
        ('\u{ce2}', '\u{ce3}', GeneralCategory::NonspacingMark), ('\u{ce6}', '\u{cef}',
        GeneralCategory::DecimalNumber), ('\u{cf1}', '\u{cf2}', GeneralCategory::OtherLetter),
        ('\u{cf3}', '\u{cf3}', GeneralCategory::SpacingMark), ('\u{d00}', '\u{d01}',
        GeneralCategory::NonspacingMark), ('\u{d02}', '\u{d03}', GeneralCategory::SpacingMark),
        ('\u{d04}', '\u{d0c}', GeneralCategory::OtherLetter), ('\u{d0e}', '\u{d10}',
        GeneralCategory::OtherLetter), ('\u{d12}', '\u{d3a}', GeneralCategory::OtherLetter),
        ('\u{d3b}', '\u{d3c}', GeneralCategory::NonspacingMark), ('\u{d3d}', '\u{d3d}',
        GeneralCategory::OtherLetter), ('\u{d3e}', '\u{d40}', GeneralCategory::SpacingMark),
        ('\u{d41}', '\u{d44}', GeneralCategory::NonspacingMark), ('\u{d46}', '\u{d48}',
        GeneralCategory::SpacingMark), ('\u{d4a}', '\u{d4c}', GeneralCategory::SpacingMark),
        ('\u{d4d}', '\u{d4d}', GeneralCategory::NonspacingMark), ('\u{d4e}', '\u{d4e}',
        GeneralCategory::OtherLetter), ('\u{d4f}', '\u{d4f}', GeneralCategory::OtherSymbol),
        ('\u{d54}', '\u{d56}', GeneralCategory::OtherLetter), ('\u{d57}', '\u{d57}',
        GeneralCategory::SpacingMark), ('\u{d58}', '\u{d5e}', GeneralCategory::OtherNumber),
        ('\u{d5f}', '\u{d61}', GeneralCategory::OtherLetter), ('\u{d62}', '\u{d63}',
        GeneralCategory::NonspacingMark), ('\u{d66}', '\u{d6f}', GeneralCategory::DecimalNumber),
        ('\u{d70}', '\u{d78}', GeneralCategory::OtherNumber), ('\u{d79}', '\u{d79}',
        GeneralCategory::OtherSymbol), ('\u{d7a}', '\u{d7f}', GeneralCategory::OtherLetter),
        ('\u{d81}', '\u{d81}', GeneralCategory::NonspacingMark), ('\u{d82}', '\u{d83}',
        GeneralCategory::SpacingMark), ('\u{d85}', '\u{d96}', GeneralCategory::OtherLetter),
        ('\u{d9a}', '\u{db1}', GeneralCategory::OtherLetter), ('\u{db3}', '\u{dbb}',
        GeneralCategory::OtherLetter), ('\u{dbd}', '\u{dbd}', GeneralCategory::OtherLetter),
        ('\u{dc0}', '\u{dc6}', GeneralCategory::OtherLetter), ('\u{dca}', '\u{dca}',
        GeneralCategory::NonspacingMark), ('\u{dcf}', '\u{dd1}', GeneralCategory::SpacingMark),
        ('\u{dd2}', '\u{dd4}', GeneralCategory::NonspacingMark), ('\u{dd6}', '\u{dd6}',
        GeneralCategory::NonspacingMark), ('\u{dd8}', '\u{ddf}', GeneralCategory::SpacingMark),
        ('\u{de6}', '\u{def}', GeneralCategory::DecimalNumber), ('\u{df2}', '\u{df3}',
        GeneralCategory::SpacingMark), ('\u{df4}', '\u{df4}', GeneralCategory::OtherPunctuation),
        ('\u{e01}', '\u{e30}', GeneralCategory::OtherLetter), ('\u{e31}', '\u{e31}',
        GeneralCategory::NonspacingMark), ('\u{e32}', '\u{e33}', GeneralCategory::OtherLetter),
        ('\u{e34}', '\u{e3a}', GeneralCategory::NonspacingMark), ('\u{e3f}', '\u{e3f}',
        GeneralCategory::CurrencySymbol), ('\u{e40}', '\u{e45}', GeneralCategory::OtherLetter),
        ('\u{e46}', '\u{e46}', GeneralCategory::ModifierLetter), ('\u{e47}', '\u{e4e}',
        GeneralCategory::NonspacingMark), ('\u{e4f}', '\u{e4f}', GeneralCategory::OtherPunctuation),
        ('\u{e50}', '\u{e59}', GeneralCategory::DecimalNumber), ('\u{e5a}', '\u{e5b}',
        GeneralCategory::OtherPunctuation), ('\u{e81}', '\u{e82}', GeneralCategory::OtherLetter),
        ('\u{e84}', '\u{e84}', GeneralCategory::OtherLetter), ('\u{e86}', '\u{e8a}',
        GeneralCategory::OtherLetter), ('\u{e8c}', '\u{ea3}', GeneralCategory::OtherLetter),
        ('\u{ea5}', '\u{ea5}', GeneralCategory::OtherLetter), ('\u{ea7}', '\u{eb0}',
        GeneralCategory::OtherLetter), ('\u{eb1}', '\u{eb1}', GeneralCategory::NonspacingMark),
        ('\u{eb2}', '\u{eb3}', GeneralCategory::OtherLetter), ('\u{eb4}', '\u{ebc}',
        GeneralCategory::NonspacingMark), ('\u{ebd}', '\u{ebd}', GeneralCategory::OtherLetter),
        ('\u{ec0}', '\u{ec4}', GeneralCategory::OtherLetter), ('\u{ec6}', '\u{ec6}',
        GeneralCategory::ModifierLetter), ('\u{ec8}', '\u{ece}', GeneralCategory::NonspacingMark),
        ('\u{ed0}', '\u{ed9}', GeneralCategory::DecimalNumber), ('\u{edc}', '\u{edf}',
        GeneralCategory::OtherLetter), ('\u{f00}', '\u{f00}', GeneralCategory::OtherLetter),
        ('\u{f01}', '\u{f03}', GeneralCategory::OtherSymbol), ('\u{f04}', '\u{f12}',
        GeneralCategory::OtherPunctuation), ('\u{f13}', '\u{f13}', GeneralCategory::OtherSymbol),
        ('\u{f14}', '\u{f14}', GeneralCategory::OtherPunctuation), ('\u{f15}', '\u{f17}',
        GeneralCategory::OtherSymbol), ('\u{f18}', '\u{f19}', GeneralCategory::NonspacingMark),
        ('\u{f1a}', '\u{f1f}', GeneralCategory::OtherSymbol), ('\u{f20}', '\u{f29}',
        GeneralCategory::DecimalNumber), ('\u{f2a}', '\u{f33}', GeneralCategory::OtherNumber),
        ('\u{f34}', '\u{f34}', GeneralCategory::OtherSymbol), ('\u{f35}', '\u{f35}',
        GeneralCategory::NonspacingMark), ('\u{f36}', '\u{f36}', GeneralCategory::OtherSymbol),
        ('\u{f37}', '\u{f37}', GeneralCategory::NonspacingMark), ('\u{f38}', '\u{f38}',
        GeneralCategory::OtherSymbol), ('\u{f39}', '\u{f39}', GeneralCategory::NonspacingMark),
        ('\u{f3a}', '\u{f3a}', GeneralCategory::OpenPunctuation), ('\u{f3b}', '\u{f3b}',
        GeneralCategory::ClosePunctuation), ('\u{f3c}', '\u{f3c}',
        GeneralCategory::OpenPunctuation), ('\u{f3d}', '\u{f3d}',
        GeneralCategory::ClosePunctuation), ('\u{f3e}', '\u{f3f}', GeneralCategory::SpacingMark),
        ('\u{f40}', '\u{f47}', GeneralCategory::OtherLetter), ('\u{f49}', '\u{f6c}',
        GeneralCategory::OtherLetter), ('\u{f71}', '\u{f7e}', GeneralCategory::NonspacingMark),
        ('\u{f7f}', '\u{f7f}', GeneralCategory::SpacingMark), ('\u{f80}', '\u{f84}',
        GeneralCategory::NonspacingMark), ('\u{f85}', '\u{f85}', GeneralCategory::OtherPunctuation),
        ('\u{f86}', '\u{f87}', GeneralCategory::NonspacingMark), ('\u{f88}', '\u{f8c}',
        GeneralCategory::OtherLetter), ('\u{f8d}', '\u{f97}', GeneralCategory::NonspacingMark),
        ('\u{f99}', '\u{fbc}', GeneralCategory::NonspacingMark), ('\u{fbe}', '\u{fc5}',
        GeneralCategory::OtherSymbol), ('\u{fc6}', '\u{fc6}', GeneralCategory::NonspacingMark),
        ('\u{fc7}', '\u{fcc}', GeneralCategory::OtherSymbol), ('\u{fce}', '\u{fcf}',
        GeneralCategory::OtherSymbol), ('\u{fd0}', '\u{fd4}', GeneralCategory::OtherPunctuation),
        ('\u{fd5}', '\u{fd8}', GeneralCategory::OtherSymbol), ('\u{fd9}', '\u{fda}',
        GeneralCategory::OtherPunctuation), ('\u{1000}', '\u{102a}', GeneralCategory::OtherLetter),
        ('\u{102b}', '\u{102c}', GeneralCategory::SpacingMark), ('\u{102d}', '\u{1030}',
        GeneralCategory::NonspacingMark), ('\u{1031}', '\u{1031}', GeneralCategory::SpacingMark),
        ('\u{1032}', '\u{1037}', GeneralCategory::NonspacingMark), ('\u{1038}', '\u{1038}',
        GeneralCategory::SpacingMark), ('\u{1039}', '\u{103a}', GeneralCategory::NonspacingMark),
        ('\u{103b}', '\u{103c}', GeneralCategory::SpacingMark), ('\u{103d}', '\u{103e}',
        GeneralCategory::NonspacingMark), ('\u{103f}', '\u{103f}', GeneralCategory::OtherLetter),
        ('\u{1040}', '\u{1049}', GeneralCategory::DecimalNumber), ('\u{104a}', '\u{104f}',
        GeneralCategory::OtherPunctuation), ('\u{1050}', '\u{1055}', GeneralCategory::OtherLetter),
        ('\u{1056}', '\u{1057}', GeneralCategory::SpacingMark), ('\u{1058}', '\u{1059}',
        GeneralCategory::NonspacingMark), ('\u{105a}', '\u{105d}', GeneralCategory::OtherLetter),
        ('\u{105e}', '\u{1060}', GeneralCategory::NonspacingMark), ('\u{1061}', '\u{1061}',
        GeneralCategory::OtherLetter), ('\u{1062}', '\u{1064}', GeneralCategory::SpacingMark),
        ('\u{1065}', '\u{1066}', GeneralCategory::OtherLetter), ('\u{1067}', '\u{106d}',
        GeneralCategory::SpacingMark), ('\u{106e}', '\u{1070}', GeneralCategory::OtherLetter),
        ('\u{1071}', '\u{1074}', GeneralCategory::NonspacingMark), ('\u{1075}', '\u{1081}',
        GeneralCategory::OtherLetter), ('\u{1082}', '\u{1082}', GeneralCategory::NonspacingMark),
        ('\u{1083}', '\u{1084}', GeneralCategory::SpacingMark), ('\u{1085}', '\u{1086}',
        GeneralCategory::NonspacingMark), ('\u{1087}', '\u{108c}', GeneralCategory::SpacingMark),
        ('\u{108d}', '\u{108d}', GeneralCategory::NonspacingMark), ('\u{108e}', '\u{108e}',
        GeneralCategory::OtherLetter), ('\u{108f}', '\u{108f}', GeneralCategory::SpacingMark),
        ('\u{1090}', '\u{1099}', GeneralCategory::DecimalNumber), ('\u{109a}', '\u{109c}',
        GeneralCategory::SpacingMark), ('\u{109d}', '\u{109d}', GeneralCategory::NonspacingMark),
        ('\u{109e}', '\u{109f}', GeneralCategory::OtherSymbol), ('\u{10a0}', '\u{10c5}',
        GeneralCategory::UppercaseLetter), ('\u{10c7}', '\u{10c7}',
        GeneralCategory::UppercaseLetter), ('\u{10cd}', '\u{10cd}',
        GeneralCategory::UppercaseLetter), ('\u{10d0}', '\u{10fa}',
        GeneralCategory::LowercaseLetter), ('\u{10fb}', '\u{10fb}',
        GeneralCategory::OtherPunctuation), ('\u{10fc}', '\u{10fc}',
        GeneralCategory::ModifierLetter), ('\u{10fd}', '\u{10ff}',
        GeneralCategory::LowercaseLetter), ('\u{1100}', '\u{1248}', GeneralCategory::OtherLetter),
        ('\u{124a}', '\u{124d}', GeneralCategory::OtherLetter), ('\u{1250}', '\u{1256}',
        GeneralCategory::OtherLetter), ('\u{1258}', '\u{1258}', GeneralCategory::OtherLetter),
        ('\u{125a}', '\u{125d}', GeneralCategory::OtherLetter), ('\u{1260}', '\u{1288}',
        GeneralCategory::OtherLetter), ('\u{128a}', '\u{128d}', GeneralCategory::OtherLetter),
        ('\u{1290}', '\u{12b0}', GeneralCategory::OtherLetter), ('\u{12b2}', '\u{12b5}',
        GeneralCategory::OtherLetter), ('\u{12b8}', '\u{12be}', GeneralCategory::OtherLetter),
        ('\u{12c0}', '\u{12c0}', GeneralCategory::OtherLetter), ('\u{12c2}', '\u{12c5}',
        GeneralCategory::OtherLetter), ('\u{12c8}', '\u{12d6}', GeneralCategory::OtherLetter),
        ('\u{12d8}', '\u{1310}', GeneralCategory::OtherLetter), ('\u{1312}', '\u{1315}',
        GeneralCategory::OtherLetter), ('\u{1318}', '\u{135a}', GeneralCategory::OtherLetter),
        ('\u{135d}', '\u{135f}', GeneralCategory::NonspacingMark), ('\u{1360}', '\u{1368}',
        GeneralCategory::OtherPunctuation), ('\u{1369}', '\u{137c}', GeneralCategory::OtherNumber),
        ('\u{1380}', '\u{138f}', GeneralCategory::OtherLetter), ('\u{1390}', '\u{1399}',
        GeneralCategory::OtherSymbol), ('\u{13a0}', '\u{13f5}', GeneralCategory::UppercaseLetter),
        ('\u{13f8}', '\u{13fd}', GeneralCategory::LowercaseLetter), ('\u{1400}', '\u{1400}',
        GeneralCategory::DashPunctuation), ('\u{1401}', '\u{166c}', GeneralCategory::OtherLetter),
        ('\u{166d}', '\u{166d}', GeneralCategory::OtherSymbol), ('\u{166e}', '\u{166e}',
        GeneralCategory::OtherPunctuation), ('\u{166f}', '\u{167f}', GeneralCategory::OtherLetter),
        ('\u{1680}', '\u{1680}', GeneralCategory::SpaceSeparator), ('\u{1681}', '\u{169a}',
        GeneralCategory::OtherLetter), ('\u{169b}', '\u{169b}', GeneralCategory::OpenPunctuation),
        ('\u{169c}', '\u{169c}', GeneralCategory::ClosePunctuation), ('\u{16a0}', '\u{16ea}',
        GeneralCategory::OtherLetter), ('\u{16eb}', '\u{16ed}', GeneralCategory::OtherPunctuation),
        ('\u{16ee}', '\u{16f0}', GeneralCategory::LetterNumber), ('\u{16f1}', '\u{16f8}',
        GeneralCategory::OtherLetter), ('\u{1700}', '\u{1711}', GeneralCategory::OtherLetter),
        ('\u{1712}', '\u{1714}', GeneralCategory::NonspacingMark), ('\u{1715}', '\u{1715}',
        GeneralCategory::SpacingMark), ('\u{171f}', '\u{1731}', GeneralCategory::OtherLetter),
        ('\u{1732}', '\u{1733}', GeneralCategory::NonspacingMark), ('\u{1734}', '\u{1734}',
        GeneralCategory::SpacingMark), ('\u{1735}', '\u{1736}', GeneralCategory::OtherPunctuation),
        ('\u{1740}', '\u{1751}', GeneralCategory::OtherLetter), ('\u{1752}', '\u{1753}',
        GeneralCategory::NonspacingMark), ('\u{1760}', '\u{176c}', GeneralCategory::OtherLetter),
        ('\u{176e}', '\u{1770}', GeneralCategory::OtherLetter), ('\u{1772}', '\u{1773}',
        GeneralCategory::NonspacingMark), ('\u{1780}', '\u{17b3}', GeneralCategory::OtherLetter),
        ('\u{17b4}', '\u{17b5}', GeneralCategory::NonspacingMark), ('\u{17b6}', '\u{17b6}',
        GeneralCategory::SpacingMark), ('\u{17b7}', '\u{17bd}', GeneralCategory::NonspacingMark),
        ('\u{17be}', '\u{17c5}', GeneralCategory::SpacingMark), ('\u{17c6}', '\u{17c6}',
        GeneralCategory::NonspacingMark), ('\u{17c7}', '\u{17c8}', GeneralCategory::SpacingMark),
        ('\u{17c9}', '\u{17d3}', GeneralCategory::NonspacingMark), ('\u{17d4}', '\u{17d6}',
        GeneralCategory::OtherPunctuation), ('\u{17d7}', '\u{17d7}',
        GeneralCategory::ModifierLetter), ('\u{17d8}', '\u{17da}',
        GeneralCategory::OtherPunctuation), ('\u{17db}', '\u{17db}',
        GeneralCategory::CurrencySymbol), ('\u{17dc}', '\u{17dc}', GeneralCategory::OtherLetter),
        ('\u{17dd}', '\u{17dd}', GeneralCategory::NonspacingMark), ('\u{17e0}', '\u{17e9}',
        GeneralCategory::DecimalNumber), ('\u{17f0}', '\u{17f9}', GeneralCategory::OtherNumber),
        ('\u{1800}', '\u{1805}', GeneralCategory::OtherPunctuation), ('\u{1806}', '\u{1806}',
        GeneralCategory::DashPunctuation), ('\u{1807}', '\u{180a}',
        GeneralCategory::OtherPunctuation), ('\u{180b}', '\u{180d}',
        GeneralCategory::NonspacingMark), ('\u{180e}', '\u{180e}', GeneralCategory::Format),
        ('\u{180f}', '\u{180f}', GeneralCategory::NonspacingMark), ('\u{1810}', '\u{1819}',
        GeneralCategory::DecimalNumber), ('\u{1820}', '\u{1842}', GeneralCategory::OtherLetter),
        ('\u{1843}', '\u{1843}', GeneralCategory::ModifierLetter), ('\u{1844}', '\u{1878}',
        GeneralCategory::OtherLetter), ('\u{1880}', '\u{1884}', GeneralCategory::OtherLetter),
        ('\u{1885}', '\u{1886}', GeneralCategory::NonspacingMark), ('\u{1887}', '\u{18a8}',
        GeneralCategory::OtherLetter), ('\u{18a9}', '\u{18a9}', GeneralCategory::NonspacingMark),
        ('\u{18aa}', '\u{18aa}', GeneralCategory::OtherLetter), ('\u{18b0}', '\u{18f5}',
        GeneralCategory::OtherLetter), ('\u{1900}', '\u{191e}', GeneralCategory::OtherLetter),
        ('\u{1920}', '\u{1922}', GeneralCategory::NonspacingMark), ('\u{1923}', '\u{1926}',
        GeneralCategory::SpacingMark), ('\u{1927}', '\u{1928}', GeneralCategory::NonspacingMark),
        ('\u{1929}', '\u{192b}', GeneralCategory::SpacingMark), ('\u{1930}', '\u{1931}',
        GeneralCategory::SpacingMark), ('\u{1932}', '\u{1932}', GeneralCategory::NonspacingMark),
        ('\u{1933}', '\u{1938}', GeneralCategory::SpacingMark), ('\u{1939}', '\u{193b}',
        GeneralCategory::NonspacingMark), ('\u{1940}', '\u{1940}', GeneralCategory::OtherSymbol),
        ('\u{1944}', '\u{1945}', GeneralCategory::OtherPunctuation), ('\u{1946}', '\u{194f}',
        GeneralCategory::DecimalNumber), ('\u{1950}', '\u{196d}', GeneralCategory::OtherLetter),
        ('\u{1970}', '\u{1974}', GeneralCategory::OtherLetter), ('\u{1980}', '\u{19ab}',
        GeneralCategory::OtherLetter), ('\u{19b0}', '\u{19c9}', GeneralCategory::OtherLetter),
        ('\u{19d0}', '\u{19d9}', GeneralCategory::DecimalNumber), ('\u{19da}', '\u{19da}',
        GeneralCategory::OtherNumber), ('\u{19de}', '\u{19ff}', GeneralCategory::OtherSymbol),
        ('\u{1a00}', '\u{1a16}', GeneralCategory::OtherLetter), ('\u{1a17}', '\u{1a18}',
        GeneralCategory::NonspacingMark), ('\u{1a19}', '\u{1a1a}', GeneralCategory::SpacingMark),
        ('\u{1a1b}', '\u{1a1b}', GeneralCategory::NonspacingMark), ('\u{1a1e}', '\u{1a1f}',
        GeneralCategory::OtherPunctuation), ('\u{1a20}', '\u{1a54}', GeneralCategory::OtherLetter),
        ('\u{1a55}', '\u{1a55}', GeneralCategory::SpacingMark), ('\u{1a56}', '\u{1a56}',
        GeneralCategory::NonspacingMark), ('\u{1a57}', '\u{1a57}', GeneralCategory::SpacingMark),
        ('\u{1a58}', '\u{1a5e}', GeneralCategory::NonspacingMark), ('\u{1a60}', '\u{1a60}',
        GeneralCategory::NonspacingMark), ('\u{1a61}', '\u{1a61}', GeneralCategory::SpacingMark),
        ('\u{1a62}', '\u{1a62}', GeneralCategory::NonspacingMark), ('\u{1a63}', '\u{1a64}',
        GeneralCategory::SpacingMark), ('\u{1a65}', '\u{1a6c}', GeneralCategory::NonspacingMark),
        ('\u{1a6d}', '\u{1a72}', GeneralCategory::SpacingMark), ('\u{1a73}', '\u{1a7c}',
        GeneralCategory::NonspacingMark), ('\u{1a7f}', '\u{1a7f}', GeneralCategory::NonspacingMark),
        ('\u{1a80}', '\u{1a89}', GeneralCategory::DecimalNumber), ('\u{1a90}', '\u{1a99}',
        GeneralCategory::DecimalNumber), ('\u{1aa0}', '\u{1aa6}',
        GeneralCategory::OtherPunctuation), ('\u{1aa7}', '\u{1aa7}',
        GeneralCategory::ModifierLetter), ('\u{1aa8}', '\u{1aad}',
        GeneralCategory::OtherPunctuation), ('\u{1ab0}', '\u{1abd}',
        GeneralCategory::NonspacingMark), ('\u{1abe}', '\u{1abe}', GeneralCategory::EnclosingMark),
        ('\u{1abf}', '\u{1ace}', GeneralCategory::NonspacingMark), ('\u{1b00}', '\u{1b03}',
        GeneralCategory::NonspacingMark), ('\u{1b04}', '\u{1b04}', GeneralCategory::SpacingMark),
        ('\u{1b05}', '\u{1b33}', GeneralCategory::OtherLetter), ('\u{1b34}', '\u{1b34}',
        GeneralCategory::NonspacingMark), ('\u{1b35}', '\u{1b35}', GeneralCategory::SpacingMark),
        ('\u{1b36}', '\u{1b3a}', GeneralCategory::NonspacingMark), ('\u{1b3b}', '\u{1b3b}',
        GeneralCategory::SpacingMark), ('\u{1b3c}', '\u{1b3c}', GeneralCategory::NonspacingMark),
        ('\u{1b3d}', '\u{1b41}', GeneralCategory::SpacingMark), ('\u{1b42}', '\u{1b42}',
        GeneralCategory::NonspacingMark), ('\u{1b43}', '\u{1b44}', GeneralCategory::SpacingMark),
        ('\u{1b45}', '\u{1b4c}', GeneralCategory::OtherLetter), ('\u{1b50}', '\u{1b59}',
        GeneralCategory::DecimalNumber), ('\u{1b5a}', '\u{1b60}',
        GeneralCategory::OtherPunctuation), ('\u{1b61}', '\u{1b6a}', GeneralCategory::OtherSymbol),
        ('\u{1b6b}', '\u{1b73}', GeneralCategory::NonspacingMark), ('\u{1b74}', '\u{1b7c}',
        GeneralCategory::OtherSymbol), ('\u{1b7d}', '\u{1b7e}', GeneralCategory::OtherPunctuation),
        ('\u{1b80}', '\u{1b81}', GeneralCategory::NonspacingMark), ('\u{1b82}', '\u{1b82}',
        GeneralCategory::SpacingMark), ('\u{1b83}', '\u{1ba0}', GeneralCategory::OtherLetter),
        ('\u{1ba1}', '\u{1ba1}', GeneralCategory::SpacingMark), ('\u{1ba2}', '\u{1ba5}',
        GeneralCategory::NonspacingMark), ('\u{1ba6}', '\u{1ba7}', GeneralCategory::SpacingMark),
        ('\u{1ba8}', '\u{1ba9}', GeneralCategory::NonspacingMark), ('\u{1baa}', '\u{1baa}',
        GeneralCategory::SpacingMark), ('\u{1bab}', '\u{1bad}', GeneralCategory::NonspacingMark),
        ('\u{1bae}', '\u{1baf}', GeneralCategory::OtherLetter), ('\u{1bb0}', '\u{1bb9}',
        GeneralCategory::DecimalNumber), ('\u{1bba}', '\u{1be5}', GeneralCategory::OtherLetter),
        ('\u{1be6}', '\u{1be6}', GeneralCategory::NonspacingMark), ('\u{1be7}', '\u{1be7}',
        GeneralCategory::SpacingMark), ('\u{1be8}', '\u{1be9}', GeneralCategory::NonspacingMark),
        ('\u{1bea}', '\u{1bec}', GeneralCategory::SpacingMark), ('\u{1bed}', '\u{1bed}',
        GeneralCategory::NonspacingMark), ('\u{1bee}', '\u{1bee}', GeneralCategory::SpacingMark),
        ('\u{1bef}', '\u{1bf1}', GeneralCategory::NonspacingMark), ('\u{1bf2}', '\u{1bf3}',
        GeneralCategory::SpacingMark), ('\u{1bfc}', '\u{1bff}', GeneralCategory::OtherPunctuation),
        ('\u{1c00}', '\u{1c23}', GeneralCategory::OtherLetter), ('\u{1c24}', '\u{1c2b}',
        GeneralCategory::SpacingMark), ('\u{1c2c}', '\u{1c33}', GeneralCategory::NonspacingMark),
        ('\u{1c34}', '\u{1c35}', GeneralCategory::SpacingMark), ('\u{1c36}', '\u{1c37}',
        GeneralCategory::NonspacingMark), ('\u{1c3b}', '\u{1c3f}',
        GeneralCategory::OtherPunctuation), ('\u{1c40}', '\u{1c49}',
        GeneralCategory::DecimalNumber), ('\u{1c4d}', '\u{1c4f}', GeneralCategory::OtherLetter),
        ('\u{1c50}', '\u{1c59}', GeneralCategory::DecimalNumber), ('\u{1c5a}', '\u{1c77}',
        GeneralCategory::OtherLetter), ('\u{1c78}', '\u{1c7d}', GeneralCategory::ModifierLetter),
        ('\u{1c7e}', '\u{1c7f}', GeneralCategory::OtherPunctuation), ('\u{1c80}', '\u{1c88}',
        GeneralCategory::LowercaseLetter), ('\u{1c90}', '\u{1cba}',
        GeneralCategory::UppercaseLetter), ('\u{1cbd}', '\u{1cbf}',
        GeneralCategory::UppercaseLetter), ('\u{1cc0}', '\u{1cc7}',
        GeneralCategory::OtherPunctuation), ('\u{1cd0}', '\u{1cd2}',
        GeneralCategory::NonspacingMark), ('\u{1cd3}', '\u{1cd3}',
        GeneralCategory::OtherPunctuation), ('\u{1cd4}', '\u{1ce0}',
        GeneralCategory::NonspacingMark), ('\u{1ce1}', '\u{1ce1}', GeneralCategory::SpacingMark),
        ('\u{1ce2}', '\u{1ce8}', GeneralCategory::NonspacingMark), ('\u{1ce9}', '\u{1cec}',
        GeneralCategory::OtherLetter), ('\u{1ced}', '\u{1ced}', GeneralCategory::NonspacingMark),
        ('\u{1cee}', '\u{1cf3}', GeneralCategory::OtherLetter), ('\u{1cf4}', '\u{1cf4}',
        GeneralCategory::NonspacingMark), ('\u{1cf5}', '\u{1cf6}', GeneralCategory::OtherLetter),
        ('\u{1cf7}', '\u{1cf7}', GeneralCategory::SpacingMark), ('\u{1cf8}', '\u{1cf9}',
        GeneralCategory::NonspacingMark), ('\u{1cfa}', '\u{1cfa}', GeneralCategory::OtherLetter),
        ('\u{1d00}', '\u{1d2b}', GeneralCategory::LowercaseLetter), ('\u{1d2c}', '\u{1d6a}',
        GeneralCategory::ModifierLetter), ('\u{1d6b}', '\u{1d77}',
        GeneralCategory::LowercaseLetter), ('\u{1d78}', '\u{1d78}',
        GeneralCategory::ModifierLetter), ('\u{1d79}', '\u{1d9a}',
        GeneralCategory::LowercaseLetter), ('\u{1d9b}', '\u{1dbf}',
        GeneralCategory::ModifierLetter), ('\u{1dc0}', '\u{1dff}', GeneralCategory::NonspacingMark),
        ('\u{1e00}', '\u{1e00}', GeneralCategory::UppercaseLetter), ('\u{1e01}', '\u{1e01}',
        GeneralCategory::LowercaseLetter), ('\u{1e02}', '\u{1e02}',
        GeneralCategory::UppercaseLetter), ('\u{1e03}', '\u{1e03}',
        GeneralCategory::LowercaseLetter), ('\u{1e04}', '\u{1e04}',
        GeneralCategory::UppercaseLetter), ('\u{1e05}', '\u{1e05}',
        GeneralCategory::LowercaseLetter), ('\u{1e06}', '\u{1e06}',
        GeneralCategory::UppercaseLetter), ('\u{1e07}', '\u{1e07}',
        GeneralCategory::LowercaseLetter), ('\u{1e08}', '\u{1e08}',
        GeneralCategory::UppercaseLetter), ('\u{1e09}', '\u{1e09}',
        GeneralCategory::LowercaseLetter), ('\u{1e0a}', '\u{1e0a}',
        GeneralCategory::UppercaseLetter), ('\u{1e0b}', '\u{1e0b}',
        GeneralCategory::LowercaseLetter), ('\u{1e0c}', '\u{1e0c}',
        GeneralCategory::UppercaseLetter), ('\u{1e0d}', '\u{1e0d}',
        GeneralCategory::LowercaseLetter), ('\u{1e0e}', '\u{1e0e}',
        GeneralCategory::UppercaseLetter), ('\u{1e0f}', '\u{1e0f}',
        GeneralCategory::LowercaseLetter), ('\u{1e10}', '\u{1e10}',
        GeneralCategory::UppercaseLetter), ('\u{1e11}', '\u{1e11}',
        GeneralCategory::LowercaseLetter), ('\u{1e12}', '\u{1e12}',
        GeneralCategory::UppercaseLetter), ('\u{1e13}', '\u{1e13}',
        GeneralCategory::LowercaseLetter), ('\u{1e14}', '\u{1e14}',
        GeneralCategory::UppercaseLetter), ('\u{1e15}', '\u{1e15}',
        GeneralCategory::LowercaseLetter), ('\u{1e16}', '\u{1e16}',
        GeneralCategory::UppercaseLetter), ('\u{1e17}', '\u{1e17}',
        GeneralCategory::LowercaseLetter), ('\u{1e18}', '\u{1e18}',
        GeneralCategory::UppercaseLetter), ('\u{1e19}', '\u{1e19}',
        GeneralCategory::LowercaseLetter), ('\u{1e1a}', '\u{1e1a}',
        GeneralCategory::UppercaseLetter), ('\u{1e1b}', '\u{1e1b}',
        GeneralCategory::LowercaseLetter), ('\u{1e1c}', '\u{1e1c}',
        GeneralCategory::UppercaseLetter), ('\u{1e1d}', '\u{1e1d}',
        GeneralCategory::LowercaseLetter), ('\u{1e1e}', '\u{1e1e}',
        GeneralCategory::UppercaseLetter), ('\u{1e1f}', '\u{1e1f}',
        GeneralCategory::LowercaseLetter), ('\u{1e20}', '\u{1e20}',
        GeneralCategory::UppercaseLetter), ('\u{1e21}', '\u{1e21}',
        GeneralCategory::LowercaseLetter), ('\u{1e22}', '\u{1e22}',
        GeneralCategory::UppercaseLetter), ('\u{1e23}', '\u{1e23}',
        GeneralCategory::LowercaseLetter), ('\u{1e24}', '\u{1e24}',
        GeneralCategory::UppercaseLetter), ('\u{1e25}', '\u{1e25}',
        GeneralCategory::LowercaseLetter), ('\u{1e26}', '\u{1e26}',
        GeneralCategory::UppercaseLetter), ('\u{1e27}', '\u{1e27}',
        GeneralCategory::LowercaseLetter), ('\u{1e28}', '\u{1e28}',
        GeneralCategory::UppercaseLetter), ('\u{1e29}', '\u{1e29}',
        GeneralCategory::LowercaseLetter), ('\u{1e2a}', '\u{1e2a}',
        GeneralCategory::UppercaseLetter), ('\u{1e2b}', '\u{1e2b}',
        GeneralCategory::LowercaseLetter), ('\u{1e2c}', '\u{1e2c}',
        GeneralCategory::UppercaseLetter), ('\u{1e2d}', '\u{1e2d}',
        GeneralCategory::LowercaseLetter), ('\u{1e2e}', '\u{1e2e}',
        GeneralCategory::UppercaseLetter), ('\u{1e2f}', '\u{1e2f}',
        GeneralCategory::LowercaseLetter), ('\u{1e30}', '\u{1e30}',
        GeneralCategory::UppercaseLetter), ('\u{1e31}', '\u{1e31}',
        GeneralCategory::LowercaseLetter), ('\u{1e32}', '\u{1e32}',
        GeneralCategory::UppercaseLetter), ('\u{1e33}', '\u{1e33}',
        GeneralCategory::LowercaseLetter), ('\u{1e34}', '\u{1e34}',
        GeneralCategory::UppercaseLetter), ('\u{1e35}', '\u{1e35}',
        GeneralCategory::LowercaseLetter), ('\u{1e36}', '\u{1e36}',
        GeneralCategory::UppercaseLetter), ('\u{1e37}', '\u{1e37}',
        GeneralCategory::LowercaseLetter), ('\u{1e38}', '\u{1e38}',
        GeneralCategory::UppercaseLetter), ('\u{1e39}', '\u{1e39}',
        GeneralCategory::LowercaseLetter), ('\u{1e3a}', '\u{1e3a}',
        GeneralCategory::UppercaseLetter), ('\u{1e3b}', '\u{1e3b}',
        GeneralCategory::LowercaseLetter), ('\u{1e3c}', '\u{1e3c}',
        GeneralCategory::UppercaseLetter), ('\u{1e3d}', '\u{1e3d}',
        GeneralCategory::LowercaseLetter), ('\u{1e3e}', '\u{1e3e}',
        GeneralCategory::UppercaseLetter), ('\u{1e3f}', '\u{1e3f}',
        GeneralCategory::LowercaseLetter), ('\u{1e40}', '\u{1e40}',
        GeneralCategory::UppercaseLetter), ('\u{1e41}', '\u{1e41}',
        GeneralCategory::LowercaseLetter), ('\u{1e42}', '\u{1e42}',
        GeneralCategory::UppercaseLetter), ('\u{1e43}', '\u{1e43}',
        GeneralCategory::LowercaseLetter), ('\u{1e44}', '\u{1e44}',
        GeneralCategory::UppercaseLetter), ('\u{1e45}', '\u{1e45}',
        GeneralCategory::LowercaseLetter), ('\u{1e46}', '\u{1e46}',
        GeneralCategory::UppercaseLetter), ('\u{1e47}', '\u{1e47}',
        GeneralCategory::LowercaseLetter), ('\u{1e48}', '\u{1e48}',
        GeneralCategory::UppercaseLetter), ('\u{1e49}', '\u{1e49}',
        GeneralCategory::LowercaseLetter), ('\u{1e4a}', '\u{1e4a}',
        GeneralCategory::UppercaseLetter), ('\u{1e4b}', '\u{1e4b}',
        GeneralCategory::LowercaseLetter), ('\u{1e4c}', '\u{1e4c}',
        GeneralCategory::UppercaseLetter), ('\u{1e4d}', '\u{1e4d}',
        GeneralCategory::LowercaseLetter), ('\u{1e4e}', '\u{1e4e}',
        GeneralCategory::UppercaseLetter), ('\u{1e4f}', '\u{1e4f}',
        GeneralCategory::LowercaseLetter), ('\u{1e50}', '\u{1e50}',
        GeneralCategory::UppercaseLetter), ('\u{1e51}', '\u{1e51}',
        GeneralCategory::LowercaseLetter), ('\u{1e52}', '\u{1e52}',
        GeneralCategory::UppercaseLetter), ('\u{1e53}', '\u{1e53}',
        GeneralCategory::LowercaseLetter), ('\u{1e54}', '\u{1e54}',
        GeneralCategory::UppercaseLetter), ('\u{1e55}', '\u{1e55}',
        GeneralCategory::LowercaseLetter), ('\u{1e56}', '\u{1e56}',
        GeneralCategory::UppercaseLetter), ('\u{1e57}', '\u{1e57}',
        GeneralCategory::LowercaseLetter), ('\u{1e58}', '\u{1e58}',
        GeneralCategory::UppercaseLetter), ('\u{1e59}', '\u{1e59}',
        GeneralCategory::LowercaseLetter), ('\u{1e5a}', '\u{1e5a}',
        GeneralCategory::UppercaseLetter), ('\u{1e5b}', '\u{1e5b}',
        GeneralCategory::LowercaseLetter), ('\u{1e5c}', '\u{1e5c}',
        GeneralCategory::UppercaseLetter), ('\u{1e5d}', '\u{1e5d}',
        GeneralCategory::LowercaseLetter), ('\u{1e5e}', '\u{1e5e}',
        GeneralCategory::UppercaseLetter), ('\u{1e5f}', '\u{1e5f}',
        GeneralCategory::LowercaseLetter), ('\u{1e60}', '\u{1e60}',
        GeneralCategory::UppercaseLetter), ('\u{1e61}', '\u{1e61}',
        GeneralCategory::LowercaseLetter), ('\u{1e62}', '\u{1e62}',
        GeneralCategory::UppercaseLetter), ('\u{1e63}', '\u{1e63}',
        GeneralCategory::LowercaseLetter), ('\u{1e64}', '\u{1e64}',
        GeneralCategory::UppercaseLetter), ('\u{1e65}', '\u{1e65}',
        GeneralCategory::LowercaseLetter), ('\u{1e66}', '\u{1e66}',
        GeneralCategory::UppercaseLetter), ('\u{1e67}', '\u{1e67}',
        GeneralCategory::LowercaseLetter), ('\u{1e68}', '\u{1e68}',
        GeneralCategory::UppercaseLetter), ('\u{1e69}', '\u{1e69}',
        GeneralCategory::LowercaseLetter), ('\u{1e6a}', '\u{1e6a}',
        GeneralCategory::UppercaseLetter), ('\u{1e6b}', '\u{1e6b}',
        GeneralCategory::LowercaseLetter), ('\u{1e6c}', '\u{1e6c}',
        GeneralCategory::UppercaseLetter), ('\u{1e6d}', '\u{1e6d}',
        GeneralCategory::LowercaseLetter), ('\u{1e6e}', '\u{1e6e}',
        GeneralCategory::UppercaseLetter), ('\u{1e6f}', '\u{1e6f}',
        GeneralCategory::LowercaseLetter), ('\u{1e70}', '\u{1e70}',
        GeneralCategory::UppercaseLetter), ('\u{1e71}', '\u{1e71}',
        GeneralCategory::LowercaseLetter), ('\u{1e72}', '\u{1e72}',
        GeneralCategory::UppercaseLetter), ('\u{1e73}', '\u{1e73}',
        GeneralCategory::LowercaseLetter), ('\u{1e74}', '\u{1e74}',
        GeneralCategory::UppercaseLetter), ('\u{1e75}', '\u{1e75}',
        GeneralCategory::LowercaseLetter), ('\u{1e76}', '\u{1e76}',
        GeneralCategory::UppercaseLetter), ('\u{1e77}', '\u{1e77}',
        GeneralCategory::LowercaseLetter), ('\u{1e78}', '\u{1e78}',
        GeneralCategory::UppercaseLetter), ('\u{1e79}', '\u{1e79}',
        GeneralCategory::LowercaseLetter), ('\u{1e7a}', '\u{1e7a}',
        GeneralCategory::UppercaseLetter), ('\u{1e7b}', '\u{1e7b}',
        GeneralCategory::LowercaseLetter), ('\u{1e7c}', '\u{1e7c}',
        GeneralCategory::UppercaseLetter), ('\u{1e7d}', '\u{1e7d}',
        GeneralCategory::LowercaseLetter), ('\u{1e7e}', '\u{1e7e}',
        GeneralCategory::UppercaseLetter), ('\u{1e7f}', '\u{1e7f}',
        GeneralCategory::LowercaseLetter), ('\u{1e80}', '\u{1e80}',
        GeneralCategory::UppercaseLetter), ('\u{1e81}', '\u{1e81}',
        GeneralCategory::LowercaseLetter), ('\u{1e82}', '\u{1e82}',
        GeneralCategory::UppercaseLetter), ('\u{1e83}', '\u{1e83}',
        GeneralCategory::LowercaseLetter), ('\u{1e84}', '\u{1e84}',
        GeneralCategory::UppercaseLetter), ('\u{1e85}', '\u{1e85}',
        GeneralCategory::LowercaseLetter), ('\u{1e86}', '\u{1e86}',
        GeneralCategory::UppercaseLetter), ('\u{1e87}', '\u{1e87}',
        GeneralCategory::LowercaseLetter), ('\u{1e88}', '\u{1e88}',
        GeneralCategory::UppercaseLetter), ('\u{1e89}', '\u{1e89}',
        GeneralCategory::LowercaseLetter), ('\u{1e8a}', '\u{1e8a}',
        GeneralCategory::UppercaseLetter), ('\u{1e8b}', '\u{1e8b}',
        GeneralCategory::LowercaseLetter), ('\u{1e8c}', '\u{1e8c}',
        GeneralCategory::UppercaseLetter), ('\u{1e8d}', '\u{1e8d}',
        GeneralCategory::LowercaseLetter), ('\u{1e8e}', '\u{1e8e}',
        GeneralCategory::UppercaseLetter), ('\u{1e8f}', '\u{1e8f}',
        GeneralCategory::LowercaseLetter), ('\u{1e90}', '\u{1e90}',
        GeneralCategory::UppercaseLetter), ('\u{1e91}', '\u{1e91}',
        GeneralCategory::LowercaseLetter), ('\u{1e92}', '\u{1e92}',
        GeneralCategory::UppercaseLetter), ('\u{1e93}', '\u{1e93}',
        GeneralCategory::LowercaseLetter), ('\u{1e94}', '\u{1e94}',
        GeneralCategory::UppercaseLetter), ('\u{1e95}', '\u{1e9d}',
        GeneralCategory::LowercaseLetter), ('\u{1e9e}', '\u{1e9e}',
        GeneralCategory::UppercaseLetter), ('\u{1e9f}', '\u{1e9f}',
        GeneralCategory::LowercaseLetter), ('\u{1ea0}', '\u{1ea0}',
        GeneralCategory::UppercaseLetter), ('\u{1ea1}', '\u{1ea1}',
        GeneralCategory::LowercaseLetter), ('\u{1ea2}', '\u{1ea2}',
        GeneralCategory::UppercaseLetter), ('\u{1ea3}', '\u{1ea3}',
        GeneralCategory::LowercaseLetter), ('\u{1ea4}', '\u{1ea4}',
        GeneralCategory::UppercaseLetter), ('\u{1ea5}', '\u{1ea5}',
        GeneralCategory::LowercaseLetter), ('\u{1ea6}', '\u{1ea6}',
        GeneralCategory::UppercaseLetter), ('\u{1ea7}', '\u{1ea7}',
        GeneralCategory::LowercaseLetter), ('\u{1ea8}', '\u{1ea8}',
        GeneralCategory::UppercaseLetter), ('\u{1ea9}', '\u{1ea9}',
        GeneralCategory::LowercaseLetter), ('\u{1eaa}', '\u{1eaa}',
        GeneralCategory::UppercaseLetter), ('\u{1eab}', '\u{1eab}',
        GeneralCategory::LowercaseLetter), ('\u{1eac}', '\u{1eac}',
        GeneralCategory::UppercaseLetter), ('\u{1ead}', '\u{1ead}',
        GeneralCategory::LowercaseLetter), ('\u{1eae}', '\u{1eae}',
        GeneralCategory::UppercaseLetter), ('\u{1eaf}', '\u{1eaf}',
        GeneralCategory::LowercaseLetter), ('\u{1eb0}', '\u{1eb0}',
        GeneralCategory::UppercaseLetter), ('\u{1eb1}', '\u{1eb1}',
        GeneralCategory::LowercaseLetter), ('\u{1eb2}', '\u{1eb2}',
        GeneralCategory::UppercaseLetter), ('\u{1eb3}', '\u{1eb3}',
        GeneralCategory::LowercaseLetter), ('\u{1eb4}', '\u{1eb4}',
        GeneralCategory::UppercaseLetter), ('\u{1eb5}', '\u{1eb5}',
        GeneralCategory::LowercaseLetter), ('\u{1eb6}', '\u{1eb6}',
        GeneralCategory::UppercaseLetter), ('\u{1eb7}', '\u{1eb7}',
        GeneralCategory::LowercaseLetter), ('\u{1eb8}', '\u{1eb8}',
        GeneralCategory::UppercaseLetter), ('\u{1eb9}', '\u{1eb9}',
        GeneralCategory::LowercaseLetter), ('\u{1eba}', '\u{1eba}',
        GeneralCategory::UppercaseLetter), ('\u{1ebb}', '\u{1ebb}',
        GeneralCategory::LowercaseLetter), ('\u{1ebc}', '\u{1ebc}',
        GeneralCategory::UppercaseLetter), ('\u{1ebd}', '\u{1ebd}',
        GeneralCategory::LowercaseLetter), ('\u{1ebe}', '\u{1ebe}',
        GeneralCategory::UppercaseLetter), ('\u{1ebf}', '\u{1ebf}',
        GeneralCategory::LowercaseLetter), ('\u{1ec0}', '\u{1ec0}',
        GeneralCategory::UppercaseLetter), ('\u{1ec1}', '\u{1ec1}',
        GeneralCategory::LowercaseLetter), ('\u{1ec2}', '\u{1ec2}',
        GeneralCategory::UppercaseLetter), ('\u{1ec3}', '\u{1ec3}',
        GeneralCategory::LowercaseLetter), ('\u{1ec4}', '\u{1ec4}',
        GeneralCategory::UppercaseLetter), ('\u{1ec5}', '\u{1ec5}',
        GeneralCategory::LowercaseLetter), ('\u{1ec6}', '\u{1ec6}',
        GeneralCategory::UppercaseLetter), ('\u{1ec7}', '\u{1ec7}',
        GeneralCategory::LowercaseLetter), ('\u{1ec8}', '\u{1ec8}',
        GeneralCategory::UppercaseLetter), ('\u{1ec9}', '\u{1ec9}',
        GeneralCategory::LowercaseLetter), ('\u{1eca}', '\u{1eca}',
        GeneralCategory::UppercaseLetter), ('\u{1ecb}', '\u{1ecb}',
        GeneralCategory::LowercaseLetter), ('\u{1ecc}', '\u{1ecc}',
        GeneralCategory::UppercaseLetter), ('\u{1ecd}', '\u{1ecd}',
        GeneralCategory::LowercaseLetter), ('\u{1ece}', '\u{1ece}',
        GeneralCategory::UppercaseLetter), ('\u{1ecf}', '\u{1ecf}',
        GeneralCategory::LowercaseLetter), ('\u{1ed0}', '\u{1ed0}',
        GeneralCategory::UppercaseLetter), ('\u{1ed1}', '\u{1ed1}',
        GeneralCategory::LowercaseLetter), ('\u{1ed2}', '\u{1ed2}',
        GeneralCategory::UppercaseLetter), ('\u{1ed3}', '\u{1ed3}',
        GeneralCategory::LowercaseLetter), ('\u{1ed4}', '\u{1ed4}',
        GeneralCategory::UppercaseLetter), ('\u{1ed5}', '\u{1ed5}',
        GeneralCategory::LowercaseLetter), ('\u{1ed6}', '\u{1ed6}',
        GeneralCategory::UppercaseLetter), ('\u{1ed7}', '\u{1ed7}',
        GeneralCategory::LowercaseLetter), ('\u{1ed8}', '\u{1ed8}',
        GeneralCategory::UppercaseLetter), ('\u{1ed9}', '\u{1ed9}',
        GeneralCategory::LowercaseLetter), ('\u{1eda}', '\u{1eda}',
        GeneralCategory::UppercaseLetter), ('\u{1edb}', '\u{1edb}',
        GeneralCategory::LowercaseLetter), ('\u{1edc}', '\u{1edc}',
        GeneralCategory::UppercaseLetter), ('\u{1edd}', '\u{1edd}',
        GeneralCategory::LowercaseLetter), ('\u{1ede}', '\u{1ede}',
        GeneralCategory::UppercaseLetter), ('\u{1edf}', '\u{1edf}',
        GeneralCategory::LowercaseLetter), ('\u{1ee0}', '\u{1ee0}',
        GeneralCategory::UppercaseLetter), ('\u{1ee1}', '\u{1ee1}',
        GeneralCategory::LowercaseLetter), ('\u{1ee2}', '\u{1ee2}',
        GeneralCategory::UppercaseLetter), ('\u{1ee3}', '\u{1ee3}',
        GeneralCategory::LowercaseLetter), ('\u{1ee4}', '\u{1ee4}',
        GeneralCategory::UppercaseLetter), ('\u{1ee5}', '\u{1ee5}',
        GeneralCategory::LowercaseLetter), ('\u{1ee6}', '\u{1ee6}',
        GeneralCategory::UppercaseLetter), ('\u{1ee7}', '\u{1ee7}',
        GeneralCategory::LowercaseLetter), ('\u{1ee8}', '\u{1ee8}',
        GeneralCategory::UppercaseLetter), ('\u{1ee9}', '\u{1ee9}',
        GeneralCategory::LowercaseLetter), ('\u{1eea}', '\u{1eea}',
        GeneralCategory::UppercaseLetter), ('\u{1eeb}', '\u{1eeb}',
        GeneralCategory::LowercaseLetter), ('\u{1eec}', '\u{1eec}',
        GeneralCategory::UppercaseLetter), ('\u{1eed}', '\u{1eed}',
        GeneralCategory::LowercaseLetter), ('\u{1eee}', '\u{1eee}',
        GeneralCategory::UppercaseLetter), ('\u{1eef}', '\u{1eef}',
        GeneralCategory::LowercaseLetter), ('\u{1ef0}', '\u{1ef0}',
        GeneralCategory::UppercaseLetter), ('\u{1ef1}', '\u{1ef1}',
        GeneralCategory::LowercaseLetter), ('\u{1ef2}', '\u{1ef2}',
        GeneralCategory::UppercaseLetter), ('\u{1ef3}', '\u{1ef3}',
        GeneralCategory::LowercaseLetter), ('\u{1ef4}', '\u{1ef4}',
        GeneralCategory::UppercaseLetter), ('\u{1ef5}', '\u{1ef5}',
        GeneralCategory::LowercaseLetter), ('\u{1ef6}', '\u{1ef6}',
        GeneralCategory::UppercaseLetter), ('\u{1ef7}', '\u{1ef7}',
        GeneralCategory::LowercaseLetter), ('\u{1ef8}', '\u{1ef8}',
        GeneralCategory::UppercaseLetter), ('\u{1ef9}', '\u{1ef9}',
        GeneralCategory::LowercaseLetter), ('\u{1efa}', '\u{1efa}',
        GeneralCategory::UppercaseLetter), ('\u{1efb}', '\u{1efb}',
        GeneralCategory::LowercaseLetter), ('\u{1efc}', '\u{1efc}',
        GeneralCategory::UppercaseLetter), ('\u{1efd}', '\u{1efd}',
        GeneralCategory::LowercaseLetter), ('\u{1efe}', '\u{1efe}',
        GeneralCategory::UppercaseLetter), ('\u{1eff}', '\u{1f07}',
        GeneralCategory::LowercaseLetter), ('\u{1f08}', '\u{1f0f}',
        GeneralCategory::UppercaseLetter), ('\u{1f10}', '\u{1f15}',
        GeneralCategory::LowercaseLetter), ('\u{1f18}', '\u{1f1d}',
        GeneralCategory::UppercaseLetter), ('\u{1f20}', '\u{1f27}',
        GeneralCategory::LowercaseLetter), ('\u{1f28}', '\u{1f2f}',
        GeneralCategory::UppercaseLetter), ('\u{1f30}', '\u{1f37}',
        GeneralCategory::LowercaseLetter), ('\u{1f38}', '\u{1f3f}',
        GeneralCategory::UppercaseLetter), ('\u{1f40}', '\u{1f45}',
        GeneralCategory::LowercaseLetter), ('\u{1f48}', '\u{1f4d}',
        GeneralCategory::UppercaseLetter), ('\u{1f50}', '\u{1f57}',
        GeneralCategory::LowercaseLetter), ('\u{1f59}', '\u{1f59}',
        GeneralCategory::UppercaseLetter), ('\u{1f5b}', '\u{1f5b}',
        GeneralCategory::UppercaseLetter), ('\u{1f5d}', '\u{1f5d}',
        GeneralCategory::UppercaseLetter), ('\u{1f5f}', '\u{1f5f}',
        GeneralCategory::UppercaseLetter), ('\u{1f60}', '\u{1f67}',
        GeneralCategory::LowercaseLetter), ('\u{1f68}', '\u{1f6f}',
        GeneralCategory::UppercaseLetter), ('\u{1f70}', '\u{1f7d}',
        GeneralCategory::LowercaseLetter), ('\u{1f80}', '\u{1f87}',
        GeneralCategory::LowercaseLetter), ('\u{1f88}', '\u{1f8f}',
        GeneralCategory::TitlecaseLetter), ('\u{1f90}', '\u{1f97}',
        GeneralCategory::LowercaseLetter), ('\u{1f98}', '\u{1f9f}',
        GeneralCategory::TitlecaseLetter), ('\u{1fa0}', '\u{1fa7}',
        GeneralCategory::LowercaseLetter), ('\u{1fa8}', '\u{1faf}',
        GeneralCategory::TitlecaseLetter), ('\u{1fb0}', '\u{1fb4}',
        GeneralCategory::LowercaseLetter), ('\u{1fb6}', '\u{1fb7}',
        GeneralCategory::LowercaseLetter), ('\u{1fb8}', '\u{1fbb}',
        GeneralCategory::UppercaseLetter), ('\u{1fbc}', '\u{1fbc}',
        GeneralCategory::TitlecaseLetter), ('\u{1fbd}', '\u{1fbd}',
        GeneralCategory::ModifierSymbol), ('\u{1fbe}', '\u{1fbe}',
        GeneralCategory::LowercaseLetter), ('\u{1fbf}', '\u{1fc1}',
        GeneralCategory::ModifierSymbol), ('\u{1fc2}', '\u{1fc4}',
        GeneralCategory::LowercaseLetter), ('\u{1fc6}', '\u{1fc7}',
        GeneralCategory::LowercaseLetter), ('\u{1fc8}', '\u{1fcb}',
        GeneralCategory::UppercaseLetter), ('\u{1fcc}', '\u{1fcc}',
        GeneralCategory::TitlecaseLetter), ('\u{1fcd}', '\u{1fcf}',
        GeneralCategory::ModifierSymbol), ('\u{1fd0}', '\u{1fd3}',
        GeneralCategory::LowercaseLetter), ('\u{1fd6}', '\u{1fd7}',
        GeneralCategory::LowercaseLetter), ('\u{1fd8}', '\u{1fdb}',
        GeneralCategory::UppercaseLetter), ('\u{1fdd}', '\u{1fdf}',
        GeneralCategory::ModifierSymbol), ('\u{1fe0}', '\u{1fe7}',
        GeneralCategory::LowercaseLetter), ('\u{1fe8}', '\u{1fec}',
        GeneralCategory::UppercaseLetter), ('\u{1fed}', '\u{1fef}',
        GeneralCategory::ModifierSymbol), ('\u{1ff2}', '\u{1ff4}',
        GeneralCategory::LowercaseLetter), ('\u{1ff6}', '\u{1ff7}',
        GeneralCategory::LowercaseLetter), ('\u{1ff8}', '\u{1ffb}',
        GeneralCategory::UppercaseLetter), ('\u{1ffc}', '\u{1ffc}',
        GeneralCategory::TitlecaseLetter), ('\u{1ffd}', '\u{1ffe}',
        GeneralCategory::ModifierSymbol), ('\u{2000}', '\u{200a}', GeneralCategory::SpaceSeparator),
        ('\u{200b}', '\u{200f}', GeneralCategory::Format), ('\u{2010}', '\u{2015}',
        GeneralCategory::DashPunctuation), ('\u{2016}', '\u{2017}',
        GeneralCategory::OtherPunctuation), ('\u{2018}', '\u{2018}',
        GeneralCategory::InitialPunctuation), ('\u{2019}', '\u{2019}',
        GeneralCategory::FinalPunctuation), ('\u{201a}', '\u{201a}',
        GeneralCategory::OpenPunctuation), ('\u{201b}', '\u{201c}',
        GeneralCategory::InitialPunctuation), ('\u{201d}', '\u{201d}',
        GeneralCategory::FinalPunctuation), ('\u{201e}', '\u{201e}',
        GeneralCategory::OpenPunctuation), ('\u{201f}', '\u{201f}',
        GeneralCategory::InitialPunctuation), ('\u{2020}', '\u{2027}',
        GeneralCategory::OtherPunctuation), ('\u{2028}', '\u{2028}',
        GeneralCategory::LineSeparator), ('\u{2029}', '\u{2029}',
        GeneralCategory::ParagraphSeparator), ('\u{202a}', '\u{202e}', GeneralCategory::Format),
        ('\u{202f}', '\u{202f}', GeneralCategory::SpaceSeparator), ('\u{2030}', '\u{2038}',
        GeneralCategory::OtherPunctuation), ('\u{2039}', '\u{2039}',
        GeneralCategory::InitialPunctuation), ('\u{203a}', '\u{203a}',
        GeneralCategory::FinalPunctuation), ('\u{203b}', '\u{203e}',
        GeneralCategory::OtherPunctuation), ('\u{203f}', '\u{2040}',
        GeneralCategory::ConnectorPunctuation), ('\u{2041}', '\u{2043}',
        GeneralCategory::OtherPunctuation), ('\u{2044}', '\u{2044}', GeneralCategory::MathSymbol),
        ('\u{2045}', '\u{2045}', GeneralCategory::OpenPunctuation), ('\u{2046}', '\u{2046}',
        GeneralCategory::ClosePunctuation), ('\u{2047}', '\u{2051}',
        GeneralCategory::OtherPunctuation), ('\u{2052}', '\u{2052}', GeneralCategory::MathSymbol),
        ('\u{2053}', '\u{2053}', GeneralCategory::OtherPunctuation), ('\u{2054}', '\u{2054}',
        GeneralCategory::ConnectorPunctuation), ('\u{2055}', '\u{205e}',
        GeneralCategory::OtherPunctuation), ('\u{205f}', '\u{205f}',
        GeneralCategory::SpaceSeparator), ('\u{2060}', '\u{2064}', GeneralCategory::Format),
        ('\u{2066}', '\u{206f}', GeneralCategory::Format), ('\u{2070}', '\u{2070}',
        GeneralCategory::OtherNumber), ('\u{2071}', '\u{2071}', GeneralCategory::ModifierLetter),
        ('\u{2074}', '\u{2079}', GeneralCategory::OtherNumber), ('\u{207a}', '\u{207c}',
        GeneralCategory::MathSymbol), ('\u{207d}', '\u{207d}', GeneralCategory::OpenPunctuation),
        ('\u{207e}', '\u{207e}', GeneralCategory::ClosePunctuation), ('\u{207f}', '\u{207f}',
        GeneralCategory::ModifierLetter), ('\u{2080}', '\u{2089}', GeneralCategory::OtherNumber),
        ('\u{208a}', '\u{208c}', GeneralCategory::MathSymbol), ('\u{208d}', '\u{208d}',
        GeneralCategory::OpenPunctuation), ('\u{208e}', '\u{208e}',
        GeneralCategory::ClosePunctuation), ('\u{2090}', '\u{209c}',
        GeneralCategory::ModifierLetter), ('\u{20a0}', '\u{20c0}', GeneralCategory::CurrencySymbol),
        ('\u{20d0}', '\u{20dc}', GeneralCategory::NonspacingMark), ('\u{20dd}', '\u{20e0}',
        GeneralCategory::EnclosingMark), ('\u{20e1}', '\u{20e1}', GeneralCategory::NonspacingMark),
        ('\u{20e2}', '\u{20e4}', GeneralCategory::EnclosingMark), ('\u{20e5}', '\u{20f0}',
        GeneralCategory::NonspacingMark), ('\u{2100}', '\u{2101}', GeneralCategory::OtherSymbol),
        ('\u{2102}', '\u{2102}', GeneralCategory::UppercaseLetter), ('\u{2103}', '\u{2106}',
        GeneralCategory::OtherSymbol), ('\u{2107}', '\u{2107}', GeneralCategory::UppercaseLetter),
        ('\u{2108}', '\u{2109}', GeneralCategory::OtherSymbol), ('\u{210a}', '\u{210a}',
        GeneralCategory::LowercaseLetter), ('\u{210b}', '\u{210d}',
        GeneralCategory::UppercaseLetter), ('\u{210e}', '\u{210f}',
        GeneralCategory::LowercaseLetter), ('\u{2110}', '\u{2112}',
        GeneralCategory::UppercaseLetter), ('\u{2113}', '\u{2113}',
        GeneralCategory::LowercaseLetter), ('\u{2114}', '\u{2114}', GeneralCategory::OtherSymbol),
        ('\u{2115}', '\u{2115}', GeneralCategory::UppercaseLetter), ('\u{2116}', '\u{2117}',
        GeneralCategory::OtherSymbol), ('\u{2118}', '\u{2118}', GeneralCategory::MathSymbol),
        ('\u{2119}', '\u{211d}', GeneralCategory::UppercaseLetter), ('\u{211e}', '\u{2123}',
        GeneralCategory::OtherSymbol), ('\u{2124}', '\u{2124}', GeneralCategory::UppercaseLetter),
        ('\u{2125}', '\u{2125}', GeneralCategory::OtherSymbol), ('\u{2126}', '\u{2126}',
        GeneralCategory::UppercaseLetter), ('\u{2127}', '\u{2127}', GeneralCategory::OtherSymbol),
        ('\u{2128}', '\u{2128}', GeneralCategory::UppercaseLetter), ('\u{2129}', '\u{2129}',
        GeneralCategory::OtherSymbol), ('\u{212a}', '\u{212d}', GeneralCategory::UppercaseLetter),
        ('\u{212e}', '\u{212e}', GeneralCategory::OtherSymbol), ('\u{212f}', '\u{212f}',
        GeneralCategory::LowercaseLetter), ('\u{2130}', '\u{2133}',
        GeneralCategory::UppercaseLetter), ('\u{2134}', '\u{2134}',
        GeneralCategory::LowercaseLetter), ('\u{2135}', '\u{2138}', GeneralCategory::OtherLetter),
        ('\u{2139}', '\u{2139}', GeneralCategory::LowercaseLetter), ('\u{213a}', '\u{213b}',
        GeneralCategory::OtherSymbol), ('\u{213c}', '\u{213d}', GeneralCategory::LowercaseLetter),
        ('\u{213e}', '\u{213f}', GeneralCategory::UppercaseLetter), ('\u{2140}', '\u{2144}',
        GeneralCategory::MathSymbol), ('\u{2145}', '\u{2145}', GeneralCategory::UppercaseLetter),
        ('\u{2146}', '\u{2149}', GeneralCategory::LowercaseLetter), ('\u{214a}', '\u{214a}',
        GeneralCategory::OtherSymbol), ('\u{214b}', '\u{214b}', GeneralCategory::MathSymbol),
        ('\u{214c}', '\u{214d}', GeneralCategory::OtherSymbol), ('\u{214e}', '\u{214e}',
        GeneralCategory::LowercaseLetter), ('\u{214f}', '\u{214f}', GeneralCategory::OtherSymbol),
        ('\u{2150}', '\u{215f}', GeneralCategory::OtherNumber), ('\u{2160}', '\u{2182}',
        GeneralCategory::LetterNumber), ('\u{2183}', '\u{2183}', GeneralCategory::UppercaseLetter),
        ('\u{2184}', '\u{2184}', GeneralCategory::LowercaseLetter), ('\u{2185}', '\u{2188}',
        GeneralCategory::LetterNumber), ('\u{2189}', '\u{2189}', GeneralCategory::OtherNumber),
        ('\u{218a}', '\u{218b}', GeneralCategory::OtherSymbol), ('\u{2190}', '\u{2194}',
        GeneralCategory::MathSymbol), ('\u{2195}', '\u{2199}', GeneralCategory::OtherSymbol),
        ('\u{219a}', '\u{219b}', GeneralCategory::MathSymbol), ('\u{219c}', '\u{219f}',
        GeneralCategory::OtherSymbol), ('\u{21a0}', '\u{21a0}', GeneralCategory::MathSymbol),
        ('\u{21a1}', '\u{21a2}', GeneralCategory::OtherSymbol), ('\u{21a3}', '\u{21a3}',
        GeneralCategory::MathSymbol), ('\u{21a4}', '\u{21a5}', GeneralCategory::OtherSymbol),
        ('\u{21a6}', '\u{21a6}', GeneralCategory::MathSymbol), ('\u{21a7}', '\u{21ad}',
        GeneralCategory::OtherSymbol), ('\u{21ae}', '\u{21ae}', GeneralCategory::MathSymbol),
        ('\u{21af}', '\u{21cd}', GeneralCategory::OtherSymbol), ('\u{21ce}', '\u{21cf}',
        GeneralCategory::MathSymbol), ('\u{21d0}', '\u{21d1}', GeneralCategory::OtherSymbol),
        ('\u{21d2}', '\u{21d2}', GeneralCategory::MathSymbol), ('\u{21d3}', '\u{21d3}',
        GeneralCategory::OtherSymbol), ('\u{21d4}', '\u{21d4}', GeneralCategory::MathSymbol),
        ('\u{21d5}', '\u{21f3}', GeneralCategory::OtherSymbol), ('\u{21f4}', '\u{22ff}',
        GeneralCategory::MathSymbol), ('\u{2300}', '\u{2307}', GeneralCategory::OtherSymbol),
        ('\u{2308}', '\u{2308}', GeneralCategory::OpenPunctuation), ('\u{2309}', '\u{2309}',
        GeneralCategory::ClosePunctuation), ('\u{230a}', '\u{230a}',
        GeneralCategory::OpenPunctuation), ('\u{230b}', '\u{230b}',
        GeneralCategory::ClosePunctuation), ('\u{230c}', '\u{231f}', GeneralCategory::OtherSymbol),
        ('\u{2320}', '\u{2321}', GeneralCategory::MathSymbol), ('\u{2322}', '\u{2328}',
        GeneralCategory::OtherSymbol), ('\u{2329}', '\u{2329}', GeneralCategory::OpenPunctuation),
        ('\u{232a}', '\u{232a}', GeneralCategory::ClosePunctuation), ('\u{232b}', '\u{237b}',
        GeneralCategory::OtherSymbol), ('\u{237c}', '\u{237c}', GeneralCategory::MathSymbol),
        ('\u{237d}', '\u{239a}', GeneralCategory::OtherSymbol), ('\u{239b}', '\u{23b3}',
        GeneralCategory::MathSymbol), ('\u{23b4}', '\u{23db}', GeneralCategory::OtherSymbol),
        ('\u{23dc}', '\u{23e1}', GeneralCategory::MathSymbol), ('\u{23e2}', '\u{2426}',
        GeneralCategory::OtherSymbol), ('\u{2440}', '\u{244a}', GeneralCategory::OtherSymbol),
        ('\u{2460}', '\u{249b}', GeneralCategory::OtherNumber), ('\u{249c}', '\u{24e9}',
        GeneralCategory::OtherSymbol), ('\u{24ea}', '\u{24ff}', GeneralCategory::OtherNumber),
        ('\u{2500}', '\u{25b6}', GeneralCategory::OtherSymbol), ('\u{25b7}', '\u{25b7}',
        GeneralCategory::MathSymbol), ('\u{25b8}', '\u{25c0}', GeneralCategory::OtherSymbol),
        ('\u{25c1}', '\u{25c1}', GeneralCategory::MathSymbol), ('\u{25c2}', '\u{25f7}',
        GeneralCategory::OtherSymbol), ('\u{25f8}', '\u{25ff}', GeneralCategory::MathSymbol),
        ('\u{2600}', '\u{266e}', GeneralCategory::OtherSymbol), ('\u{266f}', '\u{266f}',
        GeneralCategory::MathSymbol), ('\u{2670}', '\u{2767}', GeneralCategory::OtherSymbol),
        ('\u{2768}', '\u{2768}', GeneralCategory::OpenPunctuation), ('\u{2769}', '\u{2769}',
        GeneralCategory::ClosePunctuation), ('\u{276a}', '\u{276a}',
        GeneralCategory::OpenPunctuation), ('\u{276b}', '\u{276b}',
        GeneralCategory::ClosePunctuation), ('\u{276c}', '\u{276c}',
        GeneralCategory::OpenPunctuation), ('\u{276d}', '\u{276d}',
        GeneralCategory::ClosePunctuation), ('\u{276e}', '\u{276e}',
        GeneralCategory::OpenPunctuation), ('\u{276f}', '\u{276f}',
        GeneralCategory::ClosePunctuation), ('\u{2770}', '\u{2770}',
        GeneralCategory::OpenPunctuation), ('\u{2771}', '\u{2771}',
        GeneralCategory::ClosePunctuation), ('\u{2772}', '\u{2772}',
        GeneralCategory::OpenPunctuation), ('\u{2773}', '\u{2773}',
        GeneralCategory::ClosePunctuation), ('\u{2774}', '\u{2774}',
        GeneralCategory::OpenPunctuation), ('\u{2775}', '\u{2775}',
        GeneralCategory::ClosePunctuation), ('\u{2776}', '\u{2793}', GeneralCategory::OtherNumber),
        ('\u{2794}', '\u{27bf}', GeneralCategory::OtherSymbol), ('\u{27c0}', '\u{27c4}',
        GeneralCategory::MathSymbol), ('\u{27c5}', '\u{27c5}', GeneralCategory::OpenPunctuation),
        ('\u{27c6}', '\u{27c6}', GeneralCategory::ClosePunctuation), ('\u{27c7}', '\u{27e5}',
        GeneralCategory::MathSymbol), ('\u{27e6}', '\u{27e6}', GeneralCategory::OpenPunctuation),
        ('\u{27e7}', '\u{27e7}', GeneralCategory::ClosePunctuation), ('\u{27e8}', '\u{27e8}',
        GeneralCategory::OpenPunctuation), ('\u{27e9}', '\u{27e9}',
        GeneralCategory::ClosePunctuation), ('\u{27ea}', '\u{27ea}',
        GeneralCategory::OpenPunctuation), ('\u{27eb}', '\u{27eb}',
        GeneralCategory::ClosePunctuation), ('\u{27ec}', '\u{27ec}',
        GeneralCategory::OpenPunctuation), ('\u{27ed}', '\u{27ed}',
        GeneralCategory::ClosePunctuation), ('\u{27ee}', '\u{27ee}',
        GeneralCategory::OpenPunctuation), ('\u{27ef}', '\u{27ef}',
        GeneralCategory::ClosePunctuation), ('\u{27f0}', '\u{27ff}', GeneralCategory::MathSymbol),
        ('\u{2800}', '\u{28ff}', GeneralCategory::OtherSymbol), ('\u{2900}', '\u{2982}',
        GeneralCategory::MathSymbol), ('\u{2983}', '\u{2983}', GeneralCategory::OpenPunctuation),
        ('\u{2984}', '\u{2984}', GeneralCategory::ClosePunctuation), ('\u{2985}', '\u{2985}',
        GeneralCategory::OpenPunctuation), ('\u{2986}', '\u{2986}',
        GeneralCategory::ClosePunctuation), ('\u{2987}', '\u{2987}',
        GeneralCategory::OpenPunctuation), ('\u{2988}', '\u{2988}',
        GeneralCategory::ClosePunctuation), ('\u{2989}', '\u{2989}',
        GeneralCategory::OpenPunctuation), ('\u{298a}', '\u{298a}',
        GeneralCategory::ClosePunctuation), ('\u{298b}', '\u{298b}',
        GeneralCategory::OpenPunctuation), ('\u{298c}', '\u{298c}',
        GeneralCategory::ClosePunctuation), ('\u{298d}', '\u{298d}',
        GeneralCategory::OpenPunctuation), ('\u{298e}', '\u{298e}',
        GeneralCategory::ClosePunctuation), ('\u{298f}', '\u{298f}',
        GeneralCategory::OpenPunctuation), ('\u{2990}', '\u{2990}',
        GeneralCategory::ClosePunctuation), ('\u{2991}', '\u{2991}',
        GeneralCategory::OpenPunctuation), ('\u{2992}', '\u{2992}',
        GeneralCategory::ClosePunctuation), ('\u{2993}', '\u{2993}',
        GeneralCategory::OpenPunctuation), ('\u{2994}', '\u{2994}',
        GeneralCategory::ClosePunctuation), ('\u{2995}', '\u{2995}',
        GeneralCategory::OpenPunctuation), ('\u{2996}', '\u{2996}',
        GeneralCategory::ClosePunctuation), ('\u{2997}', '\u{2997}',
        GeneralCategory::OpenPunctuation), ('\u{2998}', '\u{2998}',
        GeneralCategory::ClosePunctuation), ('\u{2999}', '\u{29d7}', GeneralCategory::MathSymbol),
        ('\u{29d8}', '\u{29d8}', GeneralCategory::OpenPunctuation), ('\u{29d9}', '\u{29d9}',
        GeneralCategory::ClosePunctuation), ('\u{29da}', '\u{29da}',
        GeneralCategory::OpenPunctuation), ('\u{29db}', '\u{29db}',
        GeneralCategory::ClosePunctuation), ('\u{29dc}', '\u{29fb}', GeneralCategory::MathSymbol),
        ('\u{29fc}', '\u{29fc}', GeneralCategory::OpenPunctuation), ('\u{29fd}', '\u{29fd}',
        GeneralCategory::ClosePunctuation), ('\u{29fe}', '\u{2aff}', GeneralCategory::MathSymbol),
        ('\u{2b00}', '\u{2b2f}', GeneralCategory::OtherSymbol), ('\u{2b30}', '\u{2b44}',
        GeneralCategory::MathSymbol), ('\u{2b45}', '\u{2b46}', GeneralCategory::OtherSymbol),
        ('\u{2b47}', '\u{2b4c}', GeneralCategory::MathSymbol), ('\u{2b4d}', '\u{2b73}',
        GeneralCategory::OtherSymbol), ('\u{2b76}', '\u{2b95}', GeneralCategory::OtherSymbol),
        ('\u{2b97}', '\u{2bff}', GeneralCategory::OtherSymbol), ('\u{2c00}', '\u{2c2f}',
        GeneralCategory::UppercaseLetter), ('\u{2c30}', '\u{2c5f}',
        GeneralCategory::LowercaseLetter), ('\u{2c60}', '\u{2c60}',
        GeneralCategory::UppercaseLetter), ('\u{2c61}', '\u{2c61}',
        GeneralCategory::LowercaseLetter), ('\u{2c62}', '\u{2c64}',
        GeneralCategory::UppercaseLetter), ('\u{2c65}', '\u{2c66}',
        GeneralCategory::LowercaseLetter), ('\u{2c67}', '\u{2c67}',
        GeneralCategory::UppercaseLetter), ('\u{2c68}', '\u{2c68}',
        GeneralCategory::LowercaseLetter), ('\u{2c69}', '\u{2c69}',
        GeneralCategory::UppercaseLetter), ('\u{2c6a}', '\u{2c6a}',
        GeneralCategory::LowercaseLetter), ('\u{2c6b}', '\u{2c6b}',
        GeneralCategory::UppercaseLetter), ('\u{2c6c}', '\u{2c6c}',
        GeneralCategory::LowercaseLetter), ('\u{2c6d}', '\u{2c70}',
        GeneralCategory::UppercaseLetter), ('\u{2c71}', '\u{2c71}',
        GeneralCategory::LowercaseLetter), ('\u{2c72}', '\u{2c72}',
        GeneralCategory::UppercaseLetter), ('\u{2c73}', '\u{2c74}',
        GeneralCategory::LowercaseLetter), ('\u{2c75}', '\u{2c75}',
        GeneralCategory::UppercaseLetter), ('\u{2c76}', '\u{2c7b}',
        GeneralCategory::LowercaseLetter), ('\u{2c7c}', '\u{2c7d}',
        GeneralCategory::ModifierLetter), ('\u{2c7e}', '\u{2c80}',
        GeneralCategory::UppercaseLetter), ('\u{2c81}', '\u{2c81}',
        GeneralCategory::LowercaseLetter), ('\u{2c82}', '\u{2c82}',
        GeneralCategory::UppercaseLetter), ('\u{2c83}', '\u{2c83}',
        GeneralCategory::LowercaseLetter), ('\u{2c84}', '\u{2c84}',
        GeneralCategory::UppercaseLetter), ('\u{2c85}', '\u{2c85}',
        GeneralCategory::LowercaseLetter), ('\u{2c86}', '\u{2c86}',
        GeneralCategory::UppercaseLetter), ('\u{2c87}', '\u{2c87}',
        GeneralCategory::LowercaseLetter), ('\u{2c88}', '\u{2c88}',
        GeneralCategory::UppercaseLetter), ('\u{2c89}', '\u{2c89}',
        GeneralCategory::LowercaseLetter), ('\u{2c8a}', '\u{2c8a}',
        GeneralCategory::UppercaseLetter), ('\u{2c8b}', '\u{2c8b}',
        GeneralCategory::LowercaseLetter), ('\u{2c8c}', '\u{2c8c}',
        GeneralCategory::UppercaseLetter), ('\u{2c8d}', '\u{2c8d}',
        GeneralCategory::LowercaseLetter), ('\u{2c8e}', '\u{2c8e}',
        GeneralCategory::UppercaseLetter), ('\u{2c8f}', '\u{2c8f}',
        GeneralCategory::LowercaseLetter), ('\u{2c90}', '\u{2c90}',
        GeneralCategory::UppercaseLetter), ('\u{2c91}', '\u{2c91}',
        GeneralCategory::LowercaseLetter), ('\u{2c92}', '\u{2c92}',
        GeneralCategory::UppercaseLetter), ('\u{2c93}', '\u{2c93}',
        GeneralCategory::LowercaseLetter), ('\u{2c94}', '\u{2c94}',
        GeneralCategory::UppercaseLetter), ('\u{2c95}', '\u{2c95}',
        GeneralCategory::LowercaseLetter), ('\u{2c96}', '\u{2c96}',
        GeneralCategory::UppercaseLetter), ('\u{2c97}', '\u{2c97}',
        GeneralCategory::LowercaseLetter), ('\u{2c98}', '\u{2c98}',
        GeneralCategory::UppercaseLetter), ('\u{2c99}', '\u{2c99}',
        GeneralCategory::LowercaseLetter), ('\u{2c9a}', '\u{2c9a}',
        GeneralCategory::UppercaseLetter), ('\u{2c9b}', '\u{2c9b}',
        GeneralCategory::LowercaseLetter), ('\u{2c9c}', '\u{2c9c}',
        GeneralCategory::UppercaseLetter), ('\u{2c9d}', '\u{2c9d}',
        GeneralCategory::LowercaseLetter), ('\u{2c9e}', '\u{2c9e}',
        GeneralCategory::UppercaseLetter), ('\u{2c9f}', '\u{2c9f}',
        GeneralCategory::LowercaseLetter), ('\u{2ca0}', '\u{2ca0}',
        GeneralCategory::UppercaseLetter), ('\u{2ca1}', '\u{2ca1}',
        GeneralCategory::LowercaseLetter), ('\u{2ca2}', '\u{2ca2}',
        GeneralCategory::UppercaseLetter), ('\u{2ca3}', '\u{2ca3}',
        GeneralCategory::LowercaseLetter), ('\u{2ca4}', '\u{2ca4}',
        GeneralCategory::UppercaseLetter), ('\u{2ca5}', '\u{2ca5}',
        GeneralCategory::LowercaseLetter), ('\u{2ca6}', '\u{2ca6}',
        GeneralCategory::UppercaseLetter), ('\u{2ca7}', '\u{2ca7}',
        GeneralCategory::LowercaseLetter), ('\u{2ca8}', '\u{2ca8}',
        GeneralCategory::UppercaseLetter), ('\u{2ca9}', '\u{2ca9}',
        GeneralCategory::LowercaseLetter), ('\u{2caa}', '\u{2caa}',
        GeneralCategory::UppercaseLetter), ('\u{2cab}', '\u{2cab}',
        GeneralCategory::LowercaseLetter), ('\u{2cac}', '\u{2cac}',
        GeneralCategory::UppercaseLetter), ('\u{2cad}', '\u{2cad}',
        GeneralCategory::LowercaseLetter), ('\u{2cae}', '\u{2cae}',
        GeneralCategory::UppercaseLetter), ('\u{2caf}', '\u{2caf}',
        GeneralCategory::LowercaseLetter), ('\u{2cb0}', '\u{2cb0}',
        GeneralCategory::UppercaseLetter), ('\u{2cb1}', '\u{2cb1}',
        GeneralCategory::LowercaseLetter), ('\u{2cb2}', '\u{2cb2}',
        GeneralCategory::UppercaseLetter), ('\u{2cb3}', '\u{2cb3}',
        GeneralCategory::LowercaseLetter), ('\u{2cb4}', '\u{2cb4}',
        GeneralCategory::UppercaseLetter), ('\u{2cb5}', '\u{2cb5}',
        GeneralCategory::LowercaseLetter), ('\u{2cb6}', '\u{2cb6}',
        GeneralCategory::UppercaseLetter), ('\u{2cb7}', '\u{2cb7}',
        GeneralCategory::LowercaseLetter), ('\u{2cb8}', '\u{2cb8}',
        GeneralCategory::UppercaseLetter), ('\u{2cb9}', '\u{2cb9}',
        GeneralCategory::LowercaseLetter), ('\u{2cba}', '\u{2cba}',
        GeneralCategory::UppercaseLetter), ('\u{2cbb}', '\u{2cbb}',
        GeneralCategory::LowercaseLetter), ('\u{2cbc}', '\u{2cbc}',
        GeneralCategory::UppercaseLetter), ('\u{2cbd}', '\u{2cbd}',
        GeneralCategory::LowercaseLetter), ('\u{2cbe}', '\u{2cbe}',
        GeneralCategory::UppercaseLetter), ('\u{2cbf}', '\u{2cbf}',
        GeneralCategory::LowercaseLetter), ('\u{2cc0}', '\u{2cc0}',
        GeneralCategory::UppercaseLetter), ('\u{2cc1}', '\u{2cc1}',
        GeneralCategory::LowercaseLetter), ('\u{2cc2}', '\u{2cc2}',
        GeneralCategory::UppercaseLetter), ('\u{2cc3}', '\u{2cc3}',
        GeneralCategory::LowercaseLetter), ('\u{2cc4}', '\u{2cc4}',
        GeneralCategory::UppercaseLetter), ('\u{2cc5}', '\u{2cc5}',
        GeneralCategory::LowercaseLetter), ('\u{2cc6}', '\u{2cc6}',
        GeneralCategory::UppercaseLetter), ('\u{2cc7}', '\u{2cc7}',
        GeneralCategory::LowercaseLetter), ('\u{2cc8}', '\u{2cc8}',
        GeneralCategory::UppercaseLetter), ('\u{2cc9}', '\u{2cc9}',
        GeneralCategory::LowercaseLetter), ('\u{2cca}', '\u{2cca}',
        GeneralCategory::UppercaseLetter), ('\u{2ccb}', '\u{2ccb}',
        GeneralCategory::LowercaseLetter), ('\u{2ccc}', '\u{2ccc}',
        GeneralCategory::UppercaseLetter), ('\u{2ccd}', '\u{2ccd}',
        GeneralCategory::LowercaseLetter), ('\u{2cce}', '\u{2cce}',
        GeneralCategory::UppercaseLetter), ('\u{2ccf}', '\u{2ccf}',
        GeneralCategory::LowercaseLetter), ('\u{2cd0}', '\u{2cd0}',
        GeneralCategory::UppercaseLetter), ('\u{2cd1}', '\u{2cd1}',
        GeneralCategory::LowercaseLetter), ('\u{2cd2}', '\u{2cd2}',
        GeneralCategory::UppercaseLetter), ('\u{2cd3}', '\u{2cd3}',
        GeneralCategory::LowercaseLetter), ('\u{2cd4}', '\u{2cd4}',
        GeneralCategory::UppercaseLetter), ('\u{2cd5}', '\u{2cd5}',
        GeneralCategory::LowercaseLetter), ('\u{2cd6}', '\u{2cd6}',
        GeneralCategory::UppercaseLetter), ('\u{2cd7}', '\u{2cd7}',
        GeneralCategory::LowercaseLetter), ('\u{2cd8}', '\u{2cd8}',
        GeneralCategory::UppercaseLetter), ('\u{2cd9}', '\u{2cd9}',
        GeneralCategory::LowercaseLetter), ('\u{2cda}', '\u{2cda}',
        GeneralCategory::UppercaseLetter), ('\u{2cdb}', '\u{2cdb}',
        GeneralCategory::LowercaseLetter), ('\u{2cdc}', '\u{2cdc}',
        GeneralCategory::UppercaseLetter), ('\u{2cdd}', '\u{2cdd}',
        GeneralCategory::LowercaseLetter), ('\u{2cde}', '\u{2cde}',
        GeneralCategory::UppercaseLetter), ('\u{2cdf}', '\u{2cdf}',
        GeneralCategory::LowercaseLetter), ('\u{2ce0}', '\u{2ce0}',
        GeneralCategory::UppercaseLetter), ('\u{2ce1}', '\u{2ce1}',
        GeneralCategory::LowercaseLetter), ('\u{2ce2}', '\u{2ce2}',
        GeneralCategory::UppercaseLetter), ('\u{2ce3}', '\u{2ce4}',
        GeneralCategory::LowercaseLetter), ('\u{2ce5}', '\u{2cea}', GeneralCategory::OtherSymbol),
        ('\u{2ceb}', '\u{2ceb}', GeneralCategory::UppercaseLetter), ('\u{2cec}', '\u{2cec}',
        GeneralCategory::LowercaseLetter), ('\u{2ced}', '\u{2ced}',
        GeneralCategory::UppercaseLetter), ('\u{2cee}', '\u{2cee}',
        GeneralCategory::LowercaseLetter), ('\u{2cef}', '\u{2cf1}',
        GeneralCategory::NonspacingMark), ('\u{2cf2}', '\u{2cf2}',
        GeneralCategory::UppercaseLetter), ('\u{2cf3}', '\u{2cf3}',
        GeneralCategory::LowercaseLetter), ('\u{2cf9}', '\u{2cfc}',
        GeneralCategory::OtherPunctuation), ('\u{2cfd}', '\u{2cfd}', GeneralCategory::OtherNumber),
        ('\u{2cfe}', '\u{2cff}', GeneralCategory::OtherPunctuation), ('\u{2d00}', '\u{2d25}',
        GeneralCategory::LowercaseLetter), ('\u{2d27}', '\u{2d27}',
        GeneralCategory::LowercaseLetter), ('\u{2d2d}', '\u{2d2d}',
        GeneralCategory::LowercaseLetter), ('\u{2d30}', '\u{2d67}', GeneralCategory::OtherLetter),
        ('\u{2d6f}', '\u{2d6f}', GeneralCategory::ModifierLetter), ('\u{2d70}', '\u{2d70}',
        GeneralCategory::OtherPunctuation), ('\u{2d7f}', '\u{2d7f}',
        GeneralCategory::NonspacingMark), ('\u{2d80}', '\u{2d96}', GeneralCategory::OtherLetter),
        ('\u{2da0}', '\u{2da6}', GeneralCategory::OtherLetter), ('\u{2da8}', '\u{2dae}',
        GeneralCategory::OtherLetter), ('\u{2db0}', '\u{2db6}', GeneralCategory::OtherLetter),
        ('\u{2db8}', '\u{2dbe}', GeneralCategory::OtherLetter), ('\u{2dc0}', '\u{2dc6}',
        GeneralCategory::OtherLetter), ('\u{2dc8}', '\u{2dce}', GeneralCategory::OtherLetter),
        ('\u{2dd0}', '\u{2dd6}', GeneralCategory::OtherLetter), ('\u{2dd8}', '\u{2dde}',
        GeneralCategory::OtherLetter), ('\u{2de0}', '\u{2dff}', GeneralCategory::NonspacingMark),
        ('\u{2e00}', '\u{2e01}', GeneralCategory::OtherPunctuation), ('\u{2e02}', '\u{2e02}',
        GeneralCategory::InitialPunctuation), ('\u{2e03}', '\u{2e03}',
        GeneralCategory::FinalPunctuation), ('\u{2e04}', '\u{2e04}',
        GeneralCategory::InitialPunctuation), ('\u{2e05}', '\u{2e05}',
        GeneralCategory::FinalPunctuation), ('\u{2e06}', '\u{2e08}',
        GeneralCategory::OtherPunctuation), ('\u{2e09}', '\u{2e09}',
        GeneralCategory::InitialPunctuation), ('\u{2e0a}', '\u{2e0a}',
        GeneralCategory::FinalPunctuation), ('\u{2e0b}', '\u{2e0b}',
        GeneralCategory::OtherPunctuation), ('\u{2e0c}', '\u{2e0c}',
        GeneralCategory::InitialPunctuation), ('\u{2e0d}', '\u{2e0d}',
        GeneralCategory::FinalPunctuation), ('\u{2e0e}', '\u{2e16}',
        GeneralCategory::OtherPunctuation), ('\u{2e17}', '\u{2e17}',
        GeneralCategory::DashPunctuation), ('\u{2e18}', '\u{2e19}',
        GeneralCategory::OtherPunctuation), ('\u{2e1a}', '\u{2e1a}',
        GeneralCategory::DashPunctuation), ('\u{2e1b}', '\u{2e1b}',
        GeneralCategory::OtherPunctuation), ('\u{2e1c}', '\u{2e1c}',
        GeneralCategory::InitialPunctuation), ('\u{2e1d}', '\u{2e1d}',
        GeneralCategory::FinalPunctuation), ('\u{2e1e}', '\u{2e1f}',
        GeneralCategory::OtherPunctuation), ('\u{2e20}', '\u{2e20}',
        GeneralCategory::InitialPunctuation), ('\u{2e21}', '\u{2e21}',
        GeneralCategory::FinalPunctuation), ('\u{2e22}', '\u{2e22}',
        GeneralCategory::OpenPunctuation), ('\u{2e23}', '\u{2e23}',
        GeneralCategory::ClosePunctuation), ('\u{2e24}', '\u{2e24}',
        GeneralCategory::OpenPunctuation), ('\u{2e25}', '\u{2e25}',
        GeneralCategory::ClosePunctuation), ('\u{2e26}', '\u{2e26}',
        GeneralCategory::OpenPunctuation), ('\u{2e27}', '\u{2e27}',
        GeneralCategory::ClosePunctuation), ('\u{2e28}', '\u{2e28}',
        GeneralCategory::OpenPunctuation), ('\u{2e29}', '\u{2e29}',
        GeneralCategory::ClosePunctuation), ('\u{2e2a}', '\u{2e2e}',
        GeneralCategory::OtherPunctuation), ('\u{2e2f}', '\u{2e2f}',
        GeneralCategory::ModifierLetter), ('\u{2e30}', '\u{2e39}',
        GeneralCategory::OtherPunctuation), ('\u{2e3a}', '\u{2e3b}',
        GeneralCategory::DashPunctuation), ('\u{2e3c}', '\u{2e3f}',
        GeneralCategory::OtherPunctuation), ('\u{2e40}', '\u{2e40}',
        GeneralCategory::DashPunctuation), ('\u{2e41}', '\u{2e41}',
        GeneralCategory::OtherPunctuation), ('\u{2e42}', '\u{2e42}',
        GeneralCategory::OpenPunctuation), ('\u{2e43}', '\u{2e4f}',
        GeneralCategory::OtherPunctuation), ('\u{2e50}', '\u{2e51}', GeneralCategory::OtherSymbol),
        ('\u{2e52}', '\u{2e54}', GeneralCategory::OtherPunctuation), ('\u{2e55}', '\u{2e55}',
        GeneralCategory::OpenPunctuation), ('\u{2e56}', '\u{2e56}',
        GeneralCategory::ClosePunctuation), ('\u{2e57}', '\u{2e57}',
        GeneralCategory::OpenPunctuation), ('\u{2e58}', '\u{2e58}',
        GeneralCategory::ClosePunctuation), ('\u{2e59}', '\u{2e59}',
        GeneralCategory::OpenPunctuation), ('\u{2e5a}', '\u{2e5a}',
        GeneralCategory::ClosePunctuation), ('\u{2e5b}', '\u{2e5b}',
        GeneralCategory::OpenPunctuation), ('\u{2e5c}', '\u{2e5c}',
        GeneralCategory::ClosePunctuation), ('\u{2e5d}', '\u{2e5d}',
        GeneralCategory::DashPunctuation), ('\u{2e80}', '\u{2e99}', GeneralCategory::OtherSymbol),
        ('\u{2e9b}', '\u{2ef3}', GeneralCategory::OtherSymbol), ('\u{2f00}', '\u{2fd5}',
        GeneralCategory::OtherSymbol), ('\u{2ff0}', '\u{2fff}', GeneralCategory::OtherSymbol),
        ('\u{3000}', '\u{3000}', GeneralCategory::SpaceSeparator), ('\u{3001}', '\u{3003}',
        GeneralCategory::OtherPunctuation), ('\u{3004}', '\u{3004}', GeneralCategory::OtherSymbol),
        ('\u{3005}', '\u{3005}', GeneralCategory::ModifierLetter), ('\u{3006}', '\u{3006}',
        GeneralCategory::OtherLetter), ('\u{3007}', '\u{3007}', GeneralCategory::LetterNumber),
        ('\u{3008}', '\u{3008}', GeneralCategory::OpenPunctuation), ('\u{3009}', '\u{3009}',
        GeneralCategory::ClosePunctuation), ('\u{300a}', '\u{300a}',
        GeneralCategory::OpenPunctuation), ('\u{300b}', '\u{300b}',
        GeneralCategory::ClosePunctuation), ('\u{300c}', '\u{300c}',
        GeneralCategory::OpenPunctuation), ('\u{300d}', '\u{300d}',
        GeneralCategory::ClosePunctuation), ('\u{300e}', '\u{300e}',
        GeneralCategory::OpenPunctuation), ('\u{300f}', '\u{300f}',
        GeneralCategory::ClosePunctuation), ('\u{3010}', '\u{3010}',
        GeneralCategory::OpenPunctuation), ('\u{3011}', '\u{3011}',
        GeneralCategory::ClosePunctuation), ('\u{3012}', '\u{3013}', GeneralCategory::OtherSymbol),
        ('\u{3014}', '\u{3014}', GeneralCategory::OpenPunctuation), ('\u{3015}', '\u{3015}',
        GeneralCategory::ClosePunctuation), ('\u{3016}', '\u{3016}',
        GeneralCategory::OpenPunctuation), ('\u{3017}', '\u{3017}',
        GeneralCategory::ClosePunctuation), ('\u{3018}', '\u{3018}',
        GeneralCategory::OpenPunctuation), ('\u{3019}', '\u{3019}',
        GeneralCategory::ClosePunctuation), ('\u{301a}', '\u{301a}',
        GeneralCategory::OpenPunctuation), ('\u{301b}', '\u{301b}',
        GeneralCategory::ClosePunctuation), ('\u{301c}', '\u{301c}',
        GeneralCategory::DashPunctuation), ('\u{301d}', '\u{301d}',
        GeneralCategory::OpenPunctuation), ('\u{301e}', '\u{301f}',
        GeneralCategory::ClosePunctuation), ('\u{3020}', '\u{3020}', GeneralCategory::OtherSymbol),
        ('\u{3021}', '\u{3029}', GeneralCategory::LetterNumber), ('\u{302a}', '\u{302d}',
        GeneralCategory::NonspacingMark), ('\u{302e}', '\u{302f}', GeneralCategory::SpacingMark),
        ('\u{3030}', '\u{3030}', GeneralCategory::DashPunctuation), ('\u{3031}', '\u{3035}',
        GeneralCategory::ModifierLetter), ('\u{3036}', '\u{3037}', GeneralCategory::OtherSymbol),
        ('\u{3038}', '\u{303a}', GeneralCategory::LetterNumber), ('\u{303b}', '\u{303b}',
        GeneralCategory::ModifierLetter), ('\u{303c}', '\u{303c}', GeneralCategory::OtherLetter),
        ('\u{303d}', '\u{303d}', GeneralCategory::OtherPunctuation), ('\u{303e}', '\u{303f}',
        GeneralCategory::OtherSymbol), ('\u{3041}', '\u{3096}', GeneralCategory::OtherLetter),
        ('\u{3099}', '\u{309a}', GeneralCategory::NonspacingMark), ('\u{309b}', '\u{309c}',
        GeneralCategory::ModifierSymbol), ('\u{309d}', '\u{309e}', GeneralCategory::ModifierLetter),
        ('\u{309f}', '\u{309f}', GeneralCategory::OtherLetter), ('\u{30a0}', '\u{30a0}',
        GeneralCategory::DashPunctuation), ('\u{30a1}', '\u{30fa}', GeneralCategory::OtherLetter),
        ('\u{30fb}', '\u{30fb}', GeneralCategory::OtherPunctuation), ('\u{30fc}', '\u{30fe}',
        GeneralCategory::ModifierLetter), ('\u{30ff}', '\u{30ff}', GeneralCategory::OtherLetter),
        ('\u{3105}', '\u{312f}', GeneralCategory::OtherLetter), ('\u{3131}', '\u{318e}',
        GeneralCategory::OtherLetter), ('\u{3190}', '\u{3191}', GeneralCategory::OtherSymbol),
        ('\u{3192}', '\u{3195}', GeneralCategory::OtherNumber), ('\u{3196}', '\u{319f}',
        GeneralCategory::OtherSymbol), ('\u{31a0}', '\u{31bf}', GeneralCategory::OtherLetter),
        ('\u{31c0}', '\u{31e3}', GeneralCategory::OtherSymbol), ('\u{31ef}', '\u{31ef}',
        GeneralCategory::OtherSymbol), ('\u{31f0}', '\u{31ff}', GeneralCategory::OtherLetter),
        ('\u{3200}', '\u{321e}', GeneralCategory::OtherSymbol), ('\u{3220}', '\u{3229}',
        GeneralCategory::OtherNumber), ('\u{322a}', '\u{3247}', GeneralCategory::OtherSymbol),
        ('\u{3248}', '\u{324f}', GeneralCategory::OtherNumber), ('\u{3250}', '\u{3250}',
        GeneralCategory::OtherSymbol), ('\u{3251}', '\u{325f}', GeneralCategory::OtherNumber),
        ('\u{3260}', '\u{327f}', GeneralCategory::OtherSymbol), ('\u{3280}', '\u{3289}',
        GeneralCategory::OtherNumber), ('\u{328a}', '\u{32b0}', GeneralCategory::OtherSymbol),
        ('\u{32b1}', '\u{32bf}', GeneralCategory::OtherNumber), ('\u{32c0}', '\u{33ff}',
        GeneralCategory::OtherSymbol), ('\u{3400}', '\u{4dbf}', GeneralCategory::OtherLetter),
        ('\u{4dc0}', '\u{4dff}', GeneralCategory::OtherSymbol), ('\u{4e00}', '\u{a014}',
        GeneralCategory::OtherLetter), ('\u{a015}', '\u{a015}', GeneralCategory::ModifierLetter),
        ('\u{a016}', '\u{a48c}', GeneralCategory::OtherLetter), ('\u{a490}', '\u{a4c6}',
        GeneralCategory::OtherSymbol), ('\u{a4d0}', '\u{a4f7}', GeneralCategory::OtherLetter),
        ('\u{a4f8}', '\u{a4fd}', GeneralCategory::ModifierLetter), ('\u{a4fe}', '\u{a4ff}',
        GeneralCategory::OtherPunctuation), ('\u{a500}', '\u{a60b}', GeneralCategory::OtherLetter),
        ('\u{a60c}', '\u{a60c}', GeneralCategory::ModifierLetter), ('\u{a60d}', '\u{a60f}',
        GeneralCategory::OtherPunctuation), ('\u{a610}', '\u{a61f}', GeneralCategory::OtherLetter),
        ('\u{a620}', '\u{a629}', GeneralCategory::DecimalNumber), ('\u{a62a}', '\u{a62b}',
        GeneralCategory::OtherLetter), ('\u{a640}', '\u{a640}', GeneralCategory::UppercaseLetter),
        ('\u{a641}', '\u{a641}', GeneralCategory::LowercaseLetter), ('\u{a642}', '\u{a642}',
        GeneralCategory::UppercaseLetter), ('\u{a643}', '\u{a643}',
        GeneralCategory::LowercaseLetter), ('\u{a644}', '\u{a644}',
        GeneralCategory::UppercaseLetter), ('\u{a645}', '\u{a645}',
        GeneralCategory::LowercaseLetter), ('\u{a646}', '\u{a646}',
        GeneralCategory::UppercaseLetter), ('\u{a647}', '\u{a647}',
        GeneralCategory::LowercaseLetter), ('\u{a648}', '\u{a648}',
        GeneralCategory::UppercaseLetter), ('\u{a649}', '\u{a649}',
        GeneralCategory::LowercaseLetter), ('\u{a64a}', '\u{a64a}',
        GeneralCategory::UppercaseLetter), ('\u{a64b}', '\u{a64b}',
        GeneralCategory::LowercaseLetter), ('\u{a64c}', '\u{a64c}',
        GeneralCategory::UppercaseLetter), ('\u{a64d}', '\u{a64d}',
        GeneralCategory::LowercaseLetter), ('\u{a64e}', '\u{a64e}',
        GeneralCategory::UppercaseLetter), ('\u{a64f}', '\u{a64f}',
        GeneralCategory::LowercaseLetter), ('\u{a650}', '\u{a650}',
        GeneralCategory::UppercaseLetter), ('\u{a651}', '\u{a651}',
        GeneralCategory::LowercaseLetter), ('\u{a652}', '\u{a652}',
        GeneralCategory::UppercaseLetter), ('\u{a653}', '\u{a653}',
        GeneralCategory::LowercaseLetter), ('\u{a654}', '\u{a654}',
        GeneralCategory::UppercaseLetter), ('\u{a655}', '\u{a655}',
        GeneralCategory::LowercaseLetter), ('\u{a656}', '\u{a656}',
        GeneralCategory::UppercaseLetter), ('\u{a657}', '\u{a657}',
        GeneralCategory::LowercaseLetter), ('\u{a658}', '\u{a658}',
        GeneralCategory::UppercaseLetter), ('\u{a659}', '\u{a659}',
        GeneralCategory::LowercaseLetter), ('\u{a65a}', '\u{a65a}',
        GeneralCategory::UppercaseLetter), ('\u{a65b}', '\u{a65b}',
        GeneralCategory::LowercaseLetter), ('\u{a65c}', '\u{a65c}',
        GeneralCategory::UppercaseLetter), ('\u{a65d}', '\u{a65d}',
        GeneralCategory::LowercaseLetter), ('\u{a65e}', '\u{a65e}',
        GeneralCategory::UppercaseLetter), ('\u{a65f}', '\u{a65f}',
        GeneralCategory::LowercaseLetter), ('\u{a660}', '\u{a660}',
        GeneralCategory::UppercaseLetter), ('\u{a661}', '\u{a661}',
        GeneralCategory::LowercaseLetter), ('\u{a662}', '\u{a662}',
        GeneralCategory::UppercaseLetter), ('\u{a663}', '\u{a663}',
        GeneralCategory::LowercaseLetter), ('\u{a664}', '\u{a664}',
        GeneralCategory::UppercaseLetter), ('\u{a665}', '\u{a665}',
        GeneralCategory::LowercaseLetter), ('\u{a666}', '\u{a666}',
        GeneralCategory::UppercaseLetter), ('\u{a667}', '\u{a667}',
        GeneralCategory::LowercaseLetter), ('\u{a668}', '\u{a668}',
        GeneralCategory::UppercaseLetter), ('\u{a669}', '\u{a669}',
        GeneralCategory::LowercaseLetter), ('\u{a66a}', '\u{a66a}',
        GeneralCategory::UppercaseLetter), ('\u{a66b}', '\u{a66b}',
        GeneralCategory::LowercaseLetter), ('\u{a66c}', '\u{a66c}',
        GeneralCategory::UppercaseLetter), ('\u{a66d}', '\u{a66d}',
        GeneralCategory::LowercaseLetter), ('\u{a66e}', '\u{a66e}', GeneralCategory::OtherLetter),
        ('\u{a66f}', '\u{a66f}', GeneralCategory::NonspacingMark), ('\u{a670}', '\u{a672}',
        GeneralCategory::EnclosingMark), ('\u{a673}', '\u{a673}',
        GeneralCategory::OtherPunctuation), ('\u{a674}', '\u{a67d}',
        GeneralCategory::NonspacingMark), ('\u{a67e}', '\u{a67e}',
        GeneralCategory::OtherPunctuation), ('\u{a67f}', '\u{a67f}',
        GeneralCategory::ModifierLetter), ('\u{a680}', '\u{a680}',
        GeneralCategory::UppercaseLetter), ('\u{a681}', '\u{a681}',
        GeneralCategory::LowercaseLetter), ('\u{a682}', '\u{a682}',
        GeneralCategory::UppercaseLetter), ('\u{a683}', '\u{a683}',
        GeneralCategory::LowercaseLetter), ('\u{a684}', '\u{a684}',
        GeneralCategory::UppercaseLetter), ('\u{a685}', '\u{a685}',
        GeneralCategory::LowercaseLetter), ('\u{a686}', '\u{a686}',
        GeneralCategory::UppercaseLetter), ('\u{a687}', '\u{a687}',
        GeneralCategory::LowercaseLetter), ('\u{a688}', '\u{a688}',
        GeneralCategory::UppercaseLetter), ('\u{a689}', '\u{a689}',
        GeneralCategory::LowercaseLetter), ('\u{a68a}', '\u{a68a}',
        GeneralCategory::UppercaseLetter), ('\u{a68b}', '\u{a68b}',
        GeneralCategory::LowercaseLetter), ('\u{a68c}', '\u{a68c}',
        GeneralCategory::UppercaseLetter), ('\u{a68d}', '\u{a68d}',
        GeneralCategory::LowercaseLetter), ('\u{a68e}', '\u{a68e}',
        GeneralCategory::UppercaseLetter), ('\u{a68f}', '\u{a68f}',
        GeneralCategory::LowercaseLetter), ('\u{a690}', '\u{a690}',
        GeneralCategory::UppercaseLetter), ('\u{a691}', '\u{a691}',
        GeneralCategory::LowercaseLetter), ('\u{a692}', '\u{a692}',
        GeneralCategory::UppercaseLetter), ('\u{a693}', '\u{a693}',
        GeneralCategory::LowercaseLetter), ('\u{a694}', '\u{a694}',
        GeneralCategory::UppercaseLetter), ('\u{a695}', '\u{a695}',
        GeneralCategory::LowercaseLetter), ('\u{a696}', '\u{a696}',
        GeneralCategory::UppercaseLetter), ('\u{a697}', '\u{a697}',
        GeneralCategory::LowercaseLetter), ('\u{a698}', '\u{a698}',
        GeneralCategory::UppercaseLetter), ('\u{a699}', '\u{a699}',
        GeneralCategory::LowercaseLetter), ('\u{a69a}', '\u{a69a}',
        GeneralCategory::UppercaseLetter), ('\u{a69b}', '\u{a69b}',
        GeneralCategory::LowercaseLetter), ('\u{a69c}', '\u{a69d}',
        GeneralCategory::ModifierLetter), ('\u{a69e}', '\u{a69f}', GeneralCategory::NonspacingMark),
        ('\u{a6a0}', '\u{a6e5}', GeneralCategory::OtherLetter), ('\u{a6e6}', '\u{a6ef}',
        GeneralCategory::LetterNumber), ('\u{a6f0}', '\u{a6f1}', GeneralCategory::NonspacingMark),
        ('\u{a6f2}', '\u{a6f7}', GeneralCategory::OtherPunctuation), ('\u{a700}', '\u{a716}',
        GeneralCategory::ModifierSymbol), ('\u{a717}', '\u{a71f}', GeneralCategory::ModifierLetter),
        ('\u{a720}', '\u{a721}', GeneralCategory::ModifierSymbol), ('\u{a722}', '\u{a722}',
        GeneralCategory::UppercaseLetter), ('\u{a723}', '\u{a723}',
        GeneralCategory::LowercaseLetter), ('\u{a724}', '\u{a724}',
        GeneralCategory::UppercaseLetter), ('\u{a725}', '\u{a725}',
        GeneralCategory::LowercaseLetter), ('\u{a726}', '\u{a726}',
        GeneralCategory::UppercaseLetter), ('\u{a727}', '\u{a727}',
        GeneralCategory::LowercaseLetter), ('\u{a728}', '\u{a728}',
        GeneralCategory::UppercaseLetter), ('\u{a729}', '\u{a729}',
        GeneralCategory::LowercaseLetter), ('\u{a72a}', '\u{a72a}',
        GeneralCategory::UppercaseLetter), ('\u{a72b}', '\u{a72b}',
        GeneralCategory::LowercaseLetter), ('\u{a72c}', '\u{a72c}',
        GeneralCategory::UppercaseLetter), ('\u{a72d}', '\u{a72d}',
        GeneralCategory::LowercaseLetter), ('\u{a72e}', '\u{a72e}',
        GeneralCategory::UppercaseLetter), ('\u{a72f}', '\u{a731}',
        GeneralCategory::LowercaseLetter), ('\u{a732}', '\u{a732}',
        GeneralCategory::UppercaseLetter), ('\u{a733}', '\u{a733}',
        GeneralCategory::LowercaseLetter), ('\u{a734}', '\u{a734}',
        GeneralCategory::UppercaseLetter), ('\u{a735}', '\u{a735}',
        GeneralCategory::LowercaseLetter), ('\u{a736}', '\u{a736}',
        GeneralCategory::UppercaseLetter), ('\u{a737}', '\u{a737}',
        GeneralCategory::LowercaseLetter), ('\u{a738}', '\u{a738}',
        GeneralCategory::UppercaseLetter), ('\u{a739}', '\u{a739}',
        GeneralCategory::LowercaseLetter), ('\u{a73a}', '\u{a73a}',
        GeneralCategory::UppercaseLetter), ('\u{a73b}', '\u{a73b}',
        GeneralCategory::LowercaseLetter), ('\u{a73c}', '\u{a73c}',
        GeneralCategory::UppercaseLetter), ('\u{a73d}', '\u{a73d}',
        GeneralCategory::LowercaseLetter), ('\u{a73e}', '\u{a73e}',
        GeneralCategory::UppercaseLetter), ('\u{a73f}', '\u{a73f}',
        GeneralCategory::LowercaseLetter), ('\u{a740}', '\u{a740}',
        GeneralCategory::UppercaseLetter), ('\u{a741}', '\u{a741}',
        GeneralCategory::LowercaseLetter), ('\u{a742}', '\u{a742}',
        GeneralCategory::UppercaseLetter), ('\u{a743}', '\u{a743}',
        GeneralCategory::LowercaseLetter), ('\u{a744}', '\u{a744}',
        GeneralCategory::UppercaseLetter), ('\u{a745}', '\u{a745}',
        GeneralCategory::LowercaseLetter), ('\u{a746}', '\u{a746}',
        GeneralCategory::UppercaseLetter), ('\u{a747}', '\u{a747}',
        GeneralCategory::LowercaseLetter), ('\u{a748}', '\u{a748}',
        GeneralCategory::UppercaseLetter), ('\u{a749}', '\u{a749}',
        GeneralCategory::LowercaseLetter), ('\u{a74a}', '\u{a74a}',
        GeneralCategory::UppercaseLetter), ('\u{a74b}', '\u{a74b}',
        GeneralCategory::LowercaseLetter), ('\u{a74c}', '\u{a74c}',
        GeneralCategory::UppercaseLetter), ('\u{a74d}', '\u{a74d}',
        GeneralCategory::LowercaseLetter), ('\u{a74e}', '\u{a74e}',
        GeneralCategory::UppercaseLetter), ('\u{a74f}', '\u{a74f}',
        GeneralCategory::LowercaseLetter), ('\u{a750}', '\u{a750}',
        GeneralCategory::UppercaseLetter), ('\u{a751}', '\u{a751}',
        GeneralCategory::LowercaseLetter), ('\u{a752}', '\u{a752}',
        GeneralCategory::UppercaseLetter), ('\u{a753}', '\u{a753}',
        GeneralCategory::LowercaseLetter), ('\u{a754}', '\u{a754}',
        GeneralCategory::UppercaseLetter), ('\u{a755}', '\u{a755}',
        GeneralCategory::LowercaseLetter), ('\u{a756}', '\u{a756}',
        GeneralCategory::UppercaseLetter), ('\u{a757}', '\u{a757}',
        GeneralCategory::LowercaseLetter), ('\u{a758}', '\u{a758}',
        GeneralCategory::UppercaseLetter), ('\u{a759}', '\u{a759}',
        GeneralCategory::LowercaseLetter), ('\u{a75a}', '\u{a75a}',
        GeneralCategory::UppercaseLetter), ('\u{a75b}', '\u{a75b}',
        GeneralCategory::LowercaseLetter), ('\u{a75c}', '\u{a75c}',
        GeneralCategory::UppercaseLetter), ('\u{a75d}', '\u{a75d}',
        GeneralCategory::LowercaseLetter), ('\u{a75e}', '\u{a75e}',
        GeneralCategory::UppercaseLetter), ('\u{a75f}', '\u{a75f}',
        GeneralCategory::LowercaseLetter), ('\u{a760}', '\u{a760}',
        GeneralCategory::UppercaseLetter), ('\u{a761}', '\u{a761}',
        GeneralCategory::LowercaseLetter), ('\u{a762}', '\u{a762}',
        GeneralCategory::UppercaseLetter), ('\u{a763}', '\u{a763}',
        GeneralCategory::LowercaseLetter), ('\u{a764}', '\u{a764}',
        GeneralCategory::UppercaseLetter), ('\u{a765}', '\u{a765}',
        GeneralCategory::LowercaseLetter), ('\u{a766}', '\u{a766}',
        GeneralCategory::UppercaseLetter), ('\u{a767}', '\u{a767}',
        GeneralCategory::LowercaseLetter), ('\u{a768}', '\u{a768}',
        GeneralCategory::UppercaseLetter), ('\u{a769}', '\u{a769}',
        GeneralCategory::LowercaseLetter), ('\u{a76a}', '\u{a76a}',
        GeneralCategory::UppercaseLetter), ('\u{a76b}', '\u{a76b}',
        GeneralCategory::LowercaseLetter), ('\u{a76c}', '\u{a76c}',
        GeneralCategory::UppercaseLetter), ('\u{a76d}', '\u{a76d}',
        GeneralCategory::LowercaseLetter), ('\u{a76e}', '\u{a76e}',
        GeneralCategory::UppercaseLetter), ('\u{a76f}', '\u{a76f}',
        GeneralCategory::LowercaseLetter), ('\u{a770}', '\u{a770}',
        GeneralCategory::ModifierLetter), ('\u{a771}', '\u{a778}',
        GeneralCategory::LowercaseLetter), ('\u{a779}', '\u{a779}',
        GeneralCategory::UppercaseLetter), ('\u{a77a}', '\u{a77a}',
        GeneralCategory::LowercaseLetter), ('\u{a77b}', '\u{a77b}',
        GeneralCategory::UppercaseLetter), ('\u{a77c}', '\u{a77c}',
        GeneralCategory::LowercaseLetter), ('\u{a77d}', '\u{a77e}',
        GeneralCategory::UppercaseLetter), ('\u{a77f}', '\u{a77f}',
        GeneralCategory::LowercaseLetter), ('\u{a780}', '\u{a780}',
        GeneralCategory::UppercaseLetter), ('\u{a781}', '\u{a781}',
        GeneralCategory::LowercaseLetter), ('\u{a782}', '\u{a782}',
        GeneralCategory::UppercaseLetter), ('\u{a783}', '\u{a783}',
        GeneralCategory::LowercaseLetter), ('\u{a784}', '\u{a784}',
        GeneralCategory::UppercaseLetter), ('\u{a785}', '\u{a785}',
        GeneralCategory::LowercaseLetter), ('\u{a786}', '\u{a786}',
        GeneralCategory::UppercaseLetter), ('\u{a787}', '\u{a787}',
        GeneralCategory::LowercaseLetter), ('\u{a788}', '\u{a788}',
        GeneralCategory::ModifierLetter), ('\u{a789}', '\u{a78a}', GeneralCategory::ModifierSymbol),
        ('\u{a78b}', '\u{a78b}', GeneralCategory::UppercaseLetter), ('\u{a78c}', '\u{a78c}',
        GeneralCategory::LowercaseLetter), ('\u{a78d}', '\u{a78d}',
        GeneralCategory::UppercaseLetter), ('\u{a78e}', '\u{a78e}',
        GeneralCategory::LowercaseLetter), ('\u{a78f}', '\u{a78f}', GeneralCategory::OtherLetter),
        ('\u{a790}', '\u{a790}', GeneralCategory::UppercaseLetter), ('\u{a791}', '\u{a791}',
        GeneralCategory::LowercaseLetter), ('\u{a792}', '\u{a792}',
        GeneralCategory::UppercaseLetter), ('\u{a793}', '\u{a795}',
        GeneralCategory::LowercaseLetter), ('\u{a796}', '\u{a796}',
        GeneralCategory::UppercaseLetter), ('\u{a797}', '\u{a797}',
        GeneralCategory::LowercaseLetter), ('\u{a798}', '\u{a798}',
        GeneralCategory::UppercaseLetter), ('\u{a799}', '\u{a799}',
        GeneralCategory::LowercaseLetter), ('\u{a79a}', '\u{a79a}',
        GeneralCategory::UppercaseLetter), ('\u{a79b}', '\u{a79b}',
        GeneralCategory::LowercaseLetter), ('\u{a79c}', '\u{a79c}',
        GeneralCategory::UppercaseLetter), ('\u{a79d}', '\u{a79d}',
        GeneralCategory::LowercaseLetter), ('\u{a79e}', '\u{a79e}',
        GeneralCategory::UppercaseLetter), ('\u{a79f}', '\u{a79f}',
        GeneralCategory::LowercaseLetter), ('\u{a7a0}', '\u{a7a0}',
        GeneralCategory::UppercaseLetter), ('\u{a7a1}', '\u{a7a1}',
        GeneralCategory::LowercaseLetter), ('\u{a7a2}', '\u{a7a2}',
        GeneralCategory::UppercaseLetter), ('\u{a7a3}', '\u{a7a3}',
        GeneralCategory::LowercaseLetter), ('\u{a7a4}', '\u{a7a4}',
        GeneralCategory::UppercaseLetter), ('\u{a7a5}', '\u{a7a5}',
        GeneralCategory::LowercaseLetter), ('\u{a7a6}', '\u{a7a6}',
        GeneralCategory::UppercaseLetter), ('\u{a7a7}', '\u{a7a7}',
        GeneralCategory::LowercaseLetter), ('\u{a7a8}', '\u{a7a8}',
        GeneralCategory::UppercaseLetter), ('\u{a7a9}', '\u{a7a9}',
        GeneralCategory::LowercaseLetter), ('\u{a7aa}', '\u{a7ae}',
        GeneralCategory::UppercaseLetter), ('\u{a7af}', '\u{a7af}',
        GeneralCategory::LowercaseLetter), ('\u{a7b0}', '\u{a7b4}',
        GeneralCategory::UppercaseLetter), ('\u{a7b5}', '\u{a7b5}',
        GeneralCategory::LowercaseLetter), ('\u{a7b6}', '\u{a7b6}',
        GeneralCategory::UppercaseLetter), ('\u{a7b7}', '\u{a7b7}',
        GeneralCategory::LowercaseLetter), ('\u{a7b8}', '\u{a7b8}',
        GeneralCategory::UppercaseLetter), ('\u{a7b9}', '\u{a7b9}',
        GeneralCategory::LowercaseLetter), ('\u{a7ba}', '\u{a7ba}',
        GeneralCategory::UppercaseLetter), ('\u{a7bb}', '\u{a7bb}',
        GeneralCategory::LowercaseLetter), ('\u{a7bc}', '\u{a7bc}',
        GeneralCategory::UppercaseLetter), ('\u{a7bd}', '\u{a7bd}',
        GeneralCategory::LowercaseLetter), ('\u{a7be}', '\u{a7be}',
        GeneralCategory::UppercaseLetter), ('\u{a7bf}', '\u{a7bf}',
        GeneralCategory::LowercaseLetter), ('\u{a7c0}', '\u{a7c0}',
        GeneralCategory::UppercaseLetter), ('\u{a7c1}', '\u{a7c1}',
        GeneralCategory::LowercaseLetter), ('\u{a7c2}', '\u{a7c2}',
        GeneralCategory::UppercaseLetter), ('\u{a7c3}', '\u{a7c3}',
        GeneralCategory::LowercaseLetter), ('\u{a7c4}', '\u{a7c7}',
        GeneralCategory::UppercaseLetter), ('\u{a7c8}', '\u{a7c8}',
        GeneralCategory::LowercaseLetter), ('\u{a7c9}', '\u{a7c9}',
        GeneralCategory::UppercaseLetter), ('\u{a7ca}', '\u{a7ca}',
        GeneralCategory::LowercaseLetter), ('\u{a7d0}', '\u{a7d0}',
        GeneralCategory::UppercaseLetter), ('\u{a7d1}', '\u{a7d1}',
        GeneralCategory::LowercaseLetter), ('\u{a7d3}', '\u{a7d3}',
        GeneralCategory::LowercaseLetter), ('\u{a7d5}', '\u{a7d5}',
        GeneralCategory::LowercaseLetter), ('\u{a7d6}', '\u{a7d6}',
        GeneralCategory::UppercaseLetter), ('\u{a7d7}', '\u{a7d7}',
        GeneralCategory::LowercaseLetter), ('\u{a7d8}', '\u{a7d8}',
        GeneralCategory::UppercaseLetter), ('\u{a7d9}', '\u{a7d9}',
        GeneralCategory::LowercaseLetter), ('\u{a7f2}', '\u{a7f4}',
        GeneralCategory::ModifierLetter), ('\u{a7f5}', '\u{a7f5}',
        GeneralCategory::UppercaseLetter), ('\u{a7f6}', '\u{a7f6}',
        GeneralCategory::LowercaseLetter), ('\u{a7f7}', '\u{a7f7}', GeneralCategory::OtherLetter),
        ('\u{a7f8}', '\u{a7f9}', GeneralCategory::ModifierLetter), ('\u{a7fa}', '\u{a7fa}',
        GeneralCategory::LowercaseLetter), ('\u{a7fb}', '\u{a801}', GeneralCategory::OtherLetter),
        ('\u{a802}', '\u{a802}', GeneralCategory::NonspacingMark), ('\u{a803}', '\u{a805}',
        GeneralCategory::OtherLetter), ('\u{a806}', '\u{a806}', GeneralCategory::NonspacingMark),
        ('\u{a807}', '\u{a80a}', GeneralCategory::OtherLetter), ('\u{a80b}', '\u{a80b}',
        GeneralCategory::NonspacingMark), ('\u{a80c}', '\u{a822}', GeneralCategory::OtherLetter),
        ('\u{a823}', '\u{a824}', GeneralCategory::SpacingMark), ('\u{a825}', '\u{a826}',
        GeneralCategory::NonspacingMark), ('\u{a827}', '\u{a827}', GeneralCategory::SpacingMark),
        ('\u{a828}', '\u{a82b}', GeneralCategory::OtherSymbol), ('\u{a82c}', '\u{a82c}',
        GeneralCategory::NonspacingMark), ('\u{a830}', '\u{a835}', GeneralCategory::OtherNumber),
        ('\u{a836}', '\u{a837}', GeneralCategory::OtherSymbol), ('\u{a838}', '\u{a838}',
        GeneralCategory::CurrencySymbol), ('\u{a839}', '\u{a839}', GeneralCategory::OtherSymbol),
        ('\u{a840}', '\u{a873}', GeneralCategory::OtherLetter), ('\u{a874}', '\u{a877}',
        GeneralCategory::OtherPunctuation), ('\u{a880}', '\u{a881}', GeneralCategory::SpacingMark),
        ('\u{a882}', '\u{a8b3}', GeneralCategory::OtherLetter), ('\u{a8b4}', '\u{a8c3}',
        GeneralCategory::SpacingMark), ('\u{a8c4}', '\u{a8c5}', GeneralCategory::NonspacingMark),
        ('\u{a8ce}', '\u{a8cf}', GeneralCategory::OtherPunctuation), ('\u{a8d0}', '\u{a8d9}',
        GeneralCategory::DecimalNumber), ('\u{a8e0}', '\u{a8f1}', GeneralCategory::NonspacingMark),
        ('\u{a8f2}', '\u{a8f7}', GeneralCategory::OtherLetter), ('\u{a8f8}', '\u{a8fa}',
        GeneralCategory::OtherPunctuation), ('\u{a8fb}', '\u{a8fb}', GeneralCategory::OtherLetter),
        ('\u{a8fc}', '\u{a8fc}', GeneralCategory::OtherPunctuation), ('\u{a8fd}', '\u{a8fe}',
        GeneralCategory::OtherLetter), ('\u{a8ff}', '\u{a8ff}', GeneralCategory::NonspacingMark),
        ('\u{a900}', '\u{a909}', GeneralCategory::DecimalNumber), ('\u{a90a}', '\u{a925}',
        GeneralCategory::OtherLetter), ('\u{a926}', '\u{a92d}', GeneralCategory::NonspacingMark),
        ('\u{a92e}', '\u{a92f}', GeneralCategory::OtherPunctuation), ('\u{a930}', '\u{a946}',
        GeneralCategory::OtherLetter), ('\u{a947}', '\u{a951}', GeneralCategory::NonspacingMark),
        ('\u{a952}', '\u{a953}', GeneralCategory::SpacingMark), ('\u{a95f}', '\u{a95f}',
        GeneralCategory::OtherPunctuation), ('\u{a960}', '\u{a97c}', GeneralCategory::OtherLetter),
        ('\u{a980}', '\u{a982}', GeneralCategory::NonspacingMark), ('\u{a983}', '\u{a983}',
        GeneralCategory::SpacingMark), ('\u{a984}', '\u{a9b2}', GeneralCategory::OtherLetter),
        ('\u{a9b3}', '\u{a9b3}', GeneralCategory::NonspacingMark), ('\u{a9b4}', '\u{a9b5}',
        GeneralCategory::SpacingMark), ('\u{a9b6}', '\u{a9b9}', GeneralCategory::NonspacingMark),
        ('\u{a9ba}', '\u{a9bb}', GeneralCategory::SpacingMark), ('\u{a9bc}', '\u{a9bd}',
        GeneralCategory::NonspacingMark), ('\u{a9be}', '\u{a9c0}', GeneralCategory::SpacingMark),
        ('\u{a9c1}', '\u{a9cd}', GeneralCategory::OtherPunctuation), ('\u{a9cf}', '\u{a9cf}',
        GeneralCategory::ModifierLetter), ('\u{a9d0}', '\u{a9d9}', GeneralCategory::DecimalNumber),
        ('\u{a9de}', '\u{a9df}', GeneralCategory::OtherPunctuation), ('\u{a9e0}', '\u{a9e4}',
        GeneralCategory::OtherLetter), ('\u{a9e5}', '\u{a9e5}', GeneralCategory::NonspacingMark),
        ('\u{a9e6}', '\u{a9e6}', GeneralCategory::ModifierLetter), ('\u{a9e7}', '\u{a9ef}',
        GeneralCategory::OtherLetter), ('\u{a9f0}', '\u{a9f9}', GeneralCategory::DecimalNumber),
        ('\u{a9fa}', '\u{a9fe}', GeneralCategory::OtherLetter), ('\u{aa00}', '\u{aa28}',
        GeneralCategory::OtherLetter), ('\u{aa29}', '\u{aa2e}', GeneralCategory::NonspacingMark),
        ('\u{aa2f}', '\u{aa30}', GeneralCategory::SpacingMark), ('\u{aa31}', '\u{aa32}',
        GeneralCategory::NonspacingMark), ('\u{aa33}', '\u{aa34}', GeneralCategory::SpacingMark),
        ('\u{aa35}', '\u{aa36}', GeneralCategory::NonspacingMark), ('\u{aa40}', '\u{aa42}',
        GeneralCategory::OtherLetter), ('\u{aa43}', '\u{aa43}', GeneralCategory::NonspacingMark),
        ('\u{aa44}', '\u{aa4b}', GeneralCategory::OtherLetter), ('\u{aa4c}', '\u{aa4c}',
        GeneralCategory::NonspacingMark), ('\u{aa4d}', '\u{aa4d}', GeneralCategory::SpacingMark),
        ('\u{aa50}', '\u{aa59}', GeneralCategory::DecimalNumber), ('\u{aa5c}', '\u{aa5f}',
        GeneralCategory::OtherPunctuation), ('\u{aa60}', '\u{aa6f}', GeneralCategory::OtherLetter),
        ('\u{aa70}', '\u{aa70}', GeneralCategory::ModifierLetter), ('\u{aa71}', '\u{aa76}',
        GeneralCategory::OtherLetter), ('\u{aa77}', '\u{aa79}', GeneralCategory::OtherSymbol),
        ('\u{aa7a}', '\u{aa7a}', GeneralCategory::OtherLetter), ('\u{aa7b}', '\u{aa7b}',
        GeneralCategory::SpacingMark), ('\u{aa7c}', '\u{aa7c}', GeneralCategory::NonspacingMark),
        ('\u{aa7d}', '\u{aa7d}', GeneralCategory::SpacingMark), ('\u{aa7e}', '\u{aaaf}',
        GeneralCategory::OtherLetter), ('\u{aab0}', '\u{aab0}', GeneralCategory::NonspacingMark),
        ('\u{aab1}', '\u{aab1}', GeneralCategory::OtherLetter), ('\u{aab2}', '\u{aab4}',
        GeneralCategory::NonspacingMark), ('\u{aab5}', '\u{aab6}', GeneralCategory::OtherLetter),
        ('\u{aab7}', '\u{aab8}', GeneralCategory::NonspacingMark), ('\u{aab9}', '\u{aabd}',
        GeneralCategory::OtherLetter), ('\u{aabe}', '\u{aabf}', GeneralCategory::NonspacingMark),
        ('\u{aac0}', '\u{aac0}', GeneralCategory::OtherLetter), ('\u{aac1}', '\u{aac1}',
        GeneralCategory::NonspacingMark), ('\u{aac2}', '\u{aac2}', GeneralCategory::OtherLetter),
        ('\u{aadb}', '\u{aadc}', GeneralCategory::OtherLetter), ('\u{aadd}', '\u{aadd}',
        GeneralCategory::ModifierLetter), ('\u{aade}', '\u{aadf}',
        GeneralCategory::OtherPunctuation), ('\u{aae0}', '\u{aaea}', GeneralCategory::OtherLetter),
        ('\u{aaeb}', '\u{aaeb}', GeneralCategory::SpacingMark), ('\u{aaec}', '\u{aaed}',
        GeneralCategory::NonspacingMark), ('\u{aaee}', '\u{aaef}', GeneralCategory::SpacingMark),
        ('\u{aaf0}', '\u{aaf1}', GeneralCategory::OtherPunctuation), ('\u{aaf2}', '\u{aaf2}',
        GeneralCategory::OtherLetter), ('\u{aaf3}', '\u{aaf4}', GeneralCategory::ModifierLetter),
        ('\u{aaf5}', '\u{aaf5}', GeneralCategory::SpacingMark), ('\u{aaf6}', '\u{aaf6}',
        GeneralCategory::NonspacingMark), ('\u{ab01}', '\u{ab06}', GeneralCategory::OtherLetter),
        ('\u{ab09}', '\u{ab0e}', GeneralCategory::OtherLetter), ('\u{ab11}', '\u{ab16}',
        GeneralCategory::OtherLetter), ('\u{ab20}', '\u{ab26}', GeneralCategory::OtherLetter),
        ('\u{ab28}', '\u{ab2e}', GeneralCategory::OtherLetter), ('\u{ab30}', '\u{ab5a}',
        GeneralCategory::LowercaseLetter), ('\u{ab5b}', '\u{ab5b}',
        GeneralCategory::ModifierSymbol), ('\u{ab5c}', '\u{ab5f}', GeneralCategory::ModifierLetter),
        ('\u{ab60}', '\u{ab68}', GeneralCategory::LowercaseLetter), ('\u{ab69}', '\u{ab69}',
        GeneralCategory::ModifierLetter), ('\u{ab6a}', '\u{ab6b}', GeneralCategory::ModifierSymbol),
        ('\u{ab70}', '\u{abbf}', GeneralCategory::LowercaseLetter), ('\u{abc0}', '\u{abe2}',
        GeneralCategory::OtherLetter), ('\u{abe3}', '\u{abe4}', GeneralCategory::SpacingMark),
        ('\u{abe5}', '\u{abe5}', GeneralCategory::NonspacingMark), ('\u{abe6}', '\u{abe7}',
        GeneralCategory::SpacingMark), ('\u{abe8}', '\u{abe8}', GeneralCategory::NonspacingMark),
        ('\u{abe9}', '\u{abea}', GeneralCategory::SpacingMark), ('\u{abeb}', '\u{abeb}',
        GeneralCategory::OtherPunctuation), ('\u{abec}', '\u{abec}', GeneralCategory::SpacingMark),
        ('\u{abed}', '\u{abed}', GeneralCategory::NonspacingMark), ('\u{abf0}', '\u{abf9}',
        GeneralCategory::DecimalNumber), ('\u{ac00}', '\u{d7a3}', GeneralCategory::OtherLetter),
        ('\u{d7b0}', '\u{d7c6}', GeneralCategory::OtherLetter), ('\u{d7cb}', '\u{d7fb}',
        GeneralCategory::OtherLetter), ('\u{e000}', '\u{f8ff}', GeneralCategory::PrivateUse),
        ('\u{f900}', '\u{fa6d}', GeneralCategory::OtherLetter), ('\u{fa70}', '\u{fad9}',
        GeneralCategory::OtherLetter), ('\u{fb00}', '\u{fb06}', GeneralCategory::LowercaseLetter),
        ('\u{fb13}', '\u{fb17}', GeneralCategory::LowercaseLetter), ('\u{fb1d}', '\u{fb1d}',
        GeneralCategory::OtherLetter), ('\u{fb1e}', '\u{fb1e}', GeneralCategory::NonspacingMark),
        ('\u{fb1f}', '\u{fb28}', GeneralCategory::OtherLetter), ('\u{fb29}', '\u{fb29}',
        GeneralCategory::MathSymbol), ('\u{fb2a}', '\u{fb36}', GeneralCategory::OtherLetter),
        ('\u{fb38}', '\u{fb3c}', GeneralCategory::OtherLetter), ('\u{fb3e}', '\u{fb3e}',
        GeneralCategory::OtherLetter), ('\u{fb40}', '\u{fb41}', GeneralCategory::OtherLetter),
        ('\u{fb43}', '\u{fb44}', GeneralCategory::OtherLetter), ('\u{fb46}', '\u{fbb1}',
        GeneralCategory::OtherLetter), ('\u{fbb2}', '\u{fbc2}', GeneralCategory::ModifierSymbol),
        ('\u{fbd3}', '\u{fd3d}', GeneralCategory::OtherLetter), ('\u{fd3e}', '\u{fd3e}',
        GeneralCategory::ClosePunctuation), ('\u{fd3f}', '\u{fd3f}',
        GeneralCategory::OpenPunctuation), ('\u{fd40}', '\u{fd4f}', GeneralCategory::OtherSymbol),
        ('\u{fd50}', '\u{fd8f}', GeneralCategory::OtherLetter), ('\u{fd92}', '\u{fdc7}',
        GeneralCategory::OtherLetter), ('\u{fdcf}', '\u{fdcf}', GeneralCategory::OtherSymbol),
        ('\u{fdf0}', '\u{fdfb}', GeneralCategory::OtherLetter), ('\u{fdfc}', '\u{fdfc}',
        GeneralCategory::CurrencySymbol), ('\u{fdfd}', '\u{fdff}', GeneralCategory::OtherSymbol),
        ('\u{fe00}', '\u{fe0f}', GeneralCategory::NonspacingMark), ('\u{fe10}', '\u{fe16}',
        GeneralCategory::OtherPunctuation), ('\u{fe17}', '\u{fe17}',
        GeneralCategory::OpenPunctuation), ('\u{fe18}', '\u{fe18}',
        GeneralCategory::ClosePunctuation), ('\u{fe19}', '\u{fe19}',
        GeneralCategory::OtherPunctuation), ('\u{fe20}', '\u{fe2f}',
        GeneralCategory::NonspacingMark), ('\u{fe30}', '\u{fe30}',
        GeneralCategory::OtherPunctuation), ('\u{fe31}', '\u{fe32}',
        GeneralCategory::DashPunctuation), ('\u{fe33}', '\u{fe34}',
        GeneralCategory::ConnectorPunctuation), ('\u{fe35}', '\u{fe35}',
        GeneralCategory::OpenPunctuation), ('\u{fe36}', '\u{fe36}',
        GeneralCategory::ClosePunctuation), ('\u{fe37}', '\u{fe37}',
        GeneralCategory::OpenPunctuation), ('\u{fe38}', '\u{fe38}',
        GeneralCategory::ClosePunctuation), ('\u{fe39}', '\u{fe39}',
        GeneralCategory::OpenPunctuation), ('\u{fe3a}', '\u{fe3a}',
        GeneralCategory::ClosePunctuation), ('\u{fe3b}', '\u{fe3b}',
        GeneralCategory::OpenPunctuation), ('\u{fe3c}', '\u{fe3c}',
        GeneralCategory::ClosePunctuation), ('\u{fe3d}', '\u{fe3d}',
        GeneralCategory::OpenPunctuation), ('\u{fe3e}', '\u{fe3e}',
        GeneralCategory::ClosePunctuation), ('\u{fe3f}', '\u{fe3f}',
        GeneralCategory::OpenPunctuation), ('\u{fe40}', '\u{fe40}',
        GeneralCategory::ClosePunctuation), ('\u{fe41}', '\u{fe41}',
        GeneralCategory::OpenPunctuation), ('\u{fe42}', '\u{fe42}',
        GeneralCategory::ClosePunctuation), ('\u{fe43}', '\u{fe43}',
        GeneralCategory::OpenPunctuation), ('\u{fe44}', '\u{fe44}',
        GeneralCategory::ClosePunctuation), ('\u{fe45}', '\u{fe46}',
        GeneralCategory::OtherPunctuation), ('\u{fe47}', '\u{fe47}',
        GeneralCategory::OpenPunctuation), ('\u{fe48}', '\u{fe48}',
        GeneralCategory::ClosePunctuation), ('\u{fe49}', '\u{fe4c}',
        GeneralCategory::OtherPunctuation), ('\u{fe4d}', '\u{fe4f}',
        GeneralCategory::ConnectorPunctuation), ('\u{fe50}', '\u{fe52}',
        GeneralCategory::OtherPunctuation), ('\u{fe54}', '\u{fe57}',
        GeneralCategory::OtherPunctuation), ('\u{fe58}', '\u{fe58}',
        GeneralCategory::DashPunctuation), ('\u{fe59}', '\u{fe59}',
        GeneralCategory::OpenPunctuation), ('\u{fe5a}', '\u{fe5a}',
        GeneralCategory::ClosePunctuation), ('\u{fe5b}', '\u{fe5b}',
        GeneralCategory::OpenPunctuation), ('\u{fe5c}', '\u{fe5c}',
        GeneralCategory::ClosePunctuation), ('\u{fe5d}', '\u{fe5d}',
        GeneralCategory::OpenPunctuation), ('\u{fe5e}', '\u{fe5e}',
        GeneralCategory::ClosePunctuation), ('\u{fe5f}', '\u{fe61}',
        GeneralCategory::OtherPunctuation), ('\u{fe62}', '\u{fe62}', GeneralCategory::MathSymbol),
        ('\u{fe63}', '\u{fe63}', GeneralCategory::DashPunctuation), ('\u{fe64}', '\u{fe66}',
        GeneralCategory::MathSymbol), ('\u{fe68}', '\u{fe68}', GeneralCategory::OtherPunctuation),
        ('\u{fe69}', '\u{fe69}', GeneralCategory::CurrencySymbol), ('\u{fe6a}', '\u{fe6b}',
        GeneralCategory::OtherPunctuation), ('\u{fe70}', '\u{fe74}', GeneralCategory::OtherLetter),
        ('\u{fe76}', '\u{fefc}', GeneralCategory::OtherLetter), ('\u{feff}', '\u{feff}',
        GeneralCategory::Format), ('\u{ff01}', '\u{ff03}', GeneralCategory::OtherPunctuation),
        ('\u{ff04}', '\u{ff04}', GeneralCategory::CurrencySymbol), ('\u{ff05}', '\u{ff07}',
        GeneralCategory::OtherPunctuation), ('\u{ff08}', '\u{ff08}',
        GeneralCategory::OpenPunctuation), ('\u{ff09}', '\u{ff09}',
        GeneralCategory::ClosePunctuation), ('\u{ff0a}', '\u{ff0a}',
        GeneralCategory::OtherPunctuation), ('\u{ff0b}', '\u{ff0b}', GeneralCategory::MathSymbol),
        ('\u{ff0c}', '\u{ff0c}', GeneralCategory::OtherPunctuation), ('\u{ff0d}', '\u{ff0d}',
        GeneralCategory::DashPunctuation), ('\u{ff0e}', '\u{ff0f}',
        GeneralCategory::OtherPunctuation), ('\u{ff10}', '\u{ff19}',
        GeneralCategory::DecimalNumber), ('\u{ff1a}', '\u{ff1b}',
        GeneralCategory::OtherPunctuation), ('\u{ff1c}', '\u{ff1e}', GeneralCategory::MathSymbol),
        ('\u{ff1f}', '\u{ff20}', GeneralCategory::OtherPunctuation), ('\u{ff21}', '\u{ff3a}',
        GeneralCategory::UppercaseLetter), ('\u{ff3b}', '\u{ff3b}',
        GeneralCategory::OpenPunctuation), ('\u{ff3c}', '\u{ff3c}',
        GeneralCategory::OtherPunctuation), ('\u{ff3d}', '\u{ff3d}',
        GeneralCategory::ClosePunctuation), ('\u{ff3e}', '\u{ff3e}',
        GeneralCategory::ModifierSymbol), ('\u{ff3f}', '\u{ff3f}',
        GeneralCategory::ConnectorPunctuation), ('\u{ff40}', '\u{ff40}',
        GeneralCategory::ModifierSymbol), ('\u{ff41}', '\u{ff5a}',
        GeneralCategory::LowercaseLetter), ('\u{ff5b}', '\u{ff5b}',
        GeneralCategory::OpenPunctuation), ('\u{ff5c}', '\u{ff5c}', GeneralCategory::MathSymbol),
        ('\u{ff5d}', '\u{ff5d}', GeneralCategory::ClosePunctuation), ('\u{ff5e}', '\u{ff5e}',
        GeneralCategory::MathSymbol), ('\u{ff5f}', '\u{ff5f}', GeneralCategory::OpenPunctuation),
        ('\u{ff60}', '\u{ff60}', GeneralCategory::ClosePunctuation), ('\u{ff61}', '\u{ff61}',
        GeneralCategory::OtherPunctuation), ('\u{ff62}', '\u{ff62}',
        GeneralCategory::OpenPunctuation), ('\u{ff63}', '\u{ff63}',
        GeneralCategory::ClosePunctuation), ('\u{ff64}', '\u{ff65}',
        GeneralCategory::OtherPunctuation), ('\u{ff66}', '\u{ff6f}', GeneralCategory::OtherLetter),
        ('\u{ff70}', '\u{ff70}', GeneralCategory::ModifierLetter), ('\u{ff71}', '\u{ff9d}',
        GeneralCategory::OtherLetter), ('\u{ff9e}', '\u{ff9f}', GeneralCategory::ModifierLetter),
        ('\u{ffa0}', '\u{ffbe}', GeneralCategory::OtherLetter), ('\u{ffc2}', '\u{ffc7}',
        GeneralCategory::OtherLetter), ('\u{ffca}', '\u{ffcf}', GeneralCategory::OtherLetter),
        ('\u{ffd2}', '\u{ffd7}', GeneralCategory::OtherLetter), ('\u{ffda}', '\u{ffdc}',
        GeneralCategory::OtherLetter), ('\u{ffe0}', '\u{ffe1}', GeneralCategory::CurrencySymbol),
        ('\u{ffe2}', '\u{ffe2}', GeneralCategory::MathSymbol), ('\u{ffe3}', '\u{ffe3}',
        GeneralCategory::ModifierSymbol), ('\u{ffe4}', '\u{ffe4}', GeneralCategory::OtherSymbol),
        ('\u{ffe5}', '\u{ffe6}', GeneralCategory::CurrencySymbol), ('\u{ffe8}', '\u{ffe8}',
        GeneralCategory::OtherSymbol), ('\u{ffe9}', '\u{ffec}', GeneralCategory::MathSymbol),
        ('\u{ffed}', '\u{ffee}', GeneralCategory::OtherSymbol), ('\u{fff9}', '\u{fffb}',
        GeneralCategory::Format), ('\u{fffc}', '\u{fffd}', GeneralCategory::OtherSymbol),
        ('\u{10000}', '\u{1000b}', GeneralCategory::OtherLetter), ('\u{1000d}', '\u{10026}',
        GeneralCategory::OtherLetter), ('\u{10028}', '\u{1003a}', GeneralCategory::OtherLetter),
        ('\u{1003c}', '\u{1003d}', GeneralCategory::OtherLetter), ('\u{1003f}', '\u{1004d}',
        GeneralCategory::OtherLetter), ('\u{10050}', '\u{1005d}', GeneralCategory::OtherLetter),
        ('\u{10080}', '\u{100fa}', GeneralCategory::OtherLetter), ('\u{10100}', '\u{10102}',
        GeneralCategory::OtherPunctuation), ('\u{10107}', '\u{10133}',
        GeneralCategory::OtherNumber), ('\u{10137}', '\u{1013f}', GeneralCategory::OtherSymbol),
        ('\u{10140}', '\u{10174}', GeneralCategory::LetterNumber), ('\u{10175}', '\u{10178}',
        GeneralCategory::OtherNumber), ('\u{10179}', '\u{10189}', GeneralCategory::OtherSymbol),
        ('\u{1018a}', '\u{1018b}', GeneralCategory::OtherNumber), ('\u{1018c}', '\u{1018e}',
        GeneralCategory::OtherSymbol), ('\u{10190}', '\u{1019c}', GeneralCategory::OtherSymbol),
        ('\u{101a0}', '\u{101a0}', GeneralCategory::OtherSymbol), ('\u{101d0}', '\u{101fc}',
        GeneralCategory::OtherSymbol), ('\u{101fd}', '\u{101fd}', GeneralCategory::NonspacingMark),
        ('\u{10280}', '\u{1029c}', GeneralCategory::OtherLetter), ('\u{102a0}', '\u{102d0}',
        GeneralCategory::OtherLetter), ('\u{102e0}', '\u{102e0}', GeneralCategory::NonspacingMark),
        ('\u{102e1}', '\u{102fb}', GeneralCategory::OtherNumber), ('\u{10300}', '\u{1031f}',
        GeneralCategory::OtherLetter), ('\u{10320}', '\u{10323}', GeneralCategory::OtherNumber),
        ('\u{1032d}', '\u{10340}', GeneralCategory::OtherLetter), ('\u{10341}', '\u{10341}',
        GeneralCategory::LetterNumber), ('\u{10342}', '\u{10349}', GeneralCategory::OtherLetter),
        ('\u{1034a}', '\u{1034a}', GeneralCategory::LetterNumber), ('\u{10350}', '\u{10375}',
        GeneralCategory::OtherLetter), ('\u{10376}', '\u{1037a}', GeneralCategory::NonspacingMark),
        ('\u{10380}', '\u{1039d}', GeneralCategory::OtherLetter), ('\u{1039f}', '\u{1039f}',
        GeneralCategory::OtherPunctuation), ('\u{103a0}', '\u{103c3}',
        GeneralCategory::OtherLetter), ('\u{103c8}', '\u{103cf}', GeneralCategory::OtherLetter),
        ('\u{103d0}', '\u{103d0}', GeneralCategory::OtherPunctuation), ('\u{103d1}', '\u{103d5}',
        GeneralCategory::LetterNumber), ('\u{10400}', '\u{10427}',
        GeneralCategory::UppercaseLetter), ('\u{10428}', '\u{1044f}',
        GeneralCategory::LowercaseLetter), ('\u{10450}', '\u{1049d}', GeneralCategory::OtherLetter),
        ('\u{104a0}', '\u{104a9}', GeneralCategory::DecimalNumber), ('\u{104b0}', '\u{104d3}',
        GeneralCategory::UppercaseLetter), ('\u{104d8}', '\u{104fb}',
        GeneralCategory::LowercaseLetter), ('\u{10500}', '\u{10527}', GeneralCategory::OtherLetter),
        ('\u{10530}', '\u{10563}', GeneralCategory::OtherLetter), ('\u{1056f}', '\u{1056f}',
        GeneralCategory::OtherPunctuation), ('\u{10570}', '\u{1057a}',
        GeneralCategory::UppercaseLetter), ('\u{1057c}', '\u{1058a}',
        GeneralCategory::UppercaseLetter), ('\u{1058c}', '\u{10592}',
        GeneralCategory::UppercaseLetter), ('\u{10594}', '\u{10595}',
        GeneralCategory::UppercaseLetter), ('\u{10597}', '\u{105a1}',
        GeneralCategory::LowercaseLetter), ('\u{105a3}', '\u{105b1}',
        GeneralCategory::LowercaseLetter), ('\u{105b3}', '\u{105b9}',
        GeneralCategory::LowercaseLetter), ('\u{105bb}', '\u{105bc}',
        GeneralCategory::LowercaseLetter), ('\u{10600}', '\u{10736}', GeneralCategory::OtherLetter),
        ('\u{10740}', '\u{10755}', GeneralCategory::OtherLetter), ('\u{10760}', '\u{10767}',
        GeneralCategory::OtherLetter), ('\u{10780}', '\u{10785}', GeneralCategory::ModifierLetter),
        ('\u{10787}', '\u{107b0}', GeneralCategory::ModifierLetter), ('\u{107b2}', '\u{107ba}',
        GeneralCategory::ModifierLetter), ('\u{10800}', '\u{10805}', GeneralCategory::OtherLetter),
        ('\u{10808}', '\u{10808}', GeneralCategory::OtherLetter), ('\u{1080a}', '\u{10835}',
        GeneralCategory::OtherLetter), ('\u{10837}', '\u{10838}', GeneralCategory::OtherLetter),
        ('\u{1083c}', '\u{1083c}', GeneralCategory::OtherLetter), ('\u{1083f}', '\u{10855}',
        GeneralCategory::OtherLetter), ('\u{10857}', '\u{10857}',
        GeneralCategory::OtherPunctuation), ('\u{10858}', '\u{1085f}',
        GeneralCategory::OtherNumber), ('\u{10860}', '\u{10876}', GeneralCategory::OtherLetter),
        ('\u{10877}', '\u{10878}', GeneralCategory::OtherSymbol), ('\u{10879}', '\u{1087f}',
        GeneralCategory::OtherNumber), ('\u{10880}', '\u{1089e}', GeneralCategory::OtherLetter),
        ('\u{108a7}', '\u{108af}', GeneralCategory::OtherNumber), ('\u{108e0}', '\u{108f2}',
        GeneralCategory::OtherLetter), ('\u{108f4}', '\u{108f5}', GeneralCategory::OtherLetter),
        ('\u{108fb}', '\u{108ff}', GeneralCategory::OtherNumber), ('\u{10900}', '\u{10915}',
        GeneralCategory::OtherLetter), ('\u{10916}', '\u{1091b}', GeneralCategory::OtherNumber),
        ('\u{1091f}', '\u{1091f}', GeneralCategory::OtherPunctuation), ('\u{10920}', '\u{10939}',
        GeneralCategory::OtherLetter), ('\u{1093f}', '\u{1093f}',
        GeneralCategory::OtherPunctuation), ('\u{10980}', '\u{109b7}',
        GeneralCategory::OtherLetter), ('\u{109bc}', '\u{109bd}', GeneralCategory::OtherNumber),
        ('\u{109be}', '\u{109bf}', GeneralCategory::OtherLetter), ('\u{109c0}', '\u{109cf}',
        GeneralCategory::OtherNumber), ('\u{109d2}', '\u{109ff}', GeneralCategory::OtherNumber),
        ('\u{10a00}', '\u{10a00}', GeneralCategory::OtherLetter), ('\u{10a01}', '\u{10a03}',
        GeneralCategory::NonspacingMark), ('\u{10a05}', '\u{10a06}',
        GeneralCategory::NonspacingMark), ('\u{10a0c}', '\u{10a0f}',
        GeneralCategory::NonspacingMark), ('\u{10a10}', '\u{10a13}', GeneralCategory::OtherLetter),
        ('\u{10a15}', '\u{10a17}', GeneralCategory::OtherLetter), ('\u{10a19}', '\u{10a35}',
        GeneralCategory::OtherLetter), ('\u{10a38}', '\u{10a3a}', GeneralCategory::NonspacingMark),
        ('\u{10a3f}', '\u{10a3f}', GeneralCategory::NonspacingMark), ('\u{10a40}', '\u{10a48}',
        GeneralCategory::OtherNumber), ('\u{10a50}', '\u{10a58}',
        GeneralCategory::OtherPunctuation), ('\u{10a60}', '\u{10a7c}',
        GeneralCategory::OtherLetter), ('\u{10a7d}', '\u{10a7e}', GeneralCategory::OtherNumber),
        ('\u{10a7f}', '\u{10a7f}', GeneralCategory::OtherPunctuation), ('\u{10a80}', '\u{10a9c}',
        GeneralCategory::OtherLetter), ('\u{10a9d}', '\u{10a9f}', GeneralCategory::OtherNumber),
        ('\u{10ac0}', '\u{10ac7}', GeneralCategory::OtherLetter), ('\u{10ac8}', '\u{10ac8}',
        GeneralCategory::OtherSymbol), ('\u{10ac9}', '\u{10ae4}', GeneralCategory::OtherLetter),
        ('\u{10ae5}', '\u{10ae6}', GeneralCategory::NonspacingMark), ('\u{10aeb}', '\u{10aef}',
        GeneralCategory::OtherNumber), ('\u{10af0}', '\u{10af6}',
        GeneralCategory::OtherPunctuation), ('\u{10b00}', '\u{10b35}',
        GeneralCategory::OtherLetter), ('\u{10b39}', '\u{10b3f}',
        GeneralCategory::OtherPunctuation), ('\u{10b40}', '\u{10b55}',
        GeneralCategory::OtherLetter), ('\u{10b58}', '\u{10b5f}', GeneralCategory::OtherNumber),
        ('\u{10b60}', '\u{10b72}', GeneralCategory::OtherLetter), ('\u{10b78}', '\u{10b7f}',
        GeneralCategory::OtherNumber), ('\u{10b80}', '\u{10b91}', GeneralCategory::OtherLetter),
        ('\u{10b99}', '\u{10b9c}', GeneralCategory::OtherPunctuation), ('\u{10ba9}', '\u{10baf}',
        GeneralCategory::OtherNumber), ('\u{10c00}', '\u{10c48}', GeneralCategory::OtherLetter),
        ('\u{10c80}', '\u{10cb2}', GeneralCategory::UppercaseLetter), ('\u{10cc0}', '\u{10cf2}',
        GeneralCategory::LowercaseLetter), ('\u{10cfa}', '\u{10cff}', GeneralCategory::OtherNumber),
        ('\u{10d00}', '\u{10d23}', GeneralCategory::OtherLetter), ('\u{10d24}', '\u{10d27}',
        GeneralCategory::NonspacingMark), ('\u{10d30}', '\u{10d39}',
        GeneralCategory::DecimalNumber), ('\u{10e60}', '\u{10e7e}', GeneralCategory::OtherNumber),
        ('\u{10e80}', '\u{10ea9}', GeneralCategory::OtherLetter), ('\u{10eab}', '\u{10eac}',
        GeneralCategory::NonspacingMark), ('\u{10ead}', '\u{10ead}',
        GeneralCategory::DashPunctuation), ('\u{10eb0}', '\u{10eb1}', GeneralCategory::OtherLetter),
        ('\u{10efd}', '\u{10eff}', GeneralCategory::NonspacingMark), ('\u{10f00}', '\u{10f1c}',
        GeneralCategory::OtherLetter), ('\u{10f1d}', '\u{10f26}', GeneralCategory::OtherNumber),
        ('\u{10f27}', '\u{10f27}', GeneralCategory::OtherLetter), ('\u{10f30}', '\u{10f45}',
        GeneralCategory::OtherLetter), ('\u{10f46}', '\u{10f50}', GeneralCategory::NonspacingMark),
        ('\u{10f51}', '\u{10f54}', GeneralCategory::OtherNumber), ('\u{10f55}', '\u{10f59}',
        GeneralCategory::OtherPunctuation), ('\u{10f70}', '\u{10f81}',
        GeneralCategory::OtherLetter), ('\u{10f82}', '\u{10f85}', GeneralCategory::NonspacingMark),
        ('\u{10f86}', '\u{10f89}', GeneralCategory::OtherPunctuation), ('\u{10fb0}', '\u{10fc4}',
        GeneralCategory::OtherLetter), ('\u{10fc5}', '\u{10fcb}', GeneralCategory::OtherNumber),
        ('\u{10fe0}', '\u{10ff6}', GeneralCategory::OtherLetter), ('\u{11000}', '\u{11000}',
        GeneralCategory::SpacingMark), ('\u{11001}', '\u{11001}', GeneralCategory::NonspacingMark),
        ('\u{11002}', '\u{11002}', GeneralCategory::SpacingMark), ('\u{11003}', '\u{11037}',
        GeneralCategory::OtherLetter), ('\u{11038}', '\u{11046}', GeneralCategory::NonspacingMark),
        ('\u{11047}', '\u{1104d}', GeneralCategory::OtherPunctuation), ('\u{11052}', '\u{11065}',
        GeneralCategory::OtherNumber), ('\u{11066}', '\u{1106f}', GeneralCategory::DecimalNumber),
        ('\u{11070}', '\u{11070}', GeneralCategory::NonspacingMark), ('\u{11071}', '\u{11072}',
        GeneralCategory::OtherLetter), ('\u{11073}', '\u{11074}', GeneralCategory::NonspacingMark),
        ('\u{11075}', '\u{11075}', GeneralCategory::OtherLetter), ('\u{1107f}', '\u{11081}',
        GeneralCategory::NonspacingMark), ('\u{11082}', '\u{11082}', GeneralCategory::SpacingMark),
        ('\u{11083}', '\u{110af}', GeneralCategory::OtherLetter), ('\u{110b0}', '\u{110b2}',
        GeneralCategory::SpacingMark), ('\u{110b3}', '\u{110b6}', GeneralCategory::NonspacingMark),
        ('\u{110b7}', '\u{110b8}', GeneralCategory::SpacingMark), ('\u{110b9}', '\u{110ba}',
        GeneralCategory::NonspacingMark), ('\u{110bb}', '\u{110bc}',
        GeneralCategory::OtherPunctuation), ('\u{110bd}', '\u{110bd}', GeneralCategory::Format),
        ('\u{110be}', '\u{110c1}', GeneralCategory::OtherPunctuation), ('\u{110c2}', '\u{110c2}',
        GeneralCategory::NonspacingMark), ('\u{110cd}', '\u{110cd}', GeneralCategory::Format),
        ('\u{110d0}', '\u{110e8}', GeneralCategory::OtherLetter), ('\u{110f0}', '\u{110f9}',
        GeneralCategory::DecimalNumber), ('\u{11100}', '\u{11102}',
        GeneralCategory::NonspacingMark), ('\u{11103}', '\u{11126}', GeneralCategory::OtherLetter),
        ('\u{11127}', '\u{1112b}', GeneralCategory::NonspacingMark), ('\u{1112c}', '\u{1112c}',
        GeneralCategory::SpacingMark), ('\u{1112d}', '\u{11134}', GeneralCategory::NonspacingMark),
        ('\u{11136}', '\u{1113f}', GeneralCategory::DecimalNumber), ('\u{11140}', '\u{11143}',
        GeneralCategory::OtherPunctuation), ('\u{11144}', '\u{11144}',
        GeneralCategory::OtherLetter), ('\u{11145}', '\u{11146}', GeneralCategory::SpacingMark),
        ('\u{11147}', '\u{11147}', GeneralCategory::OtherLetter), ('\u{11150}', '\u{11172}',
        GeneralCategory::OtherLetter), ('\u{11173}', '\u{11173}', GeneralCategory::NonspacingMark),
        ('\u{11174}', '\u{11175}', GeneralCategory::OtherPunctuation), ('\u{11176}', '\u{11176}',
        GeneralCategory::OtherLetter), ('\u{11180}', '\u{11181}', GeneralCategory::NonspacingMark),
        ('\u{11182}', '\u{11182}', GeneralCategory::SpacingMark), ('\u{11183}', '\u{111b2}',
        GeneralCategory::OtherLetter), ('\u{111b3}', '\u{111b5}', GeneralCategory::SpacingMark),
        ('\u{111b6}', '\u{111be}', GeneralCategory::NonspacingMark), ('\u{111bf}', '\u{111c0}',
        GeneralCategory::SpacingMark), ('\u{111c1}', '\u{111c4}', GeneralCategory::OtherLetter),
        ('\u{111c5}', '\u{111c8}', GeneralCategory::OtherPunctuation), ('\u{111c9}', '\u{111cc}',
        GeneralCategory::NonspacingMark), ('\u{111cd}', '\u{111cd}',
        GeneralCategory::OtherPunctuation), ('\u{111ce}', '\u{111ce}',
        GeneralCategory::SpacingMark), ('\u{111cf}', '\u{111cf}', GeneralCategory::NonspacingMark),
        ('\u{111d0}', '\u{111d9}', GeneralCategory::DecimalNumber), ('\u{111da}', '\u{111da}',
        GeneralCategory::OtherLetter), ('\u{111db}', '\u{111db}',
        GeneralCategory::OtherPunctuation), ('\u{111dc}', '\u{111dc}',
        GeneralCategory::OtherLetter), ('\u{111dd}', '\u{111df}',
        GeneralCategory::OtherPunctuation), ('\u{111e1}', '\u{111f4}',
        GeneralCategory::OtherNumber), ('\u{11200}', '\u{11211}', GeneralCategory::OtherLetter),
        ('\u{11213}', '\u{1122b}', GeneralCategory::OtherLetter), ('\u{1122c}', '\u{1122e}',
        GeneralCategory::SpacingMark), ('\u{1122f}', '\u{11231}', GeneralCategory::NonspacingMark),
        ('\u{11232}', '\u{11233}', GeneralCategory::SpacingMark), ('\u{11234}', '\u{11234}',
        GeneralCategory::NonspacingMark), ('\u{11235}', '\u{11235}', GeneralCategory::SpacingMark),
        ('\u{11236}', '\u{11237}', GeneralCategory::NonspacingMark), ('\u{11238}', '\u{1123d}',
        GeneralCategory::OtherPunctuation), ('\u{1123e}', '\u{1123e}',
        GeneralCategory::NonspacingMark), ('\u{1123f}', '\u{11240}', GeneralCategory::OtherLetter),
        ('\u{11241}', '\u{11241}', GeneralCategory::NonspacingMark), ('\u{11280}', '\u{11286}',
        GeneralCategory::OtherLetter), ('\u{11288}', '\u{11288}', GeneralCategory::OtherLetter),
        ('\u{1128a}', '\u{1128d}', GeneralCategory::OtherLetter), ('\u{1128f}', '\u{1129d}',
        GeneralCategory::OtherLetter), ('\u{1129f}', '\u{112a8}', GeneralCategory::OtherLetter),
        ('\u{112a9}', '\u{112a9}', GeneralCategory::OtherPunctuation), ('\u{112b0}', '\u{112de}',
        GeneralCategory::OtherLetter), ('\u{112df}', '\u{112df}', GeneralCategory::NonspacingMark),
        ('\u{112e0}', '\u{112e2}', GeneralCategory::SpacingMark), ('\u{112e3}', '\u{112ea}',
        GeneralCategory::NonspacingMark), ('\u{112f0}', '\u{112f9}',
        GeneralCategory::DecimalNumber), ('\u{11300}', '\u{11301}',
        GeneralCategory::NonspacingMark), ('\u{11302}', '\u{11303}', GeneralCategory::SpacingMark),
        ('\u{11305}', '\u{1130c}', GeneralCategory::OtherLetter), ('\u{1130f}', '\u{11310}',
        GeneralCategory::OtherLetter), ('\u{11313}', '\u{11328}', GeneralCategory::OtherLetter),
        ('\u{1132a}', '\u{11330}', GeneralCategory::OtherLetter), ('\u{11332}', '\u{11333}',
        GeneralCategory::OtherLetter), ('\u{11335}', '\u{11339}', GeneralCategory::OtherLetter),
        ('\u{1133b}', '\u{1133c}', GeneralCategory::NonspacingMark), ('\u{1133d}', '\u{1133d}',
        GeneralCategory::OtherLetter), ('\u{1133e}', '\u{1133f}', GeneralCategory::SpacingMark),
        ('\u{11340}', '\u{11340}', GeneralCategory::NonspacingMark), ('\u{11341}', '\u{11344}',
        GeneralCategory::SpacingMark), ('\u{11347}', '\u{11348}', GeneralCategory::SpacingMark),
        ('\u{1134b}', '\u{1134d}', GeneralCategory::SpacingMark), ('\u{11350}', '\u{11350}',
        GeneralCategory::OtherLetter), ('\u{11357}', '\u{11357}', GeneralCategory::SpacingMark),
        ('\u{1135d}', '\u{11361}', GeneralCategory::OtherLetter), ('\u{11362}', '\u{11363}',
        GeneralCategory::SpacingMark), ('\u{11366}', '\u{1136c}', GeneralCategory::NonspacingMark),
        ('\u{11370}', '\u{11374}', GeneralCategory::NonspacingMark), ('\u{11400}', '\u{11434}',
        GeneralCategory::OtherLetter), ('\u{11435}', '\u{11437}', GeneralCategory::SpacingMark),
        ('\u{11438}', '\u{1143f}', GeneralCategory::NonspacingMark), ('\u{11440}', '\u{11441}',
        GeneralCategory::SpacingMark), ('\u{11442}', '\u{11444}', GeneralCategory::NonspacingMark),
        ('\u{11445}', '\u{11445}', GeneralCategory::SpacingMark), ('\u{11446}', '\u{11446}',
        GeneralCategory::NonspacingMark), ('\u{11447}', '\u{1144a}', GeneralCategory::OtherLetter),
        ('\u{1144b}', '\u{1144f}', GeneralCategory::OtherPunctuation), ('\u{11450}', '\u{11459}',
        GeneralCategory::DecimalNumber), ('\u{1145a}', '\u{1145b}',
        GeneralCategory::OtherPunctuation), ('\u{1145d}', '\u{1145d}',
        GeneralCategory::OtherPunctuation), ('\u{1145e}', '\u{1145e}',
        GeneralCategory::NonspacingMark), ('\u{1145f}', '\u{11461}', GeneralCategory::OtherLetter),
        ('\u{11480}', '\u{114af}', GeneralCategory::OtherLetter), ('\u{114b0}', '\u{114b2}',
        GeneralCategory::SpacingMark), ('\u{114b3}', '\u{114b8}', GeneralCategory::NonspacingMark),
        ('\u{114b9}', '\u{114b9}', GeneralCategory::SpacingMark), ('\u{114ba}', '\u{114ba}',
        GeneralCategory::NonspacingMark), ('\u{114bb}', '\u{114be}', GeneralCategory::SpacingMark),
        ('\u{114bf}', '\u{114c0}', GeneralCategory::NonspacingMark), ('\u{114c1}', '\u{114c1}',
        GeneralCategory::SpacingMark), ('\u{114c2}', '\u{114c3}', GeneralCategory::NonspacingMark),
        ('\u{114c4}', '\u{114c5}', GeneralCategory::OtherLetter), ('\u{114c6}', '\u{114c6}',
        GeneralCategory::OtherPunctuation), ('\u{114c7}', '\u{114c7}',
        GeneralCategory::OtherLetter), ('\u{114d0}', '\u{114d9}', GeneralCategory::DecimalNumber),
        ('\u{11580}', '\u{115ae}', GeneralCategory::OtherLetter), ('\u{115af}', '\u{115b1}',
        GeneralCategory::SpacingMark), ('\u{115b2}', '\u{115b5}', GeneralCategory::NonspacingMark),
        ('\u{115b8}', '\u{115bb}', GeneralCategory::SpacingMark), ('\u{115bc}', '\u{115bd}',
        GeneralCategory::NonspacingMark), ('\u{115be}', '\u{115be}', GeneralCategory::SpacingMark),
        ('\u{115bf}', '\u{115c0}', GeneralCategory::NonspacingMark), ('\u{115c1}', '\u{115d7}',
        GeneralCategory::OtherPunctuation), ('\u{115d8}', '\u{115db}',
        GeneralCategory::OtherLetter), ('\u{115dc}', '\u{115dd}', GeneralCategory::NonspacingMark),
        ('\u{11600}', '\u{1162f}', GeneralCategory::OtherLetter), ('\u{11630}', '\u{11632}',
        GeneralCategory::SpacingMark), ('\u{11633}', '\u{1163a}', GeneralCategory::NonspacingMark),
        ('\u{1163b}', '\u{1163c}', GeneralCategory::SpacingMark), ('\u{1163d}', '\u{1163d}',
        GeneralCategory::NonspacingMark), ('\u{1163e}', '\u{1163e}', GeneralCategory::SpacingMark),
        ('\u{1163f}', '\u{11640}', GeneralCategory::NonspacingMark), ('\u{11641}', '\u{11643}',
        GeneralCategory::OtherPunctuation), ('\u{11644}', '\u{11644}',
        GeneralCategory::OtherLetter), ('\u{11650}', '\u{11659}', GeneralCategory::DecimalNumber),
        ('\u{11660}', '\u{1166c}', GeneralCategory::OtherPunctuation), ('\u{11680}', '\u{116aa}',
        GeneralCategory::OtherLetter), ('\u{116ab}', '\u{116ab}', GeneralCategory::NonspacingMark),
        ('\u{116ac}', '\u{116ac}', GeneralCategory::SpacingMark), ('\u{116ad}', '\u{116ad}',
        GeneralCategory::NonspacingMark), ('\u{116ae}', '\u{116af}', GeneralCategory::SpacingMark),
        ('\u{116b0}', '\u{116b5}', GeneralCategory::NonspacingMark), ('\u{116b6}', '\u{116b6}',
        GeneralCategory::SpacingMark), ('\u{116b7}', '\u{116b7}', GeneralCategory::NonspacingMark),
        ('\u{116b8}', '\u{116b8}', GeneralCategory::OtherLetter), ('\u{116b9}', '\u{116b9}',
        GeneralCategory::OtherPunctuation), ('\u{116c0}', '\u{116c9}',
        GeneralCategory::DecimalNumber), ('\u{11700}', '\u{1171a}', GeneralCategory::OtherLetter),
        ('\u{1171d}', '\u{1171f}', GeneralCategory::NonspacingMark), ('\u{11720}', '\u{11721}',
        GeneralCategory::SpacingMark), ('\u{11722}', '\u{11725}', GeneralCategory::NonspacingMark),
        ('\u{11726}', '\u{11726}', GeneralCategory::SpacingMark), ('\u{11727}', '\u{1172b}',
        GeneralCategory::NonspacingMark), ('\u{11730}', '\u{11739}',
        GeneralCategory::DecimalNumber), ('\u{1173a}', '\u{1173b}', GeneralCategory::OtherNumber),
        ('\u{1173c}', '\u{1173e}', GeneralCategory::OtherPunctuation), ('\u{1173f}', '\u{1173f}',
        GeneralCategory::OtherSymbol), ('\u{11740}', '\u{11746}', GeneralCategory::OtherLetter),
        ('\u{11800}', '\u{1182b}', GeneralCategory::OtherLetter), ('\u{1182c}', '\u{1182e}',
        GeneralCategory::SpacingMark), ('\u{1182f}', '\u{11837}', GeneralCategory::NonspacingMark),
        ('\u{11838}', '\u{11838}', GeneralCategory::SpacingMark), ('\u{11839}', '\u{1183a}',
        GeneralCategory::NonspacingMark), ('\u{1183b}', '\u{1183b}',
        GeneralCategory::OtherPunctuation), ('\u{118a0}', '\u{118bf}',
        GeneralCategory::UppercaseLetter), ('\u{118c0}', '\u{118df}',
        GeneralCategory::LowercaseLetter), ('\u{118e0}', '\u{118e9}',
        GeneralCategory::DecimalNumber), ('\u{118ea}', '\u{118f2}', GeneralCategory::OtherNumber),
        ('\u{118ff}', '\u{11906}', GeneralCategory::OtherLetter), ('\u{11909}', '\u{11909}',
        GeneralCategory::OtherLetter), ('\u{1190c}', '\u{11913}', GeneralCategory::OtherLetter),
        ('\u{11915}', '\u{11916}', GeneralCategory::OtherLetter), ('\u{11918}', '\u{1192f}',
        GeneralCategory::OtherLetter), ('\u{11930}', '\u{11935}', GeneralCategory::SpacingMark),
        ('\u{11937}', '\u{11938}', GeneralCategory::SpacingMark), ('\u{1193b}', '\u{1193c}',
        GeneralCategory::NonspacingMark), ('\u{1193d}', '\u{1193d}', GeneralCategory::SpacingMark),
        ('\u{1193e}', '\u{1193e}', GeneralCategory::NonspacingMark), ('\u{1193f}', '\u{1193f}',
        GeneralCategory::OtherLetter), ('\u{11940}', '\u{11940}', GeneralCategory::SpacingMark),
        ('\u{11941}', '\u{11941}', GeneralCategory::OtherLetter), ('\u{11942}', '\u{11942}',
        GeneralCategory::SpacingMark), ('\u{11943}', '\u{11943}', GeneralCategory::NonspacingMark),
        ('\u{11944}', '\u{11946}', GeneralCategory::OtherPunctuation), ('\u{11950}', '\u{11959}',
        GeneralCategory::DecimalNumber), ('\u{119a0}', '\u{119a7}', GeneralCategory::OtherLetter),
        ('\u{119aa}', '\u{119d0}', GeneralCategory::OtherLetter), ('\u{119d1}', '\u{119d3}',
        GeneralCategory::SpacingMark), ('\u{119d4}', '\u{119d7}', GeneralCategory::NonspacingMark),
        ('\u{119da}', '\u{119db}', GeneralCategory::NonspacingMark), ('\u{119dc}', '\u{119df}',
        GeneralCategory::SpacingMark), ('\u{119e0}', '\u{119e0}', GeneralCategory::NonspacingMark),
        ('\u{119e1}', '\u{119e1}', GeneralCategory::OtherLetter), ('\u{119e2}', '\u{119e2}',
        GeneralCategory::OtherPunctuation), ('\u{119e3}', '\u{119e3}',
        GeneralCategory::OtherLetter), ('\u{119e4}', '\u{119e4}', GeneralCategory::SpacingMark),
        ('\u{11a00}', '\u{11a00}', GeneralCategory::OtherLetter), ('\u{11a01}', '\u{11a0a}',
        GeneralCategory::NonspacingMark), ('\u{11a0b}', '\u{11a32}', GeneralCategory::OtherLetter),
        ('\u{11a33}', '\u{11a38}', GeneralCategory::NonspacingMark), ('\u{11a39}', '\u{11a39}',
        GeneralCategory::SpacingMark), ('\u{11a3a}', '\u{11a3a}', GeneralCategory::OtherLetter),
        ('\u{11a3b}', '\u{11a3e}', GeneralCategory::NonspacingMark), ('\u{11a3f}', '\u{11a46}',
        GeneralCategory::OtherPunctuation), ('\u{11a47}', '\u{11a47}',
        GeneralCategory::NonspacingMark), ('\u{11a50}', '\u{11a50}', GeneralCategory::OtherLetter),
        ('\u{11a51}', '\u{11a56}', GeneralCategory::NonspacingMark), ('\u{11a57}', '\u{11a58}',
        GeneralCategory::SpacingMark), ('\u{11a59}', '\u{11a5b}', GeneralCategory::NonspacingMark),
        ('\u{11a5c}', '\u{11a89}', GeneralCategory::OtherLetter), ('\u{11a8a}', '\u{11a96}',
        GeneralCategory::NonspacingMark), ('\u{11a97}', '\u{11a97}', GeneralCategory::SpacingMark),
        ('\u{11a98}', '\u{11a99}', GeneralCategory::NonspacingMark), ('\u{11a9a}', '\u{11a9c}',
        GeneralCategory::OtherPunctuation), ('\u{11a9d}', '\u{11a9d}',
        GeneralCategory::OtherLetter), ('\u{11a9e}', '\u{11aa2}',
        GeneralCategory::OtherPunctuation), ('\u{11ab0}', '\u{11af8}',
        GeneralCategory::OtherLetter), ('\u{11b00}', '\u{11b09}',
        GeneralCategory::OtherPunctuation), ('\u{11c00}', '\u{11c08}',
        GeneralCategory::OtherLetter), ('\u{11c0a}', '\u{11c2e}', GeneralCategory::OtherLetter),
        ('\u{11c2f}', '\u{11c2f}', GeneralCategory::SpacingMark), ('\u{11c30}', '\u{11c36}',
        GeneralCategory::NonspacingMark), ('\u{11c38}', '\u{11c3d}',
        GeneralCategory::NonspacingMark), ('\u{11c3e}', '\u{11c3e}', GeneralCategory::SpacingMark),
        ('\u{11c3f}', '\u{11c3f}', GeneralCategory::NonspacingMark), ('\u{11c40}', '\u{11c40}',
        GeneralCategory::OtherLetter), ('\u{11c41}', '\u{11c45}',
        GeneralCategory::OtherPunctuation), ('\u{11c50}', '\u{11c59}',
        GeneralCategory::DecimalNumber), ('\u{11c5a}', '\u{11c6c}', GeneralCategory::OtherNumber),
        ('\u{11c70}', '\u{11c71}', GeneralCategory::OtherPunctuation), ('\u{11c72}', '\u{11c8f}',
        GeneralCategory::OtherLetter), ('\u{11c92}', '\u{11ca7}', GeneralCategory::NonspacingMark),
        ('\u{11ca9}', '\u{11ca9}', GeneralCategory::SpacingMark), ('\u{11caa}', '\u{11cb0}',
        GeneralCategory::NonspacingMark), ('\u{11cb1}', '\u{11cb1}', GeneralCategory::SpacingMark),
        ('\u{11cb2}', '\u{11cb3}', GeneralCategory::NonspacingMark), ('\u{11cb4}', '\u{11cb4}',
        GeneralCategory::SpacingMark), ('\u{11cb5}', '\u{11cb6}', GeneralCategory::NonspacingMark),
        ('\u{11d00}', '\u{11d06}', GeneralCategory::OtherLetter), ('\u{11d08}', '\u{11d09}',
        GeneralCategory::OtherLetter), ('\u{11d0b}', '\u{11d30}', GeneralCategory::OtherLetter),
        ('\u{11d31}', '\u{11d36}', GeneralCategory::NonspacingMark), ('\u{11d3a}', '\u{11d3a}',
        GeneralCategory::NonspacingMark), ('\u{11d3c}', '\u{11d3d}',
        GeneralCategory::NonspacingMark), ('\u{11d3f}', '\u{11d45}',
        GeneralCategory::NonspacingMark), ('\u{11d46}', '\u{11d46}', GeneralCategory::OtherLetter),
        ('\u{11d47}', '\u{11d47}', GeneralCategory::NonspacingMark), ('\u{11d50}', '\u{11d59}',
        GeneralCategory::DecimalNumber), ('\u{11d60}', '\u{11d65}', GeneralCategory::OtherLetter),
        ('\u{11d67}', '\u{11d68}', GeneralCategory::OtherLetter), ('\u{11d6a}', '\u{11d89}',
        GeneralCategory::OtherLetter), ('\u{11d8a}', '\u{11d8e}', GeneralCategory::SpacingMark),
        ('\u{11d90}', '\u{11d91}', GeneralCategory::NonspacingMark), ('\u{11d93}', '\u{11d94}',
        GeneralCategory::SpacingMark), ('\u{11d95}', '\u{11d95}', GeneralCategory::NonspacingMark),
        ('\u{11d96}', '\u{11d96}', GeneralCategory::SpacingMark), ('\u{11d97}', '\u{11d97}',
        GeneralCategory::NonspacingMark), ('\u{11d98}', '\u{11d98}', GeneralCategory::OtherLetter),
        ('\u{11da0}', '\u{11da9}', GeneralCategory::DecimalNumber), ('\u{11ee0}', '\u{11ef2}',
        GeneralCategory::OtherLetter), ('\u{11ef3}', '\u{11ef4}', GeneralCategory::NonspacingMark),
        ('\u{11ef5}', '\u{11ef6}', GeneralCategory::SpacingMark), ('\u{11ef7}', '\u{11ef8}',
        GeneralCategory::OtherPunctuation), ('\u{11f00}', '\u{11f01}',
        GeneralCategory::NonspacingMark), ('\u{11f02}', '\u{11f02}', GeneralCategory::OtherLetter),
        ('\u{11f03}', '\u{11f03}', GeneralCategory::SpacingMark), ('\u{11f04}', '\u{11f10}',
        GeneralCategory::OtherLetter), ('\u{11f12}', '\u{11f33}', GeneralCategory::OtherLetter),
        ('\u{11f34}', '\u{11f35}', GeneralCategory::SpacingMark), ('\u{11f36}', '\u{11f3a}',
        GeneralCategory::NonspacingMark), ('\u{11f3e}', '\u{11f3f}', GeneralCategory::SpacingMark),
        ('\u{11f40}', '\u{11f40}', GeneralCategory::NonspacingMark), ('\u{11f41}', '\u{11f41}',
        GeneralCategory::SpacingMark), ('\u{11f42}', '\u{11f42}', GeneralCategory::NonspacingMark),
        ('\u{11f43}', '\u{11f4f}', GeneralCategory::OtherPunctuation), ('\u{11f50}', '\u{11f59}',
        GeneralCategory::DecimalNumber), ('\u{11fb0}', '\u{11fb0}', GeneralCategory::OtherLetter),
        ('\u{11fc0}', '\u{11fd4}', GeneralCategory::OtherNumber), ('\u{11fd5}', '\u{11fdc}',
        GeneralCategory::OtherSymbol), ('\u{11fdd}', '\u{11fe0}', GeneralCategory::CurrencySymbol),
        ('\u{11fe1}', '\u{11ff1}', GeneralCategory::OtherSymbol), ('\u{11fff}', '\u{11fff}',
        GeneralCategory::OtherPunctuation), ('\u{12000}', '\u{12399}',
        GeneralCategory::OtherLetter), ('\u{12400}', '\u{1246e}', GeneralCategory::LetterNumber),
        ('\u{12470}', '\u{12474}', GeneralCategory::OtherPunctuation), ('\u{12480}', '\u{12543}',
        GeneralCategory::OtherLetter), ('\u{12f90}', '\u{12ff0}', GeneralCategory::OtherLetter),
        ('\u{12ff1}', '\u{12ff2}', GeneralCategory::OtherPunctuation), ('\u{13000}', '\u{1342f}',
        GeneralCategory::OtherLetter), ('\u{13430}', '\u{1343f}', GeneralCategory::Format),
        ('\u{13440}', '\u{13440}', GeneralCategory::NonspacingMark), ('\u{13441}', '\u{13446}',
        GeneralCategory::OtherLetter), ('\u{13447}', '\u{13455}', GeneralCategory::NonspacingMark),
        ('\u{14400}', '\u{14646}', GeneralCategory::OtherLetter), ('\u{16800}', '\u{16a38}',
        GeneralCategory::OtherLetter), ('\u{16a40}', '\u{16a5e}', GeneralCategory::OtherLetter),
        ('\u{16a60}', '\u{16a69}', GeneralCategory::DecimalNumber), ('\u{16a6e}', '\u{16a6f}',
        GeneralCategory::OtherPunctuation), ('\u{16a70}', '\u{16abe}',
        GeneralCategory::OtherLetter), ('\u{16ac0}', '\u{16ac9}', GeneralCategory::DecimalNumber),
        ('\u{16ad0}', '\u{16aed}', GeneralCategory::OtherLetter), ('\u{16af0}', '\u{16af4}',
        GeneralCategory::NonspacingMark), ('\u{16af5}', '\u{16af5}',
        GeneralCategory::OtherPunctuation), ('\u{16b00}', '\u{16b2f}',
        GeneralCategory::OtherLetter), ('\u{16b30}', '\u{16b36}', GeneralCategory::NonspacingMark),
        ('\u{16b37}', '\u{16b3b}', GeneralCategory::OtherPunctuation), ('\u{16b3c}', '\u{16b3f}',
        GeneralCategory::OtherSymbol), ('\u{16b40}', '\u{16b43}', GeneralCategory::ModifierLetter),
        ('\u{16b44}', '\u{16b44}', GeneralCategory::OtherPunctuation), ('\u{16b45}', '\u{16b45}',
        GeneralCategory::OtherSymbol), ('\u{16b50}', '\u{16b59}', GeneralCategory::DecimalNumber),
        ('\u{16b5b}', '\u{16b61}', GeneralCategory::OtherNumber), ('\u{16b63}', '\u{16b77}',
        GeneralCategory::OtherLetter), ('\u{16b7d}', '\u{16b8f}', GeneralCategory::OtherLetter),
        ('\u{16e40}', '\u{16e5f}', GeneralCategory::UppercaseLetter), ('\u{16e60}', '\u{16e7f}',
        GeneralCategory::LowercaseLetter), ('\u{16e80}', '\u{16e96}', GeneralCategory::OtherNumber),
        ('\u{16e97}', '\u{16e9a}', GeneralCategory::OtherPunctuation), ('\u{16f00}', '\u{16f4a}',
        GeneralCategory::OtherLetter), ('\u{16f4f}', '\u{16f4f}', GeneralCategory::NonspacingMark),
        ('\u{16f50}', '\u{16f50}', GeneralCategory::OtherLetter), ('\u{16f51}', '\u{16f87}',
        GeneralCategory::SpacingMark), ('\u{16f8f}', '\u{16f92}', GeneralCategory::NonspacingMark),
        ('\u{16f93}', '\u{16f9f}', GeneralCategory::ModifierLetter), ('\u{16fe0}', '\u{16fe1}',
        GeneralCategory::ModifierLetter), ('\u{16fe2}', '\u{16fe2}',
        GeneralCategory::OtherPunctuation), ('\u{16fe3}', '\u{16fe3}',
        GeneralCategory::ModifierLetter), ('\u{16fe4}', '\u{16fe4}',
        GeneralCategory::NonspacingMark), ('\u{16ff0}', '\u{16ff1}', GeneralCategory::SpacingMark),
        ('\u{17000}', '\u{187f7}', GeneralCategory::OtherLetter), ('\u{18800}', '\u{18cd5}',
        GeneralCategory::OtherLetter), ('\u{18d00}', '\u{18d08}', GeneralCategory::OtherLetter),
        ('\u{1aff0}', '\u{1aff3}', GeneralCategory::ModifierLetter), ('\u{1aff5}', '\u{1affb}',
        GeneralCategory::ModifierLetter), ('\u{1affd}', '\u{1affe}',
        GeneralCategory::ModifierLetter), ('\u{1b000}', '\u{1b122}', GeneralCategory::OtherLetter),
        ('\u{1b132}', '\u{1b132}', GeneralCategory::OtherLetter), ('\u{1b150}', '\u{1b152}',
        GeneralCategory::OtherLetter), ('\u{1b155}', '\u{1b155}', GeneralCategory::OtherLetter),
        ('\u{1b164}', '\u{1b167}', GeneralCategory::OtherLetter), ('\u{1b170}', '\u{1b2fb}',
        GeneralCategory::OtherLetter), ('\u{1bc00}', '\u{1bc6a}', GeneralCategory::OtherLetter),
        ('\u{1bc70}', '\u{1bc7c}', GeneralCategory::OtherLetter), ('\u{1bc80}', '\u{1bc88}',
        GeneralCategory::OtherLetter), ('\u{1bc90}', '\u{1bc99}', GeneralCategory::OtherLetter),
        ('\u{1bc9c}', '\u{1bc9c}', GeneralCategory::OtherSymbol), ('\u{1bc9d}', '\u{1bc9e}',
        GeneralCategory::NonspacingMark), ('\u{1bc9f}', '\u{1bc9f}',
        GeneralCategory::OtherPunctuation), ('\u{1bca0}', '\u{1bca3}', GeneralCategory::Format),
        ('\u{1cf00}', '\u{1cf2d}', GeneralCategory::NonspacingMark), ('\u{1cf30}', '\u{1cf46}',
        GeneralCategory::NonspacingMark), ('\u{1cf50}', '\u{1cfc3}', GeneralCategory::OtherSymbol),
        ('\u{1d000}', '\u{1d0f5}', GeneralCategory::OtherSymbol), ('\u{1d100}', '\u{1d126}',
        GeneralCategory::OtherSymbol), ('\u{1d129}', '\u{1d164}', GeneralCategory::OtherSymbol),
        ('\u{1d165}', '\u{1d166}', GeneralCategory::SpacingMark), ('\u{1d167}', '\u{1d169}',
        GeneralCategory::NonspacingMark), ('\u{1d16a}', '\u{1d16c}', GeneralCategory::OtherSymbol),
        ('\u{1d16d}', '\u{1d172}', GeneralCategory::SpacingMark), ('\u{1d173}', '\u{1d17a}',
        GeneralCategory::Format), ('\u{1d17b}', '\u{1d182}', GeneralCategory::NonspacingMark),
        ('\u{1d183}', '\u{1d184}', GeneralCategory::OtherSymbol), ('\u{1d185}', '\u{1d18b}',
        GeneralCategory::NonspacingMark), ('\u{1d18c}', '\u{1d1a9}', GeneralCategory::OtherSymbol),
        ('\u{1d1aa}', '\u{1d1ad}', GeneralCategory::NonspacingMark), ('\u{1d1ae}', '\u{1d1ea}',
        GeneralCategory::OtherSymbol), ('\u{1d200}', '\u{1d241}', GeneralCategory::OtherSymbol),
        ('\u{1d242}', '\u{1d244}', GeneralCategory::NonspacingMark), ('\u{1d245}', '\u{1d245}',
        GeneralCategory::OtherSymbol), ('\u{1d2c0}', '\u{1d2d3}', GeneralCategory::OtherNumber),
        ('\u{1d2e0}', '\u{1d2f3}', GeneralCategory::OtherNumber), ('\u{1d300}', '\u{1d356}',
        GeneralCategory::OtherSymbol), ('\u{1d360}', '\u{1d378}', GeneralCategory::OtherNumber),
        ('\u{1d400}', '\u{1d419}', GeneralCategory::UppercaseLetter), ('\u{1d41a}', '\u{1d433}',
        GeneralCategory::LowercaseLetter), ('\u{1d434}', '\u{1d44d}',
        GeneralCategory::UppercaseLetter), ('\u{1d44e}', '\u{1d454}',
        GeneralCategory::LowercaseLetter), ('\u{1d456}', '\u{1d467}',
        GeneralCategory::LowercaseLetter), ('\u{1d468}', '\u{1d481}',
        GeneralCategory::UppercaseLetter), ('\u{1d482}', '\u{1d49b}',
        GeneralCategory::LowercaseLetter), ('\u{1d49c}', '\u{1d49c}',
        GeneralCategory::UppercaseLetter), ('\u{1d49e}', '\u{1d49f}',
        GeneralCategory::UppercaseLetter), ('\u{1d4a2}', '\u{1d4a2}',
        GeneralCategory::UppercaseLetter), ('\u{1d4a5}', '\u{1d4a6}',
        GeneralCategory::UppercaseLetter), ('\u{1d4a9}', '\u{1d4ac}',
        GeneralCategory::UppercaseLetter), ('\u{1d4ae}', '\u{1d4b5}',
        GeneralCategory::UppercaseLetter), ('\u{1d4b6}', '\u{1d4b9}',
        GeneralCategory::LowercaseLetter), ('\u{1d4bb}', '\u{1d4bb}',
        GeneralCategory::LowercaseLetter), ('\u{1d4bd}', '\u{1d4c3}',
        GeneralCategory::LowercaseLetter), ('\u{1d4c5}', '\u{1d4cf}',
        GeneralCategory::LowercaseLetter), ('\u{1d4d0}', '\u{1d4e9}',
        GeneralCategory::UppercaseLetter), ('\u{1d4ea}', '\u{1d503}',
        GeneralCategory::LowercaseLetter), ('\u{1d504}', '\u{1d505}',
        GeneralCategory::UppercaseLetter), ('\u{1d507}', '\u{1d50a}',
        GeneralCategory::UppercaseLetter), ('\u{1d50d}', '\u{1d514}',
        GeneralCategory::UppercaseLetter), ('\u{1d516}', '\u{1d51c}',
        GeneralCategory::UppercaseLetter), ('\u{1d51e}', '\u{1d537}',
        GeneralCategory::LowercaseLetter), ('\u{1d538}', '\u{1d539}',
        GeneralCategory::UppercaseLetter), ('\u{1d53b}', '\u{1d53e}',
        GeneralCategory::UppercaseLetter), ('\u{1d540}', '\u{1d544}',
        GeneralCategory::UppercaseLetter), ('\u{1d546}', '\u{1d546}',
        GeneralCategory::UppercaseLetter), ('\u{1d54a}', '\u{1d550}',
        GeneralCategory::UppercaseLetter), ('\u{1d552}', '\u{1d56b}',
        GeneralCategory::LowercaseLetter), ('\u{1d56c}', '\u{1d585}',
        GeneralCategory::UppercaseLetter), ('\u{1d586}', '\u{1d59f}',
        GeneralCategory::LowercaseLetter), ('\u{1d5a0}', '\u{1d5b9}',
        GeneralCategory::UppercaseLetter), ('\u{1d5ba}', '\u{1d5d3}',
        GeneralCategory::LowercaseLetter), ('\u{1d5d4}', '\u{1d5ed}',
        GeneralCategory::UppercaseLetter), ('\u{1d5ee}', '\u{1d607}',
        GeneralCategory::LowercaseLetter), ('\u{1d608}', '\u{1d621}',
        GeneralCategory::UppercaseLetter), ('\u{1d622}', '\u{1d63b}',
        GeneralCategory::LowercaseLetter), ('\u{1d63c}', '\u{1d655}',
        GeneralCategory::UppercaseLetter), ('\u{1d656}', '\u{1d66f}',
        GeneralCategory::LowercaseLetter), ('\u{1d670}', '\u{1d689}',
        GeneralCategory::UppercaseLetter), ('\u{1d68a}', '\u{1d6a5}',
        GeneralCategory::LowercaseLetter), ('\u{1d6a8}', '\u{1d6c0}',
        GeneralCategory::UppercaseLetter), ('\u{1d6c1}', '\u{1d6c1}', GeneralCategory::MathSymbol),
        ('\u{1d6c2}', '\u{1d6da}', GeneralCategory::LowercaseLetter), ('\u{1d6db}', '\u{1d6db}',
        GeneralCategory::MathSymbol), ('\u{1d6dc}', '\u{1d6e1}', GeneralCategory::LowercaseLetter),
        ('\u{1d6e2}', '\u{1d6fa}', GeneralCategory::UppercaseLetter), ('\u{1d6fb}', '\u{1d6fb}',
        GeneralCategory::MathSymbol), ('\u{1d6fc}', '\u{1d714}', GeneralCategory::LowercaseLetter),
        ('\u{1d715}', '\u{1d715}', GeneralCategory::MathSymbol), ('\u{1d716}', '\u{1d71b}',
        GeneralCategory::LowercaseLetter), ('\u{1d71c}', '\u{1d734}',
        GeneralCategory::UppercaseLetter), ('\u{1d735}', '\u{1d735}', GeneralCategory::MathSymbol),
        ('\u{1d736}', '\u{1d74e}', GeneralCategory::LowercaseLetter), ('\u{1d74f}', '\u{1d74f}',
        GeneralCategory::MathSymbol), ('\u{1d750}', '\u{1d755}', GeneralCategory::LowercaseLetter),
        ('\u{1d756}', '\u{1d76e}', GeneralCategory::UppercaseLetter), ('\u{1d76f}', '\u{1d76f}',
        GeneralCategory::MathSymbol), ('\u{1d770}', '\u{1d788}', GeneralCategory::LowercaseLetter),
        ('\u{1d789}', '\u{1d789}', GeneralCategory::MathSymbol), ('\u{1d78a}', '\u{1d78f}',
        GeneralCategory::LowercaseLetter), ('\u{1d790}', '\u{1d7a8}',
        GeneralCategory::UppercaseLetter), ('\u{1d7a9}', '\u{1d7a9}', GeneralCategory::MathSymbol),
        ('\u{1d7aa}', '\u{1d7c2}', GeneralCategory::LowercaseLetter), ('\u{1d7c3}', '\u{1d7c3}',
        GeneralCategory::MathSymbol), ('\u{1d7c4}', '\u{1d7c9}', GeneralCategory::LowercaseLetter),
        ('\u{1d7ca}', '\u{1d7ca}', GeneralCategory::UppercaseLetter), ('\u{1d7cb}', '\u{1d7cb}',
        GeneralCategory::LowercaseLetter), ('\u{1d7ce}', '\u{1d7ff}',
        GeneralCategory::DecimalNumber), ('\u{1d800}', '\u{1d9ff}', GeneralCategory::OtherSymbol),
        ('\u{1da00}', '\u{1da36}', GeneralCategory::NonspacingMark), ('\u{1da37}', '\u{1da3a}',
        GeneralCategory::OtherSymbol), ('\u{1da3b}', '\u{1da6c}', GeneralCategory::NonspacingMark),
        ('\u{1da6d}', '\u{1da74}', GeneralCategory::OtherSymbol), ('\u{1da75}', '\u{1da75}',
        GeneralCategory::NonspacingMark), ('\u{1da76}', '\u{1da83}', GeneralCategory::OtherSymbol),
        ('\u{1da84}', '\u{1da84}', GeneralCategory::NonspacingMark), ('\u{1da85}', '\u{1da86}',
        GeneralCategory::OtherSymbol), ('\u{1da87}', '\u{1da8b}',
        GeneralCategory::OtherPunctuation), ('\u{1da9b}', '\u{1da9f}',
        GeneralCategory::NonspacingMark), ('\u{1daa1}', '\u{1daaf}',
        GeneralCategory::NonspacingMark), ('\u{1df00}', '\u{1df09}',
        GeneralCategory::LowercaseLetter), ('\u{1df0a}', '\u{1df0a}', GeneralCategory::OtherLetter),
        ('\u{1df0b}', '\u{1df1e}', GeneralCategory::LowercaseLetter), ('\u{1df25}', '\u{1df2a}',
        GeneralCategory::LowercaseLetter), ('\u{1e000}', '\u{1e006}',
        GeneralCategory::NonspacingMark), ('\u{1e008}', '\u{1e018}',
        GeneralCategory::NonspacingMark), ('\u{1e01b}', '\u{1e021}',
        GeneralCategory::NonspacingMark), ('\u{1e023}', '\u{1e024}',
        GeneralCategory::NonspacingMark), ('\u{1e026}', '\u{1e02a}',
        GeneralCategory::NonspacingMark), ('\u{1e030}', '\u{1e06d}',
        GeneralCategory::ModifierLetter), ('\u{1e08f}', '\u{1e08f}',
        GeneralCategory::NonspacingMark), ('\u{1e100}', '\u{1e12c}', GeneralCategory::OtherLetter),
        ('\u{1e130}', '\u{1e136}', GeneralCategory::NonspacingMark), ('\u{1e137}', '\u{1e13d}',
        GeneralCategory::ModifierLetter), ('\u{1e140}', '\u{1e149}',
        GeneralCategory::DecimalNumber), ('\u{1e14e}', '\u{1e14e}', GeneralCategory::OtherLetter),
        ('\u{1e14f}', '\u{1e14f}', GeneralCategory::OtherSymbol), ('\u{1e290}', '\u{1e2ad}',
        GeneralCategory::OtherLetter), ('\u{1e2ae}', '\u{1e2ae}', GeneralCategory::NonspacingMark),
        ('\u{1e2c0}', '\u{1e2eb}', GeneralCategory::OtherLetter), ('\u{1e2ec}', '\u{1e2ef}',
        GeneralCategory::NonspacingMark), ('\u{1e2f0}', '\u{1e2f9}',
        GeneralCategory::DecimalNumber), ('\u{1e2ff}', '\u{1e2ff}',
        GeneralCategory::CurrencySymbol), ('\u{1e4d0}', '\u{1e4ea}', GeneralCategory::OtherLetter),
        ('\u{1e4eb}', '\u{1e4eb}', GeneralCategory::ModifierLetter), ('\u{1e4ec}', '\u{1e4ef}',
        GeneralCategory::NonspacingMark), ('\u{1e4f0}', '\u{1e4f9}',
        GeneralCategory::DecimalNumber), ('\u{1e7e0}', '\u{1e7e6}', GeneralCategory::OtherLetter),
        ('\u{1e7e8}', '\u{1e7eb}', GeneralCategory::OtherLetter), ('\u{1e7ed}', '\u{1e7ee}',
        GeneralCategory::OtherLetter), ('\u{1e7f0}', '\u{1e7fe}', GeneralCategory::OtherLetter),
        ('\u{1e800}', '\u{1e8c4}', GeneralCategory::OtherLetter), ('\u{1e8c7}', '\u{1e8cf}',
        GeneralCategory::OtherNumber), ('\u{1e8d0}', '\u{1e8d6}', GeneralCategory::NonspacingMark),
        ('\u{1e900}', '\u{1e921}', GeneralCategory::UppercaseLetter), ('\u{1e922}', '\u{1e943}',
        GeneralCategory::LowercaseLetter), ('\u{1e944}', '\u{1e94a}',
        GeneralCategory::NonspacingMark), ('\u{1e94b}', '\u{1e94b}',
        GeneralCategory::ModifierLetter), ('\u{1e950}', '\u{1e959}',
        GeneralCategory::DecimalNumber), ('\u{1e95e}', '\u{1e95f}',
        GeneralCategory::OtherPunctuation), ('\u{1ec71}', '\u{1ecab}',
        GeneralCategory::OtherNumber), ('\u{1ecac}', '\u{1ecac}', GeneralCategory::OtherSymbol),
        ('\u{1ecad}', '\u{1ecaf}', GeneralCategory::OtherNumber), ('\u{1ecb0}', '\u{1ecb0}',
        GeneralCategory::CurrencySymbol), ('\u{1ecb1}', '\u{1ecb4}', GeneralCategory::OtherNumber),
        ('\u{1ed01}', '\u{1ed2d}', GeneralCategory::OtherNumber), ('\u{1ed2e}', '\u{1ed2e}',
        GeneralCategory::OtherSymbol), ('\u{1ed2f}', '\u{1ed3d}', GeneralCategory::OtherNumber),
        ('\u{1ee00}', '\u{1ee03}', GeneralCategory::OtherLetter), ('\u{1ee05}', '\u{1ee1f}',
        GeneralCategory::OtherLetter), ('\u{1ee21}', '\u{1ee22}', GeneralCategory::OtherLetter),
        ('\u{1ee24}', '\u{1ee24}', GeneralCategory::OtherLetter), ('\u{1ee27}', '\u{1ee27}',
        GeneralCategory::OtherLetter), ('\u{1ee29}', '\u{1ee32}', GeneralCategory::OtherLetter),
        ('\u{1ee34}', '\u{1ee37}', GeneralCategory::OtherLetter), ('\u{1ee39}', '\u{1ee39}',
        GeneralCategory::OtherLetter), ('\u{1ee3b}', '\u{1ee3b}', GeneralCategory::OtherLetter),
        ('\u{1ee42}', '\u{1ee42}', GeneralCategory::OtherLetter), ('\u{1ee47}', '\u{1ee47}',
        GeneralCategory::OtherLetter), ('\u{1ee49}', '\u{1ee49}', GeneralCategory::OtherLetter),
        ('\u{1ee4b}', '\u{1ee4b}', GeneralCategory::OtherLetter), ('\u{1ee4d}', '\u{1ee4f}',
        GeneralCategory::OtherLetter), ('\u{1ee51}', '\u{1ee52}', GeneralCategory::OtherLetter),
        ('\u{1ee54}', '\u{1ee54}', GeneralCategory::OtherLetter), ('\u{1ee57}', '\u{1ee57}',
        GeneralCategory::OtherLetter), ('\u{1ee59}', '\u{1ee59}', GeneralCategory::OtherLetter),
        ('\u{1ee5b}', '\u{1ee5b}', GeneralCategory::OtherLetter), ('\u{1ee5d}', '\u{1ee5d}',
        GeneralCategory::OtherLetter), ('\u{1ee5f}', '\u{1ee5f}', GeneralCategory::OtherLetter),
        ('\u{1ee61}', '\u{1ee62}', GeneralCategory::OtherLetter), ('\u{1ee64}', '\u{1ee64}',
        GeneralCategory::OtherLetter), ('\u{1ee67}', '\u{1ee6a}', GeneralCategory::OtherLetter),
        ('\u{1ee6c}', '\u{1ee72}', GeneralCategory::OtherLetter), ('\u{1ee74}', '\u{1ee77}',
        GeneralCategory::OtherLetter), ('\u{1ee79}', '\u{1ee7c}', GeneralCategory::OtherLetter),
        ('\u{1ee7e}', '\u{1ee7e}', GeneralCategory::OtherLetter), ('\u{1ee80}', '\u{1ee89}',
        GeneralCategory::OtherLetter), ('\u{1ee8b}', '\u{1ee9b}', GeneralCategory::OtherLetter),
        ('\u{1eea1}', '\u{1eea3}', GeneralCategory::OtherLetter), ('\u{1eea5}', '\u{1eea9}',
        GeneralCategory::OtherLetter), ('\u{1eeab}', '\u{1eebb}', GeneralCategory::OtherLetter),
        ('\u{1eef0}', '\u{1eef1}', GeneralCategory::MathSymbol), ('\u{1f000}', '\u{1f02b}',
        GeneralCategory::OtherSymbol), ('\u{1f030}', '\u{1f093}', GeneralCategory::OtherSymbol),
        ('\u{1f0a0}', '\u{1f0ae}', GeneralCategory::OtherSymbol), ('\u{1f0b1}', '\u{1f0bf}',
        GeneralCategory::OtherSymbol), ('\u{1f0c1}', '\u{1f0cf}', GeneralCategory::OtherSymbol),
        ('\u{1f0d1}', '\u{1f0f5}', GeneralCategory::OtherSymbol), ('\u{1f100}', '\u{1f10c}',
        GeneralCategory::OtherNumber), ('\u{1f10d}', '\u{1f1ad}', GeneralCategory::OtherSymbol),
        ('\u{1f1e6}', '\u{1f202}', GeneralCategory::OtherSymbol), ('\u{1f210}', '\u{1f23b}',
        GeneralCategory::OtherSymbol), ('\u{1f240}', '\u{1f248}', GeneralCategory::OtherSymbol),
        ('\u{1f250}', '\u{1f251}', GeneralCategory::OtherSymbol), ('\u{1f260}', '\u{1f265}',
        GeneralCategory::OtherSymbol), ('\u{1f300}', '\u{1f3fa}', GeneralCategory::OtherSymbol),
        ('\u{1f3fb}', '\u{1f3ff}', GeneralCategory::ModifierSymbol), ('\u{1f400}', '\u{1f6d7}',
        GeneralCategory::OtherSymbol), ('\u{1f6dc}', '\u{1f6ec}', GeneralCategory::OtherSymbol),
        ('\u{1f6f0}', '\u{1f6fc}', GeneralCategory::OtherSymbol), ('\u{1f700}', '\u{1f776}',
        GeneralCategory::OtherSymbol), ('\u{1f77b}', '\u{1f7d9}', GeneralCategory::OtherSymbol),
        ('\u{1f7e0}', '\u{1f7eb}', GeneralCategory::OtherSymbol), ('\u{1f7f0}', '\u{1f7f0}',
        GeneralCategory::OtherSymbol), ('\u{1f800}', '\u{1f80b}', GeneralCategory::OtherSymbol),
        ('\u{1f810}', '\u{1f847}', GeneralCategory::OtherSymbol), ('\u{1f850}', '\u{1f859}',
        GeneralCategory::OtherSymbol), ('\u{1f860}', '\u{1f887}', GeneralCategory::OtherSymbol),
        ('\u{1f890}', '\u{1f8ad}', GeneralCategory::OtherSymbol), ('\u{1f8b0}', '\u{1f8b1}',
        GeneralCategory::OtherSymbol), ('\u{1f900}', '\u{1fa53}', GeneralCategory::OtherSymbol),
        ('\u{1fa60}', '\u{1fa6d}', GeneralCategory::OtherSymbol), ('\u{1fa70}', '\u{1fa7c}',
        GeneralCategory::OtherSymbol), ('\u{1fa80}', '\u{1fa88}', GeneralCategory::OtherSymbol),
        ('\u{1fa90}', '\u{1fabd}', GeneralCategory::OtherSymbol), ('\u{1fabf}', '\u{1fac5}',
        GeneralCategory::OtherSymbol), ('\u{1face}', '\u{1fadb}', GeneralCategory::OtherSymbol),
        ('\u{1fae0}', '\u{1fae8}', GeneralCategory::OtherSymbol), ('\u{1faf0}', '\u{1faf8}',
        GeneralCategory::OtherSymbol), ('\u{1fb00}', '\u{1fb92}', GeneralCategory::OtherSymbol),
        ('\u{1fb94}', '\u{1fbca}', GeneralCategory::OtherSymbol), ('\u{1fbf0}', '\u{1fbf9}',
        GeneralCategory::DecimalNumber), ('\u{20000}', '\u{2a6df}', GeneralCategory::OtherLetter),
        ('\u{2a700}', '\u{2b739}', GeneralCategory::OtherLetter), ('\u{2b740}', '\u{2b81d}',
        GeneralCategory::OtherLetter), ('\u{2b820}', '\u{2cea1}', GeneralCategory::OtherLetter),
        ('\u{2ceb0}', '\u{2ebe0}', GeneralCategory::OtherLetter), ('\u{2ebf0}', '\u{2ee5d}',
        GeneralCategory::OtherLetter), ('\u{2f800}', '\u{2fa1d}', GeneralCategory::OtherLetter),
        ('\u{30000}', '\u{3134a}', GeneralCategory::OtherLetter), ('\u{31350}', '\u{323af}',
        GeneralCategory::OtherLetter), ('\u{e0001}', '\u{e0001}', GeneralCategory::Format),
        ('\u{e0020}', '\u{e007f}', GeneralCategory::Format), ('\u{e0100}', '\u{e01ef}',
        GeneralCategory::NonspacingMark), ('\u{f0000}', '\u{ffffd}', GeneralCategory::PrivateUse),
        ('\u{100000}', '\u{10fffd}', GeneralCategory::PrivateUse)
    ];

}

#[cfg(feature = "emoji")]
pub mod emoji {

    #[derive(Copy, Clone, Hash, Eq, PartialEq, Ord, PartialOrd, Debug)]
    #[non_exhaustive]
    /// The emoji character properties of a character.
    pub enum EmojiStatus {
        /// `Emoji=NO`, `Emoji_Component=NO`
        NonEmoji,
        /// `Emoji=NO`, `Emoji_Component=YES`
        NonEmojiButEmojiComponent,
        /// `Emoji=YES`, `Emoji_Component=NO`;`Emoji_Presentation=YES`
        EmojiPresentation,
        /// `Emoji=YES`, `Emoji_Component=NO`;`Emoji_Modifier_Base=YES`
        EmojiModifierBase,
        /// `Emoji=YES`, `Emoji_Component=NO`;`Emoji_Presentation=YES`, `Emoji_Modifier_Base=YES`
        EmojiPresentationAndModifierBase,
        /// `Emoji=YES`, `Emoji_Component=NO`
        EmojiOther,
        /// `Emoji=YES`, `Emoji_Component=YES`;`Emoji_Presentation=YES`
        EmojiPresentationAndEmojiComponent,
        /// `Emoji=YES`, `Emoji_Component=YES`;`Emoji_Presentation=YES`, `Emoji_Modifier=YES`
        EmojiPresentationAndModifierAndEmojiComponent,
        /// `Emoji=YES`, `Emoji_Component=YES`
        EmojiOtherAndEmojiComponent,
    }
    #[inline]
    pub(crate) fn emoji_status(c: char) -> EmojiStatus {
        // FIXME: do we want to special case ASCII here?
        super::util::bsearch_range_value_table(c, EMOJI_STATUS).unwrap()
    }
    #[inline]
    pub(crate) fn is_emoji_status_for_emoji_char_or_emoji_component(s: EmojiStatus) -> bool {
        !matches!(s, EmojiStatus::NonEmoji)
    }
    #[inline]
    pub(crate) fn is_emoji_status_for_emoji_char(s: EmojiStatus) -> bool {
        !matches!(s, EmojiStatus::NonEmoji | EmojiStatus::NonEmojiButEmojiComponent)
    }
    #[inline]
    pub(crate) fn is_emoji_status_for_emoji_component(s: EmojiStatus) -> bool {
        matches!(s, EmojiStatus::EmojiPresentationAndEmojiComponent |
            EmojiStatus::EmojiPresentationAndModifierAndEmojiComponent |
            EmojiStatus::EmojiOtherAndEmojiComponent)
    }
    // Emoji status table:
    const EMOJI_STATUS: &[(char, char, EmojiStatus)] = &[
        ('\u{0}', '\u{22}', EmojiStatus::NonEmoji), ('\u{23}', '\u{23}',
        EmojiStatus::EmojiOtherAndEmojiComponent), ('\u{24}', '\u{29}', EmojiStatus::NonEmoji),
        ('\u{2a}', '\u{2a}', EmojiStatus::EmojiOtherAndEmojiComponent), ('\u{2b}', '\u{2f}',
        EmojiStatus::NonEmoji), ('\u{30}', '\u{39}', EmojiStatus::EmojiOtherAndEmojiComponent),
        ('\u{3a}', '\u{a8}', EmojiStatus::NonEmoji), ('\u{a9}', '\u{a9}', EmojiStatus::EmojiOther),
        ('\u{aa}', '\u{ad}', EmojiStatus::NonEmoji), ('\u{ae}', '\u{ae}', EmojiStatus::EmojiOther),
        ('\u{af}', '\u{200c}', EmojiStatus::NonEmoji), ('\u{200d}', '\u{200d}',
        EmojiStatus::NonEmojiButEmojiComponent), ('\u{200e}', '\u{203b}', EmojiStatus::NonEmoji),
        ('\u{203c}', '\u{203c}', EmojiStatus::EmojiOther), ('\u{203d}', '\u{2048}',
        EmojiStatus::NonEmoji), ('\u{2049}', '\u{2049}', EmojiStatus::EmojiOther), ('\u{204a}',
        '\u{20e2}', EmojiStatus::NonEmoji), ('\u{20e3}', '\u{20e3}',
        EmojiStatus::NonEmojiButEmojiComponent), ('\u{20e4}', '\u{2121}', EmojiStatus::NonEmoji),
        ('\u{2122}', '\u{2122}', EmojiStatus::EmojiOther), ('\u{2123}', '\u{2138}',
        EmojiStatus::NonEmoji), ('\u{2139}', '\u{2139}', EmojiStatus::EmojiOther), ('\u{213a}',
        '\u{2193}', EmojiStatus::NonEmoji), ('\u{2194}', '\u{2199}', EmojiStatus::EmojiOther),
        ('\u{219a}', '\u{21a8}', EmojiStatus::NonEmoji), ('\u{21a9}', '\u{21aa}',
        EmojiStatus::EmojiOther), ('\u{21ab}', '\u{2319}', EmojiStatus::NonEmoji), ('\u{231a}',
        '\u{231b}', EmojiStatus::EmojiPresentation), ('\u{231c}', '\u{2327}',
        EmojiStatus::NonEmoji), ('\u{2328}', '\u{2328}', EmojiStatus::EmojiOther), ('\u{2329}',
        '\u{23ce}', EmojiStatus::NonEmoji), ('\u{23cf}', '\u{23cf}', EmojiStatus::EmojiOther),
        ('\u{23d0}', '\u{23e8}', EmojiStatus::NonEmoji), ('\u{23e9}', '\u{23ec}',
        EmojiStatus::EmojiPresentation), ('\u{23ed}', '\u{23ee}', EmojiStatus::EmojiOther),
        ('\u{23ef}', '\u{23ef}', EmojiStatus::EmojiOther), ('\u{23f0}', '\u{23f0}',
        EmojiStatus::EmojiPresentation), ('\u{23f1}', '\u{23f2}', EmojiStatus::EmojiOther),
        ('\u{23f3}', '\u{23f3}', EmojiStatus::EmojiPresentation), ('\u{23f4}', '\u{23f7}',
        EmojiStatus::NonEmoji), ('\u{23f8}', '\u{23fa}', EmojiStatus::EmojiOther), ('\u{23fb}',
        '\u{24c1}', EmojiStatus::NonEmoji), ('\u{24c2}', '\u{24c2}', EmojiStatus::EmojiOther),
        ('\u{24c3}', '\u{25a9}', EmojiStatus::NonEmoji), ('\u{25aa}', '\u{25ab}',
        EmojiStatus::EmojiOther), ('\u{25ac}', '\u{25b5}', EmojiStatus::NonEmoji), ('\u{25b6}',
        '\u{25b6}', EmojiStatus::EmojiOther), ('\u{25b7}', '\u{25bf}', EmojiStatus::NonEmoji),
        ('\u{25c0}', '\u{25c0}', EmojiStatus::EmojiOther), ('\u{25c1}', '\u{25fa}',
        EmojiStatus::NonEmoji), ('\u{25fb}', '\u{25fc}', EmojiStatus::EmojiOther), ('\u{25fd}',
        '\u{25fe}', EmojiStatus::EmojiPresentation), ('\u{25ff}', '\u{25ff}',
        EmojiStatus::NonEmoji), ('\u{2600}', '\u{2601}', EmojiStatus::EmojiOther), ('\u{2602}',
        '\u{2603}', EmojiStatus::EmojiOther), ('\u{2604}', '\u{2604}', EmojiStatus::EmojiOther),
        ('\u{2605}', '\u{260d}', EmojiStatus::NonEmoji), ('\u{260e}', '\u{260e}',
        EmojiStatus::EmojiOther), ('\u{260f}', '\u{2610}', EmojiStatus::NonEmoji), ('\u{2611}',
        '\u{2611}', EmojiStatus::EmojiOther), ('\u{2612}', '\u{2613}', EmojiStatus::NonEmoji),
        ('\u{2614}', '\u{2615}', EmojiStatus::EmojiPresentation), ('\u{2616}', '\u{2617}',
        EmojiStatus::NonEmoji), ('\u{2618}', '\u{2618}', EmojiStatus::EmojiOther), ('\u{2619}',
        '\u{261c}', EmojiStatus::NonEmoji), ('\u{261d}', '\u{261d}',
        EmojiStatus::EmojiModifierBase), ('\u{261e}', '\u{261f}', EmojiStatus::NonEmoji),
        ('\u{2620}', '\u{2620}', EmojiStatus::EmojiOther), ('\u{2621}', '\u{2621}',
        EmojiStatus::NonEmoji), ('\u{2622}', '\u{2623}', EmojiStatus::EmojiOther), ('\u{2624}',
        '\u{2625}', EmojiStatus::NonEmoji), ('\u{2626}', '\u{2626}', EmojiStatus::EmojiOther),
        ('\u{2627}', '\u{2629}', EmojiStatus::NonEmoji), ('\u{262a}', '\u{262a}',
        EmojiStatus::EmojiOther), ('\u{262b}', '\u{262d}', EmojiStatus::NonEmoji), ('\u{262e}',
        '\u{262e}', EmojiStatus::EmojiOther), ('\u{262f}', '\u{262f}', EmojiStatus::EmojiOther),
        ('\u{2630}', '\u{2637}', EmojiStatus::NonEmoji), ('\u{2638}', '\u{2639}',
        EmojiStatus::EmojiOther), ('\u{263a}', '\u{263a}', EmojiStatus::EmojiOther), ('\u{263b}',
        '\u{263f}', EmojiStatus::NonEmoji), ('\u{2640}', '\u{2640}', EmojiStatus::EmojiOther),
        ('\u{2641}', '\u{2641}', EmojiStatus::NonEmoji), ('\u{2642}', '\u{2642}',
        EmojiStatus::EmojiOther), ('\u{2643}', '\u{2647}', EmojiStatus::NonEmoji), ('\u{2648}',
        '\u{2653}', EmojiStatus::EmojiPresentation), ('\u{2654}', '\u{265e}',
        EmojiStatus::NonEmoji), ('\u{265f}', '\u{265f}', EmojiStatus::EmojiOther), ('\u{2660}',
        '\u{2660}', EmojiStatus::EmojiOther), ('\u{2661}', '\u{2662}', EmojiStatus::NonEmoji),
        ('\u{2663}', '\u{2663}', EmojiStatus::EmojiOther), ('\u{2664}', '\u{2664}',
        EmojiStatus::NonEmoji), ('\u{2665}', '\u{2666}', EmojiStatus::EmojiOther), ('\u{2667}',
        '\u{2667}', EmojiStatus::NonEmoji), ('\u{2668}', '\u{2668}', EmojiStatus::EmojiOther),
        ('\u{2669}', '\u{267a}', EmojiStatus::NonEmoji), ('\u{267b}', '\u{267b}',
        EmojiStatus::EmojiOther), ('\u{267c}', '\u{267d}', EmojiStatus::NonEmoji), ('\u{267e}',
        '\u{267e}', EmojiStatus::EmojiOther), ('\u{267f}', '\u{267f}',
        EmojiStatus::EmojiPresentation), ('\u{2680}', '\u{2691}', EmojiStatus::NonEmoji),
        ('\u{2692}', '\u{2692}', EmojiStatus::EmojiOther), ('\u{2693}', '\u{2693}',
        EmojiStatus::EmojiPresentation), ('\u{2694}', '\u{2694}', EmojiStatus::EmojiOther),
        ('\u{2695}', '\u{2695}', EmojiStatus::EmojiOther), ('\u{2696}', '\u{2697}',
        EmojiStatus::EmojiOther), ('\u{2698}', '\u{2698}', EmojiStatus::NonEmoji), ('\u{2699}',
        '\u{2699}', EmojiStatus::EmojiOther), ('\u{269a}', '\u{269a}', EmojiStatus::NonEmoji),
        ('\u{269b}', '\u{269c}', EmojiStatus::EmojiOther), ('\u{269d}', '\u{269f}',
        EmojiStatus::NonEmoji), ('\u{26a0}', '\u{26a0}', EmojiStatus::EmojiOther), ('\u{26a1}',
        '\u{26a1}', EmojiStatus::EmojiPresentation), ('\u{26a2}', '\u{26a6}',
        EmojiStatus::NonEmoji), ('\u{26a7}', '\u{26a7}', EmojiStatus::EmojiOther), ('\u{26a8}',
        '\u{26a9}', EmojiStatus::NonEmoji), ('\u{26aa}', '\u{26ab}',
        EmojiStatus::EmojiPresentation), ('\u{26ac}', '\u{26af}', EmojiStatus::NonEmoji),
        ('\u{26b0}', '\u{26b1}', EmojiStatus::EmojiOther), ('\u{26b2}', '\u{26bc}',
        EmojiStatus::NonEmoji), ('\u{26bd}', '\u{26be}', EmojiStatus::EmojiPresentation),
        ('\u{26bf}', '\u{26c3}', EmojiStatus::NonEmoji), ('\u{26c4}', '\u{26c5}',
        EmojiStatus::EmojiPresentation), ('\u{26c6}', '\u{26c7}', EmojiStatus::NonEmoji),
        ('\u{26c8}', '\u{26c8}', EmojiStatus::EmojiOther), ('\u{26c9}', '\u{26cd}',
        EmojiStatus::NonEmoji), ('\u{26ce}', '\u{26ce}', EmojiStatus::EmojiPresentation),
        ('\u{26cf}', '\u{26cf}', EmojiStatus::EmojiOther), ('\u{26d0}', '\u{26d0}',
        EmojiStatus::NonEmoji), ('\u{26d1}', '\u{26d1}', EmojiStatus::EmojiOther), ('\u{26d2}',
        '\u{26d2}', EmojiStatus::NonEmoji), ('\u{26d3}', '\u{26d3}', EmojiStatus::EmojiOther),
        ('\u{26d4}', '\u{26d4}', EmojiStatus::EmojiPresentation), ('\u{26d5}', '\u{26e8}',
        EmojiStatus::NonEmoji), ('\u{26e9}', '\u{26e9}', EmojiStatus::EmojiOther), ('\u{26ea}',
        '\u{26ea}', EmojiStatus::EmojiPresentation), ('\u{26eb}', '\u{26ef}',
        EmojiStatus::NonEmoji), ('\u{26f0}', '\u{26f1}', EmojiStatus::EmojiOther), ('\u{26f2}',
        '\u{26f3}', EmojiStatus::EmojiPresentation), ('\u{26f4}', '\u{26f4}',
        EmojiStatus::EmojiOther), ('\u{26f5}', '\u{26f5}', EmojiStatus::EmojiPresentation),
        ('\u{26f6}', '\u{26f6}', EmojiStatus::NonEmoji), ('\u{26f7}', '\u{26f8}',
        EmojiStatus::EmojiOther), ('\u{26f9}', '\u{26f9}', EmojiStatus::EmojiModifierBase),
        ('\u{26fa}', '\u{26fa}', EmojiStatus::EmojiPresentation), ('\u{26fb}', '\u{26fc}',
        EmojiStatus::NonEmoji), ('\u{26fd}', '\u{26fd}', EmojiStatus::EmojiPresentation),
        ('\u{26fe}', '\u{2701}', EmojiStatus::NonEmoji), ('\u{2702}', '\u{2702}',
        EmojiStatus::EmojiOther), ('\u{2703}', '\u{2704}', EmojiStatus::NonEmoji), ('\u{2705}',
        '\u{2705}', EmojiStatus::EmojiPresentation), ('\u{2706}', '\u{2707}',
        EmojiStatus::NonEmoji), ('\u{2708}', '\u{2709}', EmojiStatus::EmojiOther), ('\u{270a}',
        '\u{270b}', EmojiStatus::EmojiPresentationAndModifierBase), ('\u{270c}', '\u{270c}',
        EmojiStatus::EmojiModifierBase), ('\u{270d}', '\u{270d}', EmojiStatus::EmojiModifierBase),
        ('\u{270e}', '\u{270e}', EmojiStatus::NonEmoji), ('\u{270f}', '\u{270f}',
        EmojiStatus::EmojiOther), ('\u{2710}', '\u{2711}', EmojiStatus::NonEmoji), ('\u{2712}',
        '\u{2712}', EmojiStatus::EmojiOther), ('\u{2713}', '\u{2713}', EmojiStatus::NonEmoji),
        ('\u{2714}', '\u{2714}', EmojiStatus::EmojiOther), ('\u{2715}', '\u{2715}',
        EmojiStatus::NonEmoji), ('\u{2716}', '\u{2716}', EmojiStatus::EmojiOther), ('\u{2717}',
        '\u{271c}', EmojiStatus::NonEmoji), ('\u{271d}', '\u{271d}', EmojiStatus::EmojiOther),
        ('\u{271e}', '\u{2720}', EmojiStatus::NonEmoji), ('\u{2721}', '\u{2721}',
        EmojiStatus::EmojiOther), ('\u{2722}', '\u{2727}', EmojiStatus::NonEmoji), ('\u{2728}',
        '\u{2728}', EmojiStatus::EmojiPresentation), ('\u{2729}', '\u{2732}',
        EmojiStatus::NonEmoji), ('\u{2733}', '\u{2734}', EmojiStatus::EmojiOther), ('\u{2735}',
        '\u{2743}', EmojiStatus::NonEmoji), ('\u{2744}', '\u{2744}', EmojiStatus::EmojiOther),
        ('\u{2745}', '\u{2746}', EmojiStatus::NonEmoji), ('\u{2747}', '\u{2747}',
        EmojiStatus::EmojiOther), ('\u{2748}', '\u{274b}', EmojiStatus::NonEmoji), ('\u{274c}',
        '\u{274c}', EmojiStatus::EmojiPresentation), ('\u{274d}', '\u{274d}',
        EmojiStatus::NonEmoji), ('\u{274e}', '\u{274e}', EmojiStatus::EmojiPresentation),
        ('\u{274f}', '\u{2752}', EmojiStatus::NonEmoji), ('\u{2753}', '\u{2755}',
        EmojiStatus::EmojiPresentation), ('\u{2756}', '\u{2756}', EmojiStatus::NonEmoji),
        ('\u{2757}', '\u{2757}', EmojiStatus::EmojiPresentation), ('\u{2758}', '\u{2762}',
        EmojiStatus::NonEmoji), ('\u{2763}', '\u{2763}', EmojiStatus::EmojiOther), ('\u{2764}',
        '\u{2764}', EmojiStatus::EmojiOther), ('\u{2765}', '\u{2794}', EmojiStatus::NonEmoji),
        ('\u{2795}', '\u{2797}', EmojiStatus::EmojiPresentation), ('\u{2798}', '\u{27a0}',
        EmojiStatus::NonEmoji), ('\u{27a1}', '\u{27a1}', EmojiStatus::EmojiOther), ('\u{27a2}',
        '\u{27af}', EmojiStatus::NonEmoji), ('\u{27b0}', '\u{27b0}',
        EmojiStatus::EmojiPresentation), ('\u{27b1}', '\u{27be}', EmojiStatus::NonEmoji),
        ('\u{27bf}', '\u{27bf}', EmojiStatus::EmojiPresentation), ('\u{27c0}', '\u{2933}',
        EmojiStatus::NonEmoji), ('\u{2934}', '\u{2935}', EmojiStatus::EmojiOther), ('\u{2936}',
        '\u{2b04}', EmojiStatus::NonEmoji), ('\u{2b05}', '\u{2b07}', EmojiStatus::EmojiOther),
        ('\u{2b08}', '\u{2b1a}', EmojiStatus::NonEmoji), ('\u{2b1b}', '\u{2b1c}',
        EmojiStatus::EmojiPresentation), ('\u{2b1d}', '\u{2b4f}', EmojiStatus::NonEmoji),
        ('\u{2b50}', '\u{2b50}', EmojiStatus::EmojiPresentation), ('\u{2b51}', '\u{2b54}',
        EmojiStatus::NonEmoji), ('\u{2b55}', '\u{2b55}', EmojiStatus::EmojiPresentation),
        ('\u{2b56}', '\u{302f}', EmojiStatus::NonEmoji), ('\u{3030}', '\u{3030}',
        EmojiStatus::EmojiOther), ('\u{3031}', '\u{303c}', EmojiStatus::NonEmoji), ('\u{303d}',
        '\u{303d}', EmojiStatus::EmojiOther), ('\u{303e}', '\u{3296}', EmojiStatus::NonEmoji),
        ('\u{3297}', '\u{3297}', EmojiStatus::EmojiOther), ('\u{3298}', '\u{3298}',
        EmojiStatus::NonEmoji), ('\u{3299}', '\u{3299}', EmojiStatus::EmojiOther), ('\u{329a}',
        '\u{d7ff}', EmojiStatus::NonEmoji), ('\u{e000}', '\u{fe0e}', EmojiStatus::NonEmoji),
        ('\u{fe0f}', '\u{fe0f}', EmojiStatus::NonEmojiButEmojiComponent), ('\u{fe10}', '\u{1f003}',
        EmojiStatus::NonEmoji), ('\u{1f004}', '\u{1f004}', EmojiStatus::EmojiPresentation),
        ('\u{1f005}', '\u{1f0ce}', EmojiStatus::NonEmoji), ('\u{1f0cf}', '\u{1f0cf}',
        EmojiStatus::EmojiPresentation), ('\u{1f0d0}', '\u{1f16f}', EmojiStatus::NonEmoji),
        ('\u{1f170}', '\u{1f171}', EmojiStatus::EmojiOther), ('\u{1f172}', '\u{1f17d}',
        EmojiStatus::NonEmoji), ('\u{1f17e}', '\u{1f17f}', EmojiStatus::EmojiOther), ('\u{1f180}',
        '\u{1f18d}', EmojiStatus::NonEmoji), ('\u{1f18e}', '\u{1f18e}',
        EmojiStatus::EmojiPresentation), ('\u{1f18f}', '\u{1f190}', EmojiStatus::NonEmoji),
        ('\u{1f191}', '\u{1f19a}', EmojiStatus::EmojiPresentation), ('\u{1f19b}', '\u{1f1e5}',
        EmojiStatus::NonEmoji), ('\u{1f1e6}', '\u{1f1ff}',
        EmojiStatus::EmojiPresentationAndEmojiComponent), ('\u{1f200}', '\u{1f200}',
        EmojiStatus::NonEmoji), ('\u{1f201}', '\u{1f201}', EmojiStatus::EmojiPresentation),
        ('\u{1f202}', '\u{1f202}', EmojiStatus::EmojiOther), ('\u{1f203}', '\u{1f219}',
        EmojiStatus::NonEmoji), ('\u{1f21a}', '\u{1f21a}', EmojiStatus::EmojiPresentation),
        ('\u{1f21b}', '\u{1f22e}', EmojiStatus::NonEmoji), ('\u{1f22f}', '\u{1f22f}',
        EmojiStatus::EmojiPresentation), ('\u{1f230}', '\u{1f231}', EmojiStatus::NonEmoji),
        ('\u{1f232}', '\u{1f236}', EmojiStatus::EmojiPresentation), ('\u{1f237}', '\u{1f237}',
        EmojiStatus::EmojiOther), ('\u{1f238}', '\u{1f23a}', EmojiStatus::EmojiPresentation),
        ('\u{1f23b}', '\u{1f24f}', EmojiStatus::NonEmoji), ('\u{1f250}', '\u{1f251}',
        EmojiStatus::EmojiPresentation), ('\u{1f252}', '\u{1f2ff}', EmojiStatus::NonEmoji),
        ('\u{1f300}', '\u{1f30c}', EmojiStatus::EmojiPresentation), ('\u{1f30d}', '\u{1f30e}',
        EmojiStatus::EmojiPresentation), ('\u{1f30f}', '\u{1f30f}', EmojiStatus::EmojiPresentation),
        ('\u{1f310}', '\u{1f310}', EmojiStatus::EmojiPresentation), ('\u{1f311}', '\u{1f311}',
        EmojiStatus::EmojiPresentation), ('\u{1f312}', '\u{1f312}', EmojiStatus::EmojiPresentation),
        ('\u{1f313}', '\u{1f315}', EmojiStatus::EmojiPresentation), ('\u{1f316}', '\u{1f318}',
        EmojiStatus::EmojiPresentation), ('\u{1f319}', '\u{1f319}', EmojiStatus::EmojiPresentation),
        ('\u{1f31a}', '\u{1f31a}', EmojiStatus::EmojiPresentation), ('\u{1f31b}', '\u{1f31b}',
        EmojiStatus::EmojiPresentation), ('\u{1f31c}', '\u{1f31c}', EmojiStatus::EmojiPresentation),
        ('\u{1f31d}', '\u{1f31e}', EmojiStatus::EmojiPresentation), ('\u{1f31f}', '\u{1f320}',
        EmojiStatus::EmojiPresentation), ('\u{1f321}', '\u{1f321}', EmojiStatus::EmojiOther),
        ('\u{1f322}', '\u{1f323}', EmojiStatus::NonEmoji), ('\u{1f324}', '\u{1f32c}',
        EmojiStatus::EmojiOther), ('\u{1f32d}', '\u{1f32f}', EmojiStatus::EmojiPresentation),
        ('\u{1f330}', '\u{1f331}', EmojiStatus::EmojiPresentation), ('\u{1f332}', '\u{1f333}',
        EmojiStatus::EmojiPresentation), ('\u{1f334}', '\u{1f335}', EmojiStatus::EmojiPresentation),
        ('\u{1f336}', '\u{1f336}', EmojiStatus::EmojiOther), ('\u{1f337}', '\u{1f34a}',
        EmojiStatus::EmojiPresentation), ('\u{1f34b}', '\u{1f34b}', EmojiStatus::EmojiPresentation),
        ('\u{1f34c}', '\u{1f34f}', EmojiStatus::EmojiPresentation), ('\u{1f350}', '\u{1f350}',
        EmojiStatus::EmojiPresentation), ('\u{1f351}', '\u{1f37b}', EmojiStatus::EmojiPresentation),
        ('\u{1f37c}', '\u{1f37c}', EmojiStatus::EmojiPresentation), ('\u{1f37d}', '\u{1f37d}',
        EmojiStatus::EmojiOther), ('\u{1f37e}', '\u{1f37f}', EmojiStatus::EmojiPresentation),
        ('\u{1f380}', '\u{1f384}', EmojiStatus::EmojiPresentation), ('\u{1f385}', '\u{1f385}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f386}', '\u{1f393}',
        EmojiStatus::EmojiPresentation), ('\u{1f394}', '\u{1f395}', EmojiStatus::NonEmoji),
        ('\u{1f396}', '\u{1f397}', EmojiStatus::EmojiOther), ('\u{1f398}', '\u{1f398}',
        EmojiStatus::NonEmoji), ('\u{1f399}', '\u{1f39b}', EmojiStatus::EmojiOther), ('\u{1f39c}',
        '\u{1f39d}', EmojiStatus::NonEmoji), ('\u{1f39e}', '\u{1f39f}', EmojiStatus::EmojiOther),
        ('\u{1f3a0}', '\u{1f3c1}', EmojiStatus::EmojiPresentation), ('\u{1f3c2}', '\u{1f3c4}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f3c5}', '\u{1f3c5}',
        EmojiStatus::EmojiPresentation), ('\u{1f3c6}', '\u{1f3c6}', EmojiStatus::EmojiPresentation),
        ('\u{1f3c7}', '\u{1f3c7}', EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f3c8}',
        '\u{1f3c8}', EmojiStatus::EmojiPresentation), ('\u{1f3c9}', '\u{1f3c9}',
        EmojiStatus::EmojiPresentation), ('\u{1f3ca}', '\u{1f3ca}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f3cb}', '\u{1f3cc}',
        EmojiStatus::EmojiModifierBase), ('\u{1f3cd}', '\u{1f3ce}', EmojiStatus::EmojiOther),
        ('\u{1f3cf}', '\u{1f3d3}', EmojiStatus::EmojiPresentation), ('\u{1f3d4}', '\u{1f3df}',
        EmojiStatus::EmojiOther), ('\u{1f3e0}', '\u{1f3e3}', EmojiStatus::EmojiPresentation),
        ('\u{1f3e4}', '\u{1f3e4}', EmojiStatus::EmojiPresentation), ('\u{1f3e5}', '\u{1f3f0}',
        EmojiStatus::EmojiPresentation), ('\u{1f3f1}', '\u{1f3f2}', EmojiStatus::NonEmoji),
        ('\u{1f3f3}', '\u{1f3f3}', EmojiStatus::EmojiOther), ('\u{1f3f4}', '\u{1f3f4}',
        EmojiStatus::EmojiPresentation), ('\u{1f3f5}', '\u{1f3f5}', EmojiStatus::EmojiOther),
        ('\u{1f3f6}', '\u{1f3f6}', EmojiStatus::NonEmoji), ('\u{1f3f7}', '\u{1f3f7}',
        EmojiStatus::EmojiOther), ('\u{1f3f8}', '\u{1f3fa}', EmojiStatus::EmojiPresentation),
        ('\u{1f3fb}', '\u{1f3ff}', EmojiStatus::EmojiPresentationAndModifierAndEmojiComponent),
        ('\u{1f400}', '\u{1f407}', EmojiStatus::EmojiPresentation), ('\u{1f408}', '\u{1f408}',
        EmojiStatus::EmojiPresentation), ('\u{1f409}', '\u{1f40b}', EmojiStatus::EmojiPresentation),
        ('\u{1f40c}', '\u{1f40e}', EmojiStatus::EmojiPresentation), ('\u{1f40f}', '\u{1f410}',
        EmojiStatus::EmojiPresentation), ('\u{1f411}', '\u{1f412}', EmojiStatus::EmojiPresentation),
        ('\u{1f413}', '\u{1f413}', EmojiStatus::EmojiPresentation), ('\u{1f414}', '\u{1f414}',
        EmojiStatus::EmojiPresentation), ('\u{1f415}', '\u{1f415}', EmojiStatus::EmojiPresentation),
        ('\u{1f416}', '\u{1f416}', EmojiStatus::EmojiPresentation), ('\u{1f417}', '\u{1f429}',
        EmojiStatus::EmojiPresentation), ('\u{1f42a}', '\u{1f42a}', EmojiStatus::EmojiPresentation),
        ('\u{1f42b}', '\u{1f43e}', EmojiStatus::EmojiPresentation), ('\u{1f43f}', '\u{1f43f}',
        EmojiStatus::EmojiOther), ('\u{1f440}', '\u{1f440}', EmojiStatus::EmojiPresentation),
        ('\u{1f441}', '\u{1f441}', EmojiStatus::EmojiOther), ('\u{1f442}', '\u{1f443}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f444}', '\u{1f445}',
        EmojiStatus::EmojiPresentation), ('\u{1f446}', '\u{1f450}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f451}', '\u{1f464}',
        EmojiStatus::EmojiPresentation), ('\u{1f465}', '\u{1f465}', EmojiStatus::EmojiPresentation),
        ('\u{1f466}', '\u{1f46b}', EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f46c}',
        '\u{1f46d}', EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f46e}', '\u{1f478}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f479}', '\u{1f47b}',
        EmojiStatus::EmojiPresentation), ('\u{1f47c}', '\u{1f47c}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f47d}', '\u{1f480}',
        EmojiStatus::EmojiPresentation), ('\u{1f481}', '\u{1f483}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f484}', '\u{1f484}',
        EmojiStatus::EmojiPresentation), ('\u{1f485}', '\u{1f487}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f488}', '\u{1f48e}',
        EmojiStatus::EmojiPresentation), ('\u{1f48f}', '\u{1f48f}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f490}', '\u{1f490}',
        EmojiStatus::EmojiPresentation), ('\u{1f491}', '\u{1f491}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f492}', '\u{1f4a9}',
        EmojiStatus::EmojiPresentation), ('\u{1f4aa}', '\u{1f4aa}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f4ab}', '\u{1f4ac}',
        EmojiStatus::EmojiPresentation), ('\u{1f4ad}', '\u{1f4ad}', EmojiStatus::EmojiPresentation),
        ('\u{1f4ae}', '\u{1f4b5}', EmojiStatus::EmojiPresentation), ('\u{1f4b6}', '\u{1f4b7}',
        EmojiStatus::EmojiPresentation), ('\u{1f4b8}', '\u{1f4eb}', EmojiStatus::EmojiPresentation),
        ('\u{1f4ec}', '\u{1f4ed}', EmojiStatus::EmojiPresentation), ('\u{1f4ee}', '\u{1f4ee}',
        EmojiStatus::EmojiPresentation), ('\u{1f4ef}', '\u{1f4ef}', EmojiStatus::EmojiPresentation),
        ('\u{1f4f0}', '\u{1f4f4}', EmojiStatus::EmojiPresentation), ('\u{1f4f5}', '\u{1f4f5}',
        EmojiStatus::EmojiPresentation), ('\u{1f4f6}', '\u{1f4f7}', EmojiStatus::EmojiPresentation),
        ('\u{1f4f8}', '\u{1f4f8}', EmojiStatus::EmojiPresentation), ('\u{1f4f9}', '\u{1f4fc}',
        EmojiStatus::EmojiPresentation), ('\u{1f4fd}', '\u{1f4fd}', EmojiStatus::EmojiOther),
        ('\u{1f4fe}', '\u{1f4fe}', EmojiStatus::NonEmoji), ('\u{1f4ff}', '\u{1f502}',
        EmojiStatus::EmojiPresentation), ('\u{1f503}', '\u{1f503}', EmojiStatus::EmojiPresentation),
        ('\u{1f504}', '\u{1f507}', EmojiStatus::EmojiPresentation), ('\u{1f508}', '\u{1f508}',
        EmojiStatus::EmojiPresentation), ('\u{1f509}', '\u{1f509}', EmojiStatus::EmojiPresentation),
        ('\u{1f50a}', '\u{1f514}', EmojiStatus::EmojiPresentation), ('\u{1f515}', '\u{1f515}',
        EmojiStatus::EmojiPresentation), ('\u{1f516}', '\u{1f52b}', EmojiStatus::EmojiPresentation),
        ('\u{1f52c}', '\u{1f52d}', EmojiStatus::EmojiPresentation), ('\u{1f52e}', '\u{1f53d}',
        EmojiStatus::EmojiPresentation), ('\u{1f53e}', '\u{1f548}', EmojiStatus::NonEmoji),
        ('\u{1f549}', '\u{1f54a}', EmojiStatus::EmojiOther), ('\u{1f54b}', '\u{1f54e}',
        EmojiStatus::EmojiPresentation), ('\u{1f54f}', '\u{1f54f}', EmojiStatus::NonEmoji),
        ('\u{1f550}', '\u{1f55b}', EmojiStatus::EmojiPresentation), ('\u{1f55c}', '\u{1f567}',
        EmojiStatus::EmojiPresentation), ('\u{1f568}', '\u{1f56e}', EmojiStatus::NonEmoji),
        ('\u{1f56f}', '\u{1f570}', EmojiStatus::EmojiOther), ('\u{1f571}', '\u{1f572}',
        EmojiStatus::NonEmoji), ('\u{1f573}', '\u{1f573}', EmojiStatus::EmojiOther), ('\u{1f574}',
        '\u{1f575}', EmojiStatus::EmojiModifierBase), ('\u{1f576}', '\u{1f579}',
        EmojiStatus::EmojiOther), ('\u{1f57a}', '\u{1f57a}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f57b}', '\u{1f586}',
        EmojiStatus::NonEmoji), ('\u{1f587}', '\u{1f587}', EmojiStatus::EmojiOther), ('\u{1f588}',
        '\u{1f589}', EmojiStatus::NonEmoji), ('\u{1f58a}', '\u{1f58d}', EmojiStatus::EmojiOther),
        ('\u{1f58e}', '\u{1f58f}', EmojiStatus::NonEmoji), ('\u{1f590}', '\u{1f590}',
        EmojiStatus::EmojiModifierBase), ('\u{1f591}', '\u{1f594}', EmojiStatus::NonEmoji),
        ('\u{1f595}', '\u{1f596}', EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f597}',
        '\u{1f5a3}', EmojiStatus::NonEmoji), ('\u{1f5a4}', '\u{1f5a4}',
        EmojiStatus::EmojiPresentation), ('\u{1f5a5}', '\u{1f5a5}', EmojiStatus::EmojiOther),
        ('\u{1f5a6}', '\u{1f5a7}', EmojiStatus::NonEmoji), ('\u{1f5a8}', '\u{1f5a8}',
        EmojiStatus::EmojiOther), ('\u{1f5a9}', '\u{1f5b0}', EmojiStatus::NonEmoji), ('\u{1f5b1}',
        '\u{1f5b2}', EmojiStatus::EmojiOther), ('\u{1f5b3}', '\u{1f5bb}', EmojiStatus::NonEmoji),
        ('\u{1f5bc}', '\u{1f5bc}', EmojiStatus::EmojiOther), ('\u{1f5bd}', '\u{1f5c1}',
        EmojiStatus::NonEmoji), ('\u{1f5c2}', '\u{1f5c4}', EmojiStatus::EmojiOther), ('\u{1f5c5}',
        '\u{1f5d0}', EmojiStatus::NonEmoji), ('\u{1f5d1}', '\u{1f5d3}', EmojiStatus::EmojiOther),
        ('\u{1f5d4}', '\u{1f5db}', EmojiStatus::NonEmoji), ('\u{1f5dc}', '\u{1f5de}',
        EmojiStatus::EmojiOther), ('\u{1f5df}', '\u{1f5e0}', EmojiStatus::NonEmoji), ('\u{1f5e1}',
        '\u{1f5e1}', EmojiStatus::EmojiOther), ('\u{1f5e2}', '\u{1f5e2}', EmojiStatus::NonEmoji),
        ('\u{1f5e3}', '\u{1f5e3}', EmojiStatus::EmojiOther), ('\u{1f5e4}', '\u{1f5e7}',
        EmojiStatus::NonEmoji), ('\u{1f5e8}', '\u{1f5e8}', EmojiStatus::EmojiOther), ('\u{1f5e9}',
        '\u{1f5ee}', EmojiStatus::NonEmoji), ('\u{1f5ef}', '\u{1f5ef}', EmojiStatus::EmojiOther),
        ('\u{1f5f0}', '\u{1f5f2}', EmojiStatus::NonEmoji), ('\u{1f5f3}', '\u{1f5f3}',
        EmojiStatus::EmojiOther), ('\u{1f5f4}', '\u{1f5f9}', EmojiStatus::NonEmoji), ('\u{1f5fa}',
        '\u{1f5fa}', EmojiStatus::EmojiOther), ('\u{1f5fb}', '\u{1f5ff}',
        EmojiStatus::EmojiPresentation), ('\u{1f600}', '\u{1f600}', EmojiStatus::EmojiPresentation),
        ('\u{1f601}', '\u{1f606}', EmojiStatus::EmojiPresentation), ('\u{1f607}', '\u{1f608}',
        EmojiStatus::EmojiPresentation), ('\u{1f609}', '\u{1f60d}', EmojiStatus::EmojiPresentation),
        ('\u{1f60e}', '\u{1f60e}', EmojiStatus::EmojiPresentation), ('\u{1f60f}', '\u{1f60f}',
        EmojiStatus::EmojiPresentation), ('\u{1f610}', '\u{1f610}', EmojiStatus::EmojiPresentation),
        ('\u{1f611}', '\u{1f611}', EmojiStatus::EmojiPresentation), ('\u{1f612}', '\u{1f614}',
        EmojiStatus::EmojiPresentation), ('\u{1f615}', '\u{1f615}', EmojiStatus::EmojiPresentation),
        ('\u{1f616}', '\u{1f616}', EmojiStatus::EmojiPresentation), ('\u{1f617}', '\u{1f617}',
        EmojiStatus::EmojiPresentation), ('\u{1f618}', '\u{1f618}', EmojiStatus::EmojiPresentation),
        ('\u{1f619}', '\u{1f619}', EmojiStatus::EmojiPresentation), ('\u{1f61a}', '\u{1f61a}',
        EmojiStatus::EmojiPresentation), ('\u{1f61b}', '\u{1f61b}', EmojiStatus::EmojiPresentation),
        ('\u{1f61c}', '\u{1f61e}', EmojiStatus::EmojiPresentation), ('\u{1f61f}', '\u{1f61f}',
        EmojiStatus::EmojiPresentation), ('\u{1f620}', '\u{1f625}', EmojiStatus::EmojiPresentation),
        ('\u{1f626}', '\u{1f627}', EmojiStatus::EmojiPresentation), ('\u{1f628}', '\u{1f62b}',
        EmojiStatus::EmojiPresentation), ('\u{1f62c}', '\u{1f62c}', EmojiStatus::EmojiPresentation),
        ('\u{1f62d}', '\u{1f62d}', EmojiStatus::EmojiPresentation), ('\u{1f62e}', '\u{1f62f}',
        EmojiStatus::EmojiPresentation), ('\u{1f630}', '\u{1f633}', EmojiStatus::EmojiPresentation),
        ('\u{1f634}', '\u{1f634}', EmojiStatus::EmojiPresentation), ('\u{1f635}', '\u{1f635}',
        EmojiStatus::EmojiPresentation), ('\u{1f636}', '\u{1f636}', EmojiStatus::EmojiPresentation),
        ('\u{1f637}', '\u{1f640}', EmojiStatus::EmojiPresentation), ('\u{1f641}', '\u{1f644}',
        EmojiStatus::EmojiPresentation), ('\u{1f645}', '\u{1f647}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f648}', '\u{1f64a}',
        EmojiStatus::EmojiPresentation), ('\u{1f64b}', '\u{1f64f}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f650}', '\u{1f67f}',
        EmojiStatus::NonEmoji), ('\u{1f680}', '\u{1f680}', EmojiStatus::EmojiPresentation),
        ('\u{1f681}', '\u{1f682}', EmojiStatus::EmojiPresentation), ('\u{1f683}', '\u{1f685}',
        EmojiStatus::EmojiPresentation), ('\u{1f686}', '\u{1f686}', EmojiStatus::EmojiPresentation),
        ('\u{1f687}', '\u{1f687}', EmojiStatus::EmojiPresentation), ('\u{1f688}', '\u{1f688}',
        EmojiStatus::EmojiPresentation), ('\u{1f689}', '\u{1f689}', EmojiStatus::EmojiPresentation),
        ('\u{1f68a}', '\u{1f68b}', EmojiStatus::EmojiPresentation), ('\u{1f68c}', '\u{1f68c}',
        EmojiStatus::EmojiPresentation), ('\u{1f68d}', '\u{1f68d}', EmojiStatus::EmojiPresentation),
        ('\u{1f68e}', '\u{1f68e}', EmojiStatus::EmojiPresentation), ('\u{1f68f}', '\u{1f68f}',
        EmojiStatus::EmojiPresentation), ('\u{1f690}', '\u{1f690}', EmojiStatus::EmojiPresentation),
        ('\u{1f691}', '\u{1f693}', EmojiStatus::EmojiPresentation), ('\u{1f694}', '\u{1f694}',
        EmojiStatus::EmojiPresentation), ('\u{1f695}', '\u{1f695}', EmojiStatus::EmojiPresentation),
        ('\u{1f696}', '\u{1f696}', EmojiStatus::EmojiPresentation), ('\u{1f697}', '\u{1f697}',
        EmojiStatus::EmojiPresentation), ('\u{1f698}', '\u{1f698}', EmojiStatus::EmojiPresentation),
        ('\u{1f699}', '\u{1f69a}', EmojiStatus::EmojiPresentation), ('\u{1f69b}', '\u{1f6a1}',
        EmojiStatus::EmojiPresentation), ('\u{1f6a2}', '\u{1f6a2}', EmojiStatus::EmojiPresentation),
        ('\u{1f6a3}', '\u{1f6a3}', EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f6a4}',
        '\u{1f6a5}', EmojiStatus::EmojiPresentation), ('\u{1f6a6}', '\u{1f6a6}',
        EmojiStatus::EmojiPresentation), ('\u{1f6a7}', '\u{1f6ad}', EmojiStatus::EmojiPresentation),
        ('\u{1f6ae}', '\u{1f6b1}', EmojiStatus::EmojiPresentation), ('\u{1f6b2}', '\u{1f6b2}',
        EmojiStatus::EmojiPresentation), ('\u{1f6b3}', '\u{1f6b3}', EmojiStatus::EmojiPresentation),
        ('\u{1f6b4}', '\u{1f6b5}', EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f6b6}',
        '\u{1f6b6}', EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f6b7}', '\u{1f6b8}',
        EmojiStatus::EmojiPresentation), ('\u{1f6b9}', '\u{1f6be}', EmojiStatus::EmojiPresentation),
        ('\u{1f6bf}', '\u{1f6bf}', EmojiStatus::EmojiPresentation), ('\u{1f6c0}', '\u{1f6c0}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f6c1}', '\u{1f6c5}',
        EmojiStatus::EmojiPresentation), ('\u{1f6c6}', '\u{1f6ca}', EmojiStatus::NonEmoji),
        ('\u{1f6cb}', '\u{1f6cb}', EmojiStatus::EmojiOther), ('\u{1f6cc}', '\u{1f6cc}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f6cd}', '\u{1f6cf}',
        EmojiStatus::EmojiOther), ('\u{1f6d0}', '\u{1f6d0}', EmojiStatus::EmojiPresentation),
        ('\u{1f6d1}', '\u{1f6d2}', EmojiStatus::EmojiPresentation), ('\u{1f6d3}', '\u{1f6d4}',
        EmojiStatus::NonEmoji), ('\u{1f6d5}', '\u{1f6d5}', EmojiStatus::EmojiPresentation),
        ('\u{1f6d6}', '\u{1f6d7}', EmojiStatus::EmojiPresentation), ('\u{1f6d8}', '\u{1f6db}',
        EmojiStatus::NonEmoji), ('\u{1f6dc}', '\u{1f6dc}', EmojiStatus::EmojiPresentation),
        ('\u{1f6dd}', '\u{1f6df}', EmojiStatus::EmojiPresentation), ('\u{1f6e0}', '\u{1f6e5}',
        EmojiStatus::EmojiOther), ('\u{1f6e6}', '\u{1f6e8}', EmojiStatus::NonEmoji), ('\u{1f6e9}',
        '\u{1f6e9}', EmojiStatus::EmojiOther), ('\u{1f6ea}', '\u{1f6ea}', EmojiStatus::NonEmoji),
        ('\u{1f6eb}', '\u{1f6ec}', EmojiStatus::EmojiPresentation), ('\u{1f6ed}', '\u{1f6ef}',
        EmojiStatus::NonEmoji), ('\u{1f6f0}', '\u{1f6f0}', EmojiStatus::EmojiOther), ('\u{1f6f1}',
        '\u{1f6f2}', EmojiStatus::NonEmoji), ('\u{1f6f3}', '\u{1f6f3}', EmojiStatus::EmojiOther),
        ('\u{1f6f4}', '\u{1f6f6}', EmojiStatus::EmojiPresentation), ('\u{1f6f7}', '\u{1f6f8}',
        EmojiStatus::EmojiPresentation), ('\u{1f6f9}', '\u{1f6f9}', EmojiStatus::EmojiPresentation),
        ('\u{1f6fa}', '\u{1f6fa}', EmojiStatus::EmojiPresentation), ('\u{1f6fb}', '\u{1f6fc}',
        EmojiStatus::EmojiPresentation), ('\u{1f6fd}', '\u{1f7df}', EmojiStatus::NonEmoji),
        ('\u{1f7e0}', '\u{1f7eb}', EmojiStatus::EmojiPresentation), ('\u{1f7ec}', '\u{1f7ef}',
        EmojiStatus::NonEmoji), ('\u{1f7f0}', '\u{1f7f0}', EmojiStatus::EmojiPresentation),
        ('\u{1f7f1}', '\u{1f90b}', EmojiStatus::NonEmoji), ('\u{1f90c}', '\u{1f90c}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f90d}', '\u{1f90e}',
        EmojiStatus::EmojiPresentation), ('\u{1f90f}', '\u{1f90f}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f910}', '\u{1f917}',
        EmojiStatus::EmojiPresentation), ('\u{1f918}', '\u{1f918}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f919}', '\u{1f91e}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f91f}', '\u{1f91f}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f920}', '\u{1f925}',
        EmojiStatus::EmojiPresentation), ('\u{1f926}', '\u{1f926}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f927}', '\u{1f927}',
        EmojiStatus::EmojiPresentation), ('\u{1f928}', '\u{1f92f}', EmojiStatus::EmojiPresentation),
        ('\u{1f930}', '\u{1f930}', EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f931}',
        '\u{1f932}', EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f933}', '\u{1f939}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f93a}', '\u{1f93a}',
        EmojiStatus::EmojiPresentation), ('\u{1f93b}', '\u{1f93b}', EmojiStatus::NonEmoji),
        ('\u{1f93c}', '\u{1f93e}', EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f93f}',
        '\u{1f93f}', EmojiStatus::EmojiPresentation), ('\u{1f940}', '\u{1f945}',
        EmojiStatus::EmojiPresentation), ('\u{1f946}', '\u{1f946}', EmojiStatus::NonEmoji),
        ('\u{1f947}', '\u{1f94b}', EmojiStatus::EmojiPresentation), ('\u{1f94c}', '\u{1f94c}',
        EmojiStatus::EmojiPresentation), ('\u{1f94d}', '\u{1f94f}', EmojiStatus::EmojiPresentation),
        ('\u{1f950}', '\u{1f95e}', EmojiStatus::EmojiPresentation), ('\u{1f95f}', '\u{1f96b}',
        EmojiStatus::EmojiPresentation), ('\u{1f96c}', '\u{1f970}', EmojiStatus::EmojiPresentation),
        ('\u{1f971}', '\u{1f971}', EmojiStatus::EmojiPresentation), ('\u{1f972}', '\u{1f972}',
        EmojiStatus::EmojiPresentation), ('\u{1f973}', '\u{1f976}', EmojiStatus::EmojiPresentation),
        ('\u{1f977}', '\u{1f977}', EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f978}',
        '\u{1f978}', EmojiStatus::EmojiPresentation), ('\u{1f979}', '\u{1f979}',
        EmojiStatus::EmojiPresentation), ('\u{1f97a}', '\u{1f97a}', EmojiStatus::EmojiPresentation),
        ('\u{1f97b}', '\u{1f97b}', EmojiStatus::EmojiPresentation), ('\u{1f97c}', '\u{1f97f}',
        EmojiStatus::EmojiPresentation), ('\u{1f980}', '\u{1f984}', EmojiStatus::EmojiPresentation),
        ('\u{1f985}', '\u{1f991}', EmojiStatus::EmojiPresentation), ('\u{1f992}', '\u{1f997}',
        EmojiStatus::EmojiPresentation), ('\u{1f998}', '\u{1f9a2}', EmojiStatus::EmojiPresentation),
        ('\u{1f9a3}', '\u{1f9a4}', EmojiStatus::EmojiPresentation), ('\u{1f9a5}', '\u{1f9aa}',
        EmojiStatus::EmojiPresentation), ('\u{1f9ab}', '\u{1f9ad}', EmojiStatus::EmojiPresentation),
        ('\u{1f9ae}', '\u{1f9af}', EmojiStatus::EmojiPresentation), ('\u{1f9b0}', '\u{1f9b3}',
        EmojiStatus::EmojiPresentationAndEmojiComponent), ('\u{1f9b4}', '\u{1f9b4}',
        EmojiStatus::EmojiPresentation), ('\u{1f9b5}', '\u{1f9b6}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f9b7}', '\u{1f9b7}',
        EmojiStatus::EmojiPresentation), ('\u{1f9b8}', '\u{1f9b9}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f9ba}', '\u{1f9ba}',
        EmojiStatus::EmojiPresentation), ('\u{1f9bb}', '\u{1f9bb}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f9bc}', '\u{1f9bf}',
        EmojiStatus::EmojiPresentation), ('\u{1f9c0}', '\u{1f9c0}', EmojiStatus::EmojiPresentation),
        ('\u{1f9c1}', '\u{1f9c2}', EmojiStatus::EmojiPresentation), ('\u{1f9c3}', '\u{1f9ca}',
        EmojiStatus::EmojiPresentation), ('\u{1f9cb}', '\u{1f9cb}', EmojiStatus::EmojiPresentation),
        ('\u{1f9cc}', '\u{1f9cc}', EmojiStatus::EmojiPresentation), ('\u{1f9cd}', '\u{1f9cf}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f9d0}', '\u{1f9d0}',
        EmojiStatus::EmojiPresentation), ('\u{1f9d1}', '\u{1f9dd}',
        EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1f9de}', '\u{1f9e6}',
        EmojiStatus::EmojiPresentation), ('\u{1f9e7}', '\u{1f9ff}', EmojiStatus::EmojiPresentation),
        ('\u{1fa00}', '\u{1fa6f}', EmojiStatus::NonEmoji), ('\u{1fa70}', '\u{1fa73}',
        EmojiStatus::EmojiPresentation), ('\u{1fa74}', '\u{1fa74}', EmojiStatus::EmojiPresentation),
        ('\u{1fa75}', '\u{1fa77}', EmojiStatus::EmojiPresentation), ('\u{1fa78}', '\u{1fa7a}',
        EmojiStatus::EmojiPresentation), ('\u{1fa7b}', '\u{1fa7c}', EmojiStatus::EmojiPresentation),
        ('\u{1fa7d}', '\u{1fa7f}', EmojiStatus::NonEmoji), ('\u{1fa80}', '\u{1fa82}',
        EmojiStatus::EmojiPresentation), ('\u{1fa83}', '\u{1fa86}', EmojiStatus::EmojiPresentation),
        ('\u{1fa87}', '\u{1fa88}', EmojiStatus::EmojiPresentation), ('\u{1fa89}', '\u{1fa8f}',
        EmojiStatus::NonEmoji), ('\u{1fa90}', '\u{1fa95}', EmojiStatus::EmojiPresentation),
        ('\u{1fa96}', '\u{1faa8}', EmojiStatus::EmojiPresentation), ('\u{1faa9}', '\u{1faac}',
        EmojiStatus::EmojiPresentation), ('\u{1faad}', '\u{1faaf}', EmojiStatus::EmojiPresentation),
        ('\u{1fab0}', '\u{1fab6}', EmojiStatus::EmojiPresentation), ('\u{1fab7}', '\u{1faba}',
        EmojiStatus::EmojiPresentation), ('\u{1fabb}', '\u{1fabd}', EmojiStatus::EmojiPresentation),
        ('\u{1fabe}', '\u{1fabe}', EmojiStatus::NonEmoji), ('\u{1fabf}', '\u{1fabf}',
        EmojiStatus::EmojiPresentation), ('\u{1fac0}', '\u{1fac2}', EmojiStatus::EmojiPresentation),
        ('\u{1fac3}', '\u{1fac5}', EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1fac6}',
        '\u{1facd}', EmojiStatus::NonEmoji), ('\u{1face}', '\u{1facf}',
        EmojiStatus::EmojiPresentation), ('\u{1fad0}', '\u{1fad6}', EmojiStatus::EmojiPresentation),
        ('\u{1fad7}', '\u{1fad9}', EmojiStatus::EmojiPresentation), ('\u{1fada}', '\u{1fadb}',
        EmojiStatus::EmojiPresentation), ('\u{1fadc}', '\u{1fadf}', EmojiStatus::NonEmoji),
        ('\u{1fae0}', '\u{1fae7}', EmojiStatus::EmojiPresentation), ('\u{1fae8}', '\u{1fae8}',
        EmojiStatus::EmojiPresentation), ('\u{1fae9}', '\u{1faef}', EmojiStatus::NonEmoji),
        ('\u{1faf0}', '\u{1faf6}', EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1faf7}',
        '\u{1faf8}', EmojiStatus::EmojiPresentationAndModifierBase), ('\u{1faf9}', '\u{e001f}',
        EmojiStatus::NonEmoji), ('\u{e0020}', '\u{e007f}', EmojiStatus::NonEmojiButEmojiComponent),
        ('\u{e0080}', '\u{10ffff}', EmojiStatus::NonEmoji)
    ];

}

