/***************************************************************************
 *   Copyright (C) 2010 by SIDDHARTH SHARMA siddharth.kde@gmail.com        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
* This file is Plasma Applet for Scheduled Operations Skroogoid
*
* @author Siddharth SHARMA
*/
#include "scheduledoperationskroogoid.h"

#include <QGraphicsGridLayout>
#include <KMessageBox>
#include <Plasma/Frame>

#include "skgdocumentbank.h"
#include "skgservices.h"
#include "skgtraces.h"
#include "skgmainpanel.h"

ScheduledOperationSkroogoid::ScheduledOperationSkroogoid(QObject* iParent, const QVariantList& args)
    : Plasma::Applet(iParent, args), widgetLayout(NULL)
{}

ScheduledOperationSkroogoid::~ScheduledOperationSkroogoid()
{}

void ScheduledOperationSkroogoid::init()
{
    // This is the layout for the whole widget
    if (widgetLayout == NULL) {
        widgetLayout = new QGraphicsLinearLayout(Qt::Vertical);
        setLayout(widgetLayout);

        Plasma::DataEngine* engine = dataEngine("skgdataengine");
        if (engine->isValid())  engine->connectSource("Scheduled Operations", this);
    }
}

void ScheduledOperationSkroogoid::dataUpdated(const QString& sourceName, const Plasma::DataEngine::Data& engineData)
{
    if (sourceName != "Scheduled Operations") return;
    SKGTRACEIN(1, "AdviceSkroogoid::dataUpdated");

    QHash<QString, QStringList> listData;
    for (int i = 0 ; i < engineData.keys().count() ; ++i) {
        QString operationId = engineData.keys().at(i);
        QVariant operationValuesVariant = engineData.value(operationId);
        QStringList operationValues = operationValuesVariant.value<QStringList>();

        listData[operationId] = operationValues;
    }

    // Get the current document
    SKGMainPanel* panel = SKGMainPanel::getMainPanel();
    SKGDocumentBank* doc = qobject_cast<SKGDocumentBank*> (panel->getDocument());

    // Get the primary unit
    SKGServices::SKGUnitInfo primary = doc->getPrimaryUnit();

    // Initialisation
    int nb = widgetLayout->count();
    for (int i = 0; widgetLayout && i < nb; ++i) {
        QGraphicsLayoutItem* item = widgetLayout->itemAt(0);
        if (item) {
            widgetLayout->removeAt(0);
            delete item;
        }
    }

    // Get data from the dataengine
    nb = listData.count();

    QHash<QString, Plasma::Frame*> framesHash;

    for (int i = 0; i < nb; ++i) {
        // Operation id
        QString operationId = listData.keys().at(i);

        QStringList values = listData[operationId];

        // Date
        QString date = values.at(0);

        // Payee
        QString payee = values.at(1);

        // Amount
        double amount = SKGServices::stringToDouble(values.at(2));

        // Group Scheduled operations per date in a Plasma::Frame
        Plasma::Frame* frame;
        if (!framesHash.contains(date)) {
            frame = createFrame(date);
            framesHash.insert(date, frame);
            widgetLayout->addItem(frame);
        } else {
            frame = framesHash.value(date);
        }
        QGraphicsLinearLayout* frameLayout = static_cast<QGraphicsLinearLayout*>(frame->layout());

        // Add a linear layout for the line
        QGraphicsLinearLayout* lineLayout = new QGraphicsLinearLayout(Qt::Horizontal);
        frameLayout->addItem(lineLayout);

        // The label for the payee
        Plasma::Label* payeeLabel = new Plasma::Label();
        lineLayout->addItem(payeeLabel);
        payeeLabel->setText(payee);
        payeeLabel->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);

        lineLayout->addStretch(20);

        // The label for the amount
        Plasma::Label* amountLabel = new Plasma::Label();
        amountLabel->setText(doc->formatMoney(amount, primary));
        lineLayout->addItem(amountLabel);
        amountLabel->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
    }

    widgetLayout->addStretch(20);
}

Plasma::Frame* ScheduledOperationSkroogoid::createFrame(const QString date)
{
    Plasma::Frame* dateFrame = new Plasma::Frame();
    dateFrame->setFrameShadow(Plasma::Frame::Raised);
    QGraphicsLinearLayout* frameLayout = new QGraphicsLinearLayout(Qt::Vertical);
    dateFrame->setLayout(frameLayout);

    KLocale* loc = KGlobal::locale();
    Plasma::Label* dateLabel = new Plasma::Label();
    dateLabel->setText(loc->formatDate(SKGServices::stringToTime(date).date(), KLocale::FancyShortDate));
    dateLabel->setAlignment(Qt::AlignLeft);
    QFont dateFont = dateLabel->font();
    dateFont.setBold(true);
    dateLabel->setFont(dateFont);
    frameLayout->addItem(dateLabel);

    return dateFrame;
}


#include "scheduledoperationskroogoid.moc"
