// SPDX-License-Identifier: GPL-2.0-only
/*
 * Samsung AMS427AP24 panel with S6E88A0 controller
 * Copyright (c) 2024 Jakob Hauser <jahau@rocketmail.com>
 */

#include <linux/backlight.h>
#include <linux/delay.h>
#include <linux/gpio/consumer.h>
#include <linux/mod_devicetable.h>
#include <linux/module.h>
#include <linux/property.h>
#include <linux/regulator/consumer.h>

#include <video/mipi_display.h>

#include <drm/drm_mipi_dsi.h>
#include <drm/drm_modes.h>
#include <drm/drm_panel.h>
#include <drm/drm_probe_helper.h>

#define NUM_STEPS_CANDELA	54
#define NUM_STEPS_AID		39
#define NUM_STEPS_ELVSS		17

/* length of the payload data, thereof fixed and variable */
#define FIX_LEN_AID		4
#define FIX_LEN_ELVSS		2
#define FIX_LEN_GAMMA		1
#define VAR_LEN_AID		2
#define VAR_LEN_ELVSS		1
#define VAR_LEN_GAMMA		33
#define LEN_AID			(FIX_LEN_AID + VAR_LEN_AID)
#define LEN_ELVSS		(FIX_LEN_ELVSS + VAR_LEN_ELVSS)
#define LEN_GAMMA		(FIX_LEN_GAMMA + VAR_LEN_GAMMA)

struct s6e88a0_ams427ap24 {
	struct drm_panel panel;
	struct backlight_device *bl_dev;
	struct mipi_dsi_device *dsi;
	struct regulator_bulk_data *supplies;
	struct gpio_desc *reset_gpio;
	bool flip_horizontal;
};

static const struct regulator_bulk_data s6e88a0_ams427ap24_supplies[] = {
	{ .supply = "vdd3" },
	{ .supply = "vci" },
};

static inline
struct s6e88a0_ams427ap24 *to_s6e88a0_ams427ap24(struct drm_panel *panel)
{
	return container_of(panel, struct s6e88a0_ams427ap24, panel);
}

enum candela {
	CANDELA_10CD, /* 0 */
	CANDELA_11CD,
	CANDELA_12CD,
	CANDELA_13CD,
	CANDELA_14CD,
	CANDELA_15CD,
	CANDELA_16CD,
	CANDELA_17CD,
	CANDELA_19CD,
	CANDELA_20CD,
	CANDELA_21CD,
	CANDELA_22CD,
	CANDELA_24CD,
	CANDELA_25CD,
	CANDELA_27CD,
	CANDELA_29CD,
	CANDELA_30CD,
	CANDELA_32CD,
	CANDELA_34CD,
	CANDELA_37CD,
	CANDELA_39CD,
	CANDELA_41CD,
	CANDELA_44CD,
	CANDELA_47CD,
	CANDELA_50CD,
	CANDELA_53CD,
	CANDELA_56CD,
	CANDELA_60CD,
	CANDELA_64CD,
	CANDELA_68CD,
	CANDELA_72CD,
	CANDELA_77CD,
	CANDELA_82CD,
	CANDELA_87CD,
	CANDELA_93CD,
	CANDELA_98CD,
	CANDELA_105CD,
	CANDELA_111CD,
	CANDELA_119CD,
	CANDELA_126CD,
	CANDELA_134CD,
	CANDELA_143CD,
	CANDELA_152CD,
	CANDELA_162CD,
	CANDELA_172CD,
	CANDELA_183CD,
	CANDELA_195CD,
	CANDELA_207CD,
	CANDELA_220CD,
	CANDELA_234CD,
	CANDELA_249CD,
	CANDELA_265CD,
	CANDELA_282CD,
	CANDELA_300CD, /* 53 */
};

static const int s6e88a0_ams427ap24_br_to_cd[NUM_STEPS_CANDELA] = {
	/* columns: brightness from, brightness till, candela */
	/* 0 */    10,  /* 10CD */
	/* 11 */   11,  /* 11CD */
	/* 12 */   12,  /* 12CD */
	/* 13 */   13,  /* 13CD */
	/* 14 */   14,  /* 14CD */
	/* 15 */   15,  /* 15CD */
	/* 16 */   16,  /* 16CD */
	/* 17 */   17,  /* 17CD */
	/* 18 */   18,  /* 19CD */
	/* 19 */   19,  /* 20CD */
	/* 20 */   20,  /* 21CD */
	/* 21 */   21,  /* 22CD */
	/* 22 */   22,  /* 24CD */
	/* 23 */   23,  /* 25CD */
	/* 24 */   24,  /* 27CD */
	/* 25 */   25,  /* 29CD */
	/* 26 */   26,  /* 30CD */
	/* 27 */   27,  /* 32CD */
	/* 28 */   28,  /* 34CD */
	/* 29 */   29,  /* 37CD */
	/* 30 */   30,  /* 39CD */
	/* 31 */   32,  /* 41CD */
	/* 33 */   34,  /* 44CD */
	/* 35 */   36,  /* 47CD */
	/* 37 */   38,  /* 50CD */
	/* 39 */   40,  /* 53CD */
	/* 41 */   43,  /* 56CD */
	/* 44 */   46,  /* 60CD */
	/* 47 */   49,  /* 64CD */
	/* 50 */   52,  /* 68CD */
	/* 53 */   56,  /* 72CD */
	/* 57 */   59,  /* 77CD */
	/* 60 */   63,  /* 82CD */
	/* 64 */   67,  /* 87CD */
	/* 68 */   71,  /* 93CD */
	/* 72 */   76,  /* 98CD */
	/* 77 */   80,  /* 105CD */
	/* 81 */   86,  /* 111CD */
	/* 87 */   91,  /* 119CD */
	/* 92 */   97,  /* 126CD */
	/* 98 */   104, /* 134CD */
	/* 105 */  110, /* 143CD */
	/* 111 */  118, /* 152CD */
	/* 119 */  125, /* 162CD */
	/* 126 */  133, /* 172CD */
	/* 134 */  142, /* 183CD */
	/* 143 */  150, /* 195CD */
	/* 151 */  160, /* 207CD */
	/* 161 */  170, /* 220CD */
	/* 171 */  181, /* 234CD */
	/* 182 */  205, /* 249CD */
	/* 206 */  234, /* 265CD */
	/* 235 */  254, /* 282CD */
	/* 255 */  255, /* 300CD */
};

static const u8 s6e88a0_ams427ap24_aid[NUM_STEPS_AID][VAR_LEN_AID] = {
	{ 0x03, 0x77 }, /* AOR 90.9%, 10CD */
	{ 0x03, 0x73 }, /* AOR 90.5%, 11CD */
	{ 0x03, 0x69 }, /* AOR 89.4%, 12CD */
	{ 0x03, 0x65 }, /* AOR 89.0%, 13CD */
	{ 0x03, 0x61 }, /* AOR 88.6%, 14CD */
	{ 0x03, 0x55 }, /* AOR 87.4%, 15CD */
	{ 0x03, 0x50 }, /* AOR 86.9%, 16CD */
	{ 0x03, 0x45 }, /* AOR 85.8%, 17CD */
	{ 0x03, 0x35 }, /* AOR 84.1%, 19CD */
	{ 0x03, 0x27 }, /* AOR 82.7%, 20CD */
	{ 0x03, 0x23 }, /* AOR 82.3%, 21CD */
	{ 0x03, 0x17 }, /* AOR 81.0%, 22CD */
	{ 0x03, 0x11 }, /* AOR 80.4%, 24CD */
	{ 0x03, 0x04 }, /* AOR 79.1%, 25CD */
	{ 0x02, 0xf4 }, /* AOR 77.5%, 27CD */
	{ 0x02, 0xe3 }, /* AOR 75.7%, 29CD */
	{ 0x02, 0xd7 }, /* AOR 74.5%, 30CD */
	{ 0x02, 0xc6 }, /* AOR 72.7%, 32CD */
	{ 0x02, 0xb7 }, /* AOR 71.2%, 34CD */
	{ 0x02, 0xa1 }, /* AOR 69.0%, 37CD */
	{ 0x02, 0x91 }, /* AOR 67.3%, 39CD */
	{ 0x02, 0x78 }, /* AOR 64.8%, 41CD */
	{ 0x02, 0x62 }, /* AOR 62.5%, 44CD */
	{ 0x02, 0x45 }, /* AOR 59.5%, 47CD */
	{ 0x02, 0x30 }, /* AOR 57.4%, 50CD */
	{ 0x02, 0x13 }, /* AOR 54.4%, 53CD */
	{ 0x01, 0xf5 }, /* AOR 51.3%, 56CD */
	{ 0x01, 0xd3 }, /* AOR 47.8%, 60CD */
	{ 0x01, 0xb1 }, /* AOR 44.4%, 64CD */
	{ 0x01, 0x87 }, /* AOR 40.1%, 68CD */
	{ 0x01, 0x63 }, /* AOR 36.6%, 72CD */
	{ 0x01, 0x35 }, /* AOR 31.7%, 77CD */
	{ 0x01, 0x05 }, /* AOR 26.9%, 82CD */
	{ 0x00, 0xd5 }, /* AOR 21.8%, 87CD */
	{ 0x00, 0xa1 }, /* AOR 16.5%, 93CD */
	{ 0x00, 0x6f }, /* AOR 11.4%, 98CD */
	{ 0x00, 0x31 }, /* AOR 5.0%, 105CD */
	{ 0x01, 0x86 }, /* AOR 40.0%, 111CD ~ 172CD */
	{ 0x00, 0x08 }, /* AOR 0.6%, 183CD ~ 300CD */
};

static const u8 s6e88a0_ams427ap24_elvss[NUM_STEPS_ELVSS][VAR_LEN_ELVSS] = {
	{ 0x14 }, /* 10CD ~ 111CD */
	{ 0x13 }, /* 119CD */
	{ 0x12 }, /* 126CD */
	{ 0x12 }, /* 134CD */
	{ 0x11 }, /* 143CD */
	{ 0x10 }, /* 152CD */
	{ 0x0f }, /* 162CD */
	{ 0x0e }, /* 172CD */
	{ 0x11 }, /* 183CD */
	{ 0x11 }, /* 195CD */
	{ 0x10 }, /* 207CD */
	{ 0x0f }, /* 220CD */
	{ 0x0f }, /* 234CD */
	{ 0x0e }, /* 249CD */
	{ 0x0d }, /* 265CD */
	{ 0x0c }, /* 282CD */
	{ 0x0b }, /* 300CD */
};

static const u8 s6e88a0_ams427ap24_gamma[NUM_STEPS_CANDELA][VAR_LEN_GAMMA] = {
	/* 10CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x8a, 0x8c, 0x8b,
	  0x8c, 0x87, 0x89, 0x89, 0x88, 0x87, 0x8c, 0x80, 0x82, 0x88, 0x7b,
	  0x72, 0x8c, 0x60, 0x68, 0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	/* 11CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x8a, 0x8c, 0x8b,
	  0x8c, 0x87, 0x89, 0x89, 0x88, 0x87, 0x8c, 0x80, 0x82, 0x88, 0x7b,
	  0x72, 0x8c, 0x60, 0x68, 0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	/* 12CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x8a, 0x8b, 0x8b,
	  0x8c, 0x88, 0x89, 0x8a, 0x88, 0x87, 0x8c, 0x81, 0x82, 0x87, 0x7a,
	  0x72, 0x8b, 0x60, 0x68, 0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	/* 13CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x8a, 0x8b, 0x8b,
	  0x8c, 0x88, 0x89, 0x8a, 0x88, 0x87, 0x8c, 0x81, 0x82, 0x87, 0x7a,
	  0x72, 0x8b, 0x61, 0x69, 0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	/* 14CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x89, 0x8c, 0x8b,
	  0x8c, 0x88, 0x89, 0x8a, 0x87, 0x86, 0x8a, 0x82, 0x82, 0x87, 0x79,
	  0x71, 0x89, 0x63, 0x6c, 0x8e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	/* 15CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x8a, 0x8c, 0x8c,
	  0x8c, 0x86, 0x87, 0x88, 0x85, 0x85, 0x8a, 0x83, 0x83, 0x88, 0x78,
	  0x72, 0x89, 0x64, 0x6c, 0x8e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	/* 16CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x89, 0x8c, 0x8b,
	  0x8c, 0x86, 0x88, 0x88, 0x86, 0x86, 0x8a, 0x84, 0x84, 0x88, 0x78,
	  0x72, 0x89, 0x5d, 0x67, 0x8b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	/* 17CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x89, 0x8b, 0x8b,
	  0x8b, 0x87, 0x89, 0x89, 0x86, 0x86, 0x8a, 0x84, 0x83, 0x87, 0x78,
	  0x73, 0x89, 0x64, 0x6e, 0x8e, 0x38, 0x32, 0x24, 0x00, 0x00, 0x00 },
	/* 19CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x89, 0x8b, 0x8b,
	  0x8b, 0x87, 0x89, 0x89, 0x86, 0x86, 0x89, 0x84, 0x84, 0x87, 0x77,
	  0x72, 0x88, 0x65, 0x6f, 0x8e, 0x38, 0x32, 0x24, 0x00, 0x00, 0x00 },
	/* 20CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x89, 0x8b, 0x8b,
	  0x8b, 0x88, 0x89, 0x89, 0x85, 0x85, 0x88, 0x82, 0x83, 0x85, 0x79,
	  0x73, 0x88, 0x65, 0x6f, 0x8e, 0x38, 0x32, 0x24, 0x00, 0x00, 0x00 },
	/* 21CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x89, 0x8b, 0x8b,
	  0x8b, 0x88, 0x89, 0x89, 0x85, 0x85, 0x88, 0x82, 0x83, 0x85, 0x79,
	  0x74, 0x88, 0x65, 0x6f, 0x8e, 0x38, 0x32, 0x24, 0x00, 0x00, 0x00 },
	/* 22CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x89, 0x8c, 0x8b,
	  0x8c, 0x86, 0x88, 0x87, 0x86, 0x86, 0x89, 0x82, 0x83, 0x85, 0x7c,
	  0x75, 0x87, 0x65, 0x6f, 0x8e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	/* 24CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x89, 0x8c, 0x8b,
	  0x8c, 0x86, 0x88, 0x87, 0x86, 0x86, 0x89, 0x82, 0x83, 0x85, 0x7c,
	  0x76, 0x86, 0x66, 0x6f, 0x8e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
	/* 25CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x89, 0x8b, 0x8b,
	  0x8b, 0x86, 0x89, 0x88, 0x87, 0x87, 0x89, 0x82, 0x82, 0x84, 0x7f,
	  0x7a, 0x89, 0x6b, 0x73, 0x8f, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 27CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x89, 0x8b, 0x8b,
	  0x8b, 0x86, 0x89, 0x88, 0x87, 0x87, 0x89, 0x82, 0x82, 0x84, 0x7f,
	  0x7a, 0x89, 0x6b, 0x73, 0x8f, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 29CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x89, 0x8b, 0x8b,
	  0x8b, 0x86, 0x89, 0x88, 0x85, 0x84, 0x87, 0x84, 0x85, 0x86, 0x80,
	  0x7b, 0x88, 0x6a, 0x73, 0x8f, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 30CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x89, 0x8b, 0x8b,
	  0x8b, 0x86, 0x89, 0x88, 0x85, 0x84, 0x87, 0x84, 0x85, 0x86, 0x80,
	  0x7b, 0x88, 0x6a, 0x73, 0x8f, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 32CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x89, 0x8b, 0x8b,
	  0x8b, 0x86, 0x89, 0x88, 0x85, 0x84, 0x87, 0x84, 0x85, 0x86, 0x80,
	  0x7b, 0x88, 0x6a, 0x73, 0x8f, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 34CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8c, 0x8a, 0x89, 0x8b, 0x8b,
	  0x8b, 0x86, 0x89, 0x88, 0x85, 0x84, 0x87, 0x83, 0x84, 0x84, 0x7f,
	  0x79, 0x86, 0x6c, 0x76, 0x91, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 37CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8b, 0x89, 0x89, 0x8b, 0x8b,
	  0x8b, 0x86, 0x88, 0x88, 0x87, 0x86, 0x87, 0x83, 0x84, 0x84, 0x7f,
	  0x79, 0x86, 0x6c, 0x76, 0x90, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 39CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8b, 0x89, 0x89, 0x8b, 0x8b,
	  0x8b, 0x86, 0x88, 0x87, 0x84, 0x84, 0x86, 0x83, 0x85, 0x85, 0x80,
	  0x79, 0x85, 0x6c, 0x76, 0x90, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 41CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8b, 0x89, 0x89, 0x8b, 0x8b,
	  0x8b, 0x86, 0x88, 0x87, 0x84, 0x84, 0x86, 0x81, 0x84, 0x83, 0x7f,
	  0x79, 0x84, 0x6e, 0x79, 0x93, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 44CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8b, 0x89, 0x89, 0x8b, 0x8b,
	  0x8b, 0x86, 0x88, 0x87, 0x84, 0x84, 0x86, 0x81, 0x84, 0x83, 0x7f,
	  0x79, 0x84, 0x6e, 0x79, 0x92, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 47CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8b, 0x89, 0x89, 0x8b, 0x8b,
	  0x8b, 0x86, 0x88, 0x87, 0x84, 0x85, 0x86, 0x81, 0x84, 0x83, 0x7f,
	  0x79, 0x83, 0x6f, 0x79, 0x91, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 50CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8b, 0x89, 0x89, 0x8b, 0x8b,
	  0x8b, 0x86, 0x88, 0x87, 0x84, 0x85, 0x86, 0x82, 0x84, 0x83, 0x7f,
	  0x79, 0x83, 0x6f, 0x79, 0x90, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 53CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8b, 0x89, 0x89, 0x8b, 0x8b,
	  0x8b, 0x86, 0x88, 0x87, 0x83, 0x83, 0x85, 0x84, 0x85, 0x85, 0x7f,
	  0x79, 0x83, 0x70, 0x79, 0x8f, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 56CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8b, 0x89, 0x89, 0x8b, 0x8a,
	  0x8a, 0x87, 0x89, 0x87, 0x83, 0x83, 0x85, 0x84, 0x85, 0x84, 0x7f,
	  0x79, 0x82, 0x70, 0x7a, 0x8e, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 60CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8b, 0x89, 0x89, 0x8b, 0x8a,
	  0x8a, 0x87, 0x89, 0x87, 0x83, 0x83, 0x85, 0x84, 0x85, 0x84, 0x7e,
	  0x79, 0x82, 0x71, 0x7a, 0x8d, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 64CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8b, 0x89, 0x89, 0x8b, 0x8a,
	  0x8a, 0x86, 0x88, 0x86, 0x84, 0x84, 0x86, 0x82, 0x83, 0x82, 0x80,
	  0x7a, 0x84, 0x71, 0x7a, 0x8c, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 68CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8a, 0x89, 0x89, 0x8c, 0x8a,
	  0x8a, 0x86, 0x88, 0x86, 0x84, 0x84, 0x86, 0x82, 0x84, 0x82, 0x81,
	  0x7b, 0x83, 0x72, 0x7b, 0x8b, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 72CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8a, 0x89, 0x89, 0x8c, 0x8a,
	  0x8a, 0x86, 0x88, 0x86, 0x85, 0x85, 0x86, 0x82, 0x84, 0x82, 0x81,
	  0x7b, 0x83, 0x72, 0x7c, 0x8a, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 77CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8a, 0x89, 0x89, 0x8c, 0x8a,
	  0x8a, 0x85, 0x87, 0x85, 0x85, 0x87, 0x87, 0x82, 0x84, 0x82, 0x81,
	  0x7c, 0x82, 0x72, 0x7c, 0x89, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 82CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8a, 0x89, 0x89, 0x8c, 0x8a,
	  0x8a, 0x85, 0x87, 0x85, 0x85, 0x87, 0x87, 0x82, 0x84, 0x82, 0x81,
	  0x7c, 0x82, 0x73, 0x7c, 0x88, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 87CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8a, 0x89, 0x89, 0x8c, 0x8a,
	  0x8a, 0x85, 0x87, 0x85, 0x84, 0x84, 0x86, 0x80, 0x84, 0x81, 0x80,
	  0x7a, 0x82, 0x76, 0x7f, 0x89, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 93CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8a, 0x89, 0x89, 0x8b, 0x8a,
	  0x8a, 0x86, 0x87, 0x85, 0x84, 0x85, 0x86, 0x80, 0x84, 0x80, 0x80,
	  0x7a, 0x82, 0x76, 0x80, 0x88, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 98CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x8a, 0x89, 0x89, 0x8b, 0x8a,
	  0x8a, 0x86, 0x87, 0x85, 0x85, 0x85, 0x86, 0x80, 0x84, 0x80, 0x80,
	  0x7a, 0x82, 0x76, 0x80, 0x88, 0x33, 0x2f, 0x22, 0x00, 0x00, 0x00 },
	/* 105CD */
	{ 0x00, 0xc8, 0x00, 0xc4, 0x00, 0xc5, 0x89, 0x88, 0x88, 0x8b, 0x8a,
	  0x8a, 0x84, 0x87, 0x85, 0x85, 0x85, 0x85, 0x80, 0x84, 0x80, 0x7f,
	  0x79, 0x81, 0x71, 0x7d, 0x87, 0x38, 0x32, 0x24, 0x00, 0x00, 0x00 },
	/* 111CD */
	{ 0x00, 0xdf, 0x00, 0xde, 0x00, 0xde, 0x85, 0x85, 0x84, 0x87, 0x86,
	  0x87, 0x85, 0x86, 0x85, 0x83, 0x83, 0x83, 0x81, 0x82, 0x82, 0x80,
	  0x7d, 0x82, 0x75, 0x7f, 0x86, 0x85, 0x85, 0x82, 0x00, 0x00, 0x00 },
	/* 119CD */
	{ 0x00, 0xe3, 0x00, 0xe1, 0x00, 0xe2, 0x85, 0x85, 0x84, 0x86, 0x85,
	  0x85, 0x84, 0x85, 0x84, 0x83, 0x83, 0x83, 0x82, 0x82, 0x82, 0x7e,
	  0x7b, 0x81, 0x75, 0x7f, 0x86, 0x85, 0x85, 0x82, 0x00, 0x00, 0x00 },
	/* 126CD */
	{ 0x00, 0xe6, 0x00, 0xe5, 0x00, 0xe5, 0x85, 0x84, 0x84, 0x85, 0x85,
	  0x85, 0x84, 0x84, 0x84, 0x82, 0x83, 0x83, 0x80, 0x81, 0x81, 0x80,
	  0x7f, 0x83, 0x73, 0x7c, 0x84, 0x85, 0x85, 0x82, 0x00, 0x00, 0x00 },
	/* 134CD */
	{ 0x00, 0xe9, 0x00, 0xe8, 0x00, 0xe8, 0x84, 0x84, 0x83, 0x85, 0x85,
	  0x85, 0x84, 0x84, 0x83, 0x81, 0x82, 0x82, 0x81, 0x81, 0x81, 0x7f,
	  0x7d, 0x81, 0x73, 0x7c, 0x83, 0x85, 0x85, 0x82, 0x00, 0x00, 0x00 },
	/* 143CD */
	{ 0x00, 0xed, 0x00, 0xec, 0x00, 0xec, 0x84, 0x83, 0x83, 0x84, 0x84,
	  0x84, 0x84, 0x84, 0x83, 0x82, 0x83, 0x83, 0x81, 0x80, 0x81, 0x7f,
	  0x7e, 0x81, 0x70, 0x79, 0x81, 0x85, 0x85, 0x82, 0x00, 0x00, 0x00 },
	/* 152CD */
	{ 0x00, 0xf0, 0x00, 0xf0, 0x00, 0xf0, 0x83, 0x83, 0x83, 0x83, 0x83,
	  0x83, 0x84, 0x84, 0x83, 0x81, 0x81, 0x81, 0x80, 0x80, 0x81, 0x80,
	  0x80, 0x82, 0x6f, 0x78, 0x7f, 0x85, 0x85, 0x82, 0x00, 0x00, 0x00 },
	/* 162CD */
	{ 0x00, 0xf4, 0x00, 0xf3, 0x00, 0xf4, 0x83, 0x83, 0x83, 0x83, 0x83,
	  0x83, 0x82, 0x81, 0x81, 0x81, 0x81, 0x81, 0x80, 0x80, 0x81, 0x80,
	  0x7f, 0x82, 0x6f, 0x78, 0x7f, 0x85, 0x85, 0x82, 0x00, 0x00, 0x00 },
	/* 172CD */
	{ 0x00, 0xf8, 0x00, 0xf8, 0x00, 0xf8, 0x82, 0x82, 0x82, 0x82, 0x82,
	  0x82, 0x82, 0x81, 0x81, 0x80, 0x81, 0x80, 0x80, 0x80, 0x81, 0x81,
	  0x80, 0x83, 0x6d, 0x76, 0x7d, 0x85, 0x85, 0x82, 0x00, 0x00, 0x00 },
	/* 183CD */
	{ 0x00, 0xe0, 0x00, 0xdf, 0x00, 0xdf, 0x84, 0x84, 0x83, 0x86, 0x86,
	  0x86, 0x83, 0x84, 0x83, 0x82, 0x82, 0x82, 0x81, 0x83, 0x81, 0x81,
	  0x7e, 0x81, 0x80, 0x82, 0x84, 0x85, 0x85, 0x82, 0x00, 0x00, 0x00 },
	/* 195CD */
	{ 0x00, 0xe4, 0x00, 0xe3, 0x00, 0xe3, 0x84, 0x83, 0x83, 0x85, 0x85,
	  0x85, 0x83, 0x84, 0x83, 0x81, 0x82, 0x82, 0x82, 0x83, 0x81, 0x81,
	  0x80, 0x82, 0x7d, 0x7f, 0x81, 0x85, 0x85, 0x82, 0x00, 0x00, 0x00 },
	/* 207CD */
	{ 0x00, 0xe7, 0x00, 0xe6, 0x00, 0xe6, 0x83, 0x82, 0x82, 0x85, 0x85,
	  0x85, 0x82, 0x83, 0x83, 0x82, 0x82, 0x82, 0x80, 0x81, 0x80, 0x81,
	  0x80, 0x82, 0x7d, 0x7f, 0x81, 0x85, 0x85, 0x82, 0x00, 0x00, 0x00 },
	/* 220CD */
	{ 0x00, 0xeb, 0x00, 0xea, 0x00, 0xea, 0x83, 0x83, 0x82, 0x84, 0x84,
	  0x84, 0x82, 0x83, 0x82, 0x81, 0x81, 0x82, 0x81, 0x82, 0x81, 0x80,
	  0x7e, 0x80, 0x7d, 0x7f, 0x81, 0x85, 0x85, 0x82, 0x00, 0x00, 0x00 },
	/* 234CD */
	{ 0x00, 0xef, 0x00, 0xee, 0x00, 0xee, 0x83, 0x82, 0x82, 0x83, 0x83,
	  0x83, 0x82, 0x82, 0x82, 0x81, 0x81, 0x81, 0x80, 0x80, 0x80, 0x80,
	  0x80, 0x81, 0x7b, 0x7c, 0x7f, 0x85, 0x85, 0x82, 0x00, 0x00, 0x00 },
	/* 249CD */
	{ 0x00, 0xf3, 0x00, 0xf2, 0x00, 0xf2, 0x82, 0x81, 0x81, 0x83, 0x83,
	  0x83, 0x82, 0x82, 0x82, 0x81, 0x81, 0x81, 0x80, 0x81, 0x80, 0x7f,
	  0x7e, 0x7f, 0x7b, 0x7c, 0x7f, 0x85, 0x85, 0x82, 0x00, 0x00, 0x00 },
	/* 265CD */
	{ 0x00, 0xf7, 0x00, 0xf7, 0x00, 0xf7, 0x81, 0x81, 0x80, 0x82, 0x82,
	  0x82, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x81, 0x80, 0x7f,
	  0x7e, 0x7f, 0x7b, 0x7c, 0x7f, 0x85, 0x85, 0x82, 0x00, 0x00, 0x00 },
	/* 282CD */
	{ 0x00, 0xfb, 0x00, 0xfb, 0x00, 0xfb, 0x80, 0x80, 0x80, 0x81, 0x81,
	  0x81, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x7f, 0x7f, 0x7f, 0x7f,
	  0x7f, 0x7f, 0x78, 0x79, 0x7d, 0x85, 0x85, 0x82, 0x00, 0x00, 0x00 },
	/* 300CD */
	{ 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x80, 0x80, 0x80, 0x80, 0x80,
	  0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
	  0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00 },
};

static int s6e88a0_ams427ap24_set_brightness(struct backlight_device *bd)
{
	struct s6e88a0_ams427ap24 *ctx = bl_get_data(bd);
	struct mipi_dsi_device *dsi = ctx->dsi;
	struct mipi_dsi_multi_context dsi_ctx = { .dsi = dsi };
	struct device *dev = &dsi->dev;
	int brightness = bd->props.brightness;
	int candela_enum;
	u8 b2[LEN_AID] = { 0xb2, 0x40, 0x08, 0x20, 0x00, 0x00 };
	u8 b6[LEN_ELVSS] = { 0xb6, 0x28, 0x00 };
	u8 ca[LEN_GAMMA];

	/* get candela enum from brightness */
	for (candela_enum = 0; candela_enum < NUM_STEPS_CANDELA; candela_enum++)
		if (brightness <= s6e88a0_ams427ap24_br_to_cd[candela_enum])
			break;

	/* get aid */
	switch (candela_enum) {
	case CANDELA_10CD ... CANDELA_105CD:
		memcpy(&b2[FIX_LEN_AID],
		       s6e88a0_ams427ap24_aid[candela_enum],
		       VAR_LEN_AID);
		break;
	case CANDELA_111CD ... CANDELA_172CD:
		memcpy(&b2[FIX_LEN_AID],
		       s6e88a0_ams427ap24_aid[CANDELA_111CD],
		       VAR_LEN_AID);
		break;
	case CANDELA_183CD ... CANDELA_300CD:
		memcpy(&b2[FIX_LEN_AID],
		       s6e88a0_ams427ap24_aid[CANDELA_111CD + 1],
		       VAR_LEN_AID);
		break;
	default:
		dev_err(dev, "Failed to get aid data\n");
		return -EINVAL;
	}

	/* get elvss */
	if (candela_enum <= CANDELA_111CD) {
		memcpy(&b6[FIX_LEN_ELVSS],
		       s6e88a0_ams427ap24_elvss[0],
		       VAR_LEN_ELVSS);
	} else {
		memcpy(&b6[FIX_LEN_ELVSS],
		       s6e88a0_ams427ap24_elvss[candela_enum - CANDELA_111CD],
		       VAR_LEN_ELVSS);
	}

	/* get gamma */
	ca[0] = 0xca;
	memcpy(&ca[FIX_LEN_GAMMA],
	       s6e88a0_ams427ap24_gamma[candela_enum],
	       VAR_LEN_GAMMA);

	/* write data */
	mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0xf0, 0x5a, 0x5a); // level 1 key on
	mipi_dsi_dcs_write_buffer_multi(&dsi_ctx, b2, ARRAY_SIZE(b2)); // set aid
	mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0x55, 0x00); // acl off
	mipi_dsi_dcs_write_buffer_multi(&dsi_ctx, b6, ARRAY_SIZE(b6)); // set elvss
	mipi_dsi_dcs_write_buffer_multi(&dsi_ctx, ca, ARRAY_SIZE(ca)); // set gamma
	mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0xf7, 0x03); // gamma update
	mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0xf0, 0xa5, 0xa5); // level 1 key off

	return dsi_ctx.accum_err;
}

static void s6e88a0_ams427ap24_reset(struct s6e88a0_ams427ap24 *ctx)
{
	gpiod_set_value_cansleep(ctx->reset_gpio, 0);
	usleep_range(5000, 6000);
	gpiod_set_value_cansleep(ctx->reset_gpio, 1);
	usleep_range(1000, 2000);
	gpiod_set_value_cansleep(ctx->reset_gpio, 0);
	usleep_range(18000, 19000);
}

static int s6e88a0_ams427ap24_on(struct s6e88a0_ams427ap24 *ctx)
{
	struct mipi_dsi_device *dsi = ctx->dsi;
	struct mipi_dsi_multi_context dsi_ctx = { .dsi = dsi };
	struct device *dev = &dsi->dev;
	int ret;

	dsi->mode_flags |= MIPI_DSI_MODE_LPM;

	mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0xf0, 0x5a, 0x5a); // level 1 key on
	mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0xfc, 0x5a, 0x5a); // level 2 key on
	mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0xb0, 0x11); // src latch set global 1
	mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0xfd, 0x11); // src latch set 1
	mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0xb0, 0x13); // src latch set global 2
	mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0xfd, 0x18); // src latch set 2
	mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0xb0, 0x02); // avdd set 1
	mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0xb8, 0x30); // avdd set 2

	mipi_dsi_dcs_exit_sleep_mode_multi(&dsi_ctx);
	mipi_dsi_msleep(&dsi_ctx, 20);

	mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0xf1, 0x5a, 0x5a); // level 3 key on
	mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0xcc, 0x4c); // pixel clock divider pol.
	mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0xf2, 0x03, 0x0d); // unknown
	mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0xf1, 0xa5, 0xa5); // level 3 key off

	if (ctx->flip_horizontal)
		mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0xcb, 0x0e); // flip display

	mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0xf0, 0xa5, 0xa5); // level 1 key off
	mipi_dsi_dcs_write_seq_multi(&dsi_ctx, 0xfc, 0xa5, 0xa5); // level 2 key off

	ret = s6e88a0_ams427ap24_set_brightness(ctx->bl_dev);
	if (ret < 0) {
		dev_err(dev, "Failed to set brightness: %d\n", ret);
		return ret;
	}

	mipi_dsi_dcs_set_display_on_multi(&dsi_ctx);

	return dsi_ctx.accum_err;
}

static int s6e88a0_ams427ap24_off(struct s6e88a0_ams427ap24 *ctx)
{
	struct mipi_dsi_device *dsi = ctx->dsi;
	struct mipi_dsi_multi_context dsi_ctx = { .dsi = dsi };

	dsi->mode_flags &= ~MIPI_DSI_MODE_LPM;

	mipi_dsi_dcs_set_display_off_multi(&dsi_ctx);
	mipi_dsi_dcs_enter_sleep_mode_multi(&dsi_ctx);
	mipi_dsi_msleep(&dsi_ctx, 120);

	return dsi_ctx.accum_err;
}

static int s6e88a0_ams427ap24_prepare(struct drm_panel *panel)
{
	struct s6e88a0_ams427ap24 *ctx = to_s6e88a0_ams427ap24(panel);
	struct device *dev = &ctx->dsi->dev;
	int ret;

	ret = regulator_bulk_enable(ARRAY_SIZE(s6e88a0_ams427ap24_supplies),
				    ctx->supplies);
	if (ret < 0) {
		dev_err(dev, "Failed to enable regulators: %d\n", ret);
		return ret;
	}

	s6e88a0_ams427ap24_reset(ctx);

	ret = s6e88a0_ams427ap24_on(ctx);
	if (ret < 0) {
		dev_err(dev, "Failed to initialize panel: %d\n", ret);
		gpiod_set_value_cansleep(ctx->reset_gpio, 1);
		regulator_bulk_disable(ARRAY_SIZE(s6e88a0_ams427ap24_supplies),
				       ctx->supplies);
		return ret;
	}

	return 0;
}

static int s6e88a0_ams427ap24_unprepare(struct drm_panel *panel)
{
	struct s6e88a0_ams427ap24 *ctx = to_s6e88a0_ams427ap24(panel);
	struct device *dev = &ctx->dsi->dev;
	int ret;

	ret = s6e88a0_ams427ap24_off(ctx);
	if (ret < 0)
		dev_err(dev, "Failed to un-initialize panel: %d\n", ret);

	gpiod_set_value_cansleep(ctx->reset_gpio, 1);
	regulator_bulk_disable(ARRAY_SIZE(s6e88a0_ams427ap24_supplies),
			       ctx->supplies);

	return 0;
}

static const struct drm_display_mode s6e88a0_ams427ap24_mode = {
	.clock = (540 + 94 + 4 + 18) * (960 + 12 + 1 + 3) * 60 / 1000,
	.hdisplay = 540,
	.hsync_start = 540 + 94,
	.hsync_end = 540 + 94 + 4,
	.htotal = 540 + 94 + 4 + 18,
	.vdisplay = 960,
	.vsync_start = 960 + 12,
	.vsync_end = 960 + 12 + 1,
	.vtotal = 960 + 12 + 1 + 3,
	.width_mm = 55,
	.height_mm = 95,
	.type = DRM_MODE_TYPE_DRIVER | DRM_MODE_TYPE_PREFERRED,
};

static int s6e88a0_ams427ap24_get_modes(struct drm_panel *panel,
					struct drm_connector *connector)
{
	return drm_connector_helper_get_modes_fixed(connector,
						    &s6e88a0_ams427ap24_mode);
}

static const struct drm_panel_funcs s6e88a0_ams427ap24_panel_funcs = {
	.prepare = s6e88a0_ams427ap24_prepare,
	.unprepare = s6e88a0_ams427ap24_unprepare,
	.get_modes = s6e88a0_ams427ap24_get_modes,
};

static const struct backlight_ops s6e88a0_ams427ap24_bl_ops = {
	.update_status	= s6e88a0_ams427ap24_set_brightness,
};

static int s6e88a0_ams427ap24_register_backlight(struct s6e88a0_ams427ap24 *ctx)
{
	struct backlight_properties props = {
		.type		= BACKLIGHT_RAW,
		.brightness	= 180,
		.max_brightness = 255,
	};
	struct mipi_dsi_device *dsi = ctx->dsi;
	struct device *dev = &dsi->dev;
	int ret = 0;

	ctx->bl_dev = devm_backlight_device_register(dev, dev_name(dev), dev, ctx,
						     &s6e88a0_ams427ap24_bl_ops,
						     &props);
	if (IS_ERR(ctx->bl_dev)) {
		ret = PTR_ERR(ctx->bl_dev);
		dev_err(dev, "error registering backlight device (%d)\n", ret);
	}

	return ret;
}

static int s6e88a0_ams427ap24_probe(struct mipi_dsi_device *dsi)
{
	struct device *dev = &dsi->dev;
	struct s6e88a0_ams427ap24 *ctx;
	int ret;

	ctx = devm_drm_panel_alloc(dev, struct s6e88a0_ams427ap24, panel,
				   &s6e88a0_ams427ap24_panel_funcs,
				   DRM_MODE_CONNECTOR_DSI);
	if (IS_ERR(ctx))
		return PTR_ERR(ctx);

	ret = devm_regulator_bulk_get_const(dev,
				      ARRAY_SIZE(s6e88a0_ams427ap24_supplies),
				      s6e88a0_ams427ap24_supplies,
				      &ctx->supplies);
	if (ret < 0)
		return ret;

	ctx->reset_gpio = devm_gpiod_get(dev, "reset", GPIOD_OUT_HIGH);
	if (IS_ERR(ctx->reset_gpio))
		return dev_err_probe(dev, PTR_ERR(ctx->reset_gpio),
				     "Failed to get reset-gpios\n");

	ctx->dsi = dsi;
	mipi_dsi_set_drvdata(dsi, ctx);

	dsi->lanes = 2;
	dsi->format = MIPI_DSI_FMT_RGB888;
	dsi->mode_flags = MIPI_DSI_MODE_VIDEO | MIPI_DSI_MODE_VIDEO_BURST |
			  MIPI_DSI_MODE_NO_EOT_PACKET | MIPI_DSI_MODE_VIDEO_NO_HFP;

	ctx->panel.prepare_prev_first = true;

	ctx->flip_horizontal = device_property_read_bool(dev, "flip-horizontal");

	ret = s6e88a0_ams427ap24_register_backlight(ctx);
	if (ret < 0)
		return ret;

	drm_panel_add(&ctx->panel);

	ret = mipi_dsi_attach(dsi);
	if (ret < 0) {
		dev_err(dev, "Failed to attach to DSI host: %d\n", ret);
		drm_panel_remove(&ctx->panel);
		return ret;
	}

	return 0;
}

static void s6e88a0_ams427ap24_remove(struct mipi_dsi_device *dsi)
{
	struct s6e88a0_ams427ap24 *ctx = mipi_dsi_get_drvdata(dsi);
	int ret;

	ret = mipi_dsi_detach(dsi);
	if (ret < 0)
		dev_err(&dsi->dev, "Failed to detach from DSI host: %d\n", ret);

	drm_panel_remove(&ctx->panel);
}

static const struct of_device_id s6e88a0_ams427ap24_of_match[] = {
	{ .compatible = "samsung,s6e88a0-ams427ap24" },
	{ /* sentinel */ },
};
MODULE_DEVICE_TABLE(of, s6e88a0_ams427ap24_of_match);

static struct mipi_dsi_driver s6e88a0_ams427ap24_driver = {
	.probe = s6e88a0_ams427ap24_probe,
	.remove = s6e88a0_ams427ap24_remove,
	.driver = {
		.name = "panel-s6e88a0-ams427ap24",
		.of_match_table = s6e88a0_ams427ap24_of_match,
	},
};
module_mipi_dsi_driver(s6e88a0_ams427ap24_driver);

MODULE_AUTHOR("Jakob Hauser <jahau@rocketmail.com>");
MODULE_DESCRIPTION("Samsung AMS427AP24 panel with S6E88A0 controller");
MODULE_LICENSE("GPL v2");
