/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <stdlib.h>
#include <string.h>

#include "cdw_thread.h" /* PIPE_BUFFER_SIZE */
#include "cdw_regex_dispatch.h"
#include "cdw_debug.h"
#include "cdw_dvd_rw_mediainfo_regex.h"
#include "cdw_config.h"


/* plus_16 comment:
 * somewhere below you can find comment with plus_16 in it, commenting
 * line, where mysterious value of 16 is added to other value. 16 was guessed
 * after analyzing outputs from k3b and dvd+rw-mediainfo, and was confirmed
 * after reading k3b sources. The value appears in sources of k3b in
 * k3b-1.0.4/libk3b/projects/datacd/k3bmsinfofetcher.cpp in definition of
 * K3bMsInfoFetcher::slotMediaDetectionFinished().
 *
 * For some reason the value is added to value of starting address of last
 * track on dvd disc. I'm sure that there is some good explanation for this,
 * I'm just too lazy to read T10 specs ;-)
 */


extern char stdout_pipe_buffer[PIPE_BUFFER_SIZE + 1];
/* extern char stderr_pipe_buffer[PIPE_BUFFER_SIZE + 1]; */


extern cdw_config_t global_config;
extern cdw_task_t *thread_task;

static int cdw_dvd_rw_mediainfo_pr_handle_disc_status(regex_t *regex, regmatch_t *matches);
static int cdw_dvd_rw_mediainfo_pr_handle_dvd_write_speed(regex_t *regex, regmatch_t *matches);
static int cdw_dvd_rw_mediainfo_pr_handle_mounted_media(regex_t *regex, regmatch_t *matches);
static int cdw_dvd_rw_mediainfo_pr_handle_current_write_speed(regex_t *regex, regmatch_t *matches);
static int cdw_dvd_rw_mediainfo_pr_handle_read_capacity(regex_t *regex, regmatch_t *matches);
static int cdw_dvd_rw_mediainfo_pr_handle_end_lba(regex_t *regex, regmatch_t *matches);
static int cdw_dvd_rw_mediainfo_pr_handle_track_size(regex_t *regex, regmatch_t *matches);
static int cdw_dvd_rw_mediainfo_pr_handle_dvd_rom_detected(regex_t *regex, regmatch_t *matches);

/* valid but unused */
/*
static int cdw_dvd_rw_mediainfo_pr_handle_dvd_toc(regex_t *regex, regmatch_t *matches);
static int cdw_dvd_rw_mediainfo_pr_handle_track_start_address(regex_t *regex, regmatch_t *matches);
static int cdw_dvd_rw_mediainfo_pr_handle_next_writable_address(regex_t *regex, regmatch_t *matches);
*/

static cdw_regex_t stdout_regex[] = {
	{ "write speed", 1001,
	  /*  "Write Speed #0:        4.0x1385=5540KB/s" */
	  "Write Speed #([0-9]+):([\t ]+)([0-9]+).([0-9]+)x",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_dvd_rw_mediainfo_pr_handle_dvd_write_speed },

	{ "disc status", 1002,
	  /* "Disc status:           complete" (appendable/blank/other) */
	  "Disc status:([\t ]+)(complete|appendable|blank|other)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_dvd_rw_mediainfo_pr_handle_disc_status },

	{ "mounted media", 1003,
	  /* "Mounted Media:         1Ah, DVD-RW Sequential" */
	  "Mounted Media:([ ]+)([a-zA-Z0-9]+),([ ]+)([a-zA-Z +-]*)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_dvd_rw_mediainfo_pr_handle_mounted_media },

	{ "current write speed", 1004,
	  /* "Current Write Speed:   16.0x1385=22160KB/s" */
	  "Current Write Speed:([\t ]*)([0-9]+)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_dvd_rw_mediainfo_pr_handle_current_write_speed },

	{ "read capacity", 1005,
	  /* "READ CAPACITY:          0*2048=0" */
	  /* read capacity for empty disc is 0 */
	  "READ CAPACITY:([\t ]+)([0-9]+)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_dvd_rw_mediainfo_pr_handle_read_capacity },

	{ "DVD-ROM detected", 1006,
	  /* "DVD-ROM media detected, exiting..." */
	  "DVD-ROM media detected",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_dvd_rw_mediainfo_pr_handle_dvd_rom_detected },

	{ "end LBA", 1007,
	  /*
	   Speed Descriptor#0:    00/2295103 R@4.0x1385=5540KB/s W@16.0x1385=22160KB/s
	   Speed Descriptor#1:    00/2295103 R@4.0x1385=5540KB/s W@12.0x1385=16620KB/s
	   Speed Descriptor#2:    00/2295103 R@4.0x1385=5540KB/s W@10.0x1385=13850KB/s
	   Speed Descriptor#3:    00/2295103 R@4.0x1385=5540KB/s W@8.0x1385=11080KB/s
	   Speed Descriptor#4:    00/2295103 R@4.0x1385=5540KB/s W@4.0x1385=5540KB/s

	   The most interesting thing is after 00/ - this is end LBA;
	   check comment for end_lba field in src/disc_and_drive/cdw_disc.h
	   for more information */
	  "Speed Descriptor#([0-9]+):( +)([0-9a-fA-F]+)/([0-9]+) ",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_dvd_rw_mediainfo_pr_handle_end_lba },

	{ "track size", 1008,
	  /* data printed for every track size, including the last one (empty)
	   " Track Size:            1957744*2KB" */
	  "Track Size:( +)([0-9]+)([*])2KB",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_dvd_rw_mediainfo_pr_handle_track_size },
#if 0
	/* unused regexes */

	{ "DVD toc", 1009,
	  /* " Track#2  :             14@145056"
	   information about DVD TOC, last number is start of track;
	   track number may be also AA, whatever it means in DVD context */
	  "Track#([0-9A]+)([ ]*):([ ]*)([0-9]*)@([0-9]+)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_dvd_rw_mediainfo_pr_handle_dvd_toc },

	{ "track start address", 1010,
	  /* "Track Start Address:   666512" */
	  "Track Start Address:([ ]*)([0-9]+)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_dvd_rw_mediainfo_pr_handle_track_start_address },

	{ "next writable address", 1011,
	  /* "Next Writable Address: 470672*2KB"
	     start address for next session */
	  "Next Writable Address:([ ]*)([0-9]+)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_dvd_rw_mediainfo_pr_handle_next_writable_address },
#endif

	{ "table guard", -1,
	  /* guard: debug_id = -1 */
	  "",
	  (regex_t *) NULL, (regmatch_t *) NULL, 0,
	  (cdw_regex_handler_t) NULL }
};





/**
   \brief Do some initialization and call cdw_regex_prepare_regexes_table() for stdout regexes table
*/
void dvd_rw_mediainfo_stdout_regexp_prepare(void)
{
	cdw_rv_t crv = cdw_regex_prepare_regexes_table(stdout_regex);
	if (crv != CDW_OK) {
		cdw_vdm("ERROR: failed to prepare regexes table for stdout\n");
	}
	return;
}





/**
   \brief Do some initialization and call cdw_regex_execute_regexes_table()
   for stdout regexes table and stdout pipe buffer.
*/
void dvd_rw_mediainfo_stdout_regexp_execute(void)
{
	stdout_pipe_buffer[PIPE_BUFFER_SIZE] = '\0';
	cdw_regex_execute_regexes_table(stdout_regex, stdout_pipe_buffer);

	return;
}





/**
   \brief Call cdw_regex_clean_up_regexes_table() for stdout regexes table
*/
void dvd_rw_mediainfo_stdout_regexp_destroy(void)
{
	cdw_regex_clean_up_regexes_table(stdout_regex);
	return;
}





/**
   \brief Parse writing speeds reported by dvd+rw-mediainfo

   dvd+rw-mediainfo writes to stdout writing speeds in following format:
   \verbatim
Write Speed #0:        16.0x1385=22160KB/s
Write Speed #1:        12.0x1385=16620KB/s
Write Speed #2:        10.0x1385=13850KB/s
Write Speed #3:        8.0x1385=11080KB/s
Write Speed #4:        4.0x1385=5540KB/s
\endverbatim

   All that needs to be done is to convert #0 (#1, #2...) into number,
   use the number to index thread_task->disc->write_speeds.speeds[] table,
   and store writing speed in the table.
*/
int cdw_dvd_rw_mediainfo_pr_handle_dvd_write_speed(regex_t *regex, regmatch_t *matches)
{
	/* regexp for capturing write speeds is
                            1        2       3        4
           "Write Speed #([0-9]+):([\t ]+)([0-9]+).([0-9]+)x */
	cdw_regex_assert_subex_number(regex->re_nsub, 4);
	int speed_i = 0;

	for (unsigned int i = 0; i <= regex->re_nsub; i++) {
		char submatch[PIPE_BUFFER_SIZE];
		int len = cdw_regex_get_submatch(matches, i, stdout_pipe_buffer, submatch);
		if (len == -1) {
			cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", i, len);
			continue;
		}

		if (i == 1) {
			/* number of available speed */
			speed_i = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, speed index = \"%s\" -> %d\n", i, submatch, speed_i);

			/* this will work in both cases if dvd+rw-mediainfo
			   would list speed indexes from highest to lowest and
			   from lowest to highest */
			if (speed_i + 1 > thread_task->disc->write_speeds.n_speeds
			    && speed_i < CDW_DISC_N_SPEEDS_MAX) {

				thread_task->disc->write_speeds.n_speeds = speed_i + 1; /* number of elements in table */
			}
			cdw_vdm ("INFO: speed_i = %d, n_speeds = %d, CDW_DISC_N_SPEEDS_MAX = %d\n",
				 speed_i, thread_task->disc->write_speeds.n_speeds, CDW_DISC_N_SPEEDS_MAX);
		} else if (i == 3) {
			/* speed value; write_speeds[] has limited size */
			if (speed_i < CDW_DISC_N_SPEEDS_MAX) {
				thread_task->disc->write_speeds.speeds[speed_i] = atoi(submatch);
				cdw_vdm ("INFO: submatch %d: speed %d = \"%s\" -> %d\n",
					 i, speed_i, submatch, thread_task->disc->write_speeds.speeds[speed_i]);
			}
		} else if (i == 4) {
			cdw_vdm ("INFO: submatch %d: remainder of regexp = \"%s\"\n", i, submatch);
			/* fractional part of writing speed, not too
			   interesting, since writing speeds are integers;
			   there may be DVD discs with 2.5 speed, but I haven't
			   seen one yet, so I can't test this */
		}
	}

	return 0;
}





/**
   \brief Handle string describing status of disc

   Handle string printed by dvd+rw-mediainfo that has form of:
   "Disc status:           xxxxxxx"

   \verbatim
"Disc status:           complete"
\endverbatim
 *
 * Disc status for DVD can be one of following:
 * "blank", "appendable", "complete", "other"
 */
int cdw_dvd_rw_mediainfo_pr_handle_disc_status(regex_t *regex, regmatch_t *matches)
{
	/* regexp for capturing disc status:
                           1                   2
	   "Disc status:([\t ]+)(complete|appendable|blank|other) */
	cdw_regex_assert_subex_number(regex->re_nsub, 2);

	unsigned int sub = 2; /* select second regexp submatch */
	char submatch[PIPE_BUFFER_SIZE];
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	} else {
		/* yes, using only first letter as information about
		   status reported by dvd+rw-mediainfo - all statuses
		   have distinct first letters;
		   assigning values to disc->state_* has been moved to
		   dvd_rw_mediainfo_interface.c */
		thread_task->disc->dvd_rw_mediainfo_info.disc_status = submatch[0];
		cdw_vdm ("INFO: disc status \"%s\" saved as '%c'\n",
			 submatch,
			 thread_task->disc->dvd_rw_mediainfo_info.disc_status);
		return 0;
	}
}





/**
   \brief Extract information about media type

   Extract information (media type) from dvd+rw-mediainfo
   string which looks like this:

   \verbatim
" Mounted Media:         1Bh, DVD+R"

                 1         2          3        4
"Mounted Media:([ ]+)([a-zA-Z0-9]+),([ ]+)([\-a-zA-Z +]*)"
\endverbatim

   This function sets thread_task->disc->type
*/
int cdw_dvd_rw_mediainfo_pr_handle_mounted_media(regex_t *regex, regmatch_t *matches)
{
	/* regexp for capturing media type:
                            1        2           3         4
	   "Mounted Media:([ ]+)([a-zA-Z0-9]+),([ ]+)([a-zA-Z +-]*)"
	   2 - coded value
	   4 - human-readable name */
	cdw_regex_assert_subex_number(regex->re_nsub, 4);

	unsigned int sub = 4;
	char submatch[PIPE_BUFFER_SIZE];
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	}

	cdw_vdm ("INFO submatch %d, media type = \"%s\"\n", sub, submatch);

	/* convert string value into int value so that
	   higher level code doesn't have to compare strings */
	if (!strncmp(submatch, "DVD-R Sequential", 16)) {
		thread_task->disc->type = CDW_DVD_R_SEQ;

	} else if (!strncmp(submatch, "DVD-R Restricted", 16)) {
		thread_task->disc->type = CDW_DVD_R_RES;

	} else if (!strcmp(submatch, "DVD-R")) {
		thread_task->disc->type = CDW_DVD_R;

	} else if (!strncmp(submatch, "DVD-RW Sequential", 17)) {
		thread_task->disc->type = CDW_DVD_RW_SEQ;

	} else if (!strncmp(submatch, "DVD-RW Restricted", 17)) {
		thread_task->disc->type = CDW_DVD_RW_RES;

	} else if (!strncmp(submatch, "DVD-RW", 6)) {
		thread_task->disc->type = CDW_DVD_RW;

	} else if (!strcmp(submatch, "DVD+R")) {
		thread_task->disc->type = CDW_DVD_RP;

	} else if (!strcmp(submatch, "DVD+RW")) {
		thread_task->disc->type = CDW_DVD_RWP;

	} else if (!strcmp(submatch, "DVD-ROM")) {
		thread_task->disc->type = CDW_DVD_ROM;

	} else if (!strcmp(submatch, "DVD+R Double Layer")
		   && global_config.general.support_dvd_rp_dl) {
		thread_task->disc->type = CDW_DVD_RP_DL;

	} else {
		thread_task->disc->type = CDW_DISC_TYPE_UNKNOWN;
		cdw_vdm ("WARNING: unknown disc type, submatch was \"%s\"\n", submatch);
	}

	return 0;
}





int cdw_dvd_rw_mediainfo_pr_handle_current_write_speed(regex_t *regex, regmatch_t *matches)
{
	/* regexp for capturing current write speed is
                                   1       2
	   "Current Write Speed:([\t ]*)([0-9]+)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 2);

	unsigned int sub = 2;
	char submatch[PIPE_BUFFER_SIZE];
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	}

	/* current write speed */
	thread_task->disc->write_speeds.drive_default_speed = atoi(submatch);
	cdw_vdm ("INFO: submatch %d, current write speed = \"%s\" -> %d\n",
		 sub, submatch, thread_task->disc->write_speeds.drive_default_speed);

	return 0;
}





/**
 \verbatim
"READ CAPACITY:          0*2048=0"
\endverbatim
*/
int cdw_dvd_rw_mediainfo_pr_handle_read_capacity(regex_t *regex, regmatch_t *matches)
{
	/* regexp for capturing read capacity of DVD disc is
                             1       2
	   "READ CAPACITY:([\t ]+)([0-9]+)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 2);

	unsigned int sub = 2;
	char submatch[PIPE_BUFFER_SIZE];
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	}
	thread_task->disc->dvd_rw_mediainfo_info.read_capacity = (ssize_t) atoll(submatch);

	/* since regexp code matched this submatch as ([0-9]+), then value
	   '0' possibly returned by atoi() is not a sign of incorrectly
	   formed string in its argument, but it means that the arguments
	   translates to zero */
	cdw_vdm ("INFO: submatch %d, read capacity = \"%s\" -> %zd\n",
		 sub, submatch, thread_task->disc->dvd_rw_mediainfo_info.read_capacity);
	cdw_vdm ("INFO: read capacity = %zd\n", thread_task->disc->dvd_rw_mediainfo_info.read_capacity);

	return 0;
}





int cdw_dvd_rw_mediainfo_pr_handle_end_lba(regex_t *regex, regmatch_t *matches)
{
	/* regexp for capturing current write speed is
                                1      2        3          4
	   "Speed Descriptor#([0-9]+):( +)([0-9a-fA-F]+)/([0-9]+) " */
	cdw_regex_assert_subex_number(regex->re_nsub, 4);

	unsigned int sub = 4;
	char submatch[PIPE_BUFFER_SIZE];
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	}

	long long int end_lba = strtoll(submatch, (char **) NULL, 10);
	if (end_lba > thread_task->disc->dvd_rw_mediainfo_info.end_lba) {
		cdw_vdm ("INFO: end LBA = %lld\n", end_lba);
		thread_task->disc->dvd_rw_mediainfo_info.end_lba = end_lba;
	}

	return 0;

}





int cdw_dvd_rw_mediainfo_pr_handle_track_size(regex_t *regex, regmatch_t *matches)
{
	/* regexp for capturing track size information is
                        1      2    3
	   "Track Size:( +)([0-9]+)(*)2KB" */
	cdw_regex_assert_subex_number(regex->re_nsub, 3);

	unsigned int sub = 2;
	char submatch[PIPE_BUFFER_SIZE];
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	}

	long long int track_size = strtoll(submatch, (char **) NULL, 10);
	if (thread_task->disc->dvd_rw_mediainfo_info.all_tracks_size == -1) {
		/* init */
		thread_task->disc->dvd_rw_mediainfo_info.all_tracks_size = 0;
	}
	thread_task->disc->dvd_rw_mediainfo_info.all_tracks_size += track_size;
	/* current track may be the last one */
	thread_task->disc->dvd_rw_mediainfo_info.last_track_size = track_size;
	cdw_vdm ("INFO: all tracks size = %lld\n", thread_task->disc->dvd_rw_mediainfo_info.all_tracks_size);
	cdw_vdm ("INFO: last track size = %lld\n", thread_task->disc->dvd_rw_mediainfo_info.last_track_size);

	return 0;
}





int cdw_dvd_rw_mediainfo_pr_handle_dvd_rom_detected(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	/* this isn't necessary, because
	   cdw_dvd_rw_mediainfo_pr_handle_mounted_media()
	   can recognize DVD-ROM, but it doesn't hurt either */
	thread_task->disc->type = CDW_DVD_ROM;

	return 0;
}




/* *** valid, but unused code below */

#if 0

/**
 * Handle output of dvd+rw-mediainfo when it prints TOC table
 *
 * TOC printed by dvd+rw-mediainfo looks like this:
 * \verbatim
FABRICATED TOC:
 Track#1  :             14@0
 Track#2  :             14@145056
 Track#3  :             14@309712
 Track#4  :             14@470672
 Track#AA :             14@601376
\endverbatim
 *
 * Note that there is a track number, so we can use it to index writing speed
 * table. Note also that tracks are numbered from 1.
 *
 * The function has similar effect to handle_track_start_address() but it does
 * not capture start of blank area. However it does capture start of AA area.
 */
int cdw_dvd_rw_mediainfo_pr_handle_dvd_toc(regex_t *regex, regmatch_t *matches)
{
	char track_index[3];
	char track_start_address[10];
	int i_track;

	/* collect data */
	unsigned int i;
	for (i = 0; i < (regex->re_nsub + 1); ++i) {
		int len = matches[i].rm_eo - matches[i].rm_so;
		char submatch[PIPE_BUFFER_SIZE];
		strncpy(submatch, (char *) &stdout_pipe_buffer[matches[i].rm_so], len);
		submatch[len] = '\0';
		cdw_sdm ("matching submatch: \"%s\"\n", submatch);
		if (i == 1) {
			cdw_sdm ("submatch%d: \"%s\"\n", i, submatch);

			/* track index, 1-based */
			strncpy(track_index, submatch, len + 1);

			/* track number reported by dvd+rw-mediainfo can be
			   either a number or "AA" string */
			if (!strncmp(track_index, "AA", 2)) {
				i_track = 99;
			} else {
				i_track = atoi(track_index);
			}
		} else if (i == 5) {

			cdw_sdm ("submatch%d: \"%s\"\n", i, submatch);

			/* start of i-th track */
			strncpy(track_start_address, submatch, len + 1);

			/* track numbers reported by dvd+rw-mediainfo are
			   1-based, we will keep it that way; this will be also
			compatible with cdio */
			thread_task->disc->dvd_rw_mediainfo_info.track_address[i_track] = atol(track_start_address);
		} else {
			;
		}
	}

	return 0;
}




/**
   \brief Handle dvd+rw-mediainfo strings with track start indexes

   dvd+rw-mediainfo output looks like this:
   \verbatim
   Track Start Address:   470672*2KB
   \endverbatim
   and it is repeated for every track. We use
   thread_task->disc->dvd_rw_mediainfo_info.n_tracks to index
   thread_task->disc->dvd_rw_mediainfo_info.track_address[] and point where to store
   successive track start addresses.

   The function has similar effect to handle_dvd_toc(), but it also captures
   starting sector of blank area, which is used as part of '-C' parameter
   of growisofs when writing multisession discs.

   handle_track_start_address() does not capture AA track.
   handle_dvd_toc() does it.
*/
int cdw_dvd_rw_mediainfo_pr_handle_track_start_address(regex_t *regex, regmatch_t *matches)
{
	unsigned int i;
	for (i = 0; i < (regex->re_nsub + 1); ++i) {
		int len = matches[i].rm_eo - matches[i].rm_so;
		if (len < 0) {
			cdw_assert (len < 0, "len of subexpr is negative: %d\n", len);
			continue;
		}
		char submatch[PIPE_BUFFER_SIZE];
		strncpy(submatch, (char *) &stdout_pipe_buffer[matches[i].rm_so], (size_t) len);
		submatch[len] = '\0';

		cdw_sdm ("matching submatch: \"%s\"\n", submatch);
		if (i == 2) {
			cdw_sdm ("submatch%d: \"%s\"\n", i, submatch);
			int tsa = atoi(submatch); /* track start address */
			thread_task->disc->dvd_rw_mediainfo_info.track_address[thread_task->disc->dvd_rw_mediainfo_info.n_tracks++] = tsa;
		}
	}

	return 0;
}





/**
 * Capture address of first sector of next track/session
 *
 * dvd+rw-mediainfo prints following information about "empty" track, the
 * track that follows last track written to disc:
 * \verbatim
STDOUT READ TRACK INFORMATION[#4]:
 Track State:           blank
 Track Start Address:   470672*2KB
 Next Writable Address: 470672*2KB
 Free Blocks:           1824432*2KB
 Track Size:            1824432*2KB
\endverbatim
 *
 * The string that we must capture and process is
 * "Next Writable Address: 470672*2KB". This string is written after all data
 * is processed by handle_track_start_address().
 * Therefore we can use:
 *
 * (thread_task->disc->dvd_rw_mediainfo_info.n_tracks - 1)
 * to get
 * last session start from thread_task->disc->dvd_rw_mediainfo_info.track_address[]
 *
 * (thread_task->disc->dvd_rw_mediainfo_info.n_tracks)
 * to get
 * next (blank) session start from thread_task->disc->dvd_rw_mediainfo_info.track_address[]
 *
 * However "next (blank) session start" is not quite the same as "Next Writable
 * Address":
 * It may seem that values of "Track Start Address" and
 * "Next Writable Address" are the same, but dvd+rw-mediainfo uses two
 * different expressions to calculate the values. I use
 * "Next Writable Address" (nwa) instead of '"Track Start Address" of
 * blank track' as value of next_sess_start.
 */
int cdw_dvd_rw_mediainfo_pr_handle_next_writable_address(regex_t *regex, regmatch_t *matches)
{
	long int nwa = 0;

	/* collect data */
	unsigned int i;
	for (i = 0; i < (regex->re_nsub + 1); ++i) {
		int len = matches[i].rm_eo - matches[i].rm_so;
		if (len < 0) {
			cdw_assert (len < 0, "len of subexpr is negative: %d\n", len);
			continue;
		}
		char submatch[PIPE_BUFFER_SIZE];
		strncpy(submatch, (char *) &stdout_pipe_buffer[matches[i].rm_so], (size_t) len);
		submatch[len] = '\0';

		cdw_sdm ("matching submatch: \"%s\"\n", submatch);
		if (i == 2) {
			cdw_sdm ("submatch%d: \"%s\"\n", i, submatch);

			/* next writable address */
			nwa = atol(submatch);
		} else {
			;
		}
	}


	if (thread_task->disc->dvd_rw_mediainfo_info.n_tracks == 0) {
		/* no tracks, blank disc, so both last_sess_start
		   and next_sess_start should be 0 */
		thread_task->disc->last_sess_start = 0;
		thread_task->disc->next_sess_start = 0;
	} else {
		/* see comment at the beginning of file, starting with plus_16 */
		thread_task->disc->last_sess_start = thread_task->disc->dvd_rw_mediainfo_info.track_address[thread_task->disc->dvd_rw_mediainfo_info.n_tracks - 1] + 16;
		thread_task->disc->next_sess_start = nwa;
	}


	cdw_sdm ("thread_task->disc->last_sess_start - 16 = %ld\n", thread_task->disc->last_sess_start - 16);
	cdw_sdm ("thread_task->disc->last_sess_start = %ld\n", thread_task->disc->last_sess_start);
	cdw_sdm ("thread_task->disc->next_sess_start = %ld\n", thread_task->disc->next_sess_start);

	return 0;
}

#endif
