/*
 * Author: Andrei Zavada <johnhommer@gmail.com>
 *         building on original work by Thomas Nowotny
 *
 * License: GPL-2+
 *
 * Initial version: 2008-09-02
 *
 * CModel household
 */

#include <sys/time.h>
#include <iostream>
#include <set>
#include <algorithm>

#include <regex.h>

#include "../libstilton/pointaligned-s.hh"

#include "model.hh"

#include "config.h"

using namespace std;



CNRun::CModel::
CModel( const char *inname, CIntegrate_base *inintegrator, int instatus)
      : name (inname),
	_status (instatus | CN_MDL_NOTREADY),
	_global_unit_id_reservoir (0l),
	_longest_label (1),
	_var_cnt (1),			// reserve [0] for model_time
	_cycle (0),
	_discrete_time (0.),  _discrete_dt (NAN),
	spike_threshold (0.),
	spike_lapse (5.),
	listen_dt (0),
	_dt_logger (nullptr),
	_spike_logger (nullptr),	// open these streams at first write instead in prepare_advance()
	verbosely (1)
{
	V.resize( _var_cnt), W.resize( _var_cnt);
	V[0] = 0.;

	(_integrator = inintegrator) -> model = this;

	{
		const gsl_rng_type * T;
		gsl_rng_env_setup();
		T = gsl_rng_default;
		if ( gsl_rng_default_seed == 0 ) {
			struct timeval tp = { 0L, 0L };
			gettimeofday( &tp, nullptr);
			gsl_rng_default_seed = tp.tv_usec;
		}
		_rng = gsl_rng_alloc( T);
	}

	signal( SIGINT, SIG_IGN);
}


CNRun::CModel::
~CModel()
{
	if ( verbosely > 4 )
		fprintf( stdout, "Deleting all units...\n");

	while (unit_list.size())
		if ( unit_list.back() -> is_owned() )
			delete unit_list.back();
		else
			unit_list.pop_back();

	if ( _integrator->is_owned )
		delete _integrator;

	delete _dt_logger;
	delete _spike_logger;

	while ( Sources.size() ) {
		delete Sources.back();
		Sources.pop_back();
	}

	gsl_rng_free( _rng);
}


void
CNRun::CModel::
reset( bool also_reset_params)
{
	_cycle = 0, V[0] = 0.;

	_integrator->dt = _integrator->_dt_min;

	reset_state_all_units();
	if ( also_reset_params )
		for_all_units (U)
			(*U)->reset_params();

	regular_periods.clear();
	regular_periods_last_checked.clear();
  // this will cause scheduler_update_periods_* to be recomputed by prepare_advance()
	_status |= CN_MDL_NOTREADY;

	if ( _status & CN_MDL_LOGDT ) {
		delete _dt_logger;
		string	fname = name + ".dtlog";
		_dt_logger = new ofstream( fname.data());
	}
	if ( _status & CN_MDL_LOGSPIKERS ) {
		delete _spike_logger;
		string	fname = name + ".spikes";
		_spike_logger = new ofstream( fname.data());
	}
}








CNRun::C_BaseUnit*
CNRun::CModel::
unit_by_label( const char *inlabel) const
{
	for_all_units_const (U)
		if ( strcmp( (*U)->_label, inlabel) == 0 )
			return *U;
	return nullptr;
}

CNRun::C_BaseNeuron*
CNRun::CModel::
neuron_by_label( const char *inlabel) const
{
	for_all_units_const (U)
		if ( (*U)->is_neuron() && strcmp( (*U)->_label, inlabel) == 0 )
			return static_cast<C_BaseNeuron*>(*U);
	return nullptr;
}

CNRun::C_BaseSynapse*
CNRun::CModel::
synapse_by_label( const char *inlabel) const
{
	for_all_units_const (U)
		if ( (*U)->is_synapse() && strcmp( (*U)->_label, inlabel) == 0 )
			return static_cast<C_BaseSynapse*>(*U);
	return nullptr;
}







// ----- registering units with core lists
void
CNRun::CModel::
_include_base_unit( C_BaseUnit* u)
{
	for_all_units (U)
		if ( (*U) == u ) {
			fprintf( stderr, "Unit %s found already included in model %s\n", u->_label, name.c_str());
			goto skip_ul_pushback;
		}
	unit_list.push_back( u);
skip_ul_pushback:

	if ( verbosely > 5 )
		fprintf( stdout, "  registered base unit %s\n", u->_label);

	if ( u->has_sources() )
		register_unit_with_sources( u);

	if ( u->is_listening() ) {
		for_all_listening_units (U)
			if ( (*U) == u ) {
				fprintf( stderr, "Unit \"%s\" already on listening list\n", u->_label);
				goto skip_lisn_reg;
			}
		lisn_unit_list.push_back( u);
	}
skip_lisn_reg:

	u->M = this;

	u->_serial_id = _global_unit_id_reservoir++;
}




int
CNRun::CModel::
include_unit( C_HostedNeuron *u, bool is_last)
{
	_include_base_unit( u);

	u->idx = _var_cnt;
	_var_cnt += u->v_no();

	hosted_neu_list.push_back( u);

	// if ( u->_spikelogger_agent  &&  !(u->_spikelogger_agent->_status & CN_KL_IDLE) )
	// 	spikelogging_neu_list.push_back( u);

	if ( u->is_conscious() )
		conscious_neu_list.push_back( u);

	if ( is_last )
		finalize_additions();

	return 0;
}

int
CNRun::CModel::
include_unit( C_HostedSynapse *u, bool is_last)
{
	_include_base_unit( u);

	u->idx = _var_cnt;
	_var_cnt += u->v_no();

	hosted_syn_list.push_back( u);

	if ( u->traits() & UT_MULTIPLEXING )
		mx_syn_list.push_back( u);

	if ( is_last )
		finalize_additions();

	return 0;
}



int
CNRun::CModel::
include_unit( C_StandaloneNeuron *u)
{
	_include_base_unit( u);

	// if ( u->_spikelogger_agent  &&  !(u->_spikelogger_agent->_status & CN_KL_IDLE) )
	// 	spikelogging_neu_list.push_back( u);

	if ( u->is_conscious() )
		conscious_neu_list.push_back( u);

	if ( u->is_ddtbound() )
		ddtbound_neu_list.push_back( u);
	else
		standalone_neu_list.push_back( u);

	return 0;
}


int
CNRun::CModel::
include_unit( C_StandaloneSynapse *u)
{
/*
	if ( _check_new_synapse( u) ) {
//		u->enable( false);
		u->M = nullptr;
		return -1;
	}
*/
	_include_base_unit( u);

	if ( u->is_ddtbound() )
		ddtbound_syn_list.push_back( u);
	else
		standalone_syn_list.push_back( u);

	if ( u->traits() & UT_MULTIPLEXING )
		mx_syn_list.push_back( u);

	return 0;
}



// preserve the unit if !do_delete, so it can be re-included again
CNRun::C_BaseUnit*
CNRun::CModel::
exclude_unit( C_BaseUnit *u, bool do_delete)
{
	if ( __cn_verbosely > 5 )
		fprintf( stderr, "-excluding unit \"%s\"", u->_label);

	if ( u->has_sources() )
		unregister_unit_with_sources( u);

	if ( u->is_listening() )
		u->stop_listening();  // also calls unregister_listener

	if ( u->is_synapse() && u->traits() & UT_MULTIPLEXING )
		mx_syn_list.erase( find( mx_syn_list.begin(), mx_syn_list.end(), u));

	if ( u->is_conscious() )
		conscious_neu_list.erase( find(conscious_neu_list.begin(), conscious_neu_list.end(), u));

	if ( u->is_hostable() ) {
		size_t	our_idx;
		if ( u->is_neuron() ) {
			hosted_neu_list.erase( find( hosted_neu_list.begin(), hosted_neu_list.end(), u));
			our_idx = ((C_HostedNeuron*)u) -> idx;
		} else {
			hosted_syn_list.erase( find( hosted_syn_list.begin(), hosted_syn_list.end(), u));
			our_idx = ((C_HostedSynapse*)u) -> idx;
		}

	      // shrink V
		if ( __cn_verbosely > 5 )
			fprintf( stderr, " (shrink V by %d)", u->v_no());
		for_all_hosted_neurons (N)
			if ( (*N)->idx > our_idx )
				(*N)->idx -= u->v_no();
		for_all_hosted_synapses (Y)
			if ( (*Y)->idx > our_idx )
				(*Y)->idx -= u->v_no();
		memmove( &V[our_idx], &V[our_idx+u->v_no()], (_var_cnt - our_idx - u->v_no()) * sizeof(double));
		V.resize( _var_cnt -= u->v_no());
	}
	if ( u->is_ddtbound() ) {
		if ( u->is_neuron() )
			ddtbound_neu_list.erase( find( ddtbound_neu_list.begin(), ddtbound_neu_list.end(), u));
		else
			ddtbound_syn_list.erase( find( ddtbound_syn_list.begin(), ddtbound_syn_list.end(), u));
	}
	if ( !u->is_hostable() ) {
		if ( u->is_neuron() )
			standalone_neu_list.erase( find( standalone_neu_list.begin(), standalone_neu_list.end(), u));
		else
			standalone_syn_list.erase( find( standalone_syn_list.begin(), standalone_syn_list.end(), u));
	}

	unit_list.erase( find( unit_list.begin(), unit_list.end(), u));

	if ( do_delete ) {
		delete u;
		u = nullptr;
	} else
		u->M = nullptr;

	if ( __cn_verbosely > 5 )
		fprintf( stderr, ".\n");
	return u;
}







// listeners & spikeloggers

void
CNRun::CModel::
register_listener( C_BaseUnit *u)
{
	if ( find( lisn_unit_list.begin(), lisn_unit_list.end(), u) == lisn_unit_list.end() )
		lisn_unit_list.push_back( u);
}

void
CNRun::CModel::
unregister_listener( C_BaseUnit *u)
{
	const auto& U = find( lisn_unit_list.begin(), lisn_unit_list.end(), u);
	if ( U != lisn_unit_list.end() )
		lisn_unit_list.erase( U);
}







void
CNRun::CModel::
register_spikelogger( C_BaseNeuron *n)
{
	spikelogging_neu_list.push_back( n);
	spikelogging_neu_list.sort();
	spikelogging_neu_list.unique();
}

void
CNRun::CModel::
unregister_spikelogger( C_BaseNeuron *n)
{
	for_all_spikelogging_neurons (N)
		if ( (*N) == n ) {
			spikelogging_neu_list.erase( N);
			return;
		}
}










// units with sources

void
CNRun::CModel::
register_unit_with_sources( C_BaseUnit *u)
{
	for ( auto& I : u->sources )
		if ( I.source->is_periodic() )
			units_with_periodic_sources.push_back( u);
		else
			units_with_continuous_sources.push_back( u);
	units_with_continuous_sources.unique();
	units_with_periodic_sources.unique();
}

void
CNRun::CModel::
unregister_unit_with_sources( C_BaseUnit *u)
{
start_over_1:
	for_all_units_with_contiuous_sources (U)
		if ( (*U) == u ) {
			units_with_continuous_sources.erase( U);
			if ( verbosely > 5 )
				fprintf( stderr, " (removed \"%s\" instance from units w/ continuous sources list)\n", u->_label);
			goto start_over_1;
		}
start_over_2:
	for_all_units_with_periodic_sources (U)
		if ( (*U) == u ) {
			units_with_periodic_sources.erase( U);
			if ( verbosely > 5 )
				fprintf( stderr, " (removed \"%s\" instance from units w/ periodic sources list)\n", u->_label);
			goto start_over_2;
		}
}
















CNRun::C_BaseNeuron*
CNRun::CModel::
add_neuron_species( const char *type_s, const char *label, bool finalize,
		    double x, double y, double z)
{
	TUnitType t = unit_species_by_string( type_s);
	if ( t == NT_VOID || !unit_species_is_neuron(type_s) ) {
		fprintf( stderr, "Unrecognised neuron species: \"%s\"\n", type_s);
		return nullptr;
	} else
		return add_neuron_species( t, label, finalize, x, y, z);
}

CNRun::C_BaseNeuron*
CNRun::CModel::
add_neuron_species( TUnitType type, const char *label, bool finalize,
		    double x, double y, double z)
{
	C_BaseNeuron *n;
	switch ( type ) {
	case NT_HH_D:
		n = new CNeuronHH_d( label, x, y, z, this, CN_UOWNED, finalize);
	    break;
	case NT_HH_R:
		n = new CNeuronHH_r( label, x, y, z, this, CN_UOWNED);
	    break;

	case NT_HH2_D:
		n = new CNeuronHH2_d( label, x, y, z, this, CN_UOWNED, finalize);
	    break;
	// case NT_HH2_R:
	// 	n = new CNeuronHH2_r( label, x, y, z, this, CN_UOWNED, finalize);
	//     break;
//#ifdef CN_WANT_MORE_NEURONS
	case NT_EC_D:
		n = new CNeuronEC_d( label, x, y, z, this, CN_UOWNED, finalize);
	    break;
	case NT_ECA_D:
		n = new CNeuronECA_d( label, x, y, z, this, CN_UOWNED, finalize);
	    break;
/*
	case NT_LV:
		n = new COscillatorLV( label, x, y, z, this, CN_UOWNED, finalize);
	    break;
 */
	case NT_COLPITTS:
		n = new COscillatorColpitts( label, x, y, z, this, CN_UOWNED, finalize);
	    break;
	case NT_VDPOL:
		n = new COscillatorVdPol( label, x, y, z, this, CN_UOWNED, finalize);
	    break;
//#endif
	case NT_DOTPOISSON:
		n = new COscillatorDotPoisson( label, x, y, z, this, CN_UOWNED);
	    break;
	case NT_POISSON:
		n = new COscillatorPoisson( label, x, y, z, this, CN_UOWNED);
	    break;

	case NT_DOTPULSE:
		n = new CNeuronDotPulse( label, x, y, z, this, CN_UOWNED);
	    break;

	case NT_MAP:
		n = new CNeuronMap( label, x, y, z, this, CN_UOWNED);
	    break;

	default:
		return nullptr;
	}
	if ( n && n->_status & CN_UERROR ) {
		delete n;
		return nullptr;
	}
	return n;
}








CNRun::C_BaseSynapse*
CNRun::CModel::
add_synapse_species( const char *type_s, const char *src_l, const char *tgt_l,
		     double g, bool allow_clone, bool finalize)
{
	TUnitType ytype = unit_species_by_string( type_s);
	bool	given_species = true;
	if ( ytype == NT_VOID && (given_species = false, ytype = unit_family_by_string( type_s)) == NT_VOID ) {
		fprintf( stderr, "Unrecognised synapse species or family: \"%s\"\n", type_s);
		return nullptr;
	}

	C_BaseNeuron
		*src = neuron_by_label( src_l),
		*tgt = neuron_by_label( tgt_l);
	if ( !src || !tgt ) {
		fprintf( stderr, "Phoney source (\"%s\") or target (\"%s\")\n", src_l, tgt_l);
		return nullptr;
	}

	if ( given_species )  // let lower function do the checking
		return add_synapse_species( ytype, src, tgt, g, allow_clone, finalize);

	switch ( ytype ) {
      // catch by first entry in __CNUDT, assign proper species per source and target traits
	case YT_AB_DD:
		if ( src->traits() & UT_RATEBASED && tgt->traits() & UT_RATEBASED )
			ytype = YT_AB_RR;
		else if ( src->traits() & UT_RATEBASED && !(tgt->traits() & UT_RATEBASED) )
			ytype = YT_AB_RD;
		else if ( !(src->traits() & UT_RATEBASED) && tgt->traits() & UT_RATEBASED )
			if ( src->traits() & UT_DOT )
				ytype = YT_MXAB_DR;
			else
				ytype = YT_AB_DR;
		else
			if ( src->traits() & UT_DOT )
				ytype = YT_MXAB_DD;
			else
				ytype = YT_AB_DD;
	    break;

	case YT_ABMINUS_DD:
		if ( src->traits() & UT_RATEBASED && tgt->traits() & UT_RATEBASED )
			ytype = YT_ABMINUS_RR;
		else if ( src->traits() & UT_RATEBASED && !(tgt->traits() & UT_RATEBASED) )
			ytype = YT_ABMINUS_RD;
		else if ( !(src->traits() & UT_RATEBASED) && tgt->traits() & UT_RATEBASED )
			if ( src->traits() & UT_DOT )
				ytype = YT_MXABMINUS_DR;
			else
				ytype = YT_ABMINUS_DR;
		else
			if ( src->traits() & UT_DOT )
				ytype = YT_MXABMINUS_DD;
			else
				ytype = YT_ABMINUS_DD;
	    break;

	case YT_RALL_DD:
		if ( src->traits() & UT_RATEBASED && tgt->traits() & UT_RATEBASED )
			ytype = YT_RALL_RR;
		else if ( src->traits() & UT_RATEBASED && !(tgt->traits() & UT_RATEBASED) )
			ytype = YT_RALL_RD;
		else if ( !(src->traits() & UT_RATEBASED) && tgt->traits() & UT_RATEBASED )
			if ( src->traits() & UT_DOT )
				ytype = YT_MXRALL_DR;
			else
				ytype = YT_RALL_DR;
		else
			if ( src->traits() & UT_DOT )
				ytype = YT_MXRALL_DD;
			else
				ytype = YT_RALL_DD;
	    break;

	case YT_MAP:
		if ( src->traits() & UT_DDTSET)
			if ( src->traits() & UT_DOT )
				ytype = YT_MXMAP;
			else
				ytype = YT_MAP;
		else {
			fprintf( stderr, "Map synapses can only connect Map neurons\n");
			return nullptr;
		}
	    break;
	default:
		printf( "Teleporting is fun!\n");
		return nullptr;
	}

	return add_synapse_species( ytype, src, tgt, g, allow_clone, finalize);
}




CNRun::C_BaseSynapse*
CNRun::CModel::
add_synapse_species( TUnitType ytype, C_BaseNeuron *src, C_BaseNeuron *tgt,
		     double g, bool allow_clone, bool finalize)
{
	if ( verbosely > 5 )
		printf( "add_synapse_species( \"%s\", \"%s\", \"%s\", %g, %d, %d)\n",
			__CNUDT[ytype].species, src->_label, tgt->_label, g, allow_clone, finalize);

	C_BaseSynapse *y = nullptr;

      // consider cloning
	if ( !(_status & CN_MDL_DONT_COALESCE) && allow_clone && src->_axonal_harbour.size() )
		for ( auto& L : src->_axonal_harbour )
			if ( L->_type == ytype &&
			     L->is_not_altered() )
				return L->clone_to_target( tgt, g);

	switch ( ytype ) {
      // the __CNUDT entry at first TUnitType element whose
      // 'name' matches the type id supplied, captures all cases for a given synapse family
	case YT_AB_RR:
		if (  src->traits() & UT_RATEBASED &&  tgt->traits() & UT_RATEBASED && !(src->traits() & UT_DOT) )
			y = new CSynapseAB_rr( src, tgt, g, this, CN_UOWNED, finalize);
	    break;
	case YT_AB_RD:
		if (  src->traits() & UT_RATEBASED && !(tgt->traits() & UT_RATEBASED) && !(src->traits() & UT_DOT) )
			// y = new CSynapseAB_rd( synapse_id, src, tgt, this, CN_UOWNED, false);
			fprintf( stderr, "AB_rd not implemented\n");
	    break;
	case YT_AB_DR:
		if ( !(src->traits() & UT_RATEBASED) &&  tgt->traits() & UT_RATEBASED && !(src->traits() & UT_DOT) )
			// y = new CSynapseAB_rr( synapse_id, src, tgt, this, CN_UOWNED, false);
			fprintf( stderr, "AB_dr not implemented\n");
	    break;
	case YT_AB_DD:
		if ( !(src->traits() & UT_RATEBASED) && !(tgt->traits() & UT_RATEBASED) && !(src->traits() & UT_DOT) )
			y = new CSynapseAB_dd( src, tgt, g, this, CN_UOWNED, finalize);
	    break;
	case YT_MXAB_DR:
		if ( !(src->traits() & UT_RATEBASED) &&  tgt->traits() & UT_RATEBASED &&  src->traits() & UT_DOT )
			y = new CSynapseMxAB_dr( src, tgt, g, this, CN_UOWNED, finalize);
	    break;
	case YT_MXAB_DD:
		if (  !(src->traits() & UT_RATEBASED) && !(tgt->traits() & UT_RATEBASED) &&  src->traits() & UT_DOT )
			y = new CSynapseMxAB_dd( src, tgt, g, this, CN_UOWNED, finalize);
	    break;


	case YT_ABMINUS_RR:
		if (  src->traits() & UT_RATEBASED &&  tgt->traits() & UT_RATEBASED && !(src->traits() & UT_DOT) )
			// y = new CSynapseABMINUS_rr( src, tgt, g, this, CN_UOWNED, finalize);
			fprintf( stderr, "ABMINUS_rr not implemented\n");
	    break;
	case YT_ABMINUS_RD:
		if (  src->traits() & UT_RATEBASED && !(tgt->traits() & UT_RATEBASED) && !(src->traits() & UT_DOT) )
			// y = new CSynapseABMINUS_rd( synapse_id, src, tgt, this, CN_UOWNED, false);
			fprintf( stderr, "ABMINUS_rd not implemented\n");
	    break;
	case YT_ABMINUS_DR:
		if ( !(src->traits() & UT_RATEBASED) &&  tgt->traits() & UT_RATEBASED && !(src->traits() & UT_DOT) )
			// y = new CSynapseABMINUS_rr( synapse_id, src, tgt, this, CN_UOWNED, false);
			fprintf( stderr, "ABMINUS_dr not implemented\n");
	    break;
	case YT_ABMINUS_DD:
		if ( !(src->traits() & UT_RATEBASED) && !(tgt->traits() & UT_RATEBASED) && !(src->traits() & UT_DOT) )
			y = new CSynapseABMinus_dd( src, tgt, g, this, CN_UOWNED, finalize);
	    break;
	case YT_MXABMINUS_DR:
		if ( !(src->traits() & UT_RATEBASED) &&  tgt->traits() & UT_RATEBASED &&  src->traits() & UT_DOT )
			// y = new CSynapseMxABMinus_dr( src, tgt, g, this, CN_UOWNED, finalize);
			fprintf( stderr, "MxABMinus_dr not implemented\n");
	    break;
	case YT_MXABMINUS_DD:
		if ( !(src->traits() & UT_RATEBASED) && !(tgt->traits() & UT_RATEBASED) &&  src->traits() & UT_DOT )
			// y = new CSynapseMxABMinus_dd( src, tgt, g, this, CN_UOWNED, finalize);
			fprintf( stderr, "MxABMinus_dd not implemented\n");
	    break;


	case YT_RALL_RR:
		if (  src->traits() & UT_RATEBASED &&  tgt->traits() & UT_RATEBASED && !(src->traits() & UT_DOT) )
			// y = new CSynapseRall_rr( src, tgt, g, this, CN_UOWNED, finalize);
			fprintf( stderr, "Rall_rr not implemented\n");
	    break;
	case YT_RALL_RD:
		if (  src->traits() & UT_RATEBASED && !(tgt->traits() & UT_RATEBASED) && !(src->traits() & UT_DOT) )
			// y = new CSynapseRall_rd( synapse_id, src, tgt, this, CN_UOWNED, false);
			fprintf( stderr, "Rall_rd not implemented\n");
	    break;
	case YT_RALL_DR:
		if ( !(src->traits() & UT_RATEBASED) &&  tgt->traits() & UT_RATEBASED && !(src->traits() & UT_DOT) )
			// y = new CSynapseRall_rr( synapse_id, src, tgt, this, CN_UOWNED, false);
			fprintf( stderr, "Rall_dr not implemented\n");
	    break;
	case YT_RALL_DD:
		if ( !(src->traits() & UT_RATEBASED) && !(tgt->traits() & UT_RATEBASED) && !(src->traits() & UT_DOT) )
			y = new CSynapseRall_dd( src, tgt, g, this, CN_UOWNED, finalize);
	    break;
	case YT_MXRALL_DR:
		if ( !(src->traits() & UT_RATEBASED) &&  tgt->traits() & UT_RATEBASED &&  src->traits() & UT_DOT )
			// y = new CSynapseMxRall_dr( src, tgt, g, this, CN_UOWNED, finalize);
			fprintf( stderr, "MxRall_dr not implemented\n");
	    break;
	case YT_MXRALL_DD:
		if ( !(src->traits() & UT_RATEBASED) && !(tgt->traits() & UT_RATEBASED) &&  src->traits() & UT_DOT )
			// y = new CSynapseMxRall_dd( src, tgt, g, this, CN_UOWNED, finalize);
			fprintf( stderr, "MxRall_dd not implemented\n");
	    break;


	case YT_MAP:
		if ( src->traits() & UT_DDTSET)
			if ( src->traits() & UT_DOT )
				y = new CSynapseMxMap( src, tgt, g, this, CN_UOWNED);
			else
				y = new CSynapseMap( src, tgt, g, this, CN_UOWNED);
		else
			fprintf( stderr, "Map synapses can only connect Map neurons\n");
	    break;

	default:
		return nullptr;
	}

	if ( !y || y->_status & CN_UERROR ) {
		if ( y )
			delete y;
		return nullptr;
	}

	if ( verbosely > 5 )
		printf( "new synapse \"%s->%s\"\n", y->_label, tgt->label());
	y->set_g_on_target( *tgt, g);

	return y;
}






void
CNRun::CModel::
finalize_additions()
{
	V.resize( _var_cnt),  W.resize( _var_cnt);

	for_all_hosted_neurons (N)
		(*N) -> reset_vars();
	for_all_hosted_synapses (Y)
		(*Y) -> reset_vars();

	if ( _status & CN_MDL_SORTUNITS ) {
		__C_BaseUnitCompareByLabel cmp;
		unit_list.sort( cmp);
		// hosted_neu_list.sort( cmp);
		// hosted_syn_list.sort( cmp);
		// standalone_neu_list.sort( cmp);
		// standalone_syn_list.sort( cmp);
	}

	_integrator->prepare();
}








void
CNRun::CModel::
cull_deaf_synapses()
{
	// needs fixing
      // 1. Need to traverse syn_list backwards due to shifts its vector will undergo on element deletions;
      // 2. Omit those with a param reader, scheduler or range, but only if it is connected to parameter "gsyn"
grand_restart:
	for_all_hosted_synapses (Y)
		if ( !(*Y)->has_sources() ) {
		restart:
			for ( C_BaseSynapse::lni T = (*Y)->_targets.begin(); T != (*Y)->_targets.end(); T++ ) {
				if ( (*Y)->g_on_target( **T) == 0  ) {
					if ( verbosely > 3 )
						fprintf( stderr, " (deleting dendrite to \"%s\" of a synapse \"%s\" with gsyn == 0)\n",
							 (*T)->_label, (*Y)->_label);
					(*T)->_dendrites.erase( *Y);
					(*Y)->_targets.erase( find( (*Y)->_targets.begin(), (*Y)->_targets.end(), *T));

					snprintf( (*Y)->_label, CN_MAX_LABEL_SIZE-1, "%s:%zu", (*Y)->_source->_label, (*Y)->_targets.size());
					goto restart;
				}
			}
			if ( (*Y)->_targets.size() == 0 ) {
				delete (*Y);
				goto grand_restart;
			}
		}

	// older stuff
/*
	for_all_synapses_reversed (Y) {
		int gsyn_pidx = (*Y) -> param_idx_by_sym( "gsyn");
		if ( ((*Y)->param_schedulers && device_list_concerns_parm( (*Y)->param_schedulers, gsyn_pidx)) ||
		     ((*Y)->param_readers    && device_list_concerns_parm( (*Y)->param_readers,    gsyn_pidx)) ||
		     ((*Y)->param_ranges     && device_list_concerns_parm( (*Y)->param_ranges,     gsyn_pidx)) ) {
			if ( verbosely > 2 )
				printf( " (preserving doped synapse with zero gsyn: \"%s\")\n", (*Y)->_label);
			continue;
		}
		if ( gsyn_pidx > -1 && (*Y)->param_value( gsyn_pidx) == 0. ) {
			if ( verbosely > 2 )
				printf( " (deleting synapse with zero gsyn: \"%s\")\n", (*Y)->_label);
			delete (*Y);
			cnt++;
		}
	}
	if ( verbosely > 0 && cnt )
		printf( "Deleted %zd deaf synapses\n", cnt);
*/
}



// needs to be called after a neuron is put out
void
CNRun::CModel::
cull_blind_synapses()
{
	for_all_hosted_synapses_reversed (Y)
		if ( (*Y)->_source == nullptr && !(*Y)->has_sources() ) {
			if ( verbosely > 3 )
				printf( " (deleting synapse with nullptr source: \"%s\")\n", (*Y)->_label);
			delete (*Y);
		}
	for_all_standalone_synapses_reversed (Y)
		if ( (*Y)->_source == nullptr && !(*Y)->has_sources() ) {
			if ( verbosely > 3 )
				printf( " (deleting synapse with nullptr source: \"%s\")\n", (*Y)->_label);
			delete (*Y);
		}
}



void
CNRun::CModel::
reset_state_all_units()
{
	for_all_units (U)
		(*U) -> reset_state();
}







// tags

int
CNRun::CModel::
process_listener_tags( const list<STagGroupListener> &Listeners)
{
	regex_t RE;
	for ( auto& P : Listeners ) {
		if (0 != regcomp( &RE, P.pattern.c_str(), REG_EXTENDED | REG_NOSUB)) {
			fprintf( stderr, "Invalid regexp in process_listener_tags: \"%s\"\n", P.pattern.c_str());
			return -1;
		}
		for_all_units (U) {
			if ( regexec( &RE, (*U)->_label, 0, 0, 0) == 0 ) {
				if ( P.enable ) {
					(*U) -> start_listening( P.bits);
					if ( verbosely > 3 )
						printf( " (unit \"%s\" listening%s)\n",
							(*U)->_label, P.bits & CN_ULISTENING_1VARONLY ? ", to one var only" :"");
				} else {
					(*U) -> stop_listening();
					if ( verbosely > 3 )
						printf( " (unit \"%s\" not listening)\n", (*U)->_label);
				}
			}
		}
	}

	return 0;
}


int
CNRun::CModel::
process_spikelogger_tags( const list<STagGroupSpikelogger> &Spikeloggers)
{
	regex_t RE;
	for ( auto& P : Spikeloggers ) {
		if (0 != regcomp( &RE, P.pattern.c_str(), REG_EXTENDED | REG_NOSUB)) {
			fprintf( stderr, "Invalid regexp in process_spikelogger_tags: \"%s\"\n", P.pattern.c_str());
			return -1;
		}
		for_all_standalone_neurons (N) {
			if ( regexec( &RE, (*N)->_label, 0, 0, 0) == 0 ) {
				if ( P.enable ) {
					bool log_sdf = !(P.period == 0. || P.sigma == 0.);
					if ( ( log_sdf && !(*N)->enable_spikelogging_service( P.period, P.sigma, P.from))
					     ||
					     (!log_sdf && !(*N)->enable_spikelogging_service()) ) {
						fprintf( stderr, "Cannot have \"%s\" log spikes because it is not a conductance-based neuron (of type %s)\n",
							 (*N)->_label, (*N)->species());
						return -1;
					}
				} else
					(*N)->disable_spikelogging_service();

				if ( verbosely > 3 )
					printf( " (%sabling spike logging for standalone neuron \"%s\")\n",
						P.enable ? "en" : "dis", (*N)->_label);
			}
		}
		for_all_hosted_neurons (N) {
			if ( regexec( &RE, (*N)->_label, 0, 0, 0) == 0 ) {
				if ( P.enable ) {
					bool log_sdf = !(P.period == 0. || P.sigma == 0.);
					if ( ( log_sdf && !(*N)->enable_spikelogging_service( P.period, P.sigma, P.from))
					     ||
					     (!log_sdf && !(*N)->enable_spikelogging_service()) ) {
						fprintf( stderr, "Cannot have \"%s\" log spikes because it is not a conductance-based neuron (of type %s)\n",
							 (*N)->_label, (*N)->species());
						return -1;
					}
				} else
					(*N)->disable_spikelogging_service();

				if ( verbosely > 3 )
					printf( " (%sabling spike logging for hosted neuron \"%s\")\n",
						P.enable ? "en" : "dis", (*N)->_label);
			}
		}
	}

	return 0;
}


int
CNRun::CModel::
process_putout_tags( const list<STagGroup> &ToRemove)
{
      // execute some
	regex_t RE;
	for ( auto& P : ToRemove ) {
		if (0 != regcomp( &RE, P.pattern.c_str(), REG_EXTENDED | REG_NOSUB)) {
			fprintf( stderr, "Invalid regexp in process_putout_tags: \"%s\"\n", P.pattern.c_str());
			return -1;
		}
		for_all_units (U) {
			if ( regexec( &RE, (*U)->_label, 0, 0, 0) == 0 ) {
				if ( verbosely > 2 )
					printf( " (put out unit \"%s\")\n",
						(*U)->_label);
				delete (*U);
				if ( units() > 0 )
					U = ulist_begin();
				else
					break;
			}
		}
	}

	cull_blind_synapses();

	return 0;
}


int
CNRun::CModel::
process_decimate_tags( const list<STagGroupDecimate> &ToDecimate)
{
      // decimate others
	regex_t RE;
	for ( auto& P : ToDecimate ) {
		if (0 != regcomp( &RE, P.pattern.c_str(), REG_EXTENDED | REG_NOSUB)) {
			fprintf( stderr, "Invalid regexp in process_decimate_tags: \"%s\"\n", P.pattern.c_str());
			return -1;
		}

	      // collect group
		vector<C_BaseUnit*> dcmgroup;
		for_all_units (U)
			if ( regexec( &RE, (*U)->_label, 0, 0, 0) == 0 )
				dcmgroup.push_back( *U);
		random_shuffle( dcmgroup.begin(), dcmgroup.end());

	      // execute
		size_t	to_execute = rint( dcmgroup.size() * P.fraction), n = to_execute;
		while ( n-- )
			delete dcmgroup[n];

		if ( verbosely > 3 )
			printf( " (decimated %4.1f%% (%zu units) of %s)\n", P.fraction*100, to_execute, P.pattern.c_str());

	}

	cull_blind_synapses();

	return 0;
}






int
CNRun::CModel::
process_paramset_static_tags( const list<STagGroupNeuronParmSet> &tags)
{
	regex_t RE;
	for ( auto& P : tags ) {
		if (0 != regcomp( &RE, P.pattern.c_str(), REG_EXTENDED | REG_NOSUB)) {
			fprintf( stderr, "Invalid regexp in process_paramset_static_tags: \"%s\"\n", P.pattern.c_str());
			return -1;
		}

		vector<string> current_tag_assigned_labels;

		for_all_neurons (U) {
			if ( regexec( &RE, (*U)->_label, 0, 0, 0) == 0 )
				continue;
		      // because a named parameter can map to a different param_id in different units, rather
		      // do lookup every time

			int p_d = -1;
			C_BaseUnit::TSinkType kind = (C_BaseUnit::TSinkType)-1;
			if ( (p_d = (*U)->param_idx_by_sym( P.parm.c_str())) > -1 )
				kind = C_BaseUnit::SINK_PARAM;
			else if ( (p_d = (*U)->var_idx_by_sym( P.parm.c_str())) > -1 )
				kind = C_BaseUnit::SINK_VAR;
			if ( p_d == -1 ) {
				fprintf( stderr, "%s \"%s\" (type \"%s\") has no parameter or variable named \"%s\"\n",
					 (*U)->class_name(), (*U)->label(), (*U)->species(), P.parm.c_str());
				continue;
			}

			switch ( kind ) {
			case C_BaseUnit::SINK_PARAM:
				(*U)->param_value(p_d) = P.enable ? P.value : __CNUDT[(*U)->type()].stock_param_values[p_d];
				(*U)->param_changed_hook();
			    break;
			case C_BaseUnit::SINK_VAR:
				(*U)->  var_value(p_d) = P.value;
			    break;
			}

			current_tag_assigned_labels.push_back( (*U)->label());
		}

		if ( current_tag_assigned_labels.empty() ) {
			fprintf( stderr, "No neuron labelled matching \"%s\"\n", P.pattern.c_str());
			return -2;
		}

		if ( verbosely > 3 ) {
			printf( " set ");
			for ( auto S = current_tag_assigned_labels.begin(); S != current_tag_assigned_labels.end(); S++ )
				printf( "%s%s",
					(S == current_tag_assigned_labels.begin()) ? "" : ", ", S->c_str());
			printf( " {%s} = %g\n", P.parm.c_str(), P.value);
		}
	}
	return 0;
}





int
CNRun::CModel::
process_paramset_static_tags( const list<STagGroupSynapseParmSet> &tags)
{
	for ( auto& P : tags ) {
		regex_t REsrc, REtgt;
		if (0 != regcomp( &REsrc, P.pattern.c_str(), REG_EXTENDED | REG_NOSUB) ) {  // P->pattern acting as src
			fprintf( stderr, "Invalid regexp in process_paramset_static_tags (src): \"%s\"\n", P.pattern.c_str());
			return -1;
		}
		if (0 != regcomp( &REtgt, P.target.c_str(), REG_EXTENDED | REG_NOSUB) ) {
			fprintf( stderr, "Invalid regexp in process_paramset_static_tags (tgt): \"%s\"\n", P.target.c_str());
			return -1;
		}

		vector<string> current_tag_assigned_labels;

		bool do_gsyn = (P.parm == "gsyn");

		if ( verbosely > 5 )
			printf( "== setting %s -> %s {%s} = %g...\n", P.pattern.c_str(), P.target.c_str(), P.parm.c_str(), P.value);

		for_all_neurons (Us) {
			if ( regexec( &REsrc, (*Us)->label(), 0, 0, 0) == 0 )
				continue;

			for_all_neurons (Ut) {
                if ( regexec( &REtgt, (*Ut)->label(), 0, 0, 0) == 0 ) /* || Us == Ut */
					continue;
				C_BaseSynapse *y = static_cast<C_BaseNeuron*>(*Us) -> connects_via( *static_cast<C_BaseNeuron*>(*Ut));
				if ( !y )
					continue;

				if ( do_gsyn ) {
					y->set_g_on_target( *static_cast<C_BaseNeuron*>(*Ut), P.value);
					current_tag_assigned_labels.push_back( y->label());
					continue;
				}

				int p_d = -1;
				C_BaseUnit::TSinkType kind = (C_BaseUnit::TSinkType)-1;
				if ( (p_d = y->param_idx_by_sym( P.parm.c_str())) > -1 )
					kind = C_BaseUnit::SINK_PARAM;
				else if ( (p_d = y->var_idx_by_sym( P.parm.c_str())) > -1 )
					kind = C_BaseUnit::SINK_VAR;
				if ( p_d == -1 ) {
					fprintf( stderr, "%s \"%s\" (type \"%s\") has no parameter or variable named \"%s\"\n",
						 y->class_name(), y->label(), y->species(), P.parm.c_str());
					continue;
				}

				switch ( kind ) {
				case C_BaseUnit::SINK_PARAM:
					if ( y->_targets.size() > 1 ) {
						y = y->make_clone_independent( static_cast<C_BaseNeuron*>(*Ut));  // lest brethren synapses to other targets be clobbered
					}
					y->param_value(p_d) = P.enable ? P.value : __CNUDT[y->type()].stock_param_values[p_d];
					y->param_changed_hook();
				    break;
				case C_BaseUnit::SINK_VAR:
					y->  var_value(p_d) = P.value;
				    break;
				}

				current_tag_assigned_labels.push_back( y->label());
			}
		}
		if ( current_tag_assigned_labels.empty() ) {
			fprintf( stderr, "No synapse connecting any of \"%s\" to \"%s\"\n", P.pattern.c_str(), P.target.c_str());
			return -2;
		}

		if ( verbosely > 3 ) {
			printf( " set ");
			for ( auto S = current_tag_assigned_labels.begin(); S != current_tag_assigned_labels.end(); S++ )
				printf( "%s%s",
					(S == current_tag_assigned_labels.begin()) ? "" : ", ", S->c_str());
			printf( " {%s} = %g\n", P.parm.c_str(), P.value);
		}
	}

	if ( !(_status & CN_MDL_DONT_COALESCE) )
		coalesce_synapses();

	return 0;
}


void
CNRun::CModel::
coalesce_synapses()
{
startover:
	for_all_synapses (U1) {
		C_BaseSynapse *y1 = static_cast<C_BaseSynapse*>(*U1);
		for_all_synapses (U2) {
			if ( *U2 == *U1 )
				continue;

			C_BaseSynapse *y2 = static_cast<C_BaseSynapse*>(*U2);
			if ( y1->_source == y2->_source &&
			     (*U1) -> is_identical( **U2) ) {

				if ( verbosely > 5 )
					printf( "coalescing \"%s\" and \"%s\"\n", y1->_label, y2->_label);
				for ( C_BaseSynapse::lni T = y2->_targets.begin(); T != y2->_targets.end(); T++ ) {
					y1->_targets.push_back( *T);
					(*T)->_dendrites[y1] = (*T)->_dendrites[y2];
				}
				snprintf( y1->_label, CN_MAX_LABEL_SIZE-1, "%s:%zu", y1->_source->_label, y1->_targets.size());

				delete y2;

				goto startover;
			}
		}
	}
}





int
CNRun::CModel::
process_paramset_source_tags( const list<STagGroupSource> &tags)
{
	regex_t RE;
	for ( auto& P : tags ) {
		if (0 != regcomp( &RE, P.pattern.c_str(), REG_EXTENDED | REG_NOSUB)) {
			fprintf( stderr, "Invalid regexp in process_paramset_source_tags: \"%s\"\n", P.pattern.c_str());
			return -1;
		}

		for_all_units (U) {
			if ( regexec( &RE, (*U)->label(), 0, 0, 0) == 0 )
				continue;

			int p_d = -1;
			C_BaseUnit::TSinkType kind = (C_BaseUnit::TSinkType)-1;
			if ( (p_d = (*U)->param_idx_by_sym( P.parm.c_str())) > -1 )
				kind = C_BaseUnit::SINK_PARAM;
			else if ( (p_d = (*U)->var_idx_by_sym( P.parm.c_str())) > -1 )
				kind = C_BaseUnit::SINK_VAR;
			if ( p_d == -1 ) {
				fprintf( stderr, "%s \"%s\" (type \"%s\") has no parameter or variable named \"%s\"\n",
					 (*U)->class_name(), (*U)->label(), (*U)->species(), P.parm.c_str());
				continue;
			}

			if ( P.enable ) {
				(*U) -> attach_source( P.source, kind, p_d);
				if ( verbosely > 3 )
					printf( "Connected source \"%s\" to \"%s\"{%s}\n",
						P.source->name.c_str(), (*U)->label(), P.parm.c_str());
			} else {
				(*U) -> detach_source( P.source, kind, p_d);
				if ( verbosely > 3 )
					printf( "Disconnected source \"%s\" from \"%s\"{%s}\n",
						P.source->name.c_str(), (*U)->label(), P.parm.c_str());
			}
		}
	}

	return 0;
}




inline const char*
__attribute__ ((pure))
pl_ending( size_t cnt)
{
	return cnt == 1 ? "" : "s";
}

void
CNRun::CModel::
dump_metrics( FILE *strm)
{
	fprintf( strm,
		 "\nModel \"%s\"%s:\n"
		 "  %5zd unit%s total (%zd Neuron%s, %zd Synapse%s):\n"
		 "    %5zd hosted,\n"
		 "    %5zd standalone\n"
		 "    %5zd discrete dt-bound\n"
		 "  %5zd Listening unit%s\n"
		 "  %5zd Spikelogging neuron%s\n"
		 "  %5zd Unit%s being tuned continuously\n"
		 "  %5zd Unit%s being tuned periodically\n"
		 "  %5zd Spontaneously firing neuron%s\n"
		 "  %5zd Multiplexing synapse%s\n"
		 " %6zd vars on integration vector\n\n",
		 name.c_str(), (_status & CN_MDL_DISKLESS) ? " (diskless)" : "",
		 units(), pl_ending(units()),
		 total_neuron_cnt(), pl_ending(total_neuron_cnt()),
		 total_synapse_cnt(), pl_ending(total_synapse_cnt()),
		 hosted_unit_cnt(),
		 standalone_unit_cnt(),
		 ddtbound_unit_cnt(),
		 listening_unit_cnt(), pl_ending(listening_unit_cnt()),
		 spikelogging_neuron_cnt(), pl_ending(spikelogging_neuron_cnt()),
		 unit_with_continuous_sources_cnt(), pl_ending(unit_with_continuous_sources_cnt()),
		 unit_with_periodic_sources_cnt(), pl_ending(unit_with_periodic_sources_cnt()),
		 conscious_neuron_cnt(), pl_ending(conscious_neuron_cnt()),
		 mx_syn_list.size(), pl_ending(mx_syn_list.size()),
		 _var_cnt-1);
	if ( _status & CN_MDL_HAS_DDTB_UNITS )
		fprintf( strm, "Discrete dt: %g msec\n", discrete_dt());
}

void
CNRun::CModel::
dump_state( FILE *strm)
{
	fprintf( strm,
		 "Model time: %g msec\n"
		 "Integrator dt_min: %g msec, dt_max: %g msec\n"
		 "Logging at: %g msec\n\n",
		 model_time(),
		 dt_min(), dt_max(),
		 listen_dt);
}



void
CNRun::CModel::
dump_units( FILE *strm)
{
	fprintf( strm, "\nUnit types in the model:\n");

	set<int> found_unit_types;
	unsigned p = 0;

	fprintf( strm, "\n===== Neurons:\n");
	for_all_units (U)
		if ( (*U)->is_neuron() && found_unit_types.count( (*U)->type()) == 0 ) {
			found_unit_types.insert( (*U)->type());

			fprintf( strm, "--- %s: %s\nParameters: ---\n",
				 (*U)->species(), (*U)->type_description());
			for ( p = 0; p < (*U)->p_no(); p++ )
				if ( *(*U)->param_sym(p) != '.' || verbosely > 5 )
					fprintf( strm, "%2d: %-5s\t= %s %s\n",
						 p, (*U)->param_sym(p),
						 Stilton::double_dot_aligned_s( (*U)->param_value(p), 4, 6),
						 (*U)->param_name(p));
			fprintf( strm, "Variables: ---\n");
			for ( p = 0; p < (*U)->v_no(); p++ )
				if ( *(*U)->var_sym(p) != '.' || verbosely > 5 )
					fprintf( strm, "%2d: %-5s\t= %s %s\n",
						 p, (*U)->var_sym(p),
						 Stilton::double_dot_aligned_s( (*U)->var_value(p), 4, 6),
						 (*U)->var_name(p));
		}
	fprintf( strm, "\n===== Synapses:\n");
	for_all_units (U)
		if ( (*U)->is_synapse() && found_unit_types.count( (*U)->type()) == 0 ) {
			found_unit_types.insert( (*U)->type());

			fprintf( strm, "--- %s: %s\nParameters: ---\n",
				 (*U)->species(), (*U)->type_description());
			fprintf( strm, "    parameters:\n");
			for ( p = 0; p < (*U)->p_no(); p++ )
				if ( *(*U)->param_sym(p) != '.' || verbosely > 5 )
					fprintf( strm, "%2d: %-5s\t= %s %s\n",
						 p, (*U)->param_sym(p),
						 Stilton::double_dot_aligned_s( (*U)->param_value(p), 4, 6),
						 (*U)->param_name(p));
			fprintf( strm, "Variables: ---\n");
			for ( p = 0; p < (*U)->v_no(); p++ )
				if ( *(*U)->var_sym(p) != '.' || verbosely > 5 )
					fprintf( strm, "%2d: %-5s\t= %s %s\n",
						 p, (*U)->var_sym(p),
						 Stilton::double_dot_aligned_s( (*U)->var_value(p), 4, 6),
						 (*U)->var_name(p));

		}
	fprintf( strm, "\n");
}

