/* SPDX-License-Identifier: GPL-2.0-only */

#include <security/vboot/antirollback.h>
#include <program_loading.h>
#include <security/vboot/vboot_common.h>
#include <vb2_api.h>
#include <security/tpm/tss.h>
#include <security/vboot/mrc_cache_hash_tpm.h>
#include <console/console.h>
#include <string.h>

void mrc_cache_update_hash(uint32_t index, const uint8_t *data, size_t size)
{
	uint8_t data_hash[VB2_SHA256_DIGEST_SIZE];
	static const uint8_t dead_hash[VB2_SHA256_DIGEST_SIZE] = {
		0xba, 0xad, 0xda, 0x1a, /* BAADDA1A */
		0xde, 0xad, 0xde, 0xad, /* DEADDEAD */
		0xde, 0xad, 0xda, 0x1a, /* DEADDA1A */
		0xba, 0xad, 0xba, 0xad, /* BAADBAAD */
		0xba, 0xad, 0xda, 0x1a, /* BAADDA1A */
		0xde, 0xad, 0xde, 0xad, /* DEADDEAD */
		0xde, 0xad, 0xda, 0x1a, /* DEADDA1A */
		0xba, 0xad, 0xba, 0xad, /* BAADBAAD */
	};
	const uint8_t *hash_ptr = data_hash;

	/* Initialize TPM driver. */
	if (tlcl_lib_init() != VB2_SUCCESS) {
		printk(BIOS_ERR, "MRC: TPM driver initialization failed.\n");
		return;
	}

	/* Calculate hash of data generated by MRC. */
	if (vb2_digest_buffer(data, size, VB2_HASH_SHA256, data_hash,
			      sizeof(data_hash))) {
		printk(BIOS_ERR, "MRC: SHA-256 calculation failed for data. "
		       "Not updating TPM hash space.\n");
		/*
		 * Since data is being updated in mrc cache, the hash
		 * currently stored in TPM hash space is no longer
		 * valid. If we are not able to calculate hash of the
		 * data being updated, reset all the bits in TPM hash
		 * space to pre-defined hash pattern.
		 */
		hash_ptr = dead_hash;
	}

	/* Write hash of data to TPM space. */
	if (antirollback_write_space_mrc_hash(index, hash_ptr, VB2_SHA256_DIGEST_SIZE)
	    != TPM_SUCCESS) {
		printk(BIOS_ERR, "MRC: Could not save hash to TPM.\n");
		return;
	}

	printk(BIOS_INFO, "MRC: TPM MRC hash idx 0x%x updated successfully.\n", index);
}

int mrc_cache_verify_hash(uint32_t index, const uint8_t *data, size_t size)
{
	uint8_t data_hash[VB2_SHA256_DIGEST_SIZE];
	uint8_t tpm_hash[VB2_SHA256_DIGEST_SIZE];

	/* Calculate hash of data read from MRC_CACHE. */
	if (vb2_digest_buffer(data, size, VB2_HASH_SHA256, data_hash,
			      sizeof(data_hash))) {
		printk(BIOS_ERR, "MRC: SHA-256 calculation failed for data.\n");
		return 0;
	}

	/* Initialize TPM driver. */
	if (tlcl_lib_init() != VB2_SUCCESS) {
		printk(BIOS_ERR, "MRC: TPM driver initialization failed.\n");
		return 0;
	}

	/* Read hash of MRC data saved in TPM. */
	if (antirollback_read_space_mrc_hash(index, tpm_hash, sizeof(tpm_hash))
	    != TPM_SUCCESS) {
		printk(BIOS_ERR, "MRC: Could not read hash from TPM.\n");
		return 0;
	}

	if (memcmp(tpm_hash, data_hash, sizeof(tpm_hash))) {
		printk(BIOS_ERR, "MRC: Hash comparison failed.\n");
		return 0;
	}

	printk(BIOS_INFO, "MRC: Hash idx 0x%x comparison successful.\n", index);

	return 1;
}
