/* -*- mode: c; c-basic-offset: 2; indent-tabs-mode: nil; -*- */
/* geoclue-stumbler-settings.c
 *
 * Copyright 2021 Chris Talbot <chris@talbothome.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author(s):
 *   Chris Talbot <chris@talbothome.com>
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

#define G_LOG_DOMAIN "geocluestumbler-settings"


#include "config.h"

#include "geoclue-stumbler-settings.h"

/**
 * SECTION: geocluestumbler-settings
 * @title: GeoclueStumblerSettings
 * @short_description: The Application settings
 * @include: "geocluestumbler-settings.h"
 *
 * A class that handles application specific settings, and
 * to store them to disk.
 */

struct _GeoclueStumblerSettings
{
  GObject    parent_instance;

  GSettings *app_settings;

  char *submission_url;
  int   number_of_submissions;
  gboolean show_submission_location;
};

G_DEFINE_TYPE (GeoclueStumblerSettings, geoclue_stumbler_settings, G_TYPE_OBJECT)


char*
geoclue_stumbler_settings_get_submission_url (GeoclueStumblerSettings *self)
{
  return g_strdup (self->submission_url);
}

int geoclue_stumbler_settings_get_total_number_of_submissions (GeoclueStumblerSettings *self)
{
  return self->number_of_submissions;
}

void geoclue_stumbler_settings_set_total_number_of_submissions (GeoclueStumblerSettings *self,
                                                                int  number_of_submissions)
{
  self->number_of_submissions = number_of_submissions;
  g_debug ("Numbers of Submission is set to %d", self->number_of_submissions);

  /* Set the setting right away */
  g_settings_set_int (self->app_settings, "number-of-submissions", self->number_of_submissions);
  g_settings_apply (self->app_settings);
}

void geoclue_stumbler_settings_add_total_number_of_submissions (GeoclueStumblerSettings *self,
                                                                int  number_of_submissions)
{
  int new_total = 0;

  new_total = self->number_of_submissions + number_of_submissions;
  geoclue_stumbler_settings_set_total_number_of_submissions (self, new_total);
}

void
geoclue_stumbler_settings_set_submission_url (GeoclueStumblerSettings *self,
                                              const char              *submission_url)
{
  g_free (self->submission_url);
  self->submission_url = g_strdup (submission_url);
  g_debug ("Submission URL is set to %s", self->submission_url);

  /* Set the setting right away */
  g_settings_set_string (self->app_settings, "submission-url", self->submission_url);
  g_settings_apply (self->app_settings);
}

gboolean
geoclue_stumbler_settings_get_show_submission_location (GeoclueStumblerSettings *self)
{
  return self->show_submission_location;
}

void
geoclue_stumbler_settings_set_show_submission_location (GeoclueStumblerSettings *self,
                                                        gboolean     show_submission_location)
{
  self->show_submission_location = show_submission_location;

  /* Set the setting right away */
  g_settings_set_boolean (self->app_settings, "show-submission-location", self->show_submission_location);
  g_settings_apply (self->app_settings);
}

static void
geoclue_stumbler_settings_dispose (GObject *object)
{
//  GeoclueStumblerSettings *self = (GeoclueStumblerSettings *)object;

  g_debug ("Disposing settings");

  G_OBJECT_CLASS (geoclue_stumbler_settings_parent_class)->dispose (object);
}

static void
geoclue_stumbler_settings_class_init (GeoclueStumblerSettingsClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->dispose = geoclue_stumbler_settings_dispose;
}

static void
geoclue_stumbler_settings_init (GeoclueStumblerSettings *self)
{
  g_autofree char *version = NULL;
  self->app_settings = g_settings_new (PACKAGE_ID);

  g_settings_delay (self->app_settings);
}

GeoclueStumblerSettings *
geoclue_stumbler_settings_get_default (void)
{
  static GeoclueStumblerSettings *self;

  if (!self)
    {
      self = g_object_new (GEOCLUE_STUMBLER_TYPE_SETTINGS, NULL);
      g_object_add_weak_pointer (G_OBJECT (self), (gpointer *)&self);
      self->submission_url = g_settings_get_string (self->app_settings, "submission-url");
      self->number_of_submissions = g_settings_get_int (self->app_settings, "number-of-submissions");
      self->show_submission_location = g_settings_get_boolean (self->app_settings, "show-submission-location");
    }
  return self;
}
