------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                   --
--  Copyright (C) 2006-2008 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-db.ads,v 1.22 2008/03/19 06:25:18 vagul Exp $

with OCI.Thick.Connections;
with OCI.Thick.Containers;
with OCI.Thick.Lobs;
with OCI.Thick.Date.Internal;

package OCI.Thick.DB is

   use Connections;
   use Containers;

   type Statement is new Handle_Reference with private;

   type Statement_Type is
     (Stmt_Unknown,
      Stmt_Select,
      Stmt_Update,
      Stmt_Delete,
      Stmt_Insert,
      Stmt_Create,
      Stmt_Drop,
      Stmt_Alter,
      Stmt_Begin,
      Stmt_Declare);

   type Data_Type is
     (Type_Integer,
      Type_Long_Float,
      Type_String,
      Type_Number,
      Type_Date,
      Type_Char_Lob,
      Type_Bin_Lob);

   -------------------------------------------------
   -- Undepended from Connection statement stile. --
   -------------------------------------------------

   function Prepare (Code : in String) return Statement;

   procedure Execute
     (Connect     : in     Connection;
      Stmt        : in out Statement;
      Auto_Commit : in     Boolean := False);

   procedure Execute
     (Connect     : in     Connection;
      Stmt        : in out Statement;
      Data        : in out Container_Type'Class;
      Auto_Commit : in     Boolean := False);

   procedure Describe (Connect : in Connection; Stmt : in out Statement);

   -----------------------------------------------
   -- Depended from Connection statement stile. --
   -- Like ODBC                                 --
   -----------------------------------------------

   function Prepare
     (Connect : in Connection; Code : in String) return Statement;

   procedure Execute
     (Stmt : in out Statement; Auto_Commit : in Boolean := False);

   procedure Execute
     (Stmt        : in out Statement;
      Data        : in out Container_Type'Class;
      Auto_Commit : in     Boolean := False);

   procedure Execute
     (Stmt        : in out Statement;
      Data        : in out Container_Type'Class;
      Success     :    out Boolean;
      Auto_Commit : in     Boolean := False);

   procedure Execute
     (Stmt     : in out Statement;
      In_Data  : access procedure
                          (Data      :    out Data_Holder;
                           Position  : in     Positive;
                           Iteration : in     Positive);
      Out_Data : access procedure
                          (Data      : in Data_Holder;
                           Position  : in Positive;
                           Iteration : in Positive);
      Count       : in Natural := 1;
      Auto_Commit : in Boolean := False);
   --  Count is for non select statements.
   --  Iteration in Process routine would be from 1 to Count.

   procedure Execute
     (Stmt    : in out Statement;
      In_Data : access procedure
                         (Data      :    out Data_Holder;
                          Position  : in     Positive;
                          Iteration : in     Positive);
      Out_Data : access procedure
                         (Data      : in Data_Holder;
                          Position  : in Positive;
                          Iteration : in Positive);
      Success     :    out Boolean;
      Count       : in     Natural := 1;
      Auto_Commit : in     Boolean := False);

   procedure Execute
     (Stmt        : in out Statement;
      Success     :    out Boolean;
      Auto_Commit : in     Boolean := False);

   function Execute
     (Connect     : in Connection;
      Code        : in String;
      Auto_Commit : in Boolean := False) return Statement;

   procedure Execute
     (Connect     : in Connection;
      Code        : in String;
      Auto_Commit : in Boolean := False);

   procedure Describe (Stmt : in out Statement);

   procedure Describe (Stmt : in out Statement; Success : out Boolean);

   function Type_Of_Statement (Stmt : in Statement) return Statement_Type;

   function Described (Stmt : in Statement) return Boolean;

   function Executed (Stmt : in Statement) return Boolean;
   pragma Inline (Executed);

   function Is_Executing (Stmt : in Statement) return Boolean;

   procedure Set_Connection (Stmt : in out Statement; Connect : in Connection);

   function Get_Connection (Stmt : in Statement) return Connection;
   pragma Inline (Get_Connection);

   function Fetch
     (Stmt    : in     Statement;
      Process : access procedure
                         (Data : in Data_Holder;
                          Position, Iteration : in Positive);
      Count   : in     Positive := 1) return Boolean;

   procedure Fetch
     (Stmt  : in     Statement;
      Data  :    out Container_Type'Class;
      Count : in     Positive);

   procedure Cancel (Stmt : in Statement);

   function Parse_Error_Offset (Stmt : in Statement) return Natural;

   function Number_Of_Columns (Stmt : in Statement) return Natural;

   function Rows_Processed (Stmt : in Statement) return Natural;
   --  Returns the number of rows processed so far.

   procedure Set_Blocking (Item : in out Statement; Mode : in Boolean);
   --  Set the statement connection to the spetsific blocking mode.

   --------------------------------
   --  Bind and define routines. --
   --------------------------------

   function Bind_Count (Stmt : in Statement) return Natural;

   function Bind_Name (Stmt : in Statement; Position : Positive) return String;

   function Column_Name
     (Stmt : in Statement; Position : in Positive) return String;

   function Column_Index
     (Stmt : in Statement; Name : in String) return Positive;

   procedure Bind
     (Stmt : in out Statement; Kind : in Data_Type; Position : in Positive);

   procedure Bind
     (Stmt : in out Statement; Kind : in Data_Type; Name : in String);

   function Defined (Stmt : in Statement) return Boolean;

   procedure Define
     (Stmt : in out Statement; Kind : in Data_Type; Position : in Positive);
   --  Define could be called only after Describe or first Execute.

   procedure Define
     (Stmt : in out Statement; Kind : in Data_Type; Name : in String);

   procedure Define
     (Stmt : in out Statement; Loc : in Lobs.Locator; Position : in Positive);
   --  Define with already exists lob locator.

   procedure Define
     (Stmt : in out Statement; Loc : in Lobs.Locator; Name : in String);
   --  Define with already exists lob locator.

private

   for Statement_Type use
     (Stmt_Unknown => 0, --  Undocumented OCI Stmt type.
      Stmt_Select  => OCI_STMT_SELECT,
      Stmt_Update  => OCI_STMT_UPDATE,
      Stmt_Delete  => OCI_STMT_DELETE,
      Stmt_Insert  => OCI_STMT_INSERT,
      Stmt_Create  => OCI_STMT_CREATE,
      Stmt_Drop    => OCI_STMT_DROP,
      Stmt_Alter   => OCI_STMT_ALTER,
      Stmt_Begin   => OCI_STMT_BEGIN,
      Stmt_Declare => OCI_STMT_DECLARE);
   for Statement_Type'Size use Ub2'Size;

   for Data_Type use
     (Type_String     => SQLT_STR,
      Type_Char_Lob   => SQLT_CLOB,
      Type_Bin_Lob    => SQLT_BLOB,
      Type_Integer    => SQLT_INT,
      Type_Long_Float => SQLT_FLT,
      Type_Number     => SQLT_VNU,
      Type_Date       => SQLT_DAT); -- ?? try OCIDate => SQLT_ODT
   for Data_Type'Size use Ub2'Size;

   type Context_Type;

   type Context_Access is access all Context_Type;

   type Bind_Type is record
      Name     : Bind_Names.Bounded_String;
      Kind     : Data_Type;
      Position : Positive;
      Iter     : Natural; -- To detect Oracle bug resetting iteration counter
      Was      : Boolean; -- /
      Loc      : Lobs.Locator;
      --  Need to keep *in* Loc value for set *out*.
      Bind     : aliased OCIBind := OCIBind (Empty_Handle);
      Context  : Context_Access;
   end record;

   type Define_Type is record
      Param    : aliased OCIParam  := OCIParam (System.Null_Address);
      Kind     : Data_Type;
      Position : Positive;
      Loc      : Lobs.Locator;
      --  For define with already created locator.
      Define   : aliased OCIDefine := OCIDefine (System.Null_Address);
      Context  : Context_Access;
   end record;

   type Bind_Array is array (Positive range <>) of aliased Bind_Type;
   type Bind_Array_Access is access all Bind_Array;
   type Bind_Access is access Bind_Type;

   type Define_Array is array (Positive range <>) of aliased Define_Type;
   type Define_Array_Access is access all Define_Array;
   type Define_Access is access Define_Type;

   Char_Buffer_Size : constant := 1024;

   type Buffer_Type (Kind : Data_Type := Type_String) is record
      case Kind is
      when Type_String     => Char : C.char_array (1 .. Char_Buffer_Size);
      when Type_Integer    => Int  : Integer;
      when Type_Long_Float => Flt  : Long_Float;
      when Type_Number     => Numb : OCINumber;
      when Type_Date       => Dat  : Date.Internal.Date_Time;
      when Type_Char_Lob | Type_Bin_Lob => Dummy : Integer;
      end case;
   end record;
   pragma Unchecked_Union (Buffer_Type);

   type Statement is new Handle_Reference with record
      Connect : Connection;
      Binds   : Bind_Array_Access;
      Context : Context_Access;
      Executed, Described, Executing : Boolean := False;
   end record;

   procedure Destroy (Object : in out Statement);

end OCI.Thick.DB;
