/*
 * Copyright (C) 2014 Guitarix project MOD project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * --------------------------------------------------------------------------
 */


////////////////////////////// LOCAL INCLUDES //////////////////////////

#include "gx_common.h"      // faust support and denormal protection (SSE)
#include "gx_bypass.cc"     // bypass ramping
#include "gx_cstb.h"        // define struct PortIndex
#include "gx_pluginlv2.h"   // define struct PluginLV2
#include "cstb.cc"    // dsp class generated by faust -> dsp2cc

////////////////////////////// PLUG-IN CLASS ///////////////////////////

namespace cstb {

class Gx_cstb_
{
private:
  GxBypass                     bp;
  float*                       bypass;
  DenormalProtection           MXCSR;
  // pointer to buffer
  float*      output;
  float*      input;
  // pointer to dsp class
  PluginLV2*  cstb;
  // private functions
  inline void run_dsp_(uint32_t n_samples);
  inline void connect_(uint32_t port,void* data);
  inline void init_dsp_(uint32_t rate);
  inline void connect_all__ports(uint32_t port, void* data);
  inline void activate_f();
  inline void clean_up();
  inline void deactivate_f();

public:
  // LV2 Descriptor
  static const LV2_Descriptor descriptor;
  // static wrapper to private functions
  static void deactivate(LV2_Handle instance);
  static void cleanup(LV2_Handle instance);
  static void run(LV2_Handle instance, uint32_t n_samples);
  static void activate(LV2_Handle instance);
  static void connect_port(LV2_Handle instance, uint32_t port, void* data);
  static LV2_Handle instantiate(const LV2_Descriptor* descriptor,
                                double rate, const char* bundle_path,
                                const LV2_Feature* const* features);
  Gx_cstb_();
  ~Gx_cstb_();
};

// constructor
Gx_cstb_::Gx_cstb_() :
  bp(),
  bypass(0),
  MXCSR(),
  output(NULL),
  input(NULL),
  cstb(cstb::plugin()) {};

// destructor
Gx_cstb_::~Gx_cstb_()
{
  // just to be sure the plug have given free the allocated mem
  // it didn't hurd if the mem is already given free by clean_up()
  if (cstb->activate_plugin !=0)
    cstb->activate_plugin(false, cstb);
  // delete DSP class
  cstb->delete_instance(cstb);
};

///////////////////////// PRIVATE CLASS  FUNCTIONS /////////////////////

void Gx_cstb_::init_dsp_(uint32_t rate)
{
  bp.init_bypass(rate);
  cstb->set_samplerate(rate, cstb); // init the DSP class
}

// connect the Ports used by the plug-in class
void Gx_cstb_::connect_(uint32_t port,void* data)
{
  switch ((PortIndex)port)
    {
    case EFFECTS_OUTPUT:
      output = static_cast<float*>(data);
      break;
    case EFFECTS_INPUT:
      input = static_cast<float*>(data);
      break;
    case BYPASS: 
      bypass = static_cast<float*>(data); // , 0.0, 0.0, 1.0, 1.0 
      break;
    default:
      break;
    }
}

void Gx_cstb_::activate_f()
{
  // allocate the internal DSP mem
  if (cstb->activate_plugin !=0)
    cstb->activate_plugin(true, cstb);
}

void Gx_cstb_::clean_up()
{
  // delete the internal DSP mem
  if (cstb->activate_plugin !=0)
    cstb->activate_plugin(false, cstb);
}

void Gx_cstb_::deactivate_f()
{
  // delete the internal DSP mem
  if (cstb->activate_plugin !=0)
    cstb->activate_plugin(false, cstb);
}

void Gx_cstb_::run_dsp_(uint32_t n_samples)
{
  if (n_samples< 1) return;
  MXCSR.set_();
  FAUSTFLOAT buf[n_samples];
  // do inplace processing at default
  if (output != input)
    memcpy(output, input, n_samples*sizeof(float));
  // check if bypass is pressed
  if (!bp.pre_check_bypass(bypass, buf, input, n_samples))
    cstb->mono_audio(static_cast<int>(n_samples), input, output, cstb);
  bp.post_check_bypass(buf, output, n_samples);
  MXCSR.reset_();
}

void Gx_cstb_::connect_all__ports(uint32_t port, void* data)
{
  // connect the Ports used by the plug-in class
  connect_(port,data); 
  // connect the Ports used by the DSP class
  cstb->connect_ports(port,  data, cstb);
}

////////////////////// STATIC CLASS  FUNCTIONS  ////////////////////////

LV2_Handle 
Gx_cstb_::instantiate(const LV2_Descriptor* descriptor,
                            double rate, const char* bundle_path,
                            const LV2_Feature* const* features)
{
  // init the plug-in class
  Gx_cstb_ *self = new Gx_cstb_();
  if (!self)
    {
      return NULL;
    }

  self->init_dsp_((uint32_t)rate);

  return (LV2_Handle)self;
}

void Gx_cstb_::connect_port(LV2_Handle instance, 
                                   uint32_t port, void* data)
{
  // connect all ports
  static_cast<Gx_cstb_*>(instance)->connect_all__ports(port, data);
}

void Gx_cstb_::activate(LV2_Handle instance)
{
  // allocate needed mem
  static_cast<Gx_cstb_*>(instance)->activate_f();
}

void Gx_cstb_::run(LV2_Handle instance, uint32_t n_samples)
{
  // run dsp
  static_cast<Gx_cstb_*>(instance)->run_dsp_(n_samples);
}

void Gx_cstb_::deactivate(LV2_Handle instance)
{
  // free allocated mem
  static_cast<Gx_cstb_*>(instance)->deactivate_f();
}

void Gx_cstb_::cleanup(LV2_Handle instance)
{
  // well, clean up after us
  Gx_cstb_* self = static_cast<Gx_cstb_*>(instance);
  self->clean_up();
  delete self;
}

const LV2_Descriptor Gx_cstb_::descriptor =
{
  GXPLUGIN_URI "#_cstb_",
  Gx_cstb_::instantiate,
  Gx_cstb_::connect_port,
  Gx_cstb_::activate,
  Gx_cstb_::run,
  Gx_cstb_::deactivate,
  Gx_cstb_::cleanup,
  NULL
};


} // end namespace cstb

////////////////////////// LV2 SYMBOL EXPORT ///////////////////////////

extern "C"
LV2_SYMBOL_EXPORT
const LV2_Descriptor*
lv2_descriptor(uint32_t index)
{
  switch (index)
    {
    case 0:
      return &cstb::Gx_cstb_::descriptor;
    default:
      return NULL;
    }
}

///////////////////////////// FIN //////////////////////////////////////
