/*
 * Copyright 2001-2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.commons.logging.impl;

import org.apache.commons.logging.Log;
import org.apache.log4j.Priority;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;

import java.io.Serializable;
import java.security.PrivilegedAction;
import java.security.AccessController;
import java.lang.reflect.Method;
import java.net.URLClassLoader;
import java.net.URL;
import java.util.Arrays;

/**
 * <p>Implementation of {@link org.apache.commons.logging.Log} that maps directly to a Log4J
 * <strong>Logger</strong>.  Initial configuration of the corresponding
 * Logger instances should be done in the usual manner, as outlined in
 * the Log4J documentation.</p>
 *
 * @author <a href="mailto:sanders@apache.org">Scott Sanders</a>
 * @author Rod Waldhoff
 * @author Robert Burrell Donkin
 * @author Scott.Stark@jboss.org
 * @version $Id: DefaultLog4JLogger.java,v 1.1 2006/04/19 02:42:50 starksm Exp $
 */
public class DefaultLog4JLogger implements Log, Serializable {


    // ------------------------------------------------------------- Attributes

    /** The fully qualified name of the Log4JLogger class. */
    private static final String FQCN = DefaultLog4JLogger.class.getName();

    private static final boolean is12 = Priority.class.isAssignableFrom(Level.class);
    private Level TRACE;

    /** Log to this logger */
    private transient Logger logger = null;

    /** Logger name */
    private String name = null;


   private synchronized Level getTrace()
   {
      if( TRACE == null )
      {
         TRACE = Level.DEBUG;
         // Try to resolve the jboss custom TRACE level
         PrivilegedAction action = new PrivilegedAction()
         {
            public Object run()
            {
               ClassLoader loader = Thread.currentThread().getContextClassLoader();
               try
               {
                   Class c = loader.loadClass("org.jboss.logging.XLevel");
                   Class[] sig = {String.class, Level.class};
                   Method toLevel = c.getMethod("toLevel", sig);
                   Object[] args = {"TRACE", Level.DEBUG};
                   TRACE = (Level) toLevel.invoke(null, args);
               }
               catch(ClassNotFoundException ignore)
               {
                  System.err.println("org.jboss.logging.XLevel not found, trace maps to Level.DEBUG");
               }
               catch(Throwable t)
               {
                  if( loader instanceof URLClassLoader )
                  {
                     URLClassLoader ucl = (URLClassLoader) loader;
                     URL[] cp = ucl.getURLs();
                     System.out.println("+++ Loader.getURLs: "+ Arrays.asList(cp));
                  }
                  t.printStackTrace();
               }
               return null;
            }
         };
         AccessController.doPrivileged(action);
      }
      return TRACE;
   }

    // ------------------------------------------------------------ Constructor

    public DefaultLog4JLogger() {
    }


    /**
     * Base constructor.
     */
    public DefaultLog4JLogger(String name) {
        this.name = name;
        this.logger = getLogger();
        TRACE = getTrace();
    }

    /** For use with a log4j factory.
     */
    public DefaultLog4JLogger(Logger logger ) {
        this.name = logger.getName();
        this.logger=logger;
    }


    // --------------------------------------------------------- Implementation


    /**
     * Log a message to the Log4j Logger with <code>TRACE</code> priority.
     * Currently logs to <code>DEBUG</code> level in Log4J.
     */
    public void trace(Object message) {
       Level trace = getTrace();
        if(DefaultLog4JLogger.is12) {
            getLogger().log(DefaultLog4JLogger.FQCN, (Priority) trace, message, null );
        } else {
            getLogger().log(DefaultLog4JLogger.FQCN, trace, message, null );
        }
    }


    /**
     * Log an error to the Log4j Logger with <code>TRACE</code> priority.
     * Currently logs to <code>DEBUG</code> level in Log4J.
     */
    public void trace(Object message, Throwable t) {
       Level trace = getTrace();
        if(DefaultLog4JLogger.is12) {
            getLogger().log(DefaultLog4JLogger.FQCN, (Priority) trace, message, t );
        } else {
            getLogger().log(DefaultLog4JLogger.FQCN, trace, message, t );
        }
    }


    /**
     * Log a message to the Log4j Logger with <code>DEBUG</code> priority.
     */
    public void debug(Object message) {
        if(DefaultLog4JLogger.is12) {
            getLogger().log(DefaultLog4JLogger.FQCN, (Priority) Level.DEBUG, message, null );
        } else {
            getLogger().log(DefaultLog4JLogger.FQCN, Level.DEBUG, message, null );
        }
    }

    /**
     * Log an error to the Log4j Logger with <code>DEBUG</code> priority.
     */
    public void debug(Object message, Throwable t) {
        if(DefaultLog4JLogger.is12) {
            getLogger().log(DefaultLog4JLogger.FQCN, (Priority) Level.DEBUG, message, t );
        } else {
            getLogger().log(DefaultLog4JLogger.FQCN, Level.DEBUG, message, t );
        }
    }


    /**
     * Log a message to the Log4j Logger with <code>INFO</code> priority.
     */
    public void info(Object message) {
        if(DefaultLog4JLogger.is12) {
            getLogger().log(DefaultLog4JLogger.FQCN, (Priority) Level.INFO, message, null );
        } else {
            getLogger().log(DefaultLog4JLogger.FQCN, Level.INFO, message, null );
        }
    }


    /**
     * Log an error to the Log4j Logger with <code>INFO</code> priority.
     */
    public void info(Object message, Throwable t) {
        if(DefaultLog4JLogger.is12) {
            getLogger().log(DefaultLog4JLogger.FQCN, (Priority) Level.INFO, message, t );
        } else {
            getLogger().log(DefaultLog4JLogger.FQCN, Level.INFO, message, t );
        }
    }


    /**
     * Log a message to the Log4j Logger with <code>WARN</code> priority.
     */
    public void warn(Object message) {
        if(DefaultLog4JLogger.is12) {
            getLogger().log(DefaultLog4JLogger.FQCN, (Priority) Level.WARN, message, null );
        } else {
            getLogger().log(DefaultLog4JLogger.FQCN, Level.WARN, message, null );
        }
    }


    /**
     * Log an error to the Log4j Logger with <code>WARN</code> priority.
     */
    public void warn(Object message, Throwable t) {
        if(DefaultLog4JLogger.is12) {
            getLogger().log(DefaultLog4JLogger.FQCN, (Priority) Level.WARN, message, t );
        } else {
            getLogger().log(DefaultLog4JLogger.FQCN, Level.WARN, message, t );
        }
    }


    /**
     * Log a message to the Log4j Logger with <code>ERROR</code> priority.
     */
    public void error(Object message) {
        if(DefaultLog4JLogger.is12) {
            getLogger().log(DefaultLog4JLogger.FQCN, (Priority) Level.ERROR, message, null );
        } else {
            getLogger().log(DefaultLog4JLogger.FQCN, Level.ERROR, message, null );
        }
    }


    /**
     * Log an error to the Log4j Logger with <code>ERROR</code> priority.
     */
    public void error(Object message, Throwable t) {
        if(DefaultLog4JLogger.is12) {
            getLogger().log(DefaultLog4JLogger.FQCN, (Priority) Level.ERROR, message, t );
        } else {
            getLogger().log(DefaultLog4JLogger.FQCN, Level.ERROR, message, t );
        }
    }


    /**
     * Log a message to the Log4j Logger with <code>FATAL</code> priority.
     */
    public void fatal(Object message) {
        if(DefaultLog4JLogger.is12) {
            getLogger().log(DefaultLog4JLogger.FQCN, (Priority) Level.FATAL, message, null );
        } else {
            getLogger().log(DefaultLog4JLogger.FQCN, Level.FATAL, message, null );
        }
    }


    /**
     * Log an error to the Log4j Logger with <code>FATAL</code> priority.
     */
    public void fatal(Object message, Throwable t) {
        if(DefaultLog4JLogger.is12) {
            getLogger().log(DefaultLog4JLogger.FQCN, (Priority) Level.FATAL, message, t );
        } else {
            getLogger().log(DefaultLog4JLogger.FQCN, Level.FATAL, message, t );
        }
    }


    /**
     * Return the native Logger instance we are using.
     */
    public Logger getLogger() {
        if (logger == null) {
            logger = Logger.getLogger(name);
        }
        return (this.logger);
    }


    /**
     * Check whether the Log4j Logger used is enabled for <code>DEBUG</code> priority.
     */
    public boolean isDebugEnabled() {
        return getLogger().isDebugEnabled();
    }


     /**
     * Check whether the Log4j Logger used is enabled for <code>ERROR</code> priority.
     */
    public boolean isErrorEnabled() {
        if(DefaultLog4JLogger.is12) {
            return getLogger().isEnabledFor((Priority) Level.ERROR);
        } else {
            return getLogger().isEnabledFor(Level.ERROR);
        }
    }


    /**
     * Check whether the Log4j Logger used is enabled for <code>FATAL</code> priority.
     */
    public boolean isFatalEnabled() {
        if(DefaultLog4JLogger.is12) {
            return getLogger().isEnabledFor((Priority) Level.FATAL);
        } else {
            return getLogger().isEnabledFor(Level.FATAL);
        }
    }


    /**
     * Check whether the Log4j Logger used is enabled for <code>INFO</code> priority.
     */
    public boolean isInfoEnabled() {
        return getLogger().isInfoEnabled();
    }


    /**
     * Check whether the Log4j Logger used is enabled for <code>TRACE</code> priority.
     * For Log4J, this returns the value of <code>isEnabledFor(TRACE)</code>
     */
    public boolean isTraceEnabled() {
       Level trace = getTrace();
        return getLogger().isEnabledFor(trace);
    }

    /**
     * Check whether the Log4j Logger used is enabled for <code>WARN</code> priority.
     */
    public boolean isWarnEnabled() {
        if(DefaultLog4JLogger.is12) {
            return getLogger().isEnabledFor((Priority) Level.WARN);
        } else {
            return getLogger().isEnabledFor(Level.WARN);
        }
    }
}
