import argparse
import imp
import logging
import os
import os.path
import sys
import traceback


from twisted.internet import reactor
from twisted.internet.defer import inlineCallbacks
import yaml
import zookeeper

from aiki.introspect import get_zk_client_connector


LOG_LEVELS = dict(
    CRITICAL=logging.CRITICAL,
    ERROR=logging.ERROR,
    WARNING=logging.WARNING,
    INFO=logging.INFO,
    DEBUG=logging.DEBUG)


def make_arg_parser(root_parser, subcommand, **kwargs):
    if root_parser is None:
        if "help" in kwargs:
            # Not valid for a top-level parser
            del kwargs["help"]
        parser = argparse.ArgumentParser(**kwargs)
    else:
        if "help" not in kwargs:
            kwargs["help"] = kwargs["description"]  # Fallback
        parser = root_parser.add_parser(subcommand, **kwargs)
    parser.add_argument(
        "-e", "--environment", default=None, help="Environment to act upon (otherwise uses default)", metavar="ENVIRONMENT")
    parser.add_argument(
        "--loglevel", default=None, choices=LOG_LEVELS, help="Log level",
        metavar="CRITICAL|ERROR|WARNING|INFO|DEBUG")
    parser.add_argument("--verbose", "-v", default=False, action="store_true", help="Enable verbose logging")
    return parser


def setup_logging(options):
    options.loglevel = LOG_LEVELS.get(options.loglevel)
    if options.verbose:
        if options.loglevel is None:
            options.loglevel = logging.DEBUG
    else:
        # Avoid potentially voluminous ZK debug logging
        zookeeper.set_debug_level(0)
        if options.loglevel is None:
            options.loglevel = logging.INFO

    log_options = {
        "level": options.loglevel,
        "format": "%(asctime)s %(name)s:%(levelname)s %(message)s"
        }
    logging.basicConfig(**log_options)


def run_command(command, options):
    """Connect to ZK, either from admin machine or on a Juju machine, then run command"""
    # Introspect to find the right client connector; this will be
    # called, then yielded upon, to get a ZK client once async code is
    # entered
    try:
        connector = get_zk_client_connector(options)
    except Exception, e:
        print >> sys.stderr, e
        sys.exit(1)

    exit_code = [0]
    reactor.callWhenRunning(_run_command_in_reactor, command, exit_code, connector, options)
    reactor.run()
    sys.exit(exit_code[0])


@inlineCallbacks
def _run_command_in_reactor(command, exit_code, connector, options):
    try:
        yield _wrapped_command(command, exit_code, connector, options)
    except Exception, e:
        exit_code[0] = 1
        if options.verbose:
            traceback.print_exc()  # Writes to stderr
        else:
            print >> sys.stderr, e
    finally:
        reactor.stop()


@inlineCallbacks
def _wrapped_command(command, result, connector, options):
    client = yield connector()
    try:
        yield command(result, client, options)
    finally:
        yield client.close()


def parse_relation_settings(args):
    """Given args, return a list of the remaining args along with settings"""
    settings = {}
    kv_iter = iter(args)
    remaining_args = []
    for kv in kv_iter:
        if "=" not in kv:
            remaining_args = [kv]
            break
        k, v = kv.split("=", 1)
        if v.startswith("@"):
            filename = v[1:]
            with open(filename, "r") as f:
                v = f.read()
        settings[k] = yaml.safe_load(v)
    remaining_args.extend(kv_iter)
    return settings, remaining_args


def get_commands(dir):
    """Given a directory, return valid jitsu commands"""
    for cmd in os.listdir(dir):
        # Filter anything not a bare command word (so dotted) and certain
        # top-level names that can potentially look like commands
        if '.' in cmd or cmd in ('Makefile', 'aiki'):
            continue
        yield (cmd,
               dict(
                home=dir,
                symlink=os.path.islink(os.path.join(dir, cmd))))


def add_command_help(subparser, subcommand_home, cmd, details):
    """Given the subparser, adds a subcommand parser to it for use in help

    Attempts two strategies:

    1. If the script is Python (marked as such and parseable), the
       script must support a top-level `make_parser` function, which
       takes a subparser obj that help can be added to. This strategy
       is only attempted if the command is built-in jitsu.

    2. Otherwise, attempts to read the help from the initial comments
       of the script.
    """
    filename = os.path.join(details["home"], cmd)
    attempt_python_parse = subcommand_home == details["home"]
    with open(filename) as f:
        lines = f.readlines()
    try:
        if not attempt_python_parse or lines[0].rstrip() != "#!/usr/bin/env python":
            raise ValueError("Not a Python source file")
        source = "".join(lines)
        mod_name = "jitsu_" + cmd
        mod = imp.new_module(mod_name)
        mod.__file__ = filename
        mod.__name__ = mod_name
        # NOTE: Does not set these attributes for mod: __loader__, __path__, __package__
        exec source in mod.__dict__
        mod.make_parser(subparser)
    except Exception:
        add_command_help_from_comments(subparser, cmd, details, lines)


def add_command_help_from_comments(subparser, cmd, details, lines):
    """Attempt parsing command script comments to get description and usage.

    Uses the following convention. Symlinks are resolved:

        #!/path/to/interpreter
        # command - short description
        # then any optional usage info as comments, 
        # blank lines terminating, up to a
        # Copyright notice
    """
    real_cmd = os.path.split(os.path.realpath(os.path.join(details["home"], cmd)))[1]
    cmdspec = "# " + real_cmd + " - "  # Using real_cmd resolves to the symlinked version
    help = cmd  # Fallback in case we don't see the cmdspec
    usage = []
    line_iter = iter(lines[1:])
    for line in line_iter:
        line = line.strip()
        if cmdspec in line:
            help = line.split(cmdspec)[1]
            for usage_line in line_iter:
                if usage_line.startswith("# ") and "Copyright" not in usage_line:
                    usage.append(usage_line[2:])
                else:
                    break
            break
    subparser.add_parser(
        cmd, description=cmd,
        help=help, usage="".join(usage), formatter_class=argparse.RawDescriptionHelpFormatter)


def format_states(states):
    return "[" + ", ".join(sorted(states)) + "]"


def format_port_pairs(port_pairs):
    return "[" + ", ".join("%s/%s" % (port, proto) for port, proto in sorted(port_pairs)) + "]"


def format_descriptors(*descriptors):
    if len(descriptors) == 1:
        return "peer relation %s" % descriptors[0]
    else:
        return "relation between " + " and ".join(sorted(descriptors))
