/***************************************************************************
 *   Copyright (C) 2007-2012 by Marcel Hasler                              *
 *   mahasler@gmail.com                                                    *
 *                                                                         *
 *   This file is part of KGmailNotifier.                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation, either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the          *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program. If not, see <http://www.gnu.org/licenses/>.  *
 ***************************************************************************/

#include "account.h"

#include <QTextStream>
static QTextStream cout(stdout, QIODevice::WriteOnly);
static QTextStream cerr(stderr, QIODevice::WriteOnly);

// ------------------------------------------------------------------------------------------------------------------- //
//    Construction & Destruction                                                                                       //
// ------------------------------------------------------------------------------------------------------------------- //

Account::Account(const QString &username, const QString &password)
    : QObject()
{
    // Set up
    m_atomParser = new AtomParser;
    m_lastMessageCount = 0;
    m_currentMessageCount = 0;

    m_jobTimeoutTimer = new QTimer();
    m_jobTimeoutTimer->setSingleShot(true);
    connect(m_jobTimeoutTimer, SIGNAL(timeout()), this, SLOT(slotAbortJob()));

    m_atomFileUrl = new KUrl();
    m_atomFileUrl->setProtocol("https");
    m_atomFileUrl->setHost("mail.google.com");
    m_atomFileUrl->setPath("/mail/feed/atom");
    setCredentials(username, password);

    m_tmpFileUrl = new KUrl();
    m_tmpFileUrl->setProtocol("file");
    m_tmpFileUrl->setPath("/tmp/kgn_atom_" + username);
}

// ------------------------------------------------------------------------------------------------------------------- //

Account::~Account()
{
    if (m_atomParser)
        delete m_atomParser;

    if (m_fileCopyJob)
        delete m_fileCopyJob;

    if (m_atomFileUrl)
        delete m_atomFileUrl;

    if (m_tmpFileUrl)
        delete m_tmpFileUrl;

    if (m_jobTimeoutTimer)
        delete m_jobTimeoutTimer;
}


// ------------------------------------------------------------------------------------------------------------------- //
//    Public methods                                                                                                   //
// ------------------------------------------------------------------------------------------------------------------- //

void Account::setCredentials(const QString &username, const QString &password)
{
    m_atomFileUrl->setUser(username);
    m_atomFileUrl->setPass(password);
}

// ------------------------------------------------------------------------------------------------------------------- //

void Account::checkMail(unsigned int timeout)
{
    // Start downloading the atom file and connect the slots for parsing
    m_fileCopyJob = KIO::file_copy(*m_atomFileUrl, *m_tmpFileUrl, -1, KIO::HideProgressInfo | KIO::Overwrite);
    m_fileCopyJob->addMetaData("no-auth-prompt", "true");
    connect(m_fileCopyJob, SIGNAL(result(KJob *)), this, SLOT(slotParseData(KJob *)));

    // Start timer for connection timeout
    m_jobTimeoutTimer->start(timeout*1000);
}


// ------------------------------------------------------------------------------------------------------------------- //
//    Public access methods                                                                                            //
// ------------------------------------------------------------------------------------------------------------------- //

unsigned int Account::messageCount() const
{
    return m_currentMessageCount;
}

// ------------------------------------------------------------------------------------------------------------------- //

bool Account::hasNewMail() const
{
    if (m_currentMessageCount > m_lastMessageCount)
        return true;
    else
        return false;
}

// ------------------------------------------------------------------------------------------------------------------- //

const QString& Account::lastSender() const
{
    return m_lastSender;
}

// ------------------------------------------------------------------------------------------------------------------- //

const QString& Account::lastSubject() const
{
    return m_lastSubject;
}


// ------------------------------------------------------------------------------------------------------------------- //
//    Private slots                                                                                                    //
// ------------------------------------------------------------------------------------------------------------------- //

void Account::slotAbortJob()
{
    m_jobTimeoutTimer->stop();
    m_fileCopyJob->kill();
    emit connectionError("timeout");
}

// ------------------------------------------------------------------------------------------------------------------- //

void Account::slotParseData(KJob *job)
{
    m_jobTimeoutTimer->stop();

    if (job->error())
    {
        emit connectionError("failed");
        return;
    }

    bool success = m_atomParser->parseFile(m_tmpFileUrl->path());

    if (success)
    {
        m_lastMessageCount = m_currentMessageCount;
        m_currentMessageCount = m_atomParser->messageCount();

        if (m_currentMessageCount == 0)
        {
            m_lastSender = "";
            m_lastSubject = "";
        }
        else
        {
            m_lastSender = m_atomParser->lastSender();
            m_lastSubject = m_atomParser->lastSubject();
        }
    }
    else
        cerr << "An error occured while parsing the atom feed!" << endl;

    KIO::file_delete(*m_tmpFileUrl);

    emit connectionSuccessful();
}

// ------------------------------------------------------------------------------------------------------------------- //

#include "account.moc"
