/***************************************************************************
 *   Copyright (C) 2006-2007 by Julien Lemoine, Simon Viennot              *
 *   lapinot@tuxfamily.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "calcul.h"
#include "tirage.h"

//Initialisation d'un calcul vide, par dfaut on le suppose de type additif.
calcul::calcul() {
	est_plaque=false;
	est_additif=true;
	signe=true;		//par dfaut : positif
	valeur_calcul=0;	//par dfaut : 0
}

//Initialisation d'un calcul  partir d'une plaque.
calcul::calcul(const plaque &plaque1) {
	est_plaque=true;
	pl=plaque1;
	valeur_calcul=plaque1;
	
	est_additif=true;	//par dfaut : additif
	signe=true;		//par dfaut : positif
}

//initialisation de la variable statique (par dfaut, on est au niveau suprieur)
//utilis pour ne parenthser que les niveaux infrieurs
bool calcul::niv_sup=true;

//Fonction d'affichage d'un calcul, fonctionne rcursivement.
//Elle affiche dans le sens inverse du stockage
//(on stocke dans l'ordre croissant selon l'oprateur < dfini plus bas, et on affiche dans l'ordre dcroissant selon <).
string calcul::affiche_calcul() const {
	string resultat;
	
	if (est_plaque){
		stringstream ss;
		ss << (int) pl;
		ss >> resultat;
	}
	else{
		list<calcul>::const_reverse_iterator it;
		string Plus=string("x"), Moins=string("/"), Parenth_debut, Parenth_fin;
		
		if (est_additif){
			Plus="+"; Moins="-";
			if (!calcul::niv_sup) {
				Parenth_debut="("; Parenth_fin=")";
			}
		}
		
		bool sauv=calcul::niv_sup;
		calcul::niv_sup=false;			//niv_sup passe  faux
		resultat.append(Parenth_debut);
		
		it=sous_calculs.rbegin();
		resultat.append(it->affiche_calcul());				//affichage du premier calcul additionn
		it++;
		
		while(it!=sous_calculs.rend()) {
			if (it->signe==true) {
				resultat.append(Plus);
			}
			else {
				resultat.append(Moins);
			}
			resultat.append(it->affiche_calcul());
			it++;
		}
		
		resultat.append(Parenth_fin);
		calcul::niv_sup=sauv;			//on rtablit la valeur de niv_sup
	}
	return resultat;
}

//operateur d'galit sur les "calcul", dfini rcursivement
bool calcul::operator==(const calcul& a) const{
	if (valeur_calcul!=a.valeur_calcul) {
		return false;
	}
	
	if(est_plaque!=a.est_plaque || est_additif!=a.est_additif || signe!=a.signe) {
		return false;
	}

	if(est_plaque) {
		return pl==a.pl;
	}
	
	//les calculs sont gaux si leurs sous_calculs sont gaux
	//list<calcul>::operator== appelle calcul::operator== d'o la rcursivit
	return sous_calculs==a.sous_calculs;
}

//Oprateur de comparaison sur les "calcul", dfini rcursivement.
//Rgles de comparaison
// 0.calcul ngatif < calcul positif
// 1.petite valeur < grande valeur
// 2.plaque < calcul plus compliqu
// 3.calcul additif < calcul multiplicatif
// 4.calcul court < calcul long
// 5.rcursivit sur la liste des sous-calculs
bool calcul::operator<(const calcul& a) const{
	if (signe != a.signe) {
		return !signe;				//rgle 0
	}
	
	if (valeur_calcul != a.valeur_calcul) {		//rgle 1
		return valeur_calcul<a.valeur_calcul;
	}
	
	if (est_plaque != a.est_plaque) {
		return est_plaque;			//rgle 2
	}
	
	if (est_additif != a.est_additif){
		return est_additif;			//rgle 3
	}
	
	int nb  =  sous_calculs.size();
	int nb_a=a.sous_calculs.size();
	
	if(nb!=nb_a){
		return nb<nb_a;				//rgle 4
	}
	
	if(sous_calculs!=a.sous_calculs) {
		return sous_calculs<a.sous_calculs;	//rgle 5 (rcursivit)
	}
	
	return false;					//cas d'galit
}

//oprateur d'addition, que l'on n'appellera jamais sur un calcul vide
calcul calcul::operator+(const calcul& a) const {
	calcul resultat=reunion(a, true, false);			//type_additif==true et opposition==false
	resultat.valeur_calcul=valeur_calcul+a.valeur_calcul;
	return resultat;
}

//oprateur de soustraction, on suppose que this.valeur_calcul>a.valeur_calcul
calcul calcul::operator-(const calcul& a) const {
	calcul resultat=reunion(a, true, true);				//type_additif==true et opposition==true
	resultat.valeur_calcul=valeur_calcul-a.valeur_calcul;
	return resultat;
}

//oprateur de multiplication
calcul calcul::operator*(const calcul& a) const {
	calcul resultat=reunion(a, false, false);			//type_additif==false et opposition==false	
	resultat.valeur_calcul=valeur_calcul*a.valeur_calcul;
	return resultat;
}

//oprateur de division, on suppose que this.valeur_calcul est divisible par a.valeur_calcul...
calcul calcul::operator/(const calcul& a) const {
	calcul resultat=reunion(a, false, true);			//type_additif==false et opposition==true
	resultat.valeur_calcul=valeur_calcul/a.valeur_calcul;
	return resultat;
}

//inverse les signes de la liste des sous-calculs
void calcul::oppose_sous_calculs() {
	list<calcul>::iterator Ci; 			//calcul i
	
	for(Ci=sous_calculs.begin(); Ci!=sous_calculs.end(); Ci++){
		Ci->signe=!(Ci->signe);
	}
}

//fonction unique pour raliser toutes les oprations : on a rsultat=(*this)op(a) avec op = +, -, *, /
//type_additif et opposition dcrive le type d'opration  raliser
// "+" : type_additif=true opposition=false
// "-" : type_additif=true opposition=true
// "*" : type_additif=false opposition=false
// "/" : type_additif=false opposition=true
calcul calcul::reunion(const calcul& a, bool type_additif, bool opposition) const {
	calcul resultat;
	resultat.est_additif=type_additif;
	
	//si a n'est pas un calcul de mme type que l'opration, alors a est un sous-calcul du rsultat
	//sinon ce sont ses sous-calculs qui sont des sous-calculs du rsultat
	if (a.est_plaque || a.est_additif!=type_additif) {
		resultat.sous_calculs.push_back(a);
	}else{
		resultat.sous_calculs=a.sous_calculs;
	}
	
	//dans le cas o l'opration est de type opposition, on inverse les sous-calculs
	if (opposition==true) resultat.oppose_sous_calculs();
	
	//on ajoute *this au rsultat de la mme faon que a
	if (est_plaque || est_additif!=type_additif) {		
		resultat.sous_calculs.push_back(*this);
	}else{
		resultat.sous_calculs.insert(resultat.sous_calculs.end(), sous_calculs.begin(), sous_calculs.end());
	}

	//tri des sous_calculs
	resultat.sous_calculs.sort();
	
	return resultat;
}

//--------------------------------------------------------------
//Fonctions d'analyse des calculs
//--------------------------------------------------------------

//fonction renvoyant la liste des sous_calculs au sens large d'un calcul, de profondeur 1,
//de taille >=2 car ceux de taille 1 ont dj t vrifis comme tant canoniques
list<calcul> calcul::liste_sous_calculs() const{
	list<calcul> resultat;
	if(est_plaque){
		return resultat;
	}

	int nb=sous_calculs.size();
	calcul calc_temp;
	compte valeur_calc_temp;
	calc_temp.est_additif=est_additif;
	
	//calcul des puissances de 2 indiquant quels calculs on garde
	int deux_puiss_pos=1;
	int i;
	for(i=0;i<nb;i++){
		deux_puiss_pos=deux_puiss_pos*2;
	}
	
	list<calcul>::const_iterator Ck;
	
	for(i=0;i<deux_puiss_pos;i++){
		int k=i;
		
		if (est_additif) {
			valeur_calc_temp=0;
		}
		else {
			valeur_calc_temp=1;
		}
		
		calc_temp.sous_calculs.clear();
		compte valeur_calc_div=1;
		
		for(Ck=sous_calculs.begin(); Ck!=sous_calculs.end(); Ck++){
			if(k%2==1){
				calc_temp.sous_calculs.push_back(*Ck);
				if(est_additif){
					if (Ck->signe==true) {
						valeur_calc_temp += Ck->valeur_calcul;
					}
					else {
						valeur_calc_temp -= Ck->valeur_calcul;
					}
				}
				else{
					if (Ck->signe==true) {
						valeur_calc_temp *= Ck->valeur_calcul;
					}
					else {
						valeur_calc_div *= Ck->valeur_calcul;
					}
				}
			}
			
			k=k/2;
		}
		
		//on garde les sous-calculs de taille >=2 et <taille_calcul
		bool valable=true;
		int nb_calculs=calc_temp.sous_calculs.size();
		
		if (nb_calculs>1 && nb_calculs<nb){
		
			if(est_additif && valeur_calc_temp<0){
				valeur_calc_temp=-valeur_calc_temp;
				calc_temp.oppose_sous_calculs();
			}
			
			if(!est_additif){
				if(valeur_calc_temp%valeur_calc_div==0){
					valeur_calc_temp /= valeur_calc_div;
				}
				else {
					if(valeur_calc_div%valeur_calc_temp==0){
						valeur_calc_temp = valeur_calc_div/valeur_calc_temp;
						calc_temp.oppose_sous_calculs();
					}
					else{
						valable=false;
					}
				}
			}
			
			if(valable){
				calc_temp.sous_calculs.sort();
				calc_temp.valeur_calcul=valeur_calc_temp;
				resultat.push_back(calc_temp);
			}
		}
	}
	return resultat;
}

//Etant donn un calcul, renvoie la liste des comptes obtenus en rorganisant les fils.
//Si "utiliser_tous_enfants" est  true, on peut utiliser tous les fils dans une rorganisation.
//Sinon il faut en utiliser au plus un de moins.
list<compte> calcul::liste_valeurs_combinaisons_enfants(bool utiliser_tous_enfants) const{
	list<compte> resultat;
	
	list<calcul>::const_iterator Vi;			//valeur i
	list<calcul>::const_iterator Vdeb=sous_calculs.begin();	//dbut de la liste des valeurs (optimisation)
	
	int i, i_temp, j;
	int taille_liste=sous_calculs.size();
	
	int trois_puissance=1;		//c++ c'est vraiment de la merde, y'a mme pas de fonction puissance...
	for(i=1;i<=taille_liste;i++){
		trois_puissance=trois_puissance*3;
	}
	
	bool contient_zero;
	compte somme, prod, div;
	
	for(i=1;i<trois_puissance;i++){		//magouille avec les puissances de 3
		contient_zero=false;
		somme=0; prod=1; div=1;
		i_temp=i;
		
		for (j=0, Vi=Vdeb; j<taille_liste; j++, Vi++){
			switch(i_temp%3){
				case 0:contient_zero=true; break;
				case 1: if(est_additif) {
						somme+=Vi->valeur_calcul;
					} else {
						prod*=Vi->valeur_calcul;
					}
					break;
				case 2: if(est_additif) {
						somme-=Vi->valeur_calcul;
					} else {
						div*=Vi->valeur_calcul;
					}
					break;
			}
			i_temp=i_temp/3;
		}
		
		if(est_additif && (contient_zero==true || utiliser_tous_enfants==true) && somme>0){
			resultat.push_back(somme);
		}

		if(!est_additif && (contient_zero==true || utiliser_tous_enfants==true) && prod%div==0){
			resultat.push_back(prod/div);
		}
	}
	
	return resultat;
}

//Fonction rcursive dterminant la liste des comptes obtenus en rorganisant les fratries parmi les descendants.
//A utiliser avec le boolen "utiliser_tous_enfants"  false.
list<compte> calcul::liste_valeurs_combinaisons_descendants(bool utiliser_tous_enfants) const{
	list<compte> resultat=liste_valeurs_combinaisons_enfants(utiliser_tous_enfants);
	list<calcul>::const_iterator Ci;
	
	for(Ci=sous_calculs.begin();Ci!=sous_calculs.end();Ci++){
		if(!(Ci->est_plaque)){
			list<compte> popo=(Ci->liste_valeurs_combinaisons_descendants(true));
			resultat.splice(resultat.end(),popo);
		}
	}
	
	return resultat;
}

//renvoie la liste des valeurs des sous-calculs au sens large de profondeur 1, de taille >=1 et <taille_calcul (donc differents du calcul)
//le resultat est ngatif si le sous-calcul est dans l'autre sens dans le calcul : par exemple, pour 10+2-8, le sous calcul (8-2) fournira 
//le sous-compte -6. Pour 100*3/6, le sous-calcul 6/3 fournira le sous-compte -2.
//Fonction drive de liste_sous_calculs, doit y'avoir moyen de les fusionner, on verra a plus tard.
list<compte> calcul::liste_sous_comptes() const{
	list<compte> resultat;
	if(est_plaque){
		return resultat;
	}

	int nb=sous_calculs.size();
	calcul calc_temp;
	compte valeur_calc_temp;
	calc_temp.est_additif=est_additif;
	
	//calcul des puissances de 2 indiquant quels calculs on garde
	int deux_puiss_pos=1;
	int i;
	for(i=0;i<nb;i++){
		deux_puiss_pos=deux_puiss_pos*2;
	}
	
	list<calcul>::const_iterator Ck;
	
	for(i=0;i<deux_puiss_pos;i++){
		int k=i;
		
		if (est_additif) {
			valeur_calc_temp=0;
		}
		else {
			valeur_calc_temp=1;
		}
		
		calc_temp.sous_calculs.clear();
		compte valeur_calc_div=1;
		
		for(Ck=sous_calculs.begin(); Ck!=sous_calculs.end(); Ck++){
			if(k%2==1){
				calc_temp.sous_calculs.push_back(*Ck);
				if(est_additif){
					if (Ck->signe==true) {
						valeur_calc_temp += Ck->valeur_calcul;
					}
					else {
						valeur_calc_temp -= Ck->valeur_calcul;
					}
				}
				else{
					if (Ck->signe==true) {
						valeur_calc_temp *= Ck->valeur_calcul;
					}
					else {
						valeur_calc_div *= Ck->valeur_calcul;
					}
				}
			}
			
			k=k/2;
		}
		
		//on garde les sous-calculs de taille >=1 et <taille_calcul
		bool valable=true;
		int nb_calculs=calc_temp.sous_calculs.size();
		
		if (nb_calculs>=1 && nb_calculs<nb){
			
			bool signe_resultat=true; 		//true pour un rsultat positif, false pour un ngatif
			
			if(est_additif && valeur_calc_temp<0){
				signe_resultat=false;
				//valeur_calc_temp=-valeur_calc_temp;
				//calc_temp.sous_calculs=calc_temp.oppose_sous_calculs();
			}
			
			if(!est_additif){
				if(valeur_calc_temp%valeur_calc_div==0){
					valeur_calc_temp /= valeur_calc_div;
				}
				else {
					if(valeur_calc_div%valeur_calc_temp==0){
						signe_resultat=false;
						valeur_calc_temp = -valeur_calc_div/valeur_calc_temp;
						//calc_temp.sous_calculs=calc_temp.oppose_sous_calculs();
					}
					else{
						valable=false;
					}
				}
			}
			
			if(valable){
				//calc_temp.sous_calculs.sort();
				resultat.push_back(valeur_calc_temp);
			}
		}
	}
	return resultat;
}

//Renvoie la liste des plaques utilises par un calcul
tirage calcul::liste_plaques() const{
	tirage resultat;
	if(est_plaque){
		resultat.tab[pl]=1;
	} else{
		list<calcul>::const_iterator Ci;
		for(Ci=sous_calculs.begin(); Ci!=sous_calculs.end(); Ci++){
			resultat = resultat + (Ci->liste_plaques());
		}
	}
	return resultat;
}

//renvoie la liste des valeurs des sous-calculs de profondeur 1 au sens strict
tirage calcul::liste_valeurs() const{
	tirage resultat;
	list<calcul>::const_iterator Ci;
	for(Ci=sous_calculs.begin(); Ci!=sous_calculs.end(); Ci++){
		(resultat.tab[Ci->valeur_calcul])++;
	}
	return resultat;
	
}

//Fonction renvoyant le nombre de plaques utilises par le calcul, ncessaire pour jeu::afficheBase().
int calcul::longueur() const{
	if(est_plaque){
		return 1;
	}
	else{
		int resultat=0;
		list<calcul>::const_iterator Ci;
		for(Ci=sous_calculs.begin(); Ci!=sous_calculs.end(); Ci++){
			resultat+=Ci->longueur();
		}
		return resultat;
	}
}

//--------------------------------------------------------------
//Fonctions de canonisation
//--------------------------------------------------------------

//toutes les canonisations sont actives par dfaut
bool calcul::cano1_actif=true;
bool calcul::cano2_actif=true;
bool calcul::cano3_actif=true;

// fonction renvoyant false si le calcul n'est "pas assez simple"
bool calcul::est_cano() const{
	if(est_plaque) return true;
	
	// 2+2 n'est pas canonique, 2*2 oui (plus simple  programmer que le contraire  cause de 2/2=1=utile, 2-2=0=inutile).
	if(cano1_actif && !est_cano_1()) return false;
	
	// Aucun sous-calcul au sens large ne doit disposer parmi ses descendants d'une fratrie qui, rorganise, permet
	// d'obtenir la mme valeur que celle de ce sous-calcul
	if(cano2_actif && !est_cano_2()) return false;
	
	if(cano3_actif && !est_cano_3()) return false;
	
	return true;
}

//renvoie true si aucun sous-calcul au sens large de profondeur 1 n'est de type 2+2 (comme 100-2-10/5)
//attention, est un peu plus efficace que prvu : elle vire aussi les calculs du type 2-2 et donc empite un peu
//sur les plates bandes de est_cano_2
bool calcul::est_cano_1() const{
	if(est_additif){
		list<calcul> liste=liste_sous_calculs();
		liste.insert(liste.end(), sous_calculs.begin(), sous_calculs.end());
		list<calcul>::iterator Ci,Cj,Csuiv;
		for(Ci=liste.begin();Ci!=liste.end();Ci=Csuiv){
			Csuiv=Ci; Csuiv++;
			if(Ci->valeur_calcul!=2){
				liste.erase(Ci);
			}
		}
		if(liste.size()>1){
			tirage valeurs_du_calcul=liste_valeurs();
			for(Ci=liste.begin();Ci!=liste.end();Ci++){
				for(Cj=Ci;Cj!=liste.end();Cj++){
					if((Ci->liste_valeurs() + Cj->liste_valeurs()).inclus_dans(valeurs_du_calcul)){
						return false;
					}
				}
			}
		}
	}
	return true;
}

//Aucun sous-calcul au sens large ne doit disposer parmi ses descendants d'une fratrie qui, rorganise, permet
//d'obtenir la mme valeur que celle de ce sous-calcul.
bool calcul::est_cano_2() const{
	list<compte> liste=liste_valeurs_combinaisons_descendants(false);
	list<compte>::iterator Vi;
	
	for(Vi=liste.begin();Vi!=liste.end();Vi++){
		if(*Vi==valeur_calcul){
			return false;
		}
	}
	
	list<calcul> liste_sous_calc=liste_sous_calculs();
	list<calcul>::iterator Ci;
	
	for(Ci=liste_sous_calc.begin();Ci!=liste_sous_calc.end();Ci++){
		list<compte> liste2=Ci->liste_valeurs_combinaisons_descendants(false);
		for(Vi=liste2.begin();Vi!=liste2.end();Vi++){
			if(*Vi==Ci->valeur_calcul){
				return false;
			}
		}
	}
	
	return true;
}

//Pas de factorisation possible, ie 100*7+5*7 n'est pas canonique.
bool calcul::est_cano_3() const{
	if(est_plaque || (!est_additif)){
		return true;		//facto possible uniquement pour un calcul de type additif
	}
	list<calcul>::const_iterator Ci;
	list<compte> LC;
	list<compte>::const_iterator CPTj;
	set<compte> candidats_facto;		//le set o on stocke les valeurs possibles pour une factorisation (celles issues du premier fils)
	set<compte>::iterator CPTi,CPTsuiv;
	
	//-----------si un des fils est une plaque, pas de facto possible--------
	for(Ci=sous_calculs.begin();Ci!=sous_calculs.end();Ci++) {
		if(Ci->est_plaque){
			return true;
		}
	}
	
	//-------algo principal ------------------------------------------------
	Ci=sous_calculs.begin();
	LC=Ci->liste_sous_comptes();
	for(CPTj=LC.begin();CPTj!=LC.end();CPTj++){
		candidats_facto.insert(*CPTj);
	}
	
	for(Ci++;Ci!=sous_calculs.end();Ci++){
		LC=Ci->liste_sous_comptes();
		for(CPTi=candidats_facto.begin();CPTi!=candidats_facto.end();CPTi=CPTsuiv){
			CPTsuiv=CPTi; CPTsuiv++;
			bool appartient=false;
			for(CPTj=LC.begin();CPTj!=LC.end();CPTj++){
				if(*CPTj == *CPTi){
					appartient=true;
				}
			}
			if(!appartient){
				candidats_facto.erase(CPTi);
			}
		}
	}
	
	bool trouve=false;		//sera vrai si l'on trouve un candidat  la factorisation valable
	for(CPTi=candidats_facto.begin();CPTi!=candidats_facto.end();CPTi++){		//boucle pour viter le pb de (7+7)x8-8/(1+1)
		if(*CPTi < 0){		//si le candidat divise, pas de problme
			trouve=true;
		}
		else{			//cas o le candidat multiplie
			bool correct=true;
			for(Ci=sous_calculs.begin();Ci!=sous_calculs.end();Ci++){
				if((Ci->valeur_calcul % (*CPTi))!=0){
					correct=false;
				}
			}
			if(correct){
				trouve=true;
			}
		}
	}
	
	//le calcul est canonique si on ne peut pas le factoriser
	return !trouve;
}
