# Copyright 2013-2014 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Tests for `maastest.utils`."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

__metaclass__ = type
__all__ = []

import logging
import os.path
import random

from fixtures import TempDir
from maastest import (
    proxyfixture,
    utils,
    )
from maastest.proxyfixture import LocalProxyFixture
import mock
import netifaces
import testtools
from testtools.matchers import DirExists


class TestLocalProxyFixture(testtools.TestCase):

    def setUp(self):
        super(TestLocalProxyFixture, self).setUp()
        mock_ifaddresses = mock.MagicMock()
        mock_ifaddresses.return_value = {
            netifaces.AF_INET: [{
                'addr': '192.168.0.1',
                'netmask': '255.255.255.0',
            }]}
        self.patch(netifaces, 'ifaddresses', mock_ifaddresses)

    def make_dir(self):
        """Create a temporary directory for the duration of this test."""
        return self.useFixture(TempDir()).path

    def make_fixture(self, port=None, config_dir=None, pidfile_dir=None,
                     log_dir=None):
        """Create a `LocalProxyFixture`.

        :param port: Optional port.  Defaults to a random choice.
        :param config_dir: Optional configuration directory.  Defaults to a
            temporary directory.
        :param pidfile_dir: Optional pidfile directory.  Defaults to a
            temporary directory.
        :param log_dir: Optional log directory.  Defaults to a temporary
            directory.
        """
        if port is None:
            port = self.get_random_port_number()
        if config_dir is None:
            config_dir = self.make_dir()
        if pidfile_dir is None:
            pidfile_dir = self.make_dir()
        if log_dir is None:
            log_dir = self.make_dir()
        return LocalProxyFixture(
            port=port, config_dir=config_dir, pidfile_dir=pidfile_dir,
            log_dir=log_dir)

    def get_random_port_number(self):
        return random.randint(1, 65535)

    def test_init_sets_properties(self):
        port = self.get_random_port_number()
        config_dir = self.make_dir()
        pidfile_dir = self.make_dir()
        proxy_fixture = LocalProxyFixture(
            port=port, config_dir=config_dir, pidfile_dir=pidfile_dir)
        self.assertEqual(port, proxy_fixture.port)
        self.assertEqual(config_dir, proxy_fixture.config_dir)
        self.assertEqual(
            os.path.join(pidfile_dir, 'proxy.pid'),
            proxy_fixture.pid_file)

    def test_creates_pidfile_dir(self):
        pidfile_dir = os.path.join(self.make_dir(), "maas-test-pidfiles")
        with self.make_fixture(pidfile_dir=pidfile_dir):
            self.assertThat(pidfile_dir, DirExists())

    def test_creates_log_dir(self):
        log_dir = os.path.join(self.make_dir(), "maas-test-logs")
        with self.make_fixture(log_dir=log_dir):
            self.assertThat(log_dir, DirExists())

    def test_get_url_returns_proxy_url(self):
        port = self.get_random_port_number()
        proxy_fixture = self.make_fixture(port=port)
        self.assertEqual(
            "http://192.168.0.1:%s" % port,
            proxy_fixture.get_url())

    def test_set_up_calls_all_setup_methods(self):
        self.patch(LocalProxyFixture, 'create_cache_root', mock.MagicMock())
        self.patch(LocalProxyFixture, 'start_proxy', mock.MagicMock())

        with self.make_fixture() as fixture:
            pass

        self.assertEqual(
            [[mock.call()], [mock.call()]],
            [
                fixture.create_cache_root.mock_calls,
                fixture.start_proxy.mock_calls,
            ])

    def test_start_proxy_starts_proxy(self):
        mock_run_command = mock.MagicMock()
        self.patch(utils, 'run_command', mock_run_command)
        proxy_fixture = self.make_fixture()

        proxy_fixture.start_proxy()

        self.assertEqual(
            mock.call(
                ['polipo'] + proxy_fixture._get_config_arguments(),
                check_call=True),
            mock_run_command.mock_calls[-1])

    def test_create_cache_root_creates_directory(self):
        config_dir = self.make_dir()
        proxy_fixture = self.make_fixture(config_dir=config_dir)

        proxy_fixture.create_cache_root()
        expected_cache_path = proxyfixture.DISK_CACHE_ROOT % {
            'config_dir': config_dir}
        self.assertThat(expected_cache_path, DirExists())

    def test_create_cache_root_ignores_existing_directory(self):
        disk_cache_root = self.make_dir()
        self.patch(proxyfixture, 'DISK_CACHE_ROOT', disk_cache_root)
        proxy_fixture = self.make_fixture()

        proxy_fixture.create_cache_root()

        self.assertThat(disk_cache_root, DirExists())

    def test_get_network_address(self):
        proxy_adddress = self.make_fixture()._get_network_address()
        self.assertEqual('192.168.0.1', proxy_adddress.ip.format())
        self.assertEqual(24, proxy_adddress.prefixlen)

    def test_kill_running_proxy_kills_proxy(self):
        mock_run_command = mock.MagicMock()
        self.patch(utils, 'run_command', mock_run_command)
        mock_read_file = mock.MagicMock()
        mock_read_file.return_value = bytes(self.getUniqueInteger())
        self.patch(utils, 'read_file', mock_read_file)
        self.make_fixture().kill_running_proxy()
        self.assertEqual(
            mock.call(['kill', mock_read_file.return_value], check_call=True),
            mock_run_command.mock_calls[0])

    def test_kill_running_proxy_handles_nonexistent_pid_file_gracefully(self):
        mock_log_error = mock.MagicMock()
        self.patch(logging, 'error', mock_log_error)
        mock_read_file = mock.MagicMock()
        mock_read_file.side_effect = Exception(self.getUniqueString())
        self.patch(utils, 'read_file', mock_read_file)
        self.make_fixture().kill_running_proxy()
        mock_log_error.assert_has_calls([
            mock.call("Unable to kill proxy:"),
            mock.call(mock_read_file.side_effect.message)])

    def test_kill_running_proxy_handles_errors_from_kill_gracefully(self):
        mock_log_error = mock.MagicMock()
        self.patch(logging, 'error', mock_log_error)
        mock_run_command = mock.MagicMock()
        mock_run_command.side_effect = Exception(self.getUniqueString())
        self.patch(utils, 'run_command', mock_run_command)
        mock_read_file = mock.MagicMock()
        mock_read_file.return_value = bytes(self.getUniqueInteger())
        self.patch(utils, 'read_file', mock_read_file)
        self.make_fixture().kill_running_proxy()
        mock_log_error.assert_has_calls([
            mock.call("Unable to kill proxy:"),
            mock.call(mock_run_command.side_effect.message)])

    def test_kill_running_proxy_called_on_cleanup(self):
        mock_run_command = mock.MagicMock()
        self.patch(utils, 'run_command', mock_run_command)
        mock_read_file = mock.MagicMock()
        mock_read_file.return_value = bytes(self.getUniqueInteger())
        self.patch(utils, 'read_file', mock_read_file)
        mock_kill_running_proxy = mock.MagicMock()

        proxy_fixture = self.make_fixture()
        self.patch(
            proxy_fixture, 'kill_running_proxy', mock_kill_running_proxy)

        proxy_fixture.setUp()
        proxy_fixture.cleanUp()
        mock_kill_running_proxy.assert_has_call([mock.call()])

    def test_setUp_checks_for_pidfile(self):
        mock_exists = mock.MagicMock()
        mock_exists.return_value = True
        self.patch(os.path, 'exists', mock_exists)
        proxy_fixture = self.make_fixture()
        self.assertRaises(Exception, proxy_fixture.setUp)
