/* nexp_tgn.c
   
   The "tgn" (Traffic Generator) command is implemented here. Used as in:

   tgn create ip(dst = 10.0.0.1)/tcp(syn, src = random, dst = 80)
   tgn info
   tgn start
   tgn stop
   tgn reset
   tgn destroy

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"
#include <fcntl.h>
#include "nexp_speakers.h"
#include "util-tcl.h"

#define TGN_ID_VARNAME "tgn_id"

#define NEXP_TGN_NAMELEN (16 + TCL_INTEGER_SPACE)

/*
 * Commands that can be sent to TGN threads.
 */
enum tgn_commands {
    TGN_CMDSTART,   /* Start a TGN */
    TGN_CMDSTOP,    /* Stop a TGN */
    TGN_CMDSTATUS,  /* Get status of a TGN */
    TGN_CMDRESET,   /* Reset TGN counters */
    TGN_CMDEXIT	    /* Tell a TGN to exit */
};

/*
 * Status of a TGN.
 */
enum tgn_status {
    TGN_STOPPED,
    TGN_STARTED
};

struct nexp_tgn {
    struct nexp_tgn *next;

    char name[NEXP_TGN_NAMELEN + 1];

    /*
     * Pipes for inter-thread communications.
     */
    int to_tgn_pipefds[2];
    int from_tgn_pipefds[2];

    pcb_t pcb;
    int count;
    int start_right_away;
    int loop;

    enum tgn_status status;
};

/*
 * Linked list of all active traffic generators.
 */
static struct nexp_tgn *tgns;

/*
 * Returns a pointer to the tgn structure that corresponds to the
 * TGN ID passed as a parameter.
 *
 * Returns NULL if no TGN with the specified ID exists.
 */
static struct nexp_tgn *
lookup_tgn(const char *tgn_id)
{
    struct nexp_tgn *t;

    for (t = tgns; t; t = t->next)
	if (!strcmp(t->name, tgn_id) )
	    break;

    return t;
}

static int
tgn_destroy(Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[])
{
    struct nexp_tgn *t, *previous_t;
    const char *tgn_id;
    int cmd, sent_packets;
    Tcl_Obj *obj;

    if (objc == 2) {
	/* Explicit TGN ID */
	t = lookup_tgn(Tcl_GetString(objv[1]) );
	if (t == NULL) {
	    nexp_error(interp, "Traffic Generator \"%s\" does not exist.",
		       Tcl_GetString(objv[1]) );
	    return TCL_ERROR;
	}
    } else {
	/* No explicit TGN ID; use recently created TGN */
	tgn_id = nexp_get_var(interp, TGN_ID_VARNAME);
	if (!tgn_id) {
	    nexp_error(interp, "Use tgn command to create a TGN first.");
	    return TCL_ERROR;
	}

	t = lookup_tgn(tgn_id);
	if (t == NULL) {
	    nexp_error(interp, "Traffic Generator \"%s\" does not exist.",
		       tgn_id);
	    return TCL_ERROR;
	}
    }

    /*
     * Tell the TGN thread that it's time to exit.
     */
    cmd = TGN_CMDEXIT;
    write(t->to_tgn_pipefds[1], &cmd, sizeof(cmd) );

    /*
     * Wait for confirmation that the thread is exiting. The
     * exiting thread sends via the pipe the number of packets
     * that were sent. This will be the return value of the
     * Tcl command.
     */
    read(t->from_tgn_pipefds[0], &sent_packets, sizeof(sent_packets) );

    pcb_destroy(&t->pcb);

    close(t->to_tgn_pipefds[0]); close(t->to_tgn_pipefds[1]);
    close(t->from_tgn_pipefds[0]); close(t->from_tgn_pipefds[1]);

    /* Remove TGN from linked list of TGNs. */
    if (tgns == t)
	/* Element to remove is at the beginning of the list. */
	tgns = t->next;
    else {
	/* Find linked list item previous to one we want to remove. */
	for (previous_t = tgns;
	     previous_t->next != t;
	     previous_t = previous_t->next);

	previous_t->next = t->next;
    }

    free(t);

    obj = Tcl_NewIntObj(sent_packets);
    Tcl_SetObjResult(interp, obj);

    return TCL_OK;
}

/*
 * The thread that actually does all the work related to the "tgn" command.
 */
static void
tgn_main(ClientData clientData)
{
    int retval, cmd;
    struct nexp_tgn *tgn;
    uint8_t *packet;
    int i;
    int packet_count, packets_sent = 0;
    size_t packet_size;
    fd_set readfds;
    struct timeval timeout;

    tgn = clientData;

    /*
     * We can't block while reading commands sent by the main thread.
     */
    fcntl(tgn->to_tgn_pipefds[0], F_SETFL, O_NONBLOCK);

    tgn->status = tgn->start_right_away ? TGN_STARTED : TGN_STOPPED;

    packet_count = tgn->count ? tgn->count : pb_permutation_count(tgn->pcb.pdu);
    packet = xmalloc(pb_len(tgn->pcb.pdu) );

    /*
     * The thread's main loop.
     */
    for (;;) {
	if (tgn->status == TGN_STOPPED) {
	    FD_ZERO(&readfds);
	    FD_SET(tgn->to_tgn_pipefds[0], &readfds);

	    retval = select(tgn->to_tgn_pipefds[0] + 1, &readfds, NULL,
			    NULL, NULL /* no timeout */);
	    if (retval == -1)
		/* Error; just ignore and try again. */
		continue;

	    /* Read a command sent by the main thread. */
	    read(tgn->to_tgn_pipefds[0], &cmd, sizeof(cmd) );

process_cmd:
	    if (cmd == TGN_CMDEXIT) {
		/* We're getting told to exit. */
		break;
	    } else if (cmd == TGN_CMDSTART) {
		tgn->status = TGN_STARTED;
	    } else if (cmd == TGN_CMDSTOP) {
		tgn->status = TGN_STOPPED;
	    } else if (cmd == TGN_CMDSTATUS) {
		write(tgn->from_tgn_pipefds[1], &packets_sent,
		      sizeof(packets_sent) );
	    } else if (cmd == TGN_CMDRESET) {
		packets_sent = 0;
	    }

	    continue;
	}

	for (i = 0; i < packet_count || tgn->loop; packets_sent++, i++) {
	    packet_size = pb_build(tgn->pcb.pdu, packet, NULL);

	    /*
	     * Handle inter-packet sending delay.
	     */
	    if (timerisset(&tgn->pcb.delay)
		&& timerisset(&tgn->pcb.speaker->ts) ) {

		FD_ZERO(&readfds);
		FD_SET(tgn->to_tgn_pipefds[0], &readfds);
		timeout = tgn->pcb.delay;

		retval = select(tgn->to_tgn_pipefds[0] + 1, &readfds, NULL,
				NULL, &timeout);
		if (retval == -1)
		    /* Just ignore the error */
		    continue;
		else if (retval == 0)
		    /*
		     * select() timed out, which means we've waiting long
		     * enough, so send the packet.
		     */
		    nexp_pdu_output(&tgn->pcb, packet, packet_size);
		else {
		    /* Main thread sent us a command; read it */
		    read(tgn->to_tgn_pipefds[0], &cmd, sizeof(cmd) );

		    /*
		     * Handle here commands that do not need to interrupt
		     * traffic generation.
		     */
		    if (cmd == TGN_CMDSTATUS)
			write(tgn->from_tgn_pipefds[1], &packets_sent,
			      sizeof(packets_sent) );
		    else if (cmd == TGN_CMDRESET)
			packets_sent = 0;
		    else if (cmd != TGN_CMDSTART)
			goto process_cmd;

		    /* Send packet even if it's not time to send yet */
		    nexp_pdu_output(&tgn->pcb, packet, packet_size);
		}
	    } else {
		retval = read(tgn->to_tgn_pipefds[0], &cmd, sizeof(cmd) );
		if (retval == sizeof(cmd) ) {
		    /*
		     * A command was sent to us.  Handle here commands that
		     * would interrupt traffic generation.
		     */
		    if (cmd == TGN_CMDSTATUS)
			write(tgn->from_tgn_pipefds[1], &packets_sent,
			      sizeof(packets_sent) );
		    else if (cmd == TGN_CMDRESET)
			packets_sent = 0;
		    else if (cmd != TGN_CMDSTART)
			goto process_cmd;
		}

		nexp_pdu_output(&tgn->pcb, packet, packet_size);
	    }
	}

	/* All packets have been sent; now auto-stop */
	tgn->status = TGN_STOPPED;
    } /* TGN main loop */

    free(packet);

    write(tgn->from_tgn_pipefds[1], &packets_sent, sizeof(packets_sent) );
}

static void
usage(void)
{
    fprintf(stderr, "\
usage: tgn [-o <speaker ID>] [-count <count>]\n\
          [-delay <usecs>] [-rate <pps>] <PDU definition>");
}

static int
process_options(Tcl_Interp *interp, int argc, Tcl_Obj * const *objv,
		struct nexp_tgn *tgn)
{
    int i, index;
    char *pdudef = NULL;
    double delay;
    char errbuf[PDU_ERRBUF_SIZE];
    static const char *options[] = {
	"-o", "-count", "-delay", "-rate", "-start", "-loop", NULL
    };
    enum options {
	OPT_SPEAKER, OPT_COUNT, OPT_DELAY, OPT_RATE, OPT_START, OPT_LOOP
    };

    /*
     * Parse command-line arguments.
     */
    for (i = 1; i < argc && *Tcl_GetString(objv[i]) == '-'; i++) {
	if (Tcl_GetIndexFromObj(interp, objv[i], options, "option", 0, &index)
	    != TCL_OK)
	    return -1;

	switch ( (enum options) index) {
	case OPT_SPEAKER:
	    if (++i >= argc) {
		Tcl_WrongNumArgs(interp, 1, objv, "-o speaker");
		goto error;
	    }

	    tgn->pcb.speaker = lookup_speaker(Tcl_GetString(objv[i]) );
	    if (!tgn->pcb.speaker) {
		nexp_error(interp, "No speaker named \"%s\". Use "
			   "\"spawn_network -info\" to find out existing "
			   "speakers.", Tcl_GetString(objv[i]) );
		goto error;
	    }
	    break;
	case OPT_COUNT:
	    if (++i >= argc) {
		Tcl_WrongNumArgs(interp, 1, objv, "-c npackets");
		goto error;
	    }

	    if (Tcl_GetIntFromObj(interp, objv[i], &tgn->count) != TCL_OK)
		goto error;
	    break;
	case OPT_DELAY:
	    if (++i >= argc) {
		Tcl_WrongNumArgs(interp, 1, objv, "-d seconds");
		goto error;
	    }

	    if (Tcl_GetDoubleFromObj(interp, objv[i], &delay) != TCL_OK)
		goto error;

	    tgn->pcb.delay.tv_sec = delay;
	    tgn->pcb.delay.tv_usec = (delay - tgn->pcb.delay.tv_sec)
				      *1000000UL;
	    break;
	case OPT_RATE:
	    if (++i >= argc) {
		Tcl_WrongNumArgs(interp, 1, objv, "-r PPS");
		goto error;
	    }

	    /* Convert a packets-per-second rate to a usecs delay */
	    if (Tcl_GetDoubleFromObj(interp, objv[i], &delay) != TCL_OK)
		goto error;

	    if (delay == 0.0) {
		nexp_error(interp, "Rate can't be 0 packets per second.");
		goto error;
	    }

	    delay = 1/delay;

	    tgn->pcb.delay.tv_sec = delay;
	    tgn->pcb.delay.tv_usec = (delay - tgn->pcb.delay.tv_sec)
				      *1000000UL;
	    break;
	case OPT_START:
	    tgn->start_right_away = 1;
	    break;
	case OPT_LOOP:
	    tgn->loop = 1;
	    break;
	}
    }

    /*
     * We treat whatever is left on the command line, i.e. anything that
     * is not an option (anything that doesn't start with '-'), as a PDU
     * definition.
     */
    pdudef = copy_objv(argc - i, &objv[i]);
    if (!pdudef) {
	usage();
	return -1;
    }

#ifdef DEBUG
    printf("PDU definition = %s\n", pdudef);
#endif

    if ( (tgn->pcb.pdu = pb_parsedef(pdudef, errbuf) ) == NULL) {
	nexp_error(interp, "%s", errbuf);
	goto error;
    }

    tgn->pcb.def = pdudef;

    return 0;

error:
    if (pdudef)
	free(pdudef);

    return -1;
}

static int
tgn_create(Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[])
{
    int retval;
    Tcl_ThreadId thread_id;
    struct nexp_tgn *t, *new_tgn;
    static int tgn_id;

    /*
     * A pointer to a nexp_tgn structure needs to be passed to a TGN
     * thread. However, since threads have different stacks we can't
     * use a local variable. To avoid using a global variable we allocate
     * memory for this structure in the heap.
     */
    new_tgn = xmalloc(sizeof(*new_tgn) );
    memset(new_tgn, 0, sizeof(*new_tgn) );
    snprintf(new_tgn->name, sizeof(new_tgn->name), "tgn%d", tgn_id++);

    retval = process_options(interp, objc, objv, new_tgn);
    if (retval == -1)
	goto error;

    /*
     * Make sure the PCB has an assigned speaker: if the user has not
     * explicitely specified a speaker, then we use the default speaker,
     * which is referred to by name via the Tcl variable "speaker_id"
     * (SPEAKER_SPAWN_ID_VARNAME).
     */
    if (!new_tgn->pcb.speaker) {
	new_tgn->pcb.speaker = lookup_speaker(nexp_get_var(interp,
					      SPEAKER_SPAWN_ID_VARNAME) );
	if (!new_tgn->pcb.speaker) {
	    nexp_error(interp, "Can't find a suitable speaker! Use "
			       "spawn_network to create one.");
	    goto error;
	}
    }

    /*
     * Create pipes for communications between main thread and the
     * tgn thread.
     */
    if (pipe(new_tgn->to_tgn_pipefds) == -1) {
	nexp_error(interp, "Couldn't open pipe: %s", strerror(errno) );
	goto error;
    }

    if (pipe(new_tgn->from_tgn_pipefds) == -1) {
	nexp_error(interp, "Couldn't open pipe: %s", strerror(errno) );
	goto error;
    }

    /* Launch the TGN */
    retval = Tcl_CreateThread(&thread_id, tgn_main, new_tgn,
			      TCL_THREAD_STACK_DEFAULT,
			      TCL_THREAD_NOFLAGS);
    if (retval != TCL_OK) {
	nexp_error(interp, "Tcl_CreateThread() error");
	goto error;
    }

    /* Add new TGN to linked list of TGNs. */
    if (!tgns)
	tgns = new_tgn;
    else {
	for (t = tgns; t->next; t = t->next);

	t->next = new_tgn;
    }

    /*
    * Tell user of new tgn id.
    */
    Tcl_SetVar(interp, TGN_ID_VARNAME, new_tgn->name, 0);

    return TCL_OK;

error:
#if 0
    if (new_tgn)
	tgn_destroy(new_tgn->name);
#endif
    return TCL_ERROR;
}

void
tgn_info(void)
{
    struct nexp_tgn *t;
    int i, ntgns;
    int cmd, packets_sent;

    /* Count number of traffic generators*/
    for (t = tgns, ntgns = 0; t; t = t->next, ntgns++);

    printf("Traffic Generators:\n");

    printf("  Number of traffic generators: %d\n", ntgns);

    for (t = tgns, i = 0; t; t = t->next, i++) {
	cmd = TGN_CMDSTATUS;
	write(t->to_tgn_pipefds[1], &cmd, sizeof(cmd) );

	/* This will block until the TGN thread writes back to us */
	read(t->from_tgn_pipefds[0], &packets_sent, sizeof(packets_sent) );

	printf("  TGN #%d:\n", i);
	printf("    Name: %s\n", t->name);
	printf("    PDU definition: %s\n", t->pcb.def);
	printf("    Status: %s\n",
	       t->status == TGN_STARTED ? "sending packets" : "stopped");
	printf("    Packets sent: %d\n", packets_sent);
    }
}

/*
 * The "tgn" command; a traffic generator.
 *
 * Example:
 *
 * tgn create ip(dst = 10.0.0.1)/tcp(syn, src = random, dst = 80)
 * tgn info
 * tgn start
 * tgn stop
 * tgn reset
 * tgn destroy
 */
static int
NExp_TGNObjCmd(ClientData clientData _U_, Tcl_Interp *interp, int objc,
	       Tcl_Obj *CONST objv[])
{
    int index;
    static const char *subcmds[] = {
	"create", "destroy", "info", "start", "stop", "reset", NULL
    };
    enum subcmds {
	SUBCMD_CREATE, SUBCMD_DESTROY, SUBCMD_INFO, SUBCMD_START,
	SUBCMD_STOP, SUBCMD_RESET
    };
    struct nexp_tgn *tgn;
    const char *tgn_id;
    int cmd;

    if (Tcl_GetIndexFromObj(interp, objv[1], subcmds, "subcmd", 0, &index)
	!= TCL_OK)
	return TCL_ERROR;

    switch ( (enum subcmds) index) {
    case SUBCMD_CREATE:
	return tgn_create(interp, objc - 1, &objv[1]);
    case SUBCMD_DESTROY:
	return tgn_destroy(interp, objc - 1, &objv[1]);
    case SUBCMD_INFO:
	tgn_info();
	break;
    case SUBCMD_START:
    case SUBCMD_STOP:
    case SUBCMD_RESET:
	if (objc == 3) {
	    /* Explicit TGN ID */
	    tgn = lookup_tgn(Tcl_GetString(objv[2]) );
	    if (tgn == NULL) {
		nexp_error(interp, "Traffic Generator \"%s\" does not exist.",
			   Tcl_GetString(objv[2]) );
		return TCL_ERROR;
	    }
	} else {
	    /* No explicit TGN ID; use recently created TGN */
	    tgn_id = nexp_get_var(interp, TGN_ID_VARNAME);
	    if (!tgn_id) {
		nexp_error(interp, "Use tgn command to create a TGN first.");
		return TCL_ERROR;
	    }

	    tgn = lookup_tgn(tgn_id);
	    if (tgn == NULL) {
		nexp_error(interp, "Traffic Generator \"%s\" does not exist.",
			   tgn_id);
		return TCL_ERROR;
	    }
	}

	if (index == SUBCMD_START)
	    cmd = TGN_CMDSTART;
	else if (index == SUBCMD_STOP)
	    cmd = TGN_CMDSTOP;
	else
	    cmd = TGN_CMDRESET;

	write(tgn->to_tgn_pipefds[1], &cmd, sizeof(cmd) );
	break;
    }

    return TCL_OK;
}

static struct nexp_cmd_data cmd_data[] = {
    {"tgn", NExp_TGNObjCmd, NULL, 0, 0},

    {NULL, NULL, NULL, 0, 0}
};

void
nexp_init_tgn_cmd(Tcl_Interp *interp)
{
    nexp_create_commands(interp, cmd_data);
}
