/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var isFunction = require( '@stdlib/assert/is-function' );
var hasOwnProp = require( '@stdlib/assert/has-own-property' );
var quantile = require( './../../../../../base/dists/laplace/quantile' );
var logcdf = require( './../../../../../base/dists/laplace/logcdf' );
var logpdf = require( './../../../../../base/dists/laplace/logpdf' );
var cdf = require( './../../../../../base/dists/laplace/cdf' );
var mgf = require( './../../../../../base/dists/laplace/mgf' );
var pdf = require( './../../../../../base/dists/laplace/pdf' );
var kurtosis = require( './../../../../../base/dists/laplace/kurtosis' );
var skewness = require( './../../../../../base/dists/laplace/skewness' );
var stdev = require( './../../../../../base/dists/laplace/stdev' );
var variance = require( './../../../../../base/dists/laplace/variance' );
var entropy = require( './../../../../../base/dists/laplace/entropy' );
var median = require( './../../../../../base/dists/laplace/median' );
var mode = require( './../../../../../base/dists/laplace/mode' );
var mean = require( './../../../../../base/dists/laplace/mean' );
var Laplace = require( './../lib' );


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.strictEqual( typeof Laplace, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function throws an error if provided a `mu` argument which is not a number primitive', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		NaN,
		true,
		false,
		void 0,
		null,
		{},
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			// eslint-disable-next-line no-new
			new Laplace( value, 1.0 );
		};
	}
});

tape( 'the function throws an error if provided a `b` argument which is not a positive number', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		-5.0,
		0.0,
		NaN,
		true,
		false,
		void 0,
		null,
		{},
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			// eslint-disable-next-line no-new
			new Laplace( 2.0, value );
		};
	}
});

tape( 'if provided arguments, the function requires both `mu` and `b`', function test( t ) {
	t.throws( foo, TypeError, 'throws an error' );
	t.end();

	function foo() {
		// eslint-disable-next-line no-new
		new Laplace( 2.0 );
	}
});

tape( 'the function returns a new distribution instance (default parameters)', function test( t ) {
	var laplace = new Laplace();
	t.strictEqual( laplace instanceof Laplace, true, 'returns an instance' );
	t.end();
});

tape( 'the function returns a new distribution instance (custom parameters)', function test( t ) {
	var laplace = new Laplace( 2.0, 4.0 );
	t.strictEqual( laplace instanceof Laplace, true, 'returns an instance' );
	t.end();
});

tape( 'the function can be invoked without the new operator', function test( t ) {
	// eslint-disable-next-line new-cap
	var laplace = Laplace();
	t.strictEqual( laplace instanceof Laplace, true, 'returns an instance' );

	// eslint-disable-next-line new-cap
	laplace = Laplace( 2.0, 4.0 );
	t.strictEqual( laplace instanceof Laplace, true, 'returns an instance' );

	t.end();
});

tape( 'the created distribution has a property for getting and setting `mu`', function test( t ) {
	var laplace;

	laplace = new Laplace( 2.0, 4.0 );
	t.strictEqual( hasOwnProp( laplace, 'mu' ), true, 'has property' );
	t.strictEqual( laplace.mu, 2.0, 'returns expected value' );

	laplace.mu = 3.0;
	t.strictEqual( laplace.mu, 3.0, 'returns expected value' );

	t.end();
});

tape( 'the created distribution throws an error if one attempts to set `mu` to a value which is not a positive number', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		NaN,
		true,
		false,
		void 0,
		null,
		{},
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			var laplace = new Laplace();
			laplace.mu = value;
		};
	}
});

tape( 'the created distribution has a property for getting and setting `b`', function test( t ) {
	var laplace;

	laplace = new Laplace( 2.0, 4.0 );
	t.strictEqual( hasOwnProp( laplace, 'b' ), true, 'has property' );
	t.strictEqual( laplace.b, 4.0, 'returns expected value' );

	laplace.b = 3.0;
	t.strictEqual( laplace.b, 3.0, 'returns expected value' );

	t.end();
});

tape( 'the created distribution throws an error if one attempts to set `b` to a value which is not a positive number', function test( t ) {
	var values;
	var i;

	values = [
		'5',
		-5.0,
		0.0,
		NaN,
		true,
		false,
		void 0,
		null,
		{},
		[],
		function noop() {}
	];

	for ( i = 0; i < values.length; i++ ) {
		t.throws( badValue( values[i] ), TypeError, 'throws an error when provided '+values[i] );
	}
	t.end();

	function badValue( value ) {
		return function badValue() {
			var laplace = new Laplace();
			laplace.b = value;
		};
	}
});

tape( 'the distribution prototype has a property for retrieving the distribution entropy', function test( t ) {
	var laplace;

	t.strictEqual( hasOwnProp( Laplace.prototype, 'entropy' ), true, 'has property' );

	laplace = new Laplace();
	t.strictEqual( laplace.entropy, entropy( 0.0, 1.0 ), 'returns expected value' );

	t.end();
});

tape( 'the distribution prototype has a property for retrieving the distribution kurtosis', function test( t ) {
	var laplace;

	t.strictEqual( hasOwnProp( Laplace.prototype, 'kurtosis' ), true, 'has property' );

	laplace = new Laplace( 2.0, 4.0 );
	t.strictEqual( laplace.kurtosis, kurtosis( 2.0, 4.0 ), 'returns expected value' );

	t.end();
});

tape( 'the distribution prototype has a property for retrieving the distribution mean', function test( t ) {
	var laplace;

	t.strictEqual( hasOwnProp( Laplace.prototype, 'mean' ), true, 'has property' );

	laplace = new Laplace();
	t.strictEqual( laplace.mean, mean( 0.0, 1.0 ), 'returns expected value' );

	t.end();
});

tape( 'the distribution prototype has a property for retrieving the distribution median', function test( t ) {
	var laplace;

	t.strictEqual( hasOwnProp( Laplace.prototype, 'median' ), true, 'has property' );

	laplace = new Laplace( 5.0, 2.0 );
	t.strictEqual( laplace.median, median( 5.0, 2.0 ), 'returns expected value' );

	t.end();
});

tape( 'the distribution prototype has a property for retrieving the distribution mode', function test( t ) {
	var laplace;

	t.strictEqual( hasOwnProp( Laplace.prototype, 'mode' ), true, 'has property' );

	laplace = new Laplace( 2.0, 3.0 );
	t.strictEqual( laplace.mode, mode( 2.0, 3.0 ), 'returns expected value' );

	t.end();
});

tape( 'the distribution prototype has a property for retrieving the distribution skewness', function test( t ) {
	var laplace;

	t.strictEqual( hasOwnProp( Laplace.prototype, 'skewness' ), true, 'has property' );

	laplace = new Laplace( 0.5, 0.5 );
	t.strictEqual( laplace.skewness, skewness( 0.5, 0.5 ), 'returns expected value' );

	t.end();
});

tape( 'the distribution prototype has a property for retrieving the distribution standard deviation', function test( t ) {
	var laplace;

	t.strictEqual( hasOwnProp( Laplace.prototype, 'stdev' ), true, 'has property' );

	laplace = new Laplace( 3.0, 1.0 );
	t.strictEqual( laplace.stdev, stdev( 3.0, 1.0 ), 'returns expected value' );

	t.end();
});

tape( 'the distribution prototype has a property for retrieving the distribution variance', function test( t ) {
	var laplace;

	t.strictEqual( hasOwnProp( Laplace.prototype, 'variance' ), true, 'has property' );

	laplace = new Laplace( 3.0, 1.0 );
	t.strictEqual( laplace.variance, variance( 3.0, 1.0 ), 'returns expected value' );

	t.end();
});

tape( 'the distribution prototype has a method for evaluating the cumulative distribution function (CDF)', function test( t ) {
	var laplace;
	var y;

	t.strictEqual( hasOwnProp( Laplace.prototype, 'cdf' ), true, 'has property' );
	t.strictEqual( isFunction( Laplace.prototype.cdf ), true, 'has method' );

	laplace = new Laplace();
	y = laplace.cdf( 0.5 );

	t.strictEqual( y, cdf( 0.5, 0.0, 1.0 ), 'returns expected value' );
	t.end();
});

tape( 'the distribution prototype has a method for evaluating the natural logarithm of the cumulative distribution function (logCDF)', function test( t ) {
	var laplace;
	var y;

	t.strictEqual( hasOwnProp( Laplace.prototype, 'logcdf' ), true, 'has property' );
	t.strictEqual( isFunction( Laplace.prototype.logcdf ), true, 'has method' );

	laplace = new Laplace();
	y = laplace.logcdf( 0.5 );

	t.strictEqual( y, logcdf( 0.5, 0.0, 1.0 ), 'returns expected value' );
	t.end();
});

tape( 'the distribution prototype has a method for evaluating the natural logarithm of the probability density function (logPDF)', function test( t ) {
	var laplace;
	var y;

	t.strictEqual( hasOwnProp( Laplace.prototype, 'logpdf' ), true, 'has property' );
	t.strictEqual( isFunction( Laplace.prototype.logpdf ), true, 'has method' );

	laplace = new Laplace();
	y = laplace.logpdf( 0.5 );

	t.strictEqual( y, logpdf( 0.5, 0.0, 1.0 ), 'returns expected value' );
	t.end();
});

tape( 'the distribution prototype has a method for evaluating the moment-generating function (MGF)', function test( t ) {
	var laplace;
	var y;

	t.strictEqual( hasOwnProp( Laplace.prototype, 'mgf' ), true, 'has property' );
	t.strictEqual( isFunction( Laplace.prototype.mgf ), true, 'has method' );

	laplace = new Laplace();
	y = laplace.mgf( 0.5 );

	t.strictEqual( y, mgf( 0.5, 0.0, 1.0 ), 'returns expected value' );
	t.end();
});

tape( 'the distribution prototype has a method for evaluating the probability density function (PDF)', function test( t ) {
	var laplace;
	var y;

	t.strictEqual( hasOwnProp( Laplace.prototype, 'pdf' ), true, 'has property' );
	t.strictEqual( isFunction( Laplace.prototype.pdf ), true, 'has method' );

	laplace = new Laplace();
	y = laplace.pdf( 0.2 );

	t.strictEqual( y, pdf( 0.2, 0.0, 1.0 ), 'returns expected value' );
	t.end();
});

tape( 'the distribution prototype has a method for evaluating the quantile function', function test( t ) {
	var laplace;
	var y;

	t.strictEqual( hasOwnProp( Laplace.prototype, 'quantile' ), true, 'has property' );
	t.strictEqual( isFunction( Laplace.prototype.quantile ), true, 'has method' );

	laplace = new Laplace();
	y = laplace.quantile( 0.8 );

	t.strictEqual( y, quantile( 0.8, 0.0, 1.0 ), 'returns expected value' );
	t.end();
});
