/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "OpenCTL/compiler/LexerNG.h"
#include "GTLCore/Token_p.h"

#define TEST_FOR_TOKEN( toktype ) \
  { \
    GTLCore::Token tok = lng.nextToken(); \
    GTLTEST_CHECK_EQUAL(tok.type, GTLCore::Token::toktype ); \
  }

#define TEST_FOR_TOKEN_AND_POS( toktype, _line_, _column_ ) \
  { \
    GTLCore::Token tok = lng.nextToken(); \
    GTLTEST_CHECK_EQUAL(tok.type, GTLCore::Token::toktype ); \
    GTLTEST_CHECK_EQUAL(tok.line, _line_ ); \
    GTLTEST_CHECK_EQUAL(tok.column, _column_ ); \
  }

#define TEST_FOR_TOKEN_AND_VALUE( toktype, field, value ) \
  { \
    GTLCore::Token tok = lng.nextToken(); \
    GTLTEST_CHECK_EQUAL(tok.type, GTLCore::Token::toktype ); \
    GTLTEST_CHECK_NEAR_EQUAL(tok.field, value); \
  }

#define TEST_FOR_TOKEN_AND_VALUE_STRICT( toktype, field, value ) \
  { \
    GTLCore::Token tok = lng.nextToken(); \
    GTLTEST_CHECK_EQUAL(tok.type, GTLCore::Token::toktype ); \
    GTLTEST_CHECK_EQUAL(tok.field, value); \
  }

class TestLexerNGKeywords : public GTLTest::Case {
  public:
    TestLexerNGKeywords() : GTLTest::Case("Keywords") {}
    virtual void runTest()
    {
      std::istringstream iss(" bool const float else for if import int long return short signed size struct unsigned void while namespace ctlversion half input output print uniform varying true false");
      OpenCTL::LexerNG lng( &iss);
      TEST_FOR_TOKEN( BOOL );
      TEST_FOR_TOKEN( CONST );
      TEST_FOR_TOKEN( FLOAT );
      TEST_FOR_TOKEN( ELSE );
      TEST_FOR_TOKEN( FOR );
      TEST_FOR_TOKEN( IF );
      TEST_FOR_TOKEN( IMPORT );
      TEST_FOR_TOKEN( INT );
      TEST_FOR_TOKEN( LONG );
      TEST_FOR_TOKEN( RETURN );
      TEST_FOR_TOKEN( SHORT );
      TEST_FOR_TOKEN( SIGNED );
      TEST_FOR_TOKEN( SIZE );
      TEST_FOR_TOKEN( STRUCT );
      TEST_FOR_TOKEN( UNSIGNED );
      TEST_FOR_TOKEN( VOID );
      TEST_FOR_TOKEN( WHILE );
      TEST_FOR_TOKEN( NAMESPACE );
      TEST_FOR_TOKEN( CTLVERSION );
      TEST_FOR_TOKEN( HALF );
      TEST_FOR_TOKEN( INPUT );
      TEST_FOR_TOKEN( OUTPUT );
      TEST_FOR_TOKEN( PRINT );
      TEST_FOR_TOKEN( UNIFORM );
      TEST_FOR_TOKEN( VARYING );
      TEST_FOR_TOKEN( TTRUE );
      TEST_FOR_TOKEN( TFALSE );
    }
};

class TestLexerNGConstants : public GTLTest::Case {
  public:
    TestLexerNGConstants() : GTLTest::Case("Constants") {}
    virtual void runTest()
    {
      std::istringstream iss(" 10 10.0 10e-7 20 30. 42.0 \"hello\" 8 \"escape \\\"me\\\"\"");
      OpenCTL::LexerNG lng( &iss);
      TEST_FOR_TOKEN_AND_VALUE_STRICT( INTEGER_CONSTANT, i, 10);
      TEST_FOR_TOKEN_AND_VALUE( FLOAT_CONSTANT, f, 10.0);
      TEST_FOR_TOKEN_AND_VALUE( FLOAT_CONSTANT, f, 10e-7);
      TEST_FOR_TOKEN_AND_VALUE_STRICT( INTEGER_CONSTANT, i, 20);
      TEST_FOR_TOKEN_AND_VALUE( FLOAT_CONSTANT, f, 30.);
      TEST_FOR_TOKEN_AND_VALUE( FLOAT_CONSTANT, f, 42.0);
      TEST_FOR_TOKEN_AND_VALUE_STRICT( STRING_CONSTANT, string, "hello");
      TEST_FOR_TOKEN_AND_VALUE_STRICT( INTEGER_CONSTANT, i, 8);
      TEST_FOR_TOKEN_AND_VALUE_STRICT( STRING_CONSTANT, string, "escape \\\"me\\\"");
  }
};

class TestLexerNGSpecialCharacters : public GTLTest::Case {
  public:
    TestLexerNGSpecialCharacters() : GTLTest::Case("SpecialCharacters") {}
    virtual void runTest()
    {
      std::istringstream iss(" ; :: , . { } ( ) [ ] = == ! != & && || | ^ < <= << > >= >> + ++ - -- * / % ~");
      OpenCTL::LexerNG lng( &iss);
      TEST_FOR_TOKEN( SEMI );
      TEST_FOR_TOKEN( COLONCOLON );
      TEST_FOR_TOKEN( COMA );
      TEST_FOR_TOKEN( DOT );
      TEST_FOR_TOKEN( STARTBRACE );
      TEST_FOR_TOKEN( ENDBRACE );
      TEST_FOR_TOKEN( STARTBRACKET );
      TEST_FOR_TOKEN( ENDBRACKET );
      TEST_FOR_TOKEN( STARTBOXBRACKET );
      TEST_FOR_TOKEN( ENDBOXBRACKET );
      TEST_FOR_TOKEN( EQUAL );
      TEST_FOR_TOKEN( EQUALEQUAL );
      TEST_FOR_TOKEN( NOT );
      TEST_FOR_TOKEN( DIFFERENT );
      TEST_FOR_TOKEN( BITAND );
      TEST_FOR_TOKEN( AND );
      TEST_FOR_TOKEN( OR );
      TEST_FOR_TOKEN( BITOR );
      TEST_FOR_TOKEN( BITXOR );
      TEST_FOR_TOKEN( INFERIOR );
      TEST_FOR_TOKEN( INFERIOREQUAL );
      TEST_FOR_TOKEN( LEFTSHIFT );
      TEST_FOR_TOKEN( SUPPERIOR );
      TEST_FOR_TOKEN( SUPPERIOREQUAL );
      TEST_FOR_TOKEN( RIGHTSHIFT );
      TEST_FOR_TOKEN( PLUS );
      TEST_FOR_TOKEN( PLUSPLUS );
      TEST_FOR_TOKEN( MINUS );
      TEST_FOR_TOKEN( MINUSMINUS );
      TEST_FOR_TOKEN( MULTIPLY );
      TEST_FOR_TOKEN( DIVIDE );
      TEST_FOR_TOKEN( MODULO );
      TEST_FOR_TOKEN( TILDE );
    }
};

class TestLexerNGNoSpace : public GTLTest::Case {
  public:
    TestLexerNGNoSpace() : GTLTest::Case("NoSpace") {}
    virtual void runTest()
    {
      std::istringstream iss("32)=(<<<3<==");
      OpenCTL::LexerNG lng( &iss);
      TEST_FOR_TOKEN( INTEGER_CONSTANT );
      TEST_FOR_TOKEN( ENDBRACKET );
      TEST_FOR_TOKEN( EQUAL );
      TEST_FOR_TOKEN( STARTBRACKET );
      TEST_FOR_TOKEN( LEFTSHIFT );
      TEST_FOR_TOKEN( INFERIOR );
      TEST_FOR_TOKEN( INTEGER_CONSTANT );
      TEST_FOR_TOKEN( INFERIOREQUAL );
      TEST_FOR_TOKEN( EQUAL );
  }
};

class TestLexerNGIdentifier : public GTLTest::Case {
  public:
    TestLexerNGIdentifier() : GTLTest::Case("Identifier") {}
    virtual void runTest()
    {
      std::istringstream iss(" HOHOH BOUH23 GLOB_");
      OpenCTL::LexerNG lng(&iss);
      TEST_FOR_TOKEN_AND_VALUE_STRICT( IDENTIFIER, string, "HOHOH");
      TEST_FOR_TOKEN_AND_VALUE_STRICT( IDENTIFIER, string, "BOUH23");
      TEST_FOR_TOKEN_AND_VALUE_STRICT( IDENTIFIER, string, "GLOB_");
    }
};

class TestLexerNGPos : public GTLTest::Case {
  public:
    TestLexerNGPos() : GTLTest::Case("Pos") {}
    virtual void runTest()
    {
      std::istringstream iss(" HELLO\nWORLD !");
      OpenCTL::LexerNG lng(&iss);
      TEST_FOR_TOKEN_AND_POS( IDENTIFIER, 1, 2);
      TEST_FOR_TOKEN_AND_POS( IDENTIFIER, 2, 1);
      TEST_FOR_TOKEN_AND_POS( NOT, 2, 7);
    }
};


class TestLexerNGComments : public GTLTest::Case {
  public:
    TestLexerNGComments() : GTLTest::Case("Comments") {}
    virtual void runTest()
    {
      std::istringstream iss(" /* HELLO\nWORLD ! */ + // BOUH \n -");
      OpenCTL::LexerNG lng(&iss);
      TEST_FOR_TOKEN( PLUS );
      TEST_FOR_TOKEN( MINUS );
    }
};

class TestLexerNGCtlVersion : public GTLTest::Case {
  public:
    TestLexerNGCtlVersion() : GTLTest::Case("CtlVersion") {}
    virtual void runTest()
    {
      std::istringstream iss("ctlversion 1;");
      OpenCTL::LexerNG lng(&iss);
      TEST_FOR_TOKEN( CTLVERSION );
      TEST_FOR_TOKEN( INTEGER_CONSTANT );
      TEST_FOR_TOKEN( SEMI );
    }
};

class TestLexerNG : public GTLTest::Suite {
  public:
    TestLexerNG() : GTLTest::Suite("LexerNG")
    {
      addCase(new TestLexerNGConstants );
      addCase(new TestLexerNGKeywords );
      addCase(new TestLexerNGSpecialCharacters );
      addCase(new TestLexerNGNoSpace );
      addCase(new TestLexerNGPos );
      addCase(new TestLexerNGIdentifier );
      addCase(new TestLexerNGComments );
      addCase(new TestLexerNGCtlVersion );
    }
};
