///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file AutoStart.h
 * \brief Contains the definition of the Core::AutoStart class.
 */

#ifndef __OVITO_AUTO_START_H
#define __OVITO_AUTO_START_H

#include <core/Core.h>
#include <core/plugins/PluginClass.h>

namespace Core {

/**
 * \brief Base class for plugin classes that should be activated at program startup..
 *
 * Derive a new sub-class from this AutoStart class if you want to perform some work
 * on application startup.
 *
 * On startup the application will search the class database for all classes that
 * are derived from this AutoStart class. It will create an instance of each of these classes
 * and call their startEvent() function for each initialization stage.
 * The class instances are kept until the application shuts down.
 *
 * Note that every plugin that contains an AutoStart derived class will automatically
 * be loaded on application startup.
 *
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT AutoStart : public PluginClass
{
protected:

	/// \brief The default constructor.
	AutoStart() : PluginClass() {}

public:

	/**
	 * List of events that are generated at application startup.
	 * The startEvent() method is called for each of the events.
	 */
	enum InitializationStage {
		PluginManagerInitialized,		//< This event is generated after the PluginManager has been initailized and all native plugins have been registered.
		ViewportManagerInitialized,		//< This event is generated after the ViewportManager has been initailized.
		DataSetManagerInitialized,		//< This event is generated after the DataSetManager has been initailized.
		ViewportInputManagerInitialized,//< This event is generated after the ViewportInputManager has been initailized.
		AnimManagerInitialized,			//< This event is generated after the AnimManager has been initailized.
		ActionManagerInitialized,		//< This event is generated after the ActionManager has been initailized.

		MainWindowCreated,				//< This event is generated after the main application window has been created (but not yet shown). Note that this event is only generated in GUI mode.
		InitialSceneLoaded,				//< This event is generated after the first DataSet is in place, i.e. the initial scene file has been loaded or an has been created.
		MainWindowShown,				//< This event is generated after the main window was made visible. Note that this event is only generated in GUI mode.
		ApplicationRunning,				//< This event is generated after the application has been completely initialized.
	};

	/// \brief Performs the class specific work on application startup.
	/// \param stage The current stage of application startup.
	///
	/// This method is called several time by the system on application startup.
	///
	/// \throw Exception to abort application startup and exit the program.
	virtual void startEvent(InitializationStage stage) = 0;

private:
	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(AutoStart);
};

};	// End of namespace Core

#endif // __OVITO_AUTO_START_H
