///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/SubObjectParameterUI.h>
#include <core/gui/properties/ColorControllerUI.h>
#include <core/gui/properties/IntegerPropertyUI.h>
#include <core/gui/properties/StringPropertyUI.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/gui/properties/IntegerRadioButtonPropertyUI.h>
#include <core/gui/properties/BooleanGroupBoxPropertyUI.h>
#include <core/gui/dialogs/SaveImageFileDialog.h>
#include <core/actions/ActionManager.h>
#include <core/rendering/RenderSettings.h>
#include <core/rendering/RenderSettingsEditor.h>

namespace Core {

IMPLEMENT_PLUGIN_CLASS(RenderSettingsEditor, PropertiesEditor)

// Predefined output image dimensions.
static const int imageSizePresets[][2] = {
		{ 320, 240 },
		{ 640, 480 },
		{ 800, 600 },
		{ 1024, 768 },
		{ 600, 600 },
		{ 1000, 1000 }
};

/******************************************************************************
* Constructor that creates the UI controls for the editor.
******************************************************************************/
void RenderSettingsEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create the rollout.
	QWidget* rollout = createRollout(tr("Render settings"), rolloutParams, "core.rendering.render_settings");
	
	QVBoxLayout* layout = new QVBoxLayout(rollout);
	layout->setContentsMargins(4,4,4,4);

	// Rendering range
	{
		QGroupBox* groupBox = new QGroupBox(tr("Time output"));
		layout->addWidget(groupBox);

		QVBoxLayout* layout2 = new QVBoxLayout(groupBox);
		layout2->setContentsMargins(4,4,4,4);
#ifndef Q_WS_MAC
		layout2->setSpacing(2);
#endif
		QGridLayout* layout2c = new QGridLayout();
		layout2c->setContentsMargins(0,0,0,0);
#ifndef Q_WS_MAC
		layout2c->setSpacing(2);
#endif
		layout2->addLayout(layout2c);

		IntegerRadioButtonPropertyUI* renderingRangeTypeUI = new IntegerRadioButtonPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(RenderSettings, _renderingRangeType));

		QRadioButton* currentFrameButton = renderingRangeTypeUI->addRadioButton(RenderSettings::CURRENT_FRAME, tr("Single frame"));
		layout2c->addWidget(currentFrameButton, 0, 0, 1, 5);

		QRadioButton* animationIntervalButton = renderingRangeTypeUI->addRadioButton(RenderSettings::ANIMATION_INTERVAL, tr("Complete animation"));
		layout2c->addWidget(animationIntervalButton, 1, 0, 1, 5);

		QRadioButton* customIntervalButton = renderingRangeTypeUI->addRadioButton(RenderSettings::CUSTOM_INTERVAL, tr("Range:"));
		layout2c->addWidget(customIntervalButton, 2, 0, 1, 5);

		IntegerPropertyUI* customRangeStartUI = new IntegerPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(RenderSettings, _customRangeStart));
		customRangeStartUI->setEnabled(false);
		layout2c->addLayout(customRangeStartUI->createFieldLayout(), 3, 1);
		layout2c->addWidget(new QLabel(tr("to")), 3, 2);
		IntegerPropertyUI* customRangeEndUI = new IntegerPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(RenderSettings, _customRangeEnd));
		customRangeEndUI->setEnabled(false);
		layout2c->addLayout(customRangeEndUI->createFieldLayout(), 3, 3);
		layout2c->setColumnMinimumWidth(0, 30);
		layout2c->setColumnStretch(4, 1);
		connect(customIntervalButton, SIGNAL(toggled(bool)), customRangeStartUI, SLOT(setEnabled(bool)));
		connect(customIntervalButton, SIGNAL(toggled(bool)), customRangeEndUI, SLOT(setEnabled(bool)));

		QGridLayout* layout2a = new QGridLayout();
		layout2a->setContentsMargins(0,6,0,0);
#ifndef Q_WS_MAC
		layout2a->setSpacing(2);
#endif
		layout2->addLayout(layout2a);
		IntegerPropertyUI* everyNthFrameUI = new IntegerPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(RenderSettings, _everyNthFrame));
		layout2a->addWidget(everyNthFrameUI->label(), 0, 0);
		layout2a->addLayout(everyNthFrameUI->createFieldLayout(), 0, 1);
		everyNthFrameUI->setMinValue(1);
		IntegerPropertyUI* fileNumberBaseUI = new IntegerPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(RenderSettings, _fileNumberBase));
		layout2a->addWidget(fileNumberBaseUI->label(), 1, 0);
		layout2a->addLayout(fileNumberBaseUI->createFieldLayout(), 1, 1);
		layout2a->setColumnStretch(2, 1);
		connect(currentFrameButton, SIGNAL(toggled(bool)), everyNthFrameUI, SLOT(setDisabled(bool)));
		connect(currentFrameButton, SIGNAL(toggled(bool)), fileNumberBaseUI, SLOT(setDisabled(bool)));
	}

	// Output size
	{
		QGroupBox* groupBox = new QGroupBox(tr("Output image size"));
		layout->addWidget(groupBox);
		QGridLayout* layout2 = new QGridLayout(groupBox);
		layout2->setContentsMargins(4,4,4,4);
#ifndef Q_WS_MAC
		layout2->setSpacing(2);
#endif
		layout2->setColumnStretch(1, 1);

		// Image width parameter.
		IntegerPropertyUI* imageWidthUI = new IntegerPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(RenderSettings, _outputImageWidth));
		layout2->addWidget(imageWidthUI->label(), 0, 0);
		layout2->addLayout(imageWidthUI->createFieldLayout(), 0, 1);
		imageWidthUI->setMinValue(1);
	
		// Image height parameter.
		IntegerPropertyUI* imageHeightUI = new IntegerPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(RenderSettings, _outputImageHeight));
		layout2->addWidget(imageHeightUI->label(), 1, 0);
		layout2->addLayout(imageHeightUI->createFieldLayout(), 1, 1);
		imageHeightUI->setMinValue(1);

		sizePresetsBox = new QComboBox(groupBox);
		sizePresetsBox->addItem(tr("Presets..."));
		for(int i = 0; i < sizeof(imageSizePresets)/sizeof(imageSizePresets[0]); i++)
			sizePresetsBox->addItem(tr("%1 x %2").arg(imageSizePresets[i][0]).arg(imageSizePresets[i][1]));
		connect(sizePresetsBox, SIGNAL(activated(int)), this, SLOT(onSizePresetActivated(int)));
		layout2->addWidget(sizePresetsBox, 1, 2);
	}

	// Options
	{
		QGroupBox* groupBox = new QGroupBox(tr("Options"));
		layout->addWidget(groupBox);
		QGridLayout* layout2 = new QGridLayout(groupBox);
		layout2->setContentsMargins(4,4,4,4);
#ifndef Q_WS_MAC
		layout2->setSpacing(2);
#endif
		// Background color parameter.
		ColorControllerUI* backgroundColorPUI = new ColorControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(RenderSettings, _backgroundColor));
		layout2->addWidget(backgroundColorPUI->label(), 2, 0);
		layout2->addWidget(backgroundColorPUI->colorPicker(), 2, 1, 1, 2);
	
		// Alpha channel.
		BooleanPropertyUI* generateAlphaUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(RenderSettings, _generateAlphaChannel));
		layout2->addWidget(generateAlphaUI->checkBox(), 3, 0, 1, 3);

		// Create 'Change renderer' button.
		QPushButton* changeRendererButton = new QPushButton(tr("Change renderer..."), groupBox);
		connect(changeRendererButton, SIGNAL(clicked(bool)), ACTION_MANAGER.findActionProxy(ACTION_SELECT_RENDERER_DIALOG), SLOT(trigger()));
		layout2->addWidget(changeRendererButton, 4, 0, 1, 3);
	}

	// Render output
	{
		QGroupBox* groupBox = new QGroupBox(tr("Render output"));
		layout->addWidget(groupBox);
		QGridLayout* layout2 = new QGridLayout(groupBox);
		layout2->setContentsMargins(4,4,4,4);
#ifndef Q_WS_MAC
		layout2->setSpacing(2);
#endif
		layout2->setColumnStretch(0, 1);

		BooleanPropertyUI* saveFileUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(RenderSettings, _saveToFile));
		layout2->addWidget(saveFileUI->checkBox(), 0, 0);

		QPushButton* chooseFilenameBtn = new QPushButton(tr("Choose..."), rollout);
		connect(chooseFilenameBtn, SIGNAL(clicked(bool)), this, SLOT(onChooseImageFilename()));
		layout2->addWidget(chooseFilenameBtn, 0, 1);

		// Output filename parameter.
		StringPropertyUI* imageFilenameUI = new StringPropertyUI(this, "imageFilename");
		imageFilenameUI->setEnabled(false);
		layout2->addWidget(imageFilenameUI->textBox(), 1, 0, 1, 2);

		BooleanPropertyUI* skipExistingImagesUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(RenderSettings, _skipExistingImages));
		layout2->addWidget(skipExistingImagesUI->checkBox(), 2, 0, 1, 2);

		connect(saveFileUI->checkBox(), SIGNAL(toggled(bool)), skipExistingImagesUI, SLOT(setEnabled(bool)));
	}

	// Open a sub-editor for the renderer.
	new SubObjectParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(RenderSettings, _renderer), rolloutParams.after(rollout));
}

/******************************************************************************
* Lets the user choose a filename for the output image.
******************************************************************************/
void RenderSettingsEditor::onChooseImageFilename()
{
	RenderSettings* settings = static_object_cast<RenderSettings>(editObject());
	if(!settings) return;

	try {
		SaveImageFileDialog fileDialog(container(), tr("Output image file"), settings->imageInfo());
		if(fileDialog.exec()) {
			UNDO_MANAGER.beginCompoundOperation(tr("Change output file"));
			settings->setImageInfo(fileDialog.imageInfo());
			settings->setSaveToFile(true);
			UNDO_MANAGER.endCompoundOperation();
		}
	}
	catch(const Exception& ex) {
		ex.showError();
	}
}

/******************************************************************************
* Is called when the user selects an output size preset from the drop-down list.
******************************************************************************/
void RenderSettingsEditor::onSizePresetActivated(int index)
{
	RenderSettings* settings = static_object_cast<RenderSettings>(editObject());
	if(settings && index >= 1 && index <= sizeof(imageSizePresets)/sizeof(imageSizePresets[0])) {
		UNDO_MANAGER.beginCompoundOperation(tr("Change output dimensions"));
		settings->setOutputImageWidth(imageSizePresets[index-1][0]);
		settings->setOutputImageHeight(imageSizePresets[index-1][1]);
		UNDO_MANAGER.endCompoundOperation();
	}
	sizePresetsBox->setCurrentIndex(0);
}


};
