/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Julien Moutte <julien@fluendo.com>
 */

#ifndef __PGM_COMMON_H__
#define __PGM_COMMON_H__

#include <glib-object.h>
#include <math.h> /* sin, cos, sincos, fabs, fabsf */

G_BEGIN_DECLS

/**
 * PGM_DEGREES_TO_RADIANS:
 * @theta: an angle in degrees.
 *
 * Converts from degrees to radians.
 */
#define PGM_DEGREES_TO_RADIANS(theta) ((theta) * (G_PI / 180.0f))

/**
 * PGM_RADIANS_TO_DEGREES:
 * @theta: an angle in radians.
 *
 * Converts from radians to degrees.
 */
#define PGM_RADIANS_TO_DEGREES(theta) ((theta) * (180.0f / G_PI))

/**
 * PGM_SINCOS:
 * @theta: a float representing an angle in radians.
 * @s: the address of the float where the sine value is going to be stored.
 * @c: the address of the float where the cosine value is going to be stored.
 *
 * Calculates sine and cosine of the angle @theta simultaneously.
 */
#if HAVE_SINCOSF
#define PGM_SINCOS(theta,s,c) sincosf ((theta), (s), (c))
#elif HAVE_SINF && HAVE_COSF
#define PGM_SINCOS(theta,s,c)                         \
  G_STMT_START {                                      \
    *(s) = sinf (theta);                              \
    *(c) = cosf (theta);                              \
  } G_STMT_END
#else
#define PGM_SINCOS(theta,s,c)                         \
  G_STMT_START {                                      \
    *(s) = (gfloat) sin ((gdouble) theta);            \
    *(c) = (gfloat) cos ((gdouble) theta);            \
  } G_STMT_END
#endif /* HAVE_SINCOS */

/**
 * PGM_FABSF:
 * @x: the floating-point number.
 *
 * Calculates the absolute value of float.
 */
#if HAVE_FABSF
#define PGM_FABSF(x) fabsf (x)
#else
#define PGM_FABSF(x) ((gfloat) fabs (x))
#endif /* HAVE_FABSF */

/**
 * PgmError:
 * @PGM_ERROR_X: Generic error code.
 * @PGM_ERROR_OK: No error occured.
 *
 * Every Pigment method returns an error code from that list.
 */
typedef enum {
  PGM_ERROR_X,
  PGM_ERROR_OK
} PgmError;

/**
 * pgm_lrintf:
 * @x: the input floating-point number.
 *
 * Round @x to nearest integer using the current FPU rounding mode. On platforms
 * other than Windows where the C99 lrintf function is not available, the
 * function reverts to a float-to-int cast which uses the truncate FPU rounding
 * mode.
 *
 * Returns: the rounded nearest integer.
 */
#if HAVE_LRINTF
#define pgm_lrintf(x) lrintf (x)
#elif defined (_MSC_VER)
gint32 pgm_lrintf (gfloat x);
#else
#define pgm_lrintf(x) (gint32) (x)
#endif

G_END_DECLS

#endif /* __PGM_COMMON_H__ */
