/*	File

PIRL CVS ID: File.java,v 1.16 2012/04/16 06:18:24 castalia Exp

Copyright (C) 2006-2102  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package PIRL.Utilities;

//Pirl Packages
import PIRL.Configuration.Configuration;
import PIRL.Configuration.Configuration_Exception;
import PIRL.PVL.Parameter;
import PIRL.PVL.PVL_Exception;
import PIRL.Strings.String_Buffer;

import java.io.IOException;
import java.net.URI;
import java.util.Iterator;
import java.util.Vector;

/**	This <i>File</i> extends the capabilities of the JFC {@link
	java.io.File#File(java.lang.String) File} to enable mapping the
	File's canonical path to a logical path. The mappings are defined by
	a {@link #Pathnames_Map(Configuration) pathnames map} obtained
	from a Configuration.
<p>
	@see java.io.File
	@see PIRL.Configuration.Configuration

	@author Rodney Heyd UA/PIRL/HiRISE
	@version 1.16
*/
public class File
	extends java.io.File
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.Utilities.File (1.16 2012/04/16 06:18:24)";

/** Default configuration file name: "Pathname.conf".
*/	
public static final String
	DEFAULT_CONFIGURATION_FILENAME		= "Pathnames_Map.conf";
		
/** The Parameter Group in the Configuration containing the logical pathname map.
*/	
public static final String
	PATHNAMES_MAP_GROUP					= "Pathnames_Map";


/**	The canonical to logical path map.
*/
private Configuration
	Pathnames_Map						= null;

/** The logical path for this File.
*/
private String
	Logical_Path						= null;


/** Exit Status: Success.
 */
public static final int
	EXIT_SUCCESS						= 0;

/** Exit Status: Invalid command line syntax.
*/
public static final int
	EXIT_INVALID_COMMAND_LINE_SYNTAX 	= 1;

/** Exit Status: A Configuration problem was encountered.
*/
public static final int
	EXIT_CONFIGURATION_PROBLEM			= 2;

/** Exit Status: An IO Error occured.
*/
public static final int
	EXIT_IO_ERROR						= 3;


//	Debug control
private static final int
	DEBUG_OFF				= 0,
	DEBUG_CONSTRUCTORS		= 1 << 0,
	DEBUG_ACCESSORS			= 1 << 1,
	DEBUG_HELPERS			= 1 << 4,
	DEBUG_MAIN				= 1 << 5,
	DEBUG_ALL				= -1,

	DEBUG					= DEBUG_OFF;

/*===========================================================================
	Constructors
*/
/** Construct a File from a String.
<p>
	@param	pathname	A pathname String.
	@see	java.io.File
*/
public File
	(
	String	pathname
	)
{super (pathname);}

/**	Construct a File from directory and filename Strings.
<p>
	@param parent	A directory pathname String.
	@param child	A filename String for a file in the parent directory.
	@see java.io.File
*/
public File
	(
	String	parent,
	String	child
	)
{super (parent, child);}

/** Construct a File a directory File and a filename String.
<p>
	@param parent	A directory File.
	@param child	A filename String for a file in the parent directory.
	@see java.io.File
*/
public File
	(
	File	parent,
	String	child
	)
{super (parent, child);}

/** Construct a File from a URI.
<p>
	@param uri	A URI.
	@see java.net.URI
*/
public File
	(
	URI	uri
	)
{super (uri);}


/*===========================================================================
	Accessors
*/	
/**	Set the pathnames map to be used for mapping canonical paths to logical
	paths.
<p>
	The canonical to logical pathname mappings are defined by a {@link
	#PATHNAMES_MAP_GROUP} group of Assignment Parameters of the form:
<p><blockquote>
	&lt;<i>canonical segment</i>&gt; <b>=</b> &lt;<i>logical segment</i>&gt;
</blockquote><p>
	The <i>cononical segment</i> is any substring of a pathname that
	might be returned by the {@link java.io.File#getCanonicalPath()}
	method. The <i>logical segment</i> replaces the canonical segment
	wherever it occurs in the pathname to produce the {@link
	#Logical_Path()} of the File. Initially the logical path is identical
	to the canonical path, then all mapping parameters are applied in the
	order they occur in the group. Thus a matching canonical segment may
	have been produced, in whole or in part, by a preceeding canonical to
	logical segment mapping replacement.
<p>
	If the configuration argument is null an attempt will be made to load
	a Configuration from the {@link #DEFAULT_CONFIGURATION_FILENAME}
	file. If this file can not be accessed a default Configuration is
	provided. <b>N.B.</b>: This behaviour is different than the
	{@link #Pathnames_Map(String)} method which will throw a
	Configuration_Exception if the source file can not be accessed.
<p>
	If the configuration is named {@link #PATHNAMES_MAP_GROUP} then a
	copy of the entire configuration is used as the pathnames map.
	Otherwise the configuration searched for a Parameter Group with that
	name which is copied. If no {@link #PATHNAMES_MAP_GROUP} is found
	then an empty pathnames map is provided; no canonical to logical
	pathname mapping will be done in this case.
<p>
	Any {@link Configuration#Defaults default} Configuration parameters
	are removed from the pathnames map as well as any Parameters that are
	a Group or have an Array, or no, Value.
<p>
	@param configuration	A Configuration to use as the source of the
		pathnames map.
	@return	This File.
	@throws	Configuration_Exception	If there was a problem obtaining the
		{@link #PATHNAMES_MAP_GROUP} from the map or reading the
		{@link #DEFAULT_CONFIGURATION_FILENAME}.
*/
public File Pathnames_Map
	(
	Configuration	configuration
	) 
	throws Configuration_Exception
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">>> File.Pathnames_Map: " + configuration);
Configuration
	map = null;
if (configuration == null)
	{
	if ((DEBUG & DEBUG_ACCESSORS) != 0)
		System.out.println
			("    Attempting to load configuration file "
				+ DEFAULT_CONFIGURATION_FILENAME);
	try {configuration = new Configuration (DEFAULT_CONFIGURATION_FILENAME);} 
	catch (Configuration_Exception exception)
		{
		throw new Configuration_Exception
			(ID + '\n'
			+"Unable to obtain a Pathnames Map Configuration\n"
			+"  from the \""
				+ DEFAULT_CONFIGURATION_FILENAME + "\" source file.\n"
			+ exception.getMessage ());
		}
	catch (IllegalArgumentException exception)
		{
		//	The default file could not be found; use an empty map.
		if ((DEBUG & DEBUG_ACCESSORS) != 0)
			System.out.println
				("    No default config file.");
		}
	}

if (configuration != null)
	{
	if (configuration.Name ().equalsIgnoreCase (PATHNAMES_MAP_GROUP))
		//	Copy the entire Configuration.
		map = new Configuration (configuration);
	else
		{
		synchronized (configuration)
			{
			//	Try to get the Pathnames_Map group.
			boolean
				case_sensitive = configuration.Case_Sensitive (false);
			try {map = configuration.Group (PATHNAMES_MAP_GROUP);}
			catch (Configuration_Exception exception)
				{
				configuration.Case_Sensitive (case_sensitive);
				throw new Configuration_Exception
					(ID + '\n'
					+ "Unable to obtain the "+ PATHNAMES_MAP_GROUP
						+ " from the Configuration.\n"
					+ exception.getMessage ());
				}
			configuration.Case_Sensitive (case_sensitive);
			}
		}

	if (map != null)
		{
		//	Remove Configuration Defaults and inappropriate parameters.
		Parameter
			parameter;
		Iterator
			parameters = map.iterator (),
			defaults;
		Clean_Map:
		while (parameters.hasNext ())
			{
			parameter = (Parameter)parameters.next ();
			try
				{
				if (parameter.Is_Aggregate () ||
					parameter.Value () == null ||
					parameter.Value ().Is_Array ())
					{
					parameters.remove ();
					continue Clean_Map;
					}
				}
			catch (PVL_Exception exception) {/* Can't happen */}
			defaults = Configuration.Defaults.iterator ();
			while (defaults.hasNext ())
				{
				if (parameter.Name ()
						.equals (((Parameter)defaults.next ()).Name ()))
					{
					parameters.remove ();
					continue Clean_Map;
					}
				}
			}
		}
	}

if (map == null)
	{
	//	Use and empty map.
	if ((DEBUG & DEBUG_ACCESSORS) != 0)
		System.out.println
			("    Using an empty map.");
	Pathnames_Map = new Configuration ((Parameter)null);
	Pathnames_Map.Name (PATHNAMES_MAP_GROUP);
	}
else
	Pathnames_Map = map;

Logical_Path = null;	//	Reset.
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(Pathnames_Map.Description ()
		+ "<<< File.Pathnames_Map: " + configuration);
return this;
}

/**	Set the pathnames map from a configuration file source.
<p>
	@param	source	The source of a configuration file. This may be
		a local filename, a URL, or a jar file resource name.
	@return	This File.
	@throws	Configuration_Exception	If the source file could not be
		accessed (not found or no permission to read) or there was
		a problem constructing a Configuration from the source contents.
	@see	#Pathnames_Map(Configuration)
*/
public File Pathnames_Map
	(
	String	source
	)
	throws Configuration_Exception
{
Configuration
	configuration = null;
if (source != null)
	{
	try {configuration = new Configuration (source);} 
	catch (Configuration_Exception exception)
		{
		throw new Configuration_Exception
			(ID + '\n'
			+"Unable to obtain a Pathnames Map Configuration\n"
			+"  from the \"" + source + "\" source file.\n"
			+ exception.getMessage ());
		}
	catch (IllegalArgumentException exception)
		{
		//	The source file could not be found.
		throw new Configuration_Exception
			(ID + '\n'
			+"Couldn't access the Pathnames Map Configuration\n"
			+"  from the \"" + source + "\" source file.");
		}
	}
return Pathnames_Map (configuration);
}

/**	Get the current pathnames map.
<p>
	@return	A Configuration containing the current pathnames map parameters.
		This may be null if no map has yet been assigned.
*/
public Configuration Pathnames_Map ()
{return Pathnames_Map;}

/**	Get the logical path for the file.
<p>
	The logical path is based on the File's {@link
	java.io.File#getCanonicalPath() canonical path} with the cononical
	segments listed in thepathnames map replaced with the logical
	segments to which they are assigned.
<p>
	<b>N.B.</b.: If a pathnames map has not yet been bound to this File,
	a default map will be provided.
<p>
	@return	The logical path String.
	@throws	Configuration_Exception	If a default pathnames map was needed
		but the source file could not be accessed (not found or no
		permission to read) or there was a problem constructing a
		Configuration from the source contents.
	@throws IOException	If the @link java.io.File#getCanonicalPath()
		canonical path of the File could not be obtained.
	@see	#Pathnames_Map(Configuration)
	@see	#Pathnames_Map(String)
*/
public String Logical_Path ()
	throws IOException, Configuration_Exception
{
if (Logical_Path == null)
	Set_Logical_Path ();
return Logical_Path;
}

/*------------------------------------------------------------------------------
	Helpers
*/
/**	Sets the logical path.
<p>
	The Pathnames_Map is used to determine the logical path mapping.
<p>
	@see #Pathnames_Map(Configuration)
*/
private void Set_Logical_Path ()
	throws IOException, Configuration_Exception
{
if ((DEBUG & DEBUG_HELPERS) != 0)
	System.out.println
		(">>> File.Set_Logical_Path");
if (Pathnames_Map == null)
	//	Instantiate the default map.
	Pathnames_Map ((Configuration)null);

String_Buffer
	logical_path = null;
try
	{
	// Initialize with the canonical path of this File.
	logical_path = new String_Buffer (getCanonicalPath ());
	} 
catch (IOException exception)
	{
	throw new IOException
		(ID + '\n'
		+"Unable to obtain the canonical path for \"" + getName () + "\"\n"
		+ exception.getMessage ());
	}
if ((DEBUG & DEBUG_HELPERS) != 0)
	System.out.println
		("    Canonical path: " + logical_path);

Parameter
	parameter;
Iterator
	map_entries = Pathnames_Map.iterator ();
while (map_entries.hasNext ())
	{
	parameter = (Parameter)map_entries.next ();
	try
		{
		logical_path.replace
			(0, parameter.Name (), parameter.Value ().String_Data ());
		if ((DEBUG & DEBUG_HELPERS) != 0)
			System.out.println
				("    Canonical segment: " + parameter.Name () + '\n'
				+"      Logical segment: " + parameter.Value ().String_Data () + '\n'
				+"    ---> Logical path: " + logical_path);
		}
	catch (PVL_Exception exception) {/* Shouldn't happen with clean map. */}
	}
Logical_Path = logical_path.toString ();
if ((DEBUG & DEBUG_HELPERS) != 0)
	System.out.println
		("<<< File.Set_Logical_Path: " + Logical_Path);
}

/*==============================================================================
	Application
*/
/**	Map filenames to their logical forms.
<p>
	The command line syntax is described by the {@link #Usage()
	Usage} method.
<p>
	Each filename specified will be listed along with its canonical path
	and mapped logical path.
<p>
	Exit Status Values
<p>
	0 - Success<br>
	1 - Command line syntax problem<br>
	2 - A Configuration problem was encountered<br>
	3 - An IO Error occured<br>
<p>
	@param arguments	Array of command line argument Strings.
	@see	#Usage()
 */
public static void main
	(
	String[]	arguments
	)
{
String
	configuration_filename = null;
Vector
	filenames = new Vector ();

for (int count = 0;
		 count < arguments.length;
	   ++count)
	{
	if (arguments[count].length () == 0)
		continue;
	if (arguments[count].charAt (0) == '-')
		{
		switch (arguments[count].charAt (1))
			{
			case 'C':	//	-Configuration
			case 'c':
				String 
					filename = null;
				if (++count == arguments.length ||
					arguments[count].charAt (0) == '-')
					{
					System.err.println
						("Missing Configuration filename.\n");
					Usage ();
					}
				if (configuration_filename != null)
					{
					System.err.println
						("Multiple configuration files specified:\n"
						+"  " + configuration_filename + '\n'
						+"  " + arguments[count]);
					System.exit (EXIT_INVALID_COMMAND_LINE_SYNTAX);
					}
				configuration_filename = arguments[count];
				break;
			default:
				System.err.println
					("Unrecognized command line option: "
						+ arguments[count] + '\n');
			case 'H':	//	-Help
			case 'h':
				Usage ();
			}
		}
	else
		filenames.add (arguments[count]);
	}

if (filenames.isEmpty ())
	{
	System.out.println ("No filename specified.\n");
	Usage ();
	}

File
	logical_File;
String
	canonical_path,
	logical_path;
Iterator
	list = filenames.iterator ();
try
	{
	while (list.hasNext ())
		{
		logical_File = new File ((String)list.next ())
			.Pathnames_Map (configuration_filename);

		canonical_path = logical_File.getCanonicalPath ();
		logical_path   = logical_File.Logical_Path ();
		System.out.println
			("      Filename: " + logical_File.getPath () + '\n'
			+"Canonical path: " + canonical_path + '\n'
			+"  Logical path: " + logical_path + '\n');
		}
	}
catch (Configuration_Exception exception)
	{
	System.err.println (exception.getMessage ());
	System.exit (EXIT_CONFIGURATION_PROBLEM);			
	}
catch (IOException exception) 
	{
	System.err.println (exception.getMessage ());
	System.exit (EXIT_IO_ERROR);
	}
System.exit (EXIT_SUCCESS);
}

/**	Prints the command line usage syntax.
<p>
<blockquote><pre>
Usage: <b>File</b> &lt;<i>Options</i>&gt; &lt;<i>filename</i>&gt; [...]
&nbsp;&nbsp;Options -
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>C</u>onfiguration</b> &lt;<i>filename</i>&gt;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;(default: Pathnames_Map.conf)
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>H</u>elp</b>]
</pre></blockquote>
*/
public static void Usage ()
{
System.out.println
	(
	ID + '\n' +
	"Usage: File <Options> <pathname> [...]\n" +
	"  Options -\n" +
	"    [-Configuration <filename>]" +
		" (default: " + DEFAULT_CONFIGURATION_FILENAME + ")\n" +
	"    [-Help]"
	);
System.exit (EXIT_INVALID_COMMAND_LINE_SYNTAX);
}


}
