/*	Font_Selector

PIRL CVS ID: Font_Selector.java,v 1.8 2012/04/16 06:22:58 castalia Exp

Copyright (C) 2003-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package	PIRL.Viewers;

import javax.swing.JDialog;
import javax.swing.JPanel;
import javax.swing.JComboBox;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JTextArea;
import javax.swing.JScrollPane;
import javax.swing.JLabel;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.Box;
import java.awt.Font;
import java.awt.Frame;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GraphicsEnvironment;
import java.awt.event.*;
import java.lang.Math;
import java.lang.Integer;
import java.lang.NumberFormatException;
import java.util.Vector;
import java.util.TreeSet;
import java.util.Iterator;

//	For the Test stub.
import javax.swing.JFrame;
import javax.swing.BoxLayout;


/**	A <I>Font_Selector</I> is a dialog box for selecting a named font
	and its size and style characteristics.
<P>
	@author	Bradford Castalia, UA/PIRL
	@version 1.8
*/
public class Font_Selector
	extends JDialog
	implements ItemListener
{
private static final String
	ID = "PIRL.Viewers.Font_Selector (1.8 2012/04/16 06:22:58)";

//	The original and current fonts.
private Font						Original_Font = null,
									Current_Font = null;

/**	The default font when no other default is available:
	SansSerif 12-point plain.
*/
public static final Font			DEFAULT_FONT
										= new Font ("SansSerif", Font.PLAIN, 12);

//	Font sizes.
/**	The Vector of default Integer font sizes: 9, 10, 12, 14, 16, 18, 24.
*/
public static final Vector			DEFAULT_SIZES;
static
	{
	DEFAULT_SIZES = new Vector (7);
	DEFAULT_SIZES.add (new Integer ( 9));
	DEFAULT_SIZES.add (new Integer (10));
	DEFAULT_SIZES.add (new Integer (12));
	DEFAULT_SIZES.add (new Integer (14));
	DEFAULT_SIZES.add (new Integer (16));
	DEFAULT_SIZES.add (new Integer (18));
	DEFAULT_SIZES.add (new Integer (24));
	}

/**	The minimum acceptable value for a font size: 4.
*/
public static final int				MINIMUM_SIZE
										= 4;
/**	The maximum acceptable value for a font size: 256.
*/
public static final int				MAXIMUM_SIZE
										= 256;

//	Sample text.
private static final String			DEFAULT_TEXT =
	"ABCDEFGHIJKLMNOPQRSTUVWXYZ\n" +
	"abcdefghijklmnopqrstuvwxyz\n" +
	"0123456789 ~!@#$%^&*()_+`-=[]\\{}|;':\"<>?,./";
private JTextArea					Sample_Text
										= new JTextArea (DEFAULT_TEXT);
private static final Insets			TEXT_MARGIN
										= new Insets (5, 5, 5, 5);
private	JScrollPane					Sample_Font_Panel;

//	Selection lists.
private JComboBox					Font_List = null,	// Disabled == null.
									Size_List = null,
									Style_List = null;
private static final int			MAX_LIST_LINES_SHOWING
										= 9;
private static DefaultComboBoxModel	Font_List_Model = null,
									Style_List_Model = null;
private DefaultComboBoxModel		Size_List_Model;
//	Contols when selection lists are active.
private boolean						Selection_Enabled = true;

//	Selection list labels.
private static final JLabel			Font_Label
										= new JLabel ("Font:", JLabel.RIGHT),
									Size_Label
										= new JLabel ("Size:", JLabel.RIGHT),
									Style_Label
										= new JLabel ("Style:", JLabel.RIGHT);

//	Panel for buttons.
private JPanel						Buttons_Panel;
//	Optional "Apply" button.
private JButton						Apply_Button = null;

//	Minimum size of the dialog panel.
private static final Dimension		MINIMUM_DIMENSION
										= new Dimension (250, 125);

//  DEBUG control.
private static final int
	DEBUG_OFF		= 0,
	DEBUG_SETUP		= 1 << 0,
	DEBUG_SELECT	= 1 << 1,
	DEBUG_LISTENERS	= 1 << 2,
	DEBUG_ACCESSORS	= 1 << 3,
	DEBUG_ALL		= -1,

	DEBUG			= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs a <CODE>Font_Selector</CODE> containing the
	specified font characteteristics selection lists.
<P>
	<B>Note</B>: If no selection characteristics lists are enabled, then
	the dialog will not appear when <CODE>setVisible</CODE>. The resulting
	object will still provide correct responses to all of its methods,
	however.
<P>
	@param	title	The String to be used as the title on the dialog box.
		If this is null, then the title will be "Font_Selector".
	@param	select_font	If true, a pull-down list of all available font
		family names is included
		on the selection panel. If false, no font selection is provided.
	@param	select_style	If true, a pull-down list of the four possible
		font styles - plain, bold, italic, and bold plus italic - is
		is included on the selection panel. If false, no style selection
		is provided.
	@param	select_sizes	A Vector of Integer values listing the
		allowable font sizes to included in a pull-down on the
		selection panel. The list will be numerically sorted and
		duplicate values will be removed. If the list contains two
		values and the second value is negative, then all values in the
		range [first,-second] will be included in the selection list.
		Any value less than the <CODE>{@link #MINIMUM_SIZE
		MINIMUM_SIZE}</CODE> or greater than the <CODE>{@link
		#MAXIMUM_SIZE MAXIMUM_SIZE}</CODE> is excluded. If the Vector
		is empty, or all of the listed values are outside of the
		acceptable range, then the <CODE>{@link #DEFAULT_SIZES
		DEFAULT_SIZES}</CODE> will be used. If the Vector is null, no
		size selection is provided (but the <CODE>DEFAULT_SIZES</CODE>
		will still be the effective selection sizes).
	@param	default_font	The Font to use as the original, default,
		font. If the default_font is null and there is a non-null
		owner, then the font from the owner will be used; otherwise the
		default front from the dialog will be used. If this fails to
		produce a non-null font, then the <CODE>{@link #DEFAULT_FONT
		DEFAULT_FONT}</CODE> will be used. If the size of the final
		default font is not one of the selection sizes, then the
		<CODE>{@link #Nearest_Size(int) Nearest_Size}</CODE> to the
		font size in the selection list will be used.
	@param	apply_listener	An ActionListener that will be associated
		with an "Apply" button. When the button is pressed the button
		will call the listener's
		<CODE>{#ActionListener.actionPerformed(ActionEvent)
		actionPerformed}</CODE> method. If null, then no button will
		appear; but it can be added later, if desired, by using the
		<CODE>{@link #addActionListener(ActionListener)
		addActionListener}</CODE> method.
	@param	owner	The parent Frame of the dialog. This may be null.
*/
public Font_Selector
	(
	String			title,
	boolean			select_font,
	boolean			select_style,
	Vector			select_sizes,
	Font			default_font,
	ActionListener	apply_listener,
	Frame			owner
	)
{
super (owner, ((title == null) ? "Font_Selector" : title), true);
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println
		(">>> Font_Selector: " + title + '\n'
		+"    font - " + select_font + '\n'
		+"    sizes - " + select_sizes + '\n'
		+"    style - " + select_style + '\n'
		+"    default_font - " + default_font);
if (Font_List_Model == null)
	{
	//	Initialize the list models.
	construct_Font_List_Model ();
	construct_Style_List_Model ();
	}
boolean
	select_size = true;
if (select_sizes == null)
	select_size = false;
else
	construct_Size_List_Model (select_sizes);

//	Set the initial font.
if (default_font == null &&
	owner != null)
	Original_Font = owner.getFont ();
else
	Original_Font = default_font;
Font (default_font);
Original_Font = Current_Font;

if (! (select_font || select_size || select_style))
	{
	if ((DEBUG & DEBUG_SETUP) != 0)
		System.out.println("<<< Font_Selector: no selections");
	return;
	}

JPanel panel			= new JPanel (new GridBagLayout ());
GridBagConstraints location = new GridBagConstraints ();
location.anchor			= GridBagConstraints.WEST;
location.fill			= GridBagConstraints.NONE;
location.weightx		= 0.0;
location.weighty		= 0.0;

Selection_Enabled = false;
if (select_font)
	{
	if ((DEBUG & DEBUG_SETUP) != 0)
		System.out.println
			("    Font_Selector: select font " + Current_Font.getFamily ());
	location.insets		= new Insets (5, 5, 5, 5);
	panel.add (Font_Label, location);

	Font_List = new JComboBox (Font_List_Model);
	Font_List.setMaximumRowCount (MAX_LIST_LINES_SHOWING);
	Font_List.addItemListener (this);
	Font_List.setSelectedItem (Current_Font.getFamily ());
	location.insets		= new Insets (5, 0, 5, 5);
	if (! (select_size || select_style))
		location.gridwidth = GridBagConstraints.REMAINDER;
	panel.add (Font_List, location);
	}
if (select_style)
	{
	if ((DEBUG & DEBUG_SETUP) != 0)
		System.out.println
			("    Font_Selector: select style "
			+ Style_Description (Current_Font.getStyle ())
			+ " (" + Current_Font.getStyle () + ')');
	location.insets		= new Insets (5, 5, 5, 5);
	panel.add (Style_Label, location);

	Style_List = new JComboBox (Style_List_Model);
	Style_List.setMaximumRowCount (MAX_LIST_LINES_SHOWING);
	Style_List.addItemListener (this);
	Style_List.setSelectedIndex (Current_Font.getStyle ());
	location.insets		= new Insets (5, 0, 5, 5);
	if (! select_size)
		location.gridwidth = GridBagConstraints.REMAINDER;
	panel.add (Style_List, location);
	}
if (select_size)
	{
	if ((DEBUG & DEBUG_SETUP) != 0)
		System.out.println
			("    Font_Selector: select size " + Current_Font.getSize ());
	location.insets		= new Insets (5, 5, 5, 5);
	panel.add (Size_Label, location);

	Size_List = new JComboBox (Size_List_Model);
	Size_List.setMaximumRowCount (MAX_LIST_LINES_SHOWING);
	Size_List.addItemListener (this);
	Size_List.setSelectedItem (new Integer (Current_Font.getSize ()));
	location.insets		= new Insets (5, 0, 5, 5);
	location.gridwidth	= GridBagConstraints.REMAINDER;
	panel.add (Size_List, location);
	}
Selection_Enabled = true;

//	Sample Font:

Sample_Text.setMargin (TEXT_MARGIN);
Sample_Font_Panel		= new JScrollPane (Sample_Text);
location.insets			= new Insets (0, 5, 0, 5);
location.fill			= GridBagConstraints.BOTH;
location.weightx		= 1.0;
location.weighty		= 1.0;
panel.add (Sample_Font_Panel, location);

//	Buttons:

Buttons_Panel			= new JPanel (new GridBagLayout ());
JButton button			= new JButton ("Cancel");
button.setMnemonic ('C');
button.setToolTipText ("Cancel selection");
button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event) {Cancel ();}});
location.insets			= new Insets (5, 5, 5, 0);
location.gridwidth		= 1;
location.anchor 		= GridBagConstraints.WEST;
location.fill			= GridBagConstraints.NONE;
location.weightx		= 0.0;
location.weighty		= 0.0;
Buttons_Panel.add (button, location);

location.anchor			= GridBagConstraints.CENTER;
location.fill			= GridBagConstraints.HORIZONTAL;
location.weightx		= 1.0;
Buttons_Panel.add (Box.createHorizontalGlue (), location);

location.insets			= new Insets (5, 0, 5, 5);
location.anchor 		= GridBagConstraints.EAST;
location.fill			= GridBagConstraints.NONE;
location.weightx		= 0.0;

if (apply_listener != null)
	{
	Apply_Button		= new JButton ("Apply");
	Apply_Button.setMnemonic ('A');
	Apply_Button.setToolTipText ("Apply selection");
	Apply_Button.addActionListener (apply_listener);
	Buttons_Panel.add (Apply_Button, location);
	}

button					= new JButton ("Select");
button.setMnemonic ('S');
button.setToolTipText ("Accept selection");
button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event) {Accept ();}});
location.gridwidth		= GridBagConstraints.REMAINDER;
Buttons_Panel.add (button, location);

location.insets			= new Insets (0, 0, 0, 0);
location.anchor 		= GridBagConstraints.CENTER;
location.fill			= GridBagConstraints.BOTH;
panel.add (Buttons_Panel, location);

Dimension
	dimension = panel.getPreferredSize ();
if (dimension.width  < MINIMUM_DIMENSION.width ||
	dimension.height < MINIMUM_DIMENSION.height)
	{
	dimension.width  = Math.max (dimension.width,  MINIMUM_DIMENSION.width);
	dimension.height = Math.max (dimension.height, MINIMUM_DIMENSION.height);
	panel.setPreferredSize (dimension);
	}

getContentPane ().add (panel, BorderLayout.CENTER);
pack ();
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println("<<< Font_Selector");
}

/**	Constructs a <CODE>Font_Selector</CODE> with no "Apply" button,
	and using a null owner.
<P>
	@see #Font_Selector(String, boolean, boolean, Vector, Font,
		ActionListener, Frame)
*/
public Font_Selector
	(
	String		title,
	boolean		select_font,
	boolean		select_style,
	Vector		select_sizes,
	Font		default_font
	)
{this (title, select_font, select_style, select_sizes, default_font, null, null);}

/**	Constructs a <CODE>Font_Selector</CODE> with no "Apply" button, and
	using a null default font.
<P>
	@see #Font_Selector(String, boolean, boolean, Vector, Font,
		ActionListener, Frame)
*/
public Font_Selector
	(
	String		title,
	boolean		select_font,
	boolean		select_style,
	Vector		select_sizes,
	Frame		owner
	)
{this (title, select_font, select_style, select_sizes, null, null, owner);}

/**	Constructs a <CODE>Font_Selector</CODE> with no "Apply" button, and
	using a null default font and owner.
<P>
	@see #Font_Selector(String, boolean, boolean, Vector, Font,
		ActionListener, Frame)
*/
public Font_Selector
	(
	String		title,
	boolean		select_font,
	boolean		select_style,
	Vector		select_sizes
	)
{this (title, select_font, select_style, select_sizes, null, null, null);}

/**	Constructs a <CODE>Font_Selector</CODE> with font and style
	selections enabled,  with no "Apply" button, and using the
	<CODE>DEFAULT_SIZES</CODE>.
<P>
	@see #Font_Selector(String, boolean, boolean, Vector, Font,
		ActionListener, Frame)
*/
public Font_Selector
	(
	String		title,
	Font		default_font,
	Frame		owner
	)
{this (title, true, true, DEFAULT_SIZES, default_font, null, owner);}

/**	Constructs a <CODE>Font_Selector</CODE> with font and style
	selections enabled, with no "Apply" button, and using the
	<CODE>DEFAULT_SIZES</CODE> and a null owner.
<P>
	@see #Font_Selector(String, boolean, boolean, Vector, Font,
		ActionListener, Frame)
*/
public Font_Selector
	(
	String		title,
	Font		default_font
	)
{this (title, true, true, DEFAULT_SIZES, default_font, null, null);}

/**	Constructs a <CODE>Font_Selector</CODE> with font and style
	selections enabled and using the <CODE>DEFAULT_SIZES</CODE> and a
	null default font.
<P>
	@see #Font_Selector(String, boolean, boolean, Vector, Font,
		ActionListener, Frame)
*/
public Font_Selector
	(
	String		title,
	Frame		owner
	)
{this (title, true, true, DEFAULT_SIZES, null, null, owner);}

/**	Constructs a <CODE>Font_Selector</CODE> with font and style
	selections enabled, with no "Apply" button, and using the
	<CODE>DEFAULT_SIZES</CODE> and a null default font and owner.
<P>
	@see #Font_Selector(String, boolean, boolean, Vector, Font,
		ActionListener, Frame)
*/
public Font_Selector
	(
	String		title
	)
{this (title, true, true, DEFAULT_SIZES, null, null, null);}

/**	Constructs a <CODE>Font_Selector</CODE> with font and style
	selections enabled, with no "Apply" button, and using the
	<CODE>DEFAULT_SIZES</CODE> and a null title and owner.
<P>
	@see #Font_Selector(String, boolean, boolean, Vector, Font,
		ActionListener, Frame)
*/
public Font_Selector
	(
	Font		default_font
	)
{this (null, true, true, DEFAULT_SIZES, default_font, null, null);}

/**	Constructs a <CODE>Font_Selector</CODE> with font and style
	selections enabled, with no "Apply" button, and using the
	<CODE>DEFAULT_SIZES</CODE> and null title, default font and owner.
<P>
	@see #Font_Selector(String, boolean, boolean, Vector, Font,
		ActionListener, Frame)
*/
public Font_Selector ()
{this (null, true, true, DEFAULT_SIZES, null, null, null);}

/*==============================================================================
	Actions
*/
/**	The method that implements the ItemListener interface to
	listen for list selections. This method should only be used
	by the dialog itself.
*/
public void itemStateChanged
	(
	ItemEvent	event
	)
{
if (! Selection_Enabled ||
	event.getStateChange () != ItemEvent.SELECTED)
	return;
if ((DEBUG & DEBUG_LISTENERS) != 0)
	System.out.println (">-< Font_Selector.itemStateChanged");
Object
	list = event.getSource ();
if (list == Font_List)
	{
	if ((DEBUG & DEBUG_LISTENERS) != 0)
		System.out.println
			("    Selected font " + Font_List.getSelectedItem ());
	set_font_name ((String)Font_List.getSelectedItem ());
	}
else if (list == Size_List)
	{
	if ((DEBUG & DEBUG_LISTENERS) != 0)
		System.out.println
			("    Selected size " + Size_List.getSelectedItem ());
	set_size (((Integer)Size_List.getSelectedItem ()).intValue ());
	}
else if (Style_List != null)
	{
	if ((DEBUG & DEBUG_LISTENERS) != 0)
		System.out.println
			("    Selected style " + Style_List.getSelectedItem ()
			+ " (" + Style_List.getSelectedIndex () + ')');
	set_style (Style_List.getSelectedIndex ());
	}
}

private void Accept ()
{setVisible (false);}

private void Cancel ()
{
Reset ();
setVisible (false);
}

/**	Resets the current font to the original font.
*/
public void Reset ()
{Font (Original_Font);}

/**	Sets the dialog to be visible on the display.
<P>
	<B>Note</B>: If no selection lists were enabled when this
	<CODE>Font_Selector</CODE> was constructed, then it will
	never become visible.
*/
public void setVisible
	(
	boolean		set_visible
	)
{
if (isVisible ())
	super.setVisible (set_visible);
else if (set_visible &&
	Font_List != null ||
	Size_List != null ||
	Style_List != null)
	super.setVisible (set_visible);
}

//	Apply button:

/**	Adds an <CODE>ActionListener</CODE> to the "Apply" button.

	If no "Apply" button is present, one is added.
<P>
	@param	apply_listener	The <CODE>ActionListener</CODE> for the
		"Apply" button.
*/
public void addActionListener
	(
	ActionListener	apply_listener
	)
{
if (Apply_Button == null)
	{
	make_Apply_Button (apply_listener);
	GridBagConstraints location = new GridBagConstraints ();
	location.insets		= new Insets (5, 0, 5, 5);
	location.anchor 	= GridBagConstraints.EAST;
	location.gridwidth	= 1;
	Buttons_Panel.add (Apply_Button, location,
		Buttons_Panel.getComponentCount () - 1);
	Buttons_Panel.validate ();
	}
else
	Apply_Button.addActionListener (apply_listener);
}

/**	Removes an <CODE>ActionListener</CODE>, if registered, from the
	"Apply" button, if present.
<P>
	@param	apply_listener	The <CODE>ActionListener</CODE> to be
		removed from the "Apply" button.
*/
public void removeActionListener
	(
	ActionListener	apply_listener
	)
{
if (Apply_Button != null)
	Apply_Button.removeActionListener (apply_listener);
}

/**	Gets an array of <CODE>ActionListener</CODE>s currently registered
	with the "Apply" button.
<P>
	@return	An <CODE>ActionListener</CODE> array. This will be empty
		if there is no "Apply" button, or no <CODE>ActionListener</CODE>
		is registered with the button.
*/
public ActionListener[] getActionListeners ()
{
if (Apply_Button != null)
	return Apply_Button.getActionListeners ();
return new ActionListener[0];
}

private JButton make_Apply_Button
	(
	ActionListener	apply_listener
	)
{
Apply_Button		= new JButton ("Apply");
Apply_Button.setMnemonic ('A');
Apply_Button.setToolTipText ("Apply selection");
Apply_Button.addActionListener (apply_listener);
return Apply_Button;
}

/*==============================================================================
	Accessors
*/
//	Font:

private void set_font
	(
	Font	font
	)
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println (">>> Font_Selector.set_font: " + font);
if (font == null)
	{
	if (Original_Font == null)
		Original_Font = getFont ();
	if (Original_Font == null)
		Original_Font = DEFAULT_FONT;
	font = Original_Font;
	}
if (Current_Font == font)
	{
	if ((DEBUG & DEBUG_ACCESSORS) != 0)
		System.out.println ("<<< Font_Selector.Font: font unchanged.");
	return;
	}
String
	name = font.getFamily ();
int
	size = font.getSize (),
	style = font.getStyle (),
	index;
boolean
	update = false;
if (Current_Font == null ||
	! Current_Font.getFamily ().equals (name))
	{
	update = true;
	if ((DEBUG & DEBUG_ACCESSORS) != 0)
		System.out.println
			("    Changing Current_Font name to " + name);
	if (Font_List != null)
		{
		index = Font_List_Model.getIndexOf (name);
		if ((DEBUG & DEBUG_ACCESSORS) != 0)
			System.out.println
				("    Font_List selected index " + index);
		Selection_Enabled = false;
		Font_List.setSelectedIndex (index);
		Selection_Enabled = true;
		}
	}
if (Current_Font == null ||
	Current_Font.getSize () != size)
	{
	update = true;
	if ((DEBUG & DEBUG_ACCESSORS) != 0)
		System.out.println
			("    Changed Current_Font size to " + size);
	if (Size_List != null)
		{
		index = Size_List_Model.getIndexOf (new Integer (size));
		if ((DEBUG & DEBUG_ACCESSORS) != 0)
			System.out.println
				("    Size_List selected index " + index);
		Selection_Enabled = false;
		Size_List.setSelectedIndex (index);
		Selection_Enabled = true;
		}
	}
if (Current_Font == null ||
	Current_Font.getStyle () != style)
	{
	update = true;
	if ((DEBUG & DEBUG_ACCESSORS) != 0)
		System.out.println
			("    Changed Current_Font style to "
			+ Style_Description (style) + " (" + style + ')');
	if (Style_List != null)
		{
		if ((DEBUG & DEBUG_ACCESSORS) != 0)
			System.out.println
				("    Style_List selected index " + style);
		Selection_Enabled = false;
		Style_List.setSelectedIndex (style);
		Selection_Enabled = true;
		}
	}
if (update)
	Sample_Text.setFont (Current_Font = font);
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println ("<<< Font_Selector.set_font: update " + update);
}

/**	Set the current font.
<P>
	If the font has a size that is not in the size selection list,
	then the <CODE>{@link #Nearest_Size Nearest_Size}</CODE> in
	the selection list will be applied to the font.
<P>
	@param	font	The Font to become the current font.
	@return	This Font_Selector.
*/
public Font_Selector Font
	(
	Font	font
	)
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println (">-< Font_Selector.Font: " + font);
int
	size = font.getSize (),
	nearest_size = Nearest_Size (size);
if (size != nearest_size)
	font = font.deriveFont ((float)nearest_size);
set_font (font);
return this;
}

/**	Set the current font name, style and size.
<P>
	The name must be an available font name.
<P>
	The style must be one of the <CODE>Font</CODE> class
	style constants.
<P>
	If the font has a size that is not in the size selection list,
	then the <CODE>{@link #Nearest_Size Nearest_Size}</CODE> in
	the selection list will be applied to the font.
<P>
	@param	name	The String providing a valid font family name.
	@param	style	A <CODE>Font</CODE> class style value.
	@param	size	The point size for the font.
	@return	This Font_Selector.
	@see	Font
*/
public Font_Selector Font
	(
	String	name,
	int		style,
	int		size
	)
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println (">-< Font_Selector.Font: "
		+ "name - " + name
		+ ", size - " + size
		+ ", style - " + Style_Description (style) + " (" + style + ')');
if (name != null &&
	Font_List_Model.getIndexOf (name) >= 0 &&
	Style_List_Model.getSize () > style && style >= 0)
	set_font (name, style, Nearest_Size (size));
return this;
}

private void set_font
	(
	String	name,
	int		style,
	int		size
	)
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println (">-< Font_Selector.set_font: "
		+ "name - " + name
		+ ", size - " + size
		+ ", style - " + Style_Description (style) + " (" + style + ')');
set_font (new Font (name, style, size));
}

/**	Gets the current font.
<P>
	@return	The current Font.
*/
public Font Font ()
{return Current_Font;}

/**	Gets the original font.
<P>
	@return	The original Font. This was the default font determined
		when this <CODE>Font_Selector</CODE> was constructed or when
		the <CODE>{@link #Select(Font) Select}</CODE> method was used.
*/
public Font Original_Font ()
{return Original_Font;}

/**	Sets the current font to the named family.
<P>
	@param	name	The font family name to be applied.
	@return	This Font_Selector.
*/
public Font_Selector Font_Name
	(
	String	name
	)
{return Font (name, Current_Font.getStyle (), Current_Font.getSize ());}

private void set_font_name
	(
	String	name
	)
{set_font (name, Current_Font.getStyle (), Current_Font.getSize ());}

/**	Gets the name of the current font.
<P>
	@return	The name of the current font.
*/
public String Font_Name ()
{return Current_Font.getFamily ();}

private static void construct_Font_List_Model ()
{
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println (">-< Font_Selector.construct_Font_List_Model");
Font_List_Model = new DefaultComboBoxModel
	(GraphicsEnvironment.getLocalGraphicsEnvironment ()
		.getAvailableFontFamilyNames ());
}

//	Style:

/**	Sets the style of the current font.
<P>
	@param	style	A <CODE>Font</CODE> class style value.
	@return	This Font_Selector.
*/
public Font_Selector Style
	(
	int		style
	)
{return Font (Current_Font.getFamily (), style, Current_Font.getSize ());}

private void set_style
	(
	int		style
	)
{set_font (Current_Font.getFamily (), style, Current_Font.getSize ());}

/**	Gets the style of the current font.
<P>
	@return	The <CODE>Font</CODE> class style value of the current font.
*/
public int Style ()
{return Current_Font.getStyle ();}

/**	Provides a description of a style value.
<P>
	@param	style	A <CODE>Font</CODE> class style value.
	@return	A String describing the style. This will be null if
		the style value is invalid.
*/
public static String Style_Description
	(
	int		style
	)
{
if (Style_List_Model == null)
	construct_Style_List_Model ();
if (style >= 0 &&
	style < Style_List_Model.getSize ())
	return (String)Style_List_Model.getElementAt (style);
return null;
}

private static void construct_Style_List_Model ()
{
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println (">-< Font_Selector.construct_Style_List_Model");
Style_List_Model = new DefaultComboBoxModel (new Object[]
	{"Plain", "Bold", "Italic", "Bold & Italic"});
}

//	Size:

/**	Sets the size of the current font.
<P>
	@param	size	The font size.
	@return	This Font_Selector.
	@see	#Font(String, int, int)
*/
public Font_Selector Size
	(
	int		size
	)
{return Font (Current_Font.getFamily (), Current_Font.getStyle (), size);}

private void set_size
	(
	int		size
	)
{set_font (Current_Font.getFamily (), Current_Font.getStyle (), size);}

/**	Gets the size of the current font.
<P>
	@return	The point size of the current font.
*/
public int Size ()
{return Current_Font.getSize ();}

/**	Gets the size from the selection list that is nearest to
	a specified size.
<P>
	If the size is less than the smallest selection size, then the
	smallest size will be returned. If the size is greater than the
	largest selection size, then the largest size will be returned.
	Otherwise the selection size nearest to the specified size, rounded
	up, is returned.
<P>
	@param	size	The size to use in searching for the nearest
		size in the selection list.
*/
public int Nearest_Size
	(
	int		size
	)
{
int
	total = Size_List_Model.getSize (),
	previous = ((Integer)Size_List_Model.getElementAt (0)).intValue (),
	current = previous,
	count = 1;
while (count < total && current < size)
	{
	previous = current;
	current = ((Integer)Size_List_Model.getElementAt (count++)).intValue ();
	}
if (size >= current)
	return current;
if (size <= previous)
	return previous;
if ((current - size) <= (size - previous))
	return current;
return previous;
}

private void construct_Size_List_Model
	(
	Vector		sizes
	)
{
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println
		(">>> Font_Selector.construct_Size_List_Model: sizes - " + sizes);
if (sizes == null || sizes.isEmpty ())
	sizes = DEFAULT_SIZES;
else if (sizes.size () == 2 && ((Integer)sizes.elementAt (1)).intValue () < 0)
	{
	//	Sizes in the range [size[0],-size[1]].
	int
		size = ((Integer)sizes.elementAt (0)).intValue (),
		last = -((Integer)sizes.remove (1)).intValue ();
	if (size < MINIMUM_SIZE)
		sizes.setElementAt (new Integer (size = MINIMUM_SIZE), 0);
	if (size > MAXIMUM_SIZE)
		sizes.setElementAt (new Integer (size = MAXIMUM_SIZE), 0);
	if (last < MINIMUM_SIZE)
		last = MINIMUM_SIZE;
	if (last > MAXIMUM_SIZE)
		last = MAXIMUM_SIZE;
	if (size > last)
		{
		sizes.setElementAt (new Integer (last), 0);
		last = size;
		size = ((Integer)sizes.elementAt (0)).intValue ();
		}
	else if (size != last)
		while (size <= last)
			sizes.add (new Integer (size++));
	}
else
	{
	//	Sizes list; sort and unique the list.
	TreeSet
		sorted_sizes = new TreeSet (sizes);
	sizes.clear ();
	int
		count = 0,
		size,
		previous_size = 0;
	Integer
		size_Integer;
	Iterator
		iterator = sorted_sizes.iterator ();
	while (iterator.hasNext ())
		{
		size_Integer = (Integer)iterator.next ();
		size = size_Integer.intValue ();
		if (size >= MINIMUM_SIZE &&
			size <= MAXIMUM_SIZE &&
			size != previous_size)
			//	Only add unique sizes in the acceptable range.
			sizes.add (size_Integer);
		previous_size = size;
		}
	if (sizes.isEmpty ())
		sizes = DEFAULT_SIZES;
	}
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println
		("<<< Font_Selector.construct_Size_List_Model: sizes - " + sizes);
Size_List_Model = new DefaultComboBoxModel (sizes);
}

//	Text:

/**	Sets the text to be displayed in the current font.
<P>
	@param	text	The String of text to be displayed.
	@return	This Font_Selector.
*/
public Font_Selector Text
	(
	String	text
	)
{
Sample_Text.setText (text);
return this;
}

/**	Gets the current sample text.

	The default text is:
<PRE>
	ABCDEFGHIJKLMNOPQRSTUVWXYZ
	abcdefghijklmnopqrstuvwxyz
	0123456789 ~!@#$%^&*()_+`-=[]\\{}|;':\"<>?,./
</PRE>
<P>
	@return	The String of sample text.
*/
public String Text ()
{return Sample_Text.getText ();}

//	Select:

/**	Initiates a user interactive font selection.
<P>
	This dialog is <CODE>{@link #setVisible(boolean) setVisible}</CODE>.
	When the user completes the selection or closes the dialog, the
	current font is returned. If the "Cancel" button is pressed, then
	the original font is restored before the selection operation completes.
<P>
	@param	default_font	The font to be the original, default
		font for the selection. If null, then the previous original
		font is used.
	@return	The Font that was selected. This will be the original
		font if no selection was made or the "Cancle" button was
		pressed.
*/
public Font Select
	(
	Font	default_font
	)
{
if ((DEBUG & DEBUG_SELECT) != 0)
	System.out.println
		(">>> Font_Selector.Select: " + default_font);
if (default_font == null)
	//	Start with the original font.
	Font (Original_Font);
else
	//	Start with the specified font as the original font.
	Font (Original_Font = default_font);
setVisible (true);
if ((DEBUG & DEBUG_SELECT) != 0)
	System.out.println
		("<<< Font_Selector.Select: "
		+ Font_Name ()
		+ ", size " + ((Size () == -1) ?
			"(none)" : String.valueOf (Size ()))
		+ ", style " + ((Style () == -1) ?
			"(none)" : Style_List_Model.getElementAt (Style ())));
return Current_Font;
}

/**	Initiates a <CODE>Select</CODE> operation using the previous
	original font.
<P>
	@see	#Select(Font)
*/
public Font Select ()
{return Select (null);}

/*==============================================================================
	Test stub
*/
private static String			_title_ = null;
private static boolean 			_get_font_ = true,
								_get_style_ = true;
private static Vector			_sizes_ = new Vector ();
private static final JLabel		_selected_font_ = new JLabel ("Select a font");
private static Font_Selector	_font_selector_ = null;
private static int				_click_count_ = 0;

public static void main (String arguments[])
{
final JFrame
	frame = new JFrame ("Font_Selector");
frame.addWindowListener (new WindowAdapter ()
	{public void windowClosing (WindowEvent e) {System.exit (0);}});

JPanel
	content = new JPanel ();
frame.setContentPane (content);
content.setLayout (new BoxLayout (content, BoxLayout.X_AXIS));

if (arguments.length > 0)
	{
	_title_ = arguments[0];
	if (arguments.length > 1)
		{
		_get_font_ = _get_style_ = false;
		for (int arg = 1;
			 arg < arguments.length;
			 arg++)
			{
			if (arguments[arg].equalsIgnoreCase ("font"))
				_get_font_ = true;
			else if (arguments[arg].equalsIgnoreCase ("style"))
				_get_style_ = true;
			else
				{
				try {_sizes_.add (new Integer (arguments[arg]));}
				catch (NumberFormatException exception) {}
				}
			}
		if (_sizes_.isEmpty ())
			_sizes_ = null;
		}
	}
_font_selector_ = new Font_Selector (
	_title_,
	_get_font_,
	_get_style_,
	_sizes_,
	new Font ("Serif", Font.BOLD, 14));

JButton
	button = new JButton ("Font...");
button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent e)
	{
	if (_click_count_++ == 1)
		_font_selector_.addActionListener (new ActionListener ()
			{public void actionPerformed (ActionEvent e)
			{_apply_font_ ();}});
	_font_selector_.Select ();
	_apply_font_ ();
	}});
button.setAlignmentX (JComponent.CENTER_ALIGNMENT);
content.add (button);
content.add (_selected_font_);

content.setPreferredSize (new Dimension (500, 50));
frame.pack ();
frame.setVisible (true);
}

private static void _apply_font_ ()
{
Font font = _font_selector_.Font ();
_selected_font_.setText (font.toString ());
_selected_font_.setFont (font);
}


}	//	End of Font_Selector class.
