/***************************************************************************
 *   Copyright (C) 2008 by Johan Maes                                      *
 *   on4qz@telenet.be                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "scopeplot.h"
#include <qwt_plot_curve.h>
#include <qwt_plot_grid.h>
#include <qwt_plot_marker.h>
#include <qwt_legend.h>
#include "qwt_plot_picker.h"
#include "qwt_symbol.h"
#include <QMessageBox>

scopePlot::scopePlot(QString title, QWidget *parent) : QMainWindow(parent)
{
	wd=new QWidget(this);
	ui.setupUi(wd);
	mrk1=0;
	mrk2=0;
	initActions();
	initMenuBar();
	initToolBar();
	initStatusBar();
	curve1 = new QwtPlotCurve("Curve 1");
	curve2 = new QwtPlotCurve("Curve 2");
	curve3 = new QwtPlotCurve("Curve 3");
  curve4 = new QwtPlotCurve("Curve 4");
	init(title);
	xScaleMul=1.;
	xPrimeScaleMul=1.;
	xAltScaleMul=1;
	toggleMarker=FALSE;

	showCrv1=TRUE;
	showCrv2=TRUE;
	showCrv3=TRUE;
  showCrv4=TRUE;
}

scopePlot::~scopePlot()
{
  delete curve1;
	delete curve2;
	delete curve3;
  delete curve4;
  delete toolsMenu;
}

void scopePlot::initActions()
{


  zoomAction = new QAction(QIcon(":/icons/viewmagplus.png"), tr("&Zoom"), this);
	zoomAction->setCheckable(TRUE);
  zoomAction->setStatusTip(tr("Zoom in or out"));
  connect(zoomAction, SIGNAL(toggled(bool)), this, SLOT(slotZoom(bool)));
	
}


void scopePlot::initMenuBar()
{
  toolsMenu=new QMenu(tr("&Zoom"));
	toolsMenu->addAction(zoomAction);
	menuBar()->addMenu(toolsMenu);
}

void scopePlot::initToolBar()
{
}

void scopePlot::initStatusBar()
{
}


void scopePlot::init(QString title)
{
  
	setup=TRUE;
  setCentralWidget(wd);
	connect(ui.offsetWheel, SIGNAL(valueChanged(double)),SLOT(slotOffsetChanged(double )));
	connect(ui.rangeWheel, SIGNAL(valueChanged(double)), SLOT(slotRangeChanged(double )));
	connect(ui.samplesPushButton, SIGNAL(clicked()), this, SLOT(slotSamplesButtton()));
	plW=ui.plotWindow;
	plW->setTitle(title);
	plW->setCanvasBackground(Qt::darkBlue);
	curve1->attach(plW);
	curve2->attach(plW);
	curve3->attach(plW);
  curve4->attach(plW);
	plW->setAxisTitle(QwtPlot::xBottom, "Time/seconds");
	plW->setAxisScale(QwtPlot::xBottom, 0, 100);
	plW->setAxisTitle(QwtPlot::yLeft, "Values");
	plW->setAxisScale(QwtPlot::yLeft, -1.5, 1.5);

	QwtPlotGrid *grid = new QwtPlotGrid;
	grid->enableXMin(true);
	grid->setMajPen(QPen(Qt::white, 0, Qt::DotLine));
	grid->setMinPen(QPen(Qt::gray, 0 , Qt::DotLine));
	grid->attach(plW);

	marker1=new QwtPlotMarker();
	marker1->setValue(0.0, 0.0);
	marker1->setLabel(QString::fromLatin1("mrk1"));
	marker1->setLabelAlignment(Qt::AlignRight | Qt::AlignBottom);
	marker1->setLinePen(QPen(QColor(200,150,0), 0, Qt::DashDotLine));
	marker1->setSymbol( QwtSymbol(QwtSymbol::Diamond, 
      QColor(Qt::yellow), QColor(Qt::green), QSize(7,7)));
//	marker1->hide();
	marker1->attach(plW);
	
	marker2=new QwtPlotMarker();
	marker2->setValue(0.0, 0.0);
	marker2->setLabel(QString::fromLatin1("mrk2"));
	marker2->setLabelAlignment(Qt::AlignLeft | Qt::AlignTop);
	marker2->setLinePen(QPen(QColor(200,150,0), 0, Qt::DashDotLine));
	marker2->setSymbol( QwtSymbol(QwtSymbol::Diamond, 
        QColor(Qt::yellow), QColor(Qt::yellow), QSize(7,7)));
//	marker2->hide();
	marker2->attach(plW);






	legend = new QwtLegend;
	legend->setFrameStyle(QFrame::Box|QFrame::Sunken);
  legend->setItemMode(QwtLegend::ClickableItem );
  QPalette pal(legend->palette());
  pal.setColor(QPalette::Window,Qt::darkBlue);
  pal.setColor(QPalette::WindowText,Qt::white);
  pal.setColor(QPalette::Text,Qt::white);
  legend->setPalette(pal);
	plW->insertLegend(legend, QwtPlot::BottomLegend);

  picker = new QwtPlotPicker(QwtPlot::xBottom, QwtPlot::yLeft,
        QwtPicker::PointSelection | QwtPicker::DragSelection, 
        QwtPlotPicker::CrossRubberBand, QwtPicker::AlwaysOn, 
        plW->canvas());
	picker->setRubberBandPen(QColor(Qt::green));
	picker->setRubberBand(QwtPicker::CrossRubberBand);
	picker->setTrackerPen(QColor(Qt::white));



//	connect(plW, SIGNAL(plotMouseMoved(const QMouseEvent&)),SLOT(plotMouseMoved( const QMouseEvent&)));
//  connect(picker, SIGNAL(moved(const QPoint &)),SLOT(pickerMoved(const QPoint &)));
  connect(picker, SIGNAL(selected(const QwtDoublePoint  &)),SLOT(pickerSelected(const QwtDoublePoint &)));
//	connect(plW, SIGNAL(plotMouseReleased(const QMouseEvent&)),SLOT(plotMouseReleased( const QMouseEvent&)));
	connect(plW, SIGNAL(legendClicked(QwtPlotItem *)),SLOT(legendClicked(QwtPlotItem *)));
	connect(ui.nextButton, SIGNAL(clicked()),SLOT(slotNext()));
	connect(ui.previousButton, SIGNAL(clicked()),SLOT(slotPrevious()));
}

void scopePlot::slotSamplesButtton()
{
  long i;
	if (ui.samplesPushButton->isChecked())
		{
			ui.samplesPushButton->setText("Time");
			xScaleMul=xAltScaleMul;
		}
	else
		{
			ui.samplesPushButton->setText("Samples");
			xScaleMul=xPrimeScaleMul;
		}
	for(i=0;i<x.size();i++)
		{
			x[i]=(double)i*xScaleMul;
		}
	curve1->setData(x.data(), yl.data(), x.size());
	curve2->setData(x.data(), yr.data(), x.size());
	curve3->setData(x.data(), c3.data(), x.size());
  curve4->setData(x.data(), c4.data(), x.size());
 	setupWheels(x.size());
	plW->replot();
}


void scopePlot::plot1D(short int *data, unsigned long len,QString name,QString xLabel,QString yLeftLabel)
{
    x.resize(len);
    yl.resize(len);
    for (unsigned long i = 0; i < len; i++)
      {

        x[i]=(double)i*xScaleMul;
        yl[i]=(double)data[i];
      }
    plot(name,xLabel,yLeftLabel);
}


void scopePlot::plot1D(double *data, unsigned long len,QString curveName,QString xLabel,QString yLeftLabel)
{
    x.resize(len);
    yl.resize(len);
    for (unsigned long i = 0; i < len; i++)
      {
        x[i]=(double)i*xScaleMul;
        yl[i]=data[i];
      }
   plot(curveName,xLabel,yLeftLabel);
}

void scopePlot::plot1DUpdate(double *data)
{
	 for (long i = 0; i < yl.size(); i++)
      {
         yl[i]=data[i];
      }
	curve1->setData(x.data(), yl.data(), x.size());
	plW->replot();
}

void scopePlot::plotData(unsigned int size,
											short int * iData, QString curve1Name,QString xLabel, QString yLLabel,
											double * dData, QString curve2Name, QString yRLabel)
{	
  plot1D(iData,size,curve1Name,xLabel,yLLabel);
 	addYR(dData,size,curve2Name,yRLabel);
	show();
}

void  scopePlot::plotData(unsigned int size,
											double * dData1, QString curve1Name,QString xLabel, QString yLLabel,
											double * dData2, QString curve2Name, QString yRLabel)
{	
  plot1D(dData1,size,curve1Name,xLabel,yLLabel);
 	addYR(dData2,size,curve2Name,yRLabel);
	show();
}


void scopePlot::addYR(double *data, unsigned long len,QString curveName,QString yRightLabel)
{
    yr.resize(len);
    for (unsigned long i = 0; i < len; i++)
      {
        yr[i]=data[i];
      }
   plotYR(curveName,yRightLabel);
}

void scopePlot::add3(double *data, unsigned long len,QString curveName)
{
    c3.resize(len);
    for (unsigned long i = 0; i < len; i++)
      {
       c3[i]=data[i];
      }
   plot3(curveName);
}

void scopePlot::add4(double *data, unsigned long len,QString curveName)
{
    c4.resize(len);
    for (unsigned long i = 0; i < len; i++)
      {
       c4[i]=data[i];
      }
   plot4(curveName);
}

void scopePlot::show()
{
	QMainWindow::show();
	plW->show();
	plW->replot();
}

void scopePlot::refresh()
{

	plW->replot();
}

void scopePlot::plot(QString curveName,QString xLabel,QString yLeftLabel)
{
  plW->setAxisTitle(QwtPlot::xBottom,xLabel);
  plW->setAxisTitle(QwtPlot::yLeft, yLeftLabel);
  plW->setAxisAutoScale(QwtPlot::yLeft);
  plW->setAxisAutoScale(QwtPlot::xBottom);
	curve1->setTitle(curveName);
	curve1->setPen(QPen(Qt::yellow));
	curve1->setYAxis(QwtPlot::yLeft);
	curve1->setData(x.data(), yl.data(), x.size());
 	setupWheels(x.size());
}


void scopePlot::plotYR(QString curveName,QString yRightLabel)
{

  // axes
  plW->setAxisTitle(QwtPlot::yRight,yRightLabel);
  plW->enableAxis(QwtPlot::yRight);
	curve2->setTitle(curveName);
	curve2->setPen(QPen(Qt::green));
	curve2->setYAxis(QwtPlot::yRight);
	curve2->setData(x.data(), yr.data(), x.size());
  plW->setAxisAutoScale(QwtPlot::yRight);
  plW->replot();
}


void scopePlot::plot3(QString curveName)
{

  // axes
  plW->setAxisAutoScale(QwtPlot::yLeft);
  plW->enableAxis(QwtPlot::yLeft);
	curve3->setTitle(curveName);
	curve3->setPen(QPen(Qt::red));
	curve3->setYAxis(QwtPlot::yLeft);
	curve3->setData(x.data(), c3.data(), x.size());
  plW->replot();
}

void scopePlot::plot4(QString curveName)
{

  // axes
  plW->setAxisAutoScale(QwtPlot::yLeft);
  plW->enableAxis(QwtPlot::yLeft);
  curve4->setTitle(curveName);
  curve4->setPen(QPen(Qt::white));
  curve4->setYAxis(QwtPlot::yLeft);
  curve4->setData(x.data(), c4.data(), x.size());
  plW->replot();
}


void scopePlot::slotZoom(bool)
{
}

void scopePlot::slotOffsetChanged(double ioffset)
{
	if (setup) return;
	if((ioffset-range/2)<startPoint)
		{
			ui.offsetWheel->setValue(startPoint+range/2);
			return;
		}
	if((ioffset+range/2)>endPoint)
		{
			ui.offsetWheel->setValue(endPoint-range/2);
			return;
		}
	dispCenter=ioffset;
	plW->setAxisScale(QwtPlot::xBottom,dispCenter-range/2,dispCenter+range/2);
	plW->replot();
}

void scopePlot::slotRangeChanged(double irange)
{
	if (setup) return;
	
	if((dispCenter-irange/2)<startPoint)
		{
			ui.rangeWheel->setValue((dispCenter-startPoint)*2);
			return;
		}
	if((dispCenter+irange/2)>endPoint)
		{
			ui.rangeWheel->setValue((endPoint-dispCenter)*2);
			return;
		}
	
	range=irange;
	plW->setAxisScale(QwtPlot::xBottom,dispCenter-range/2,dispCenter+range/2);
	plW->replot();
}

void scopePlot::setupWheels(int size)
{
 	setup=TRUE; 
	// offset is from 0 to size-range
	// range is from 10 to
	if (x.size()==0)
		{
			QMessageBox::warning(0,"Scope Plot",
			   	"No data in Scope Plot" ,
			   	QMessageBox::Ok,0 );
			return;
		}
	blockSignals(TRUE); 
	startPoint=x[0];
	endPoint=x[size-1];
	range=endPoint-startPoint;
	dispCenter=(endPoint+startPoint)/2;
	ui.offsetWheel->setMass(0.5);
	ui.rangeWheel->setMass(0.5);
		
	ui.offsetWheel->setRange(startPoint, endPoint, (endPoint-startPoint)/3600.);
  ui.offsetWheel->setTotalAngle(3600.0);
  ui.rangeWheel->setRange(range/200., range, range/180.);
	range=endPoint-startPoint;
	ui.rangeWheel->setTotalAngle(180.0);
	blockSignals(FALSE);
	ui.offsetWheel->setValue(dispCenter);
	ui.rangeWheel->setValue(range);
 	setup=FALSE;  
	plW->setAxisScale(QwtPlot::xBottom,dispCenter-range/2,dispCenter+range/2);
}

/*
void scopePlot::pickerMoved(const QPoint &pos)
{
	QString info;
	info.sprintf("x=%7g, yL=%7g, yR=%7g",
		plW->invTransform(QwtPlot::xBottom, pos.x()),
		plW->invTransform(QwtPlot::yLeft, pos.y()),
		plW->invTransform(QwtPlot::yRight, pos.y())
   	);
   ui.positionLabel->setText(info);
}
*/
void scopePlot::pickerSelected(const QwtDoublePoint &pos)
{
	if (!toggleMarker)
		{
			marker1->setValue(pos);
		}
	else
		{
			marker2->setValue(pos);
		}
	toggleMarker=!toggleMarker;
	showMarker();
	plW->replot();
}


void scopePlot::showMarker()
{

	QString info;
	
	ui.marker1Label->setText(info.sprintf("MRK1= %7g, %7g",marker1->xValue(),marker1->yValue()));
	ui.marker2Label->setText(info.sprintf("MRK2= %7g, %7g",marker2->xValue(),marker2->yValue()));
	ui.marker3Label->setText(info.sprintf("Diff= %7g, %7g",
	marker2->xValue()-marker1->xValue(),marker2->yValue()-marker1->yValue()));

}

void scopePlot::legendClicked(QwtPlotItem *pi)
{
	if (pi==curve1)
		{
			if (showCrv1)
				{
					curve1->setStyle(QwtPlotCurve::NoCurve);
				}
			else
				{
					curve1->setStyle(QwtPlotCurve::Lines);
				}
			showCrv1=!showCrv1;
		}
	else if (pi==curve2)
		{
			if (showCrv2)
				{
					curve2->setStyle(QwtPlotCurve::NoCurve);
				}
			else
				{
					curve2->setStyle(QwtPlotCurve::Lines);
				}
			showCrv2=!showCrv2;
		}
	else if (pi==curve3)
		{
			if (showCrv3)
				{
					curve3->setStyle(QwtPlotCurve::NoCurve);
				}
			else
				{
					curve3->setStyle(QwtPlotCurve::Lines);
				}
			showCrv3=!showCrv3;
		}
  else if (pi==curve4)
    {
      if (showCrv4)
        {
          curve4->setStyle(QwtPlotCurve::NoCurve);
        }
      else
        {
          curve4->setStyle(QwtPlotCurve::Lines);
        }
      showCrv4=!showCrv4;
    }


	plW->replot();
}
