# Copyright 2014: Mirantis Inc.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.


import ddt

from rally.plugins.task.sla import outliers
from rally.task import sla
from tests.unit import test


@ddt.ddt
class OutliersTestCase(test.TestCase):

    @ddt.data(({"max": 0, "min_iterations": 5, "sigmas": 2.5}, True),
              ({"max": -1}, False),
              ({"max": 0, "min_iterations": 2}, False),
              ({"max": 0, "sigmas": 0}, False),
              ({"foo": "bar"}, False))
    @ddt.unpack
    def test_validate(self, config, valid):
        results = sla.SLA.validate("outliers", None, None, config)
        if valid:
            self.assertEqual([], results)
        else:
            self.assertEqual(1, len(results))

    def test_result(self):
        sla1 = outliers.Outliers({"max": 1})
        sla2 = outliers.Outliers({"max": 2})
        iteration_durations = [3.1, 4.2, 3.6, 4.5, 2.8, 3.3, 4.1, 3.8, 4.3,
                               2.9, 10.2, 11.2, 3.4]  # outliers: 10.2, 11.2
        for sla_inst in [sla1, sla2]:
            for d in iteration_durations:
                sla_inst.add_iteration({"duration": d})
        self.assertFalse(sla1.result()["success"])  # 2 outliers >  1
        self.assertTrue(sla2.result()["success"])   # 2 outliers <= 2
        self.assertEqual("Failed", sla1.status())
        self.assertEqual("Passed", sla2.status())

    def test_result_large_sigmas(self):
        sla_inst = outliers.Outliers({"max": 1, "sigmas": 5})
        iteration_durations = [3.1, 4.2, 3.6, 4.5, 2.8, 3.3, 4.1, 3.8, 4.3,
                               2.9, 10.2, 11.2, 3.4]
        for d in iteration_durations:
            sla_inst.add_iteration({"duration": d})
        # NOTE(msdubov): No outliers registered since sigmas = 5 (not 2)
        self.assertTrue(sla_inst.result()["success"])
        self.assertEqual("Passed", sla_inst.status())

    def test_result_no_iterations(self):
        sla_inst = outliers.Outliers({"max": 0})
        self.assertTrue(sla_inst.result()["success"])

    def test_result_few_iterations_large_min_iterations(self):
        sla_inst = outliers.Outliers({"max": 0, "min_iterations": 10})
        iteration_durations = [3.1, 4.2, 4.7, 3.6, 15.14, 2.8]
        for d in iteration_durations:
            sla_inst.add_iteration({"duration": d})
        # NOTE(msdubov): SLA doesn't fail because it hasn't iterations < 10
        self.assertTrue(sla_inst.result()["success"])

    def test_result_few_iterations_small_min_iterations(self):
        sla_inst = outliers.Outliers({"max": 0, "min_iterations": 5})
        iteration_durations = [3.1, 4.2, 4.7, 3.6, 15.14, 2.8]
        for d in iteration_durations:
            sla_inst.add_iteration({"duration": d})
        # NOTE(msdubov): Now this SLA can fail with >= 5 iterations
        self.assertFalse(sla_inst.result()["success"])

    def test_add_iteration(self):
        sla_inst = outliers.Outliers({"max": 1})
        # NOTE(msdubov): One outlier in the first 11 iterations
        first_iterations = [3.1, 4.2, 3.6, 4.5, 2.8, 3.3, 4.1, 3.8, 4.3,
                            2.9, 10.2]
        for d in first_iterations:
            self.assertTrue(sla_inst.add_iteration({"duration": d}))
        # NOTE(msdubov): 12th iteration makes the SLA always failed
        self.assertFalse(sla_inst.add_iteration({"duration": 11.2}))
        self.assertFalse(sla_inst.add_iteration({"duration": 3.4}))

    @ddt.data([[3.1, 4.2, 3.6, 4.5, 2.8, 3.3, 4.1, 3.8, 4.3, 2.9, 10.2],
               [3.1, 4.2, 3.6, 4.5, 2.8, 3.3, 20.1, 3.8, 4.3, 2.9, 24.2],
               [3.1, 4.2, 3.6, 4.5, 2.8, 3.3, 4.1, 30.8, 4.3, 49.9, 69.2]])
    def test_merge(self, durations):

        single_sla = outliers.Outliers({"max": 1})

        for dd in durations:
            for d in dd:
                single_sla.add_iteration({"duration": d})

        slas = [outliers.Outliers({"max": 1})
                for _ in durations]

        for idx, sla_inst in enumerate(slas):
            for duration in durations[idx]:
                sla_inst.add_iteration({"duration": duration})

        merged_sla = slas[0]
        for sla_inst in slas[1:]:
            merged_sla.merge(sla_inst)

        self.assertEqual(single_sla.success, merged_sla.success)
        self.assertEqual(single_sla.iterations, merged_sla.iterations)

        # self.assertEqual(single_sla.threshold, merged_sla.threshold)

        # NOTE(ikhudoshyn): We are unable to implement
        # rally.plugins.common.sla.outliers.Outliers.merge(..) correctly
        # (see my comment for the method)
        # The assert above will fail with the majority of data
        # The line below passes with this particular data
        # but may fail as well on another data

        self.assertEqual(single_sla.outliers, merged_sla.outliers)
