// Copyright © 2024 Mikhail Hogrefe
//
// This file is part of Malachite.
//
// Malachite is free software: you can redistribute it and/or modify it under the terms of the GNU
// Lesser General Public License (LGPL) as published by the Free Software Foundation; either version
// 3 of the License, or (at your option) any later version. See <https://www.gnu.org/licenses/>.

use malachite_base::num::basic::floats::PrimitiveFloat;
use malachite_base::num::basic::traits::NegativeInfinity;
use malachite_base::num::float::NiceFloat;
use malachite_base::num::random::special_random_primitive_float_range;
use malachite_base::random::EXAMPLE_SEED;
use malachite_base::test_util::num::random::special_random_primitive_floats_helper_helper;
use malachite_base::test_util::stats::moments::{CheckedToF64, MomentStats};
use std::panic::catch_unwind;

fn special_random_primitive_float_range_helper<T: CheckedToF64 + PrimitiveFloat>(
    a: T,
    b: T,
    mean_exponent_numerator: u64,
    mean_exponent_denominator: u64,
    mean_precision_numerator: u64,
    mean_precision_denominator: u64,
    mean_zero_p_numerator: u64,
    mean_zero_p_denominator: u64,
    expected_values: &[T],
    expected_common_values: &[(T, usize)],
    expected_median: (T, Option<T>),
    expected_moment_stats: MomentStats,
) {
    special_random_primitive_floats_helper_helper(
        special_random_primitive_float_range::<T>(
            EXAMPLE_SEED,
            a,
            b,
            mean_exponent_numerator,
            mean_exponent_denominator,
            mean_precision_numerator,
            mean_precision_denominator,
            mean_zero_p_numerator,
            mean_zero_p_denominator,
        ),
        expected_values,
        expected_common_values,
        expected_median,
        expected_moment_stats,
    );
}

#[test]
fn test_special_random_primitive_float_range() {
    // f32, a = 1.0, b = 2.0, mean abs of exponent = 1, mean precision = 2, mean zero P = 1/10
    let values = &[
        1.0, 1.28125, 1.609375, 1.0, 1.5, 1.0, 1.0, 1.0, 1.5, 1.5, 1.71875, 1.0, 1.0, 1.0, 1.9375,
        1.9375, 1.9375, 1.0, 1.25, 1.375, 1.0, 1.0, 1.3125, 1.0, 1.25, 1.25, 1.75, 1.0, 1.5, 1.75,
        1.0, 1.5, 1.484375, 1.65625, 1.0, 1.0, 1.75, 1.5234375, 1.75, 1.0, 1.5, 1.25, 1.515625,
        1.0, 1.5, 1.0, 1.0, 1.4140625, 1.0, 1.875,
    ];
    let common_values = &[
        (1.0, 333222),
        (1.5, 222154),
        (1.75, 74130),
        (1.25, 74019),
        (1.375, 25091),
        (1.875, 24869),
        (1.625, 24805),
        (1.125, 24731),
        (1.4375, 8353),
        (1.8125, 8307),
        (1.5625, 8208),
        (1.6875, 8156),
        (1.0625, 8145),
        (1.3125, 8117),
        (1.1875, 8108),
        (1.9375, 8080),
        (1.03125, 2799),
        (1.96875, 2798),
        (1.15625, 2785),
        (1.28125, 2779),
    ];
    let sample_median = (1.3359375, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(1.3333245906621964),
        standard_deviation: NiceFloat(0.29800120914228473),
        skewness: NiceFloat(0.27868047434467813),
        excess_kurtosis: NiceFloat(-1.2068148586621514),
    };
    special_random_primitive_float_range_helper::<f32>(
        1.0,
        2.0,
        1,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = 1.0, b = 2.0, mean abs of exponent = 10, mean precision = 10, mean zero P = 1/100
    let values = &[
        1.4101562, 1.90625, 1.0, 1.953125, 1.875, 1.75, 1.46875, 1.3115959, 1.46875, 1.25, 1.0,
        1.5, 1.0, 1.5625, 1.5, 1.5117188, 1.1158447, 1.2253418, 1.1882324, 1.5, 1.1991673, 1.75,
        1.15625, 1.2519531, 1.0668945, 1.6660919, 1.0096436, 1.2796631, 1.875, 1.6987305,
        1.8608398, 1.3996582, 1.5, 1.1125488, 1.419461, 1.25, 1.0681152, 1.75, 1.1171875,
        1.4804688, 1.7055664, 1.3237305, 1.0, 1.0, 1.84375, 1.5, 1.125, 1.3097513, 1.0, 1.375,
    ];
    let common_values = &[
        (1.0, 100845),
        (1.5, 91973),
        (1.75, 42129),
        (1.25, 41726),
        (1.125, 19270),
        (1.625, 19149),
        (1.875, 19012),
        (1.375, 18836),
        (1.8125, 8852),
        (1.1875, 8822),
        (1.3125, 8722),
        (1.0625, 8692),
        (1.4375, 8630),
        (1.6875, 8623),
        (1.9375, 8620),
        (1.5625, 8431),
        (1.71875, 4047),
        (1.09375, 3974),
        (1.40625, 3949),
        (1.90625, 3941),
    ];
    let sample_median = (1.4912109, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(1.449574181673412),
        standard_deviation: NiceFloat(0.29612085814559613),
        skewness: NiceFloat(0.029341331332504897),
        excess_kurtosis: NiceFloat(-1.1487737412063284),
    };
    special_random_primitive_float_range_helper::<f32>(
        1.0,
        2.0,
        10,
        1,
        10,
        1,
        1,
        100,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = -0.1, b = 0.1, mean abs of exponent = 5, mean precision = 2, mean zero P = 1/10
    let values = &[
        0.09375,
        0.01171875,
        0.01171875,
        -0.0,
        -0.09375,
        -0.01171875,
        -0.01171875,
        -0.03125,
        0.03125,
        0.0625,
        -0.0625,
        -0.07525635,
        -0.0625,
        0.0,
        -0.09375,
        -0.09375,
        -0.09375,
        -0.0,
        0.07525635,
        -0.078125,
        -0.05078125,
        0.0625,
        0.09375,
        0.09375,
        -0.024414062,
        0.09375,
        -0.001953125,
        0.078125,
        0.05078125,
        -0.0,
        -0.0625,
        -0.09375,
        0.0,
        0.024414062,
        -0.03125,
        -0.0043945312,
        -0.09375,
        0.001953125,
        -0.09375,
        -0.003540039,
        -0.03125,
        0.0625,
        -0.046875,
        0.09375,
        -0.0625,
        0.03125,
        0.0,
        -0.0146484375,
        0.0043945312,
        0.09375,
    ];
    let common_values = &[
        (0.0625, 74962),
        (-0.0625, 74736),
        (-0.0, 50351),
        (0.09375, 50309),
        (-0.09375, 50178),
        (0.0, 49873),
        (0.03125, 37782),
        (-0.03125, 37678),
        (0.078125, 33219),
        (-0.078125, 33138),
        (0.046875, 25130),
        (-0.046875, 25053),
        (0.015625, 18743),
        (-0.015625, 18692),
        (0.0234375, 12593),
        (-0.0234375, 12549),
        (0.0703125, 11091),
        (0.0859375, 11088),
        (-0.0859375, 11060),
        (-0.0703125, 11055),
    ];
    let sample_median = (0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(0.0000674789649248169),
        standard_deviation: NiceFloat(0.05672019953597066),
        skewness: NiceFloat(-0.001702872258409426),
        excess_kurtosis: NiceFloat(-1.109977309372344),
    };
    special_random_primitive_float_range_helper::<f32>(
        -0.1,
        0.1,
        5,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = -0.1, b = 0.1, mean abs of exponent = 10, mean precision = 10, mean zero P = 1/100
    let values = &[
        0.0041503906,
        0.0625,
        0.00042547443,
        -0.0041503906,
        -0.0625,
        -0.00042547443,
        -0.0027618408,
        0.0027618408,
        0.09375,
        -0.09375,
        -0.083244324,
        -0.076171875,
        -0.00023269653,
        -0.00043609738,
        -0.0029296875,
        0.083244324,
        -0.011459351,
        -0.05419922,
        0.076171875,
        0.00023269653,
        0.00043609738,
        -0.000006198883,
        0.0029296875,
        -0.0008621216,
        0.011459351,
        0.05419922,
        -0.00035095215,
        -0.0625,
        0.000006198883,
        -0.0008010864,
        -0.00030517578,
        -0.00008274312,
        0.0008621216,
        -0.0029296875,
        -0.04892564,
        -0.013671875,
        0.00035095215,
        -0.0009398293,
        0.0625,
        -0.07344723,
        0.0008010864,
        -0.01953125,
        0.00030517578,
        0.00008274312,
        0.0029296875,
        0.04892564,
        -0.006134033,
        -0.0070343018,
        -0.005493164,
        -0.0000029057264,
    ];
    let common_values = &[
        (0.0625, 7060),
        (-0.0625, 7045),
        (0.09375, 6481),
        (-0.09375, 6468),
        (0.03125, 6186),
        (-0.03125, 6171),
        (0.078125, 5831),
        (-0.078125, 5813),
        (0.046875, 5527),
        (-0.046875, 5516),
        (0.015625, 5112),
        (-0.015625, 5103),
        (-0.0, 5098),
        (0.0, 4891),
        (0.0234375, 4821),
        (-0.0234375, 4810),
        (0.0078125, 4681),
        (-0.0078125, 4678),
        (0.01171875, 4123),
        (-0.01171875, 4120),
    ];
    let sample_median = (0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(0.000022248151599295377),
        standard_deviation: NiceFloat(0.035382441406785786),
        skewness: NiceFloat(0.0002929225243710825),
        excess_kurtosis: NiceFloat(1.5823949638933827),
    };
    special_random_primitive_float_range_helper::<f32>(
        -0.1,
        0.1,
        10,
        1,
        10,
        1,
        1,
        100,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = e, b = π, mean abs of exponent = 3, mean precision = 2, mean zero P = 1/10
    let values = &[
        3.0, 2.859375, 3.0234375, 3.0, 2.75, 3.0, 3.0, 3.0, 2.75, 2.75, 3.078125, 3.0, 3.0, 3.0,
        2.84375, 2.84375, 2.96875, 3.0, 3.125, 2.8125, 3.0, 3.0, 3.03125, 3.0, 3.125, 3.125, 2.875,
        3.0, 2.75, 2.875, 3.0, 2.75, 2.9140625, 2.734375, 3.0, 3.0, 3.125, 2.9257812, 3.125, 3.0,
        2.75, 3.125, 2.9921875, 3.0, 2.75, 3.0, 3.0, 3.0195312, 3.0, 2.8125,
    ];
    let common_values = &[
        (3.0, 333251),
        (2.75, 222162),
        (3.125, 74149),
        (2.875, 74011),
        (3.0625, 33262),
        (2.8125, 33146),
        (2.9375, 33097),
        (2.90625, 9548),
        (2.78125, 9377),
        (2.71875, 9327),
        (3.03125, 9327),
        (3.09375, 9323),
        (2.84375, 9315),
        (2.96875, 9265),
        (2.859375, 3186),
        (3.015625, 3185),
        (2.796875, 3180),
        (3.046875, 3160),
        (3.140625, 3157),
        (2.765625, 3152),
    ];
    let sample_median = (3.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(2.923236772231216),
        standard_deviation: NiceFloat(0.12616339323242115),
        skewness: NiceFloat(-0.17689588042510748),
        excess_kurtosis: NiceFloat(-1.2738593390011854),
    };
    special_random_primitive_float_range_helper::<f32>(
        core::f32::consts::E,
        core::f32::consts::PI,
        3,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = e, b = π, mean abs of exponent = 10, mean precision = 10, mean zero P = 1/100
    let values = &[
        2.9238281, 2.953125, 3.0, 2.8671875, 2.8125, 3.125, 3.015625, 2.8462658, 3.140625, 2.875,
        3.0, 2.75, 3.0, 2.71875, 2.75, 3.0214844, 2.970642, 3.0179443, 2.968872, 2.75, 2.9763536,
        2.875, 2.890625, 2.8388672, 2.7814941, 3.087654, 3.1065063, 2.817566, 2.8125, 2.7746582,
        2.746338, 3.0207214, 2.75, 2.9852295, 3.0, 2.9980469, 3.006958, 2.875, 2.8632812,
        2.9277344, 3.088623, 2.8376465, 3.0, 3.0, 3.0117188, 2.75, 2.8125, 3.0, 3.0, 3.140625,
    ];
    let common_values = &[
        (3.0, 103073),
        (2.75, 94168),
        (2.875, 43152),
        (3.125, 42886),
        (3.0625, 26213),
        (2.8125, 25982),
        (2.9375, 25799),
        (2.90625, 10203),
        (2.71875, 10186),
        (2.84375, 10149),
        (3.09375, 10141),
        (3.03125, 10134),
        (2.78125, 10085),
        (2.96875, 10085),
        (3.046875, 4666),
        (2.734375, 4649),
        (2.765625, 4649),
        (3.109375, 4629),
        (2.859375, 4606),
        (3.140625, 4590),
    ];
    let sample_median = (2.9375, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(2.925577550306893),
        standard_deviation: NiceFloat(0.12565616542944752),
        skewness: NiceFloat(-0.016691773531483878),
        excess_kurtosis: NiceFloat(-1.2405888218124512),
    };
    special_random_primitive_float_range_helper::<f32>(
        core::f32::consts::E,
        core::f32::consts::PI,
        10,
        1,
        10,
        1,
        1,
        100,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = 100.0, b = 101.0, mean abs of exponent = 7, mean precision = 2, mean zero P = 1/10
    let values = &[
        100.0, 100.28125, 100.609375, 100.0, 100.5, 100.0, 100.0, 100.0, 100.5, 100.5, 100.71875,
        100.0, 100.0, 100.0, 100.9375, 100.9375, 100.9375, 100.0, 100.25, 100.375, 100.0, 100.0,
        100.3125, 100.0, 100.25, 100.25, 100.75, 100.0, 100.5, 100.75, 100.0, 100.5, 100.484375,
        100.65625, 100.0, 100.0, 100.75, 100.52344, 100.75, 100.0, 100.5, 100.25, 100.515625,
        100.0, 100.5, 100.0, 100.0, 100.41406, 100.0, 100.875,
    ];
    let common_values = &[
        (100.0, 333428),
        (100.5, 222283),
        (100.25, 74406),
        (100.75, 73829),
        (100.375, 25079),
        (100.625, 25046),
        (100.875, 24767),
        (100.125, 24656),
        (100.3125, 8323),
        (100.8125, 8256),
        (100.4375, 8240),
        (100.1875, 8213),
        (100.9375, 8177),
        (100.5625, 8174),
        (100.0625, 8080),
        (100.6875, 8059),
        (100.34375, 2831),
        (100.90625, 2811),
        (100.84375, 2797),
        (100.03125, 2781),
    ];
    let sample_median = (100.33276, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(100.3331636999286),
        standard_deviation: NiceFloat(0.29794394613548436),
        skewness: NiceFloat(0.27963401436978874),
        excess_kurtosis: NiceFloat(-1.2056484413996),
    };
    special_random_primitive_float_range_helper::<f32>(
        100.0,
        101.0,
        7,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = 100.0, b = 101.0, mean abs of exponent = 10, mean precision = 10, mean zero P =
    // 1/100
    let values = &[
        100.41016, 100.90625, 100.0, 100.953125, 100.875, 100.75, 100.46875, 100.0, 100.59375,
        100.25, 100.0, 100.5, 100.0, 100.5625, 100.5, 100.24609, 100.3822, 100.41431, 100.40991,
        100.5, 100.5, 100.25, 100.46875, 100.009766, 100.78467, 100.64989, 100.06262, 100.12927,
        100.125, 100.833496, 100.01904, 100.0, 100.5, 100.11841, 100.6875, 100.75, 100.849365,
        100.25, 100.44531, 100.58984, 100.3999, 100.13916, 100.0, 100.0, 100.382324, 100.5,
        100.875, 100.3125, 100.0, 100.70117,
    ];
    let common_values = &[
        (100.0, 110530),
        (100.5, 100839),
        (100.75, 45914),
        (100.25, 45828),
        (100.625, 20971),
        (100.875, 20919),
        (100.125, 20849),
        (100.375, 20809),
        (100.8125, 9612),
        (100.3125, 9504),
        (100.9375, 9489),
        (100.1875, 9482),
        (100.0625, 9475),
        (100.6875, 9471),
        (100.4375, 9432),
        (100.5625, 9376),
        (100.40625, 4413),
        (100.59375, 4409),
        (100.03125, 4369),
        (100.21875, 4315),
    ];
    let sample_median = (100.48999, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(100.44491497797223),
        standard_deviation: NiceFloat(0.29627860462288247),
        skewness: NiceFloat(0.0334721390871067),
        excess_kurtosis: NiceFloat(-1.1467496174419634),
    };
    special_random_primitive_float_range_helper::<f32>(
        100.0,
        101.0,
        10,
        1,
        10,
        1,
        1,
        100,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = 1.0e38, b = Infinity, mean abs of exponent = 129, mean precision = 2, mean zero P =
    // 1/10
    let values = &[
        1.7014118e38,
        2.179934e38,
        2.7382097e38,
        1.2760589e38,
        1.4887354e38,
        1.2760589e38,
        1.2760589e38,
        1.2760589e38,
        1.4887354e38,
        1.0633824e38,
        1.5020276e38,
        1.7014118e38,
        1.7014118e38,
        1.7014118e38,
        1.80775e38,
        3.2964854e38,
        3.2964854e38,
        1.2760589e38,
        1.1697206e38,
        2.3394413e38,
        1.2760589e38,
        1.7014118e38,
        1.4089817e38,
        1.2760589e38,
        2.1267648e38,
        1.1697206e38,
        1.1697206e38,
        1.7014118e38,
        2.5521178e38,
        2.1267648e38,
        1.7014118e38,
        1.0633824e38,
        2.2596876e38,
        1.4754431e38,
        1.2760589e38,
        1.7014118e38,
        2.1267648e38,
        3.0705167e38,
        2.9774707e38,
        1.2760589e38,
        1.0633824e38,
        2.1267648e38,
        1.0168594e38,
        1.7014118e38,
        2.5521178e38,
        1.2760589e38,
        1.2760589e38,
        1.4654739e38,
        1.7014118e38,
        1.9140883e38,
    ];
    let common_values = &[
        (1.2760589e38, 190486),
        (1.7014118e38, 142730),
        (2.5521178e38, 95012),
        (1.4887354e38, 63724),
        (1.0633824e38, 63427),
        (2.1267648e38, 31749),
        (2.9774707e38, 31613),
        (1.3823971e38, 28489),
        (1.1697206e38, 28187),
        (1.5950736e38, 28108),
        (1.9140883e38, 10758),
        (3.1901472e38, 10750),
        (2.3394413e38, 10610),
        (2.7647942e38, 10493),
        (1.329228e38, 8248),
        (1.1165515e38, 8201),
        (1.5419045e38, 8177),
        (1.0102133e38, 8147),
        (1.4355662e38, 8096),
        (1.6482427e38, 8021),
    ];
    let sample_median = (1.5286122e38, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(1.7213254069804385e38),
        standard_deviation: NiceFloat(5.915470456567596e37),
        skewness: NiceFloat(1.0016210709928859),
        excess_kurtosis: NiceFloat(-0.11636872848929292),
    };
    special_random_primitive_float_range_helper::<f32>(
        1.0e38,
        f32::INFINITY,
        129,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = -f32::MIN_POSITIVE_SUBNORMAL, b = 3.0e-45, mean abs of exponent = 150, mean
    // precision = 2, mean zero P = 1/10
    let values = &[
        1.0e-45, 1.0e-45, 1.0e-45, -0.0, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, 1.0e-45, 1.0e-45,
        -1.0e-45, -1.0e-45, -1.0e-45, 0.0, -1.0e-45, -1.0e-45, -1.0e-45, -0.0, 1.0e-45, -1.0e-45,
        -1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, -1.0e-45, 1.0e-45, -1.0e-45, 1.0e-45, 1.0e-45, -0.0,
        -1.0e-45, -1.0e-45, 0.0, 1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, 1.0e-45, -1.0e-45,
        -1.0e-45, -1.0e-45, 1.0e-45, -1.0e-45, 1.0e-45, -1.0e-45, 1.0e-45, 0.0, -1.0e-45, 1.0e-45,
        1.0e-45,
    ];
    let common_values = &[(1.0e-45, 450531), (-1.0e-45, 449245), (-0.0, 50351), (0.0, 49873)];
    let sample_median = (0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(1.8020698251216835e-48),
        standard_deviation: NiceFloat(1.3292224464360085e-45),
        skewness: NiceFloat(-0.0025604536978191587),
        excess_kurtosis: NiceFloat(-1.8886053342505447),
    };
    special_random_primitive_float_range_helper::<f32>(
        -f32::MIN_POSITIVE_SUBNORMAL,
        3.0e-45,
        150,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = -0.0, b = 3.0e-45, mean abs of exponent = 150, mean precision = 2, mean zero P =
    // 1/10
    let values = &[
        1.0e-45, 1.0e-45, 1.0e-45, -0.0, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45,
        1.0e-45, 1.0e-45, 1.0e-45, 0.0, 1.0e-45, 1.0e-45, 1.0e-45, -0.0, 1.0e-45, 1.0e-45, 1.0e-45,
        1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, -0.0, 1.0e-45,
        1.0e-45, 0.0, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45,
        1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 1.0e-45, 0.0, 1.0e-45, 1.0e-45, 1.0e-45,
    ];
    let common_values = &[(1.0e-45, 899776), (-0.0, 50351), (0.0, 49873)];
    let sample_median = (1.0e-45, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(1.2608547270363571e-45),
        standard_deviation: NiceFloat(4.2080794564999146e-46),
        skewness: NiceFloat(-2.662524701673921),
        excess_kurtosis: NiceFloat(5.089037787023615),
    };
    special_random_primitive_float_range_helper::<f32>(
        -0.0,
        3.0e-45,
        150,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = 0.0, b = f32::MIN_POSITIVE_SUBNORMAL, mean abs of exponent = 150, mean precision =
    // 2, mean zero P = 1/10
    let values = &[0.0; 50];
    let common_values = &[(0.0, 1000000)];
    let sample_median = (0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(0.0),
        standard_deviation: NiceFloat(0.0),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_range_helper::<f32>(
        0.0,
        f32::MIN_POSITIVE_SUBNORMAL,
        150,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // -f32::MIN_POSITIVE_SUBNORMAL, b = -0.0, mean abs of exponent = 150, mean precision = 2, mean
    // zero P = 1/10
    let values = &[-1.0e-45; 50];
    let common_values = &[(-1.0e-45, 1000000)];
    let sample_median = (-1.0e-45, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(-1.401298464324817e-45),
        standard_deviation: NiceFloat(0.0),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_range_helper::<f32>(
        -f32::MIN_POSITIVE_SUBNORMAL,
        -0.0,
        150,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = -f32::MIN_POSITIVE_SUBNORMAL, b = f32::MIN_POSITIVE_SUBNORMAL, mean abs of exponent
    // = 150, mean precision = 2, mean zero P = 1/10
    let values = &[
        -1.0e-45, -1.0e-45, -1.0e-45, -0.0, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45,
        -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, 0.0, -1.0e-45, -1.0e-45, -1.0e-45, -0.0, -1.0e-45,
        -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45,
        -1.0e-45, -0.0, -1.0e-45, -1.0e-45, 0.0, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45,
        -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, -1.0e-45, 0.0,
        -1.0e-45, -1.0e-45, -1.0e-45,
    ];
    let common_values = &[(-1.0e-45, 899776), (-0.0, 50351), (0.0, 49873)];
    let sample_median = (-1.0e-45, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(-1.2608547270363571e-45),
        standard_deviation: NiceFloat(4.2080794564999146e-46),
        skewness: NiceFloat(2.662524701673921),
        excess_kurtosis: NiceFloat(5.089037787023615),
    };
    special_random_primitive_float_range_helper::<f32>(
        -f32::MIN_POSITIVE_SUBNORMAL,
        f32::MIN_POSITIVE_SUBNORMAL,
        150,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = -Infinity, b = Infinity, mean abs of exponent = 1, mean precision = 2, mean zero P =
    // 1/10
    let values = &[
        12.0,
        1.5,
        0.75,
        0.0,
        -12.0,
        -1.5,
        -0.75,
        -4.0,
        4.0,
        2.0,
        -2.0,
        -1.2041016,
        -2.0,
        f32::NEGATIVE_INFINITY,
        -0.75,
        -0.75,
        -3.0,
        0.0,
        1.2041016,
        -40.0,
        -0.6875,
        2.0,
        0.75,
        0.75,
        -1.3125,
        3.0,
        -2.0,
        40.0,
        0.6875,
        f32::NEGATIVE_INFINITY,
        -0.5,
        -1.5,
        -0.0,
        1.3125,
        -0.125,
        -1.125,
        -0.046875,
        2.0,
        -1.5,
        -1.9375,
        -1.0,
        0.5,
        -0.75,
        1.5,
        -1.0,
        0.125,
        0.0,
        -1.875,
        1.125,
        0.046875,
    ];
    let common_values = &[
        (1.0, 49888),
        (-1.0, 49745),
        (1.5, 33700),
        (f32::NEGATIVE_INFINITY, 33640),
        (-1.5, 33609),
        (0.0, 33393),
        (-0.0, 33191),
        (0.5, 25189),
        (-0.5, 25121),
        (2.0, 24770),
        (-2.0, 24716),
        (3.0, 16990),
        (-3.0, 16941),
        (0.75, 16655),
        (-0.75, 16611),
        (4.0, 12510),
        (0.25, 12465),
        (-4.0, 12461),
        (-0.25, 12430),
        (1.75, 11179),
    ];
    let sample_median = (-0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(f64::NAN),
        standard_deviation: NiceFloat(f64::NAN),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_range_helper::<f32>(
        f32::NEGATIVE_INFINITY,
        f32::INFINITY,
        1,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f32, a = -0.0, b = 0.0, mean abs of exponent = 1, mean precision = 2, mean zero P = 1/10
    let values = &[-0.0; 50];
    let common_values = &[(-0.0, 1000000)];
    let sample_median = (-0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(0.0),
        standard_deviation: NiceFloat(0.0),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_range_helper::<f32>(
        -0.0,
        0.0,
        1,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = 1.0, b = 2.0, mean abs of exponent = 1, mean precision = 2, mean zero P = 1/10
    let values = &[
        1.0, 1.28125, 1.609375, 1.0, 1.5, 1.0, 1.0, 1.0, 1.5, 1.5, 1.71875, 1.0, 1.0, 1.0, 1.9375,
        1.9375, 1.9375, 1.0, 1.25, 1.375, 1.0, 1.0, 1.3125, 1.0, 1.25, 1.25, 1.75, 1.0, 1.5, 1.75,
        1.0, 1.5, 1.484375, 1.65625, 1.0, 1.0, 1.75, 1.5234375, 1.75, 1.0, 1.5, 1.25, 1.515625,
        1.0, 1.5, 1.0, 1.0, 1.4140625, 1.0, 1.875,
    ];
    let common_values = &[
        (1.0, 333198),
        (1.5, 222142),
        (1.25, 74116),
        (1.75, 74020),
        (1.625, 24916),
        (1.375, 24899),
        (1.875, 24889),
        (1.125, 24789),
        (1.8125, 8258),
        (1.0625, 8253),
        (1.9375, 8239),
        (1.5625, 8179),
        (1.6875, 8158),
        (1.1875, 8148),
        (1.4375, 8140),
        (1.3125, 8094),
        (1.40625, 2826),
        (1.84375, 2825),
        (1.34375, 2802),
        (1.78125, 2781),
    ];
    let sample_median = (1.3359375, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(1.3334110677410675),
        standard_deviation: NiceFloat(0.29815058562815744),
        skewness: NiceFloat(0.27918114410601663),
        excess_kurtosis: NiceFloat(-1.206767297925903),
    };
    special_random_primitive_float_range_helper::<f64>(
        1.0,
        2.0,
        1,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = 1.0, b = 2.0, mean abs of exponent = 10, mean precision = 10, mean zero P = 1/100
    let values = &[
        1.41015625,
        1.90625,
        1.0,
        1.953125,
        1.875,
        1.75,
        1.46875,
        1.3115959167480469,
        1.46875,
        1.25,
        1.0,
        1.5,
        1.0,
        1.5625,
        1.5,
        1.51171875,
        1.1158447265625,
        1.225341796875,
        1.188232421875,
        1.5,
        1.199167251586914,
        1.75,
        1.15625,
        1.251953125,
        1.06689453125,
        1.6660919189453125,
        1.0096435546875,
        1.2796630859375,
        1.875,
        1.69873046875,
        1.86083984375,
        1.3819717407022836,
        1.5,
        1.802490234375,
        1.909928560256958,
        1.2957243919372559,
        1.401123046875,
        1.75,
        1.3671875,
        1.95703125,
        1.14501953125,
        1.59130859375,
        1.0,
        1.0,
        1.825472442433238,
        1.5,
        1.375,
        1.7292792797088623,
        1.0,
        1.852405808866024,
    ];
    let common_values = &[
        (1.0, 90973),
        (1.5, 83219),
        (1.75, 38074),
        (1.25, 37805),
        (1.875, 17372),
        (1.375, 17334),
        (1.625, 17115),
        (1.125, 17095),
        (1.3125, 7927),
        (1.6875, 7879),
        (1.4375, 7873),
        (1.5625, 7861),
        (1.9375, 7859),
        (1.8125, 7831),
        (1.1875, 7812),
        (1.0625, 7625),
        (1.84375, 3620),
        (1.34375, 3611),
        (1.28125, 3607),
        (1.15625, 3558),
    ];
    let sample_median = (1.4921875, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(1.4546223996766015),
        standard_deviation: NiceFloat(0.29563414647088554),
        skewness: NiceFloat(0.02445210407687687),
        excess_kurtosis: NiceFloat(-1.1474911308258557),
    };
    special_random_primitive_float_range_helper::<f64>(
        1.0,
        2.0,
        10,
        1,
        10,
        1,
        1,
        100,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = -0.1, b = 0.1, mean abs of exponent = 5, mean precision = 2, mean zero P = 1/10
    let values = &[
        0.09375,
        0.01171875,
        0.01171875,
        -0.0,
        -0.09375,
        -0.01171875,
        -0.01171875,
        -0.03125,
        0.03125,
        0.0625,
        -0.0625,
        -0.07525634765625,
        -0.0625,
        0.0,
        -0.09375,
        -0.09375,
        -0.09375,
        -0.0,
        0.07525634765625,
        -0.078125,
        -0.05078125,
        0.0625,
        0.09375,
        0.09375,
        -0.0244140625,
        0.09375,
        -0.001953125,
        0.078125,
        0.05078125,
        -0.0,
        -0.0625,
        -0.09375,
        0.0,
        0.0244140625,
        -0.03125,
        -0.00439453125,
        -0.09375,
        0.001953125,
        -0.09375,
        -0.0035400390625,
        -0.03125,
        0.0625,
        -0.046875,
        0.09375,
        -0.0625,
        0.03125,
        0.0,
        -0.0146484375,
        0.00439453125,
        0.09375,
    ];
    let common_values = &[
        (0.0625, 74957),
        (-0.0625, 74731),
        (-0.0, 50351),
        (0.09375, 50307),
        (-0.09375, 50176),
        (0.0, 49873),
        (0.03125, 37778),
        (-0.03125, 37674),
        (0.078125, 33218),
        (-0.078125, 33137),
        (0.046875, 25128),
        (-0.046875, 25051),
        (0.015625, 18742),
        (-0.015625, 18691),
        (0.0234375, 12592),
        (-0.0234375, 12548),
        (0.0859375, 11149),
        (-0.0859375, 11121),
        (0.0703125, 11027),
        (-0.0703125, 10991),
    ];
    let sample_median = (0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(0.00006747896492481727),
        standard_deviation: NiceFloat(0.05672798899804289),
        skewness: NiceFloat(-0.001703150825150012),
        excess_kurtosis: NiceFloat(-1.1100263487007116),
    };
    special_random_primitive_float_range_helper::<f64>(
        -0.1,
        0.1,
        5,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = -0.1, b = 0.1, mean abs of exponent = 10, mean precision = 10, mean zero P = 1/100
    let values = &[
        0.005245480002486147,
        0.0625,
        0.00037892110412940383,
        -0.005245480002486147,
        -0.0625,
        -0.00037892110412940383,
        -0.0025482177734375,
        0.0025482177734375,
        0.08385447226464748,
        -0.08385447226464748,
        -0.07456207275390625,
        -0.076171875,
        -0.000156402587890625,
        -0.0004252493381500244,
        -0.0029296875,
        0.07456207275390625,
        -0.012820055672818853,
        -0.05615234375,
        0.076171875,
        0.000156402587890625,
        0.0004252493381500244,
        -6.198883056640625e-6,
        0.0029296875,
        -0.00057220458984375,
        0.012820055672818853,
        0.05615234375,
        -0.0003814697265625,
        -0.0625,
        6.198883056640625e-6,
        -0.00092315673828125,
        -0.00030517578125,
        -0.00007461127825081348,
        0.00057220458984375,
        -0.0029296875,
        -0.05444455146789551,
        -0.009765625,
        0.0003814697265625,
        -0.0007852371782064438,
        0.0625,
        -0.07266771433324948,
        0.00092315673828125,
        -0.02317537716589868,
        0.00030517578125,
        0.00007461127825081348,
        0.0029296875,
        0.05444455146789551,
        -0.007110595703125,
        -0.00537109375,
        -0.0072021484375,
        -2.2798776626586914e-6,
    ];
    let common_values = &[
        (0.0625, 6407),
        (-0.0625, 6393),
        (0.09375, 5870),
        (-0.09375, 5858),
        (0.03125, 5626),
        (-0.03125, 5611),
        (0.078125, 5277),
        (-0.078125, 5260),
        (-0.0, 5098),
        (0.046875, 4988),
        (-0.046875, 4978),
        (0.0, 4891),
        (0.015625, 4680),
        (-0.015625, 4672),
        (0.0234375, 4384),
        (-0.0234375, 4374),
        (0.0078125, 4263),
        (-0.0078125, 4260),
        (0.01171875, 3729),
        (-0.01171875, 3726),
    ];
    let sample_median = (0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(0.000022298181036079024),
        standard_deviation: NiceFloat(0.035440202565071834),
        skewness: NiceFloat(0.00030138197742575316),
        excess_kurtosis: NiceFloat(1.5712669731678002),
    };
    special_random_primitive_float_range_helper::<f64>(
        -0.1,
        0.1,
        10,
        1,
        10,
        1,
        1,
        100,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = e, b = π, mean abs of exponent = 3, mean precision = 2, mean zero P = 1/10
    let values = &[
        3.0, 2.859375, 3.0234375, 3.0, 2.75, 3.0, 3.0, 3.0, 2.75, 2.75, 3.078125, 3.0, 3.0, 3.0,
        2.84375, 2.84375, 2.96875, 3.0, 3.125, 2.8125, 3.0, 3.0, 3.03125, 3.0, 3.125, 3.125, 2.875,
        3.0, 2.75, 2.875, 3.0, 2.75, 2.9140625, 2.734375, 3.0, 3.0, 3.125, 2.92578125, 3.125, 3.0,
        2.75, 3.125, 2.9921875, 3.0, 2.75, 3.0, 3.0, 3.01953125, 3.0, 2.8125,
    ];
    let common_values = &[
        (3.0, 333198),
        (2.75, 222142),
        (3.125, 74385),
        (2.875, 73751),
        (2.9375, 33356),
        (2.8125, 33246),
        (3.0625, 32891),
        (3.03125, 9479),
        (2.71875, 9381),
        (2.84375, 9352),
        (2.90625, 9344),
        (2.78125, 9342),
        (2.96875, 9313),
        (3.09375, 9258),
        (2.765625, 3221),
        (2.921875, 3214),
        (3.015625, 3204),
        (3.046875, 3175),
        (2.859375, 3165),
        (2.734375, 3163),
    ];
    let sample_median = (3.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(2.9232526079986494),
        standard_deviation: NiceFloat(0.1262248827883599),
        skewness: NiceFloat(-0.17691898540367232),
        excess_kurtosis: NiceFloat(-1.2742881322265622),
    };
    special_random_primitive_float_range_helper::<f64>(
        core::f64::consts::E,
        core::f64::consts::PI,
        3,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = e, b = π, mean abs of exponent = 10, mean precision = 10, mean zero P = 1/100
    let values = &[
        2.923828125,
        2.953125,
        3.0,
        2.8671875,
        2.8125,
        3.125,
        3.015625,
        2.8462657928466797,
        3.140625,
        2.875,
        3.0,
        2.75,
        3.0,
        2.71875,
        2.75,
        3.021484375,
        2.97064208984375,
        3.0179443359375,
        2.9688720703125,
        2.75,
        2.97635555267334,
        2.875,
        2.890625,
        2.8388671875,
        2.781494140625,
        3.0876541137695312,
        3.10650634765625,
        2.81756591796875,
        2.8125,
        2.774658203125,
        2.746337890625,
        3.086890615457378,
        2.75,
        2.7857666015625,
        2.9739707708358765,
        2.7529093623161316,
        2.7723388671875,
        3.125,
        2.94140625,
        2.916015625,
        2.776611328125,
        2.823486328125,
        3.0,
        3.0,
        3.135541538707912,
        2.75,
        3.0625,
        2.734408974647522,
        3.0,
        2.789327871054411,
    ];
    let common_values = &[
        (3.0, 91105),
        (2.75, 83336),
        (2.875, 38014),
        (3.125, 37964),
        (3.0625, 23196),
        (2.9375, 22929),
        (2.8125, 22922),
        (3.09375, 9177),
        (2.78125, 9003),
        (2.84375, 8943),
        (2.90625, 8936),
        (2.71875, 8920),
        (3.03125, 8913),
        (2.96875, 8854),
        (2.765625, 4212),
        (2.859375, 4171),
        (2.953125, 4126),
        (2.890625, 4119),
        (3.015625, 4090),
        (2.984375, 4082),
    ];
    let sample_median = (2.9375, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(2.9261256924208183),
        standard_deviation: NiceFloat(0.12519730680807895),
        skewness: NiceFloat(-0.016892719700380622),
        excess_kurtosis: NiceFloat(-1.2338524787684633),
    };
    special_random_primitive_float_range_helper::<f64>(
        core::f64::consts::E,
        core::f64::consts::PI,
        10,
        1,
        10,
        1,
        1,
        100,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = 100.0, b = 101.0, mean abs of exponent = 7, mean precision = 2, mean zero P = 1/10
    let values = &[
        100.0,
        100.28125,
        100.609375,
        100.0,
        100.5,
        100.0,
        100.0,
        100.0,
        100.5,
        100.5,
        100.71875,
        100.0,
        100.0,
        100.0,
        100.9375,
        100.9375,
        100.9375,
        100.0,
        100.25,
        100.375,
        100.0,
        100.0,
        100.3125,
        100.0,
        100.25,
        100.25,
        100.75,
        100.0,
        100.5,
        100.75,
        100.0,
        100.5,
        100.484375,
        100.65625,
        100.0,
        100.0,
        100.75,
        100.5234375,
        100.75,
        100.0,
        100.5,
        100.25,
        100.515625,
        100.0,
        100.5,
        100.0,
        100.0,
        100.4140625,
        100.0,
        100.875,
    ];
    let common_values = &[
        (100.0, 333198),
        (100.5, 222142),
        (100.25, 74116),
        (100.75, 74020),
        (100.625, 24916),
        (100.375, 24899),
        (100.875, 24889),
        (100.125, 24789),
        (100.8125, 8258),
        (100.0625, 8253),
        (100.9375, 8239),
        (100.5625, 8179),
        (100.6875, 8158),
        (100.1875, 8148),
        (100.4375, 8140),
        (100.3125, 8094),
        (100.40625, 2826),
        (100.84375, 2825),
        (100.34375, 2802),
        (100.78125, 2781),
    ];
    let sample_median = (100.3359375, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(100.33341106774094),
        standard_deviation: NiceFloat(0.29815058562814944),
        skewness: NiceFloat(0.27918114410604106),
        excess_kurtosis: NiceFloat(-1.2067672979256399),
    };
    special_random_primitive_float_range_helper::<f64>(
        100.0,
        101.0,
        7,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = 100.0, b = 101.0, mean abs of exponent = 10, mean precision = 10, mean zero P =
    // 1/100
    let values = &[
        100.41015625,
        100.90625,
        100.0,
        100.953125,
        100.875,
        100.75,
        100.46875,
        100.31159591674805,
        100.46875,
        100.25,
        100.0,
        100.5,
        100.0,
        100.5625,
        100.5,
        100.51171875,
        100.1158447265625,
        100.225341796875,
        100.188232421875,
        100.5,
        100.19916725158691,
        100.75,
        100.15625,
        100.251953125,
        100.06689453125,
        100.66609191894531,
        100.0096435546875,
        100.2796630859375,
        100.875,
        100.69873046875,
        100.86083984375,
        100.38197174070228,
        100.5,
        100.802490234375,
        100.90992856025696,
        100.83274569362402,
        100.740478515625,
        100.25,
        100.9140625,
        100.32421875,
        100.45556640625,
        100.36865234375,
        100.0,
        100.0,
        100.09102300740778,
        100.5,
        100.125,
        100.27702021598816,
        100.0,
        100.45081884413958,
    ];
    let common_values = &[
        (100.0, 91410),
        (100.5, 83666),
        (100.25, 38155),
        (100.75, 38107),
        (100.625, 17631),
        (100.125, 17378),
        (100.375, 17229),
        (100.875, 17035),
        (100.6875, 7968),
        (100.1875, 7924),
        (100.5625, 7918),
        (100.3125, 7917),
        (100.4375, 7914),
        (100.8125, 7806),
        (100.0625, 7786),
        (100.9375, 7756),
        (100.40625, 3664),
        (100.59375, 3645),
        (100.71875, 3629),
        (100.09375, 3596),
    ];
    let sample_median = (100.4921875, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(100.45411484815686),
        standard_deviation: NiceFloat(0.2957014655624591),
        skewness: NiceFloat(0.026064278269824867),
        excess_kurtosis: NiceFloat(-1.1480623934899934),
    };
    special_random_primitive_float_range_helper::<f64>(
        100.0,
        101.0,
        10,
        1,
        10,
        1,
        1,
        100,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = 1.0e38, b = Infinity, mean abs of exponent = 129, mean precision = 2, mean zero P =
    // 1/10
    let values = &[
        1.78405961588245e44,
        2.2322523270013563e41,
        7.009816758571332e40,
        2.1778071482940062e40,
        1.4887353552791058e38,
        5.575186299632656e42,
        1.2760588759535192e38,
        3.402823669209385e38,
        8.166776806102523e39,
        2.0416942015256308e39,
        6.424531087467318e41,
        1.7014118346046923e38,
        2.722258935367508e39,
        2.722258935367508e39,
        5.274376687274546e39,
        3.2964854295465914e38,
        2.4457795122442452e38,
        1.361129467683754e39,
        2.3819765684465692e39,
        1.914088313930279e38,
        1.2760588759535192e38,
        6.80564733841877e38,
        1.089966956543631e38,
        3.402823669209385e38,
        2.9774707105582116e38,
        5.954941421116423e38,
        1.169720636290726e38,
        1.7014118346046923e38,
        1.6333553612205046e40,
        2.9774707105582116e38,
        2.722258935367508e39,
        2.0416942015256308e39,
        3.9557825154559096e39,
        8.336917989562992e39,
        5.444517870735016e39,
        4.3556142965880123e40,
        1.9055812547572554e40,
        1.026164012745955e39,
        3.4028236692093846e39,
        1.2760588759535192e38,
        1.3066842889764037e41,
        4.7639531368931385e39,
        1.2295358961010472e38,
        1.7014118346046923e38,
        1.6725558898897967e43,
        1.2760588759535192e38,
        1.2760588759535192e38,
        1.6098350440189294e44,
        1.7014118346046923e38,
        3.743106036130323e39,
    ];
    let common_values = &[
        (1.2760588759535192e38, 83417),
        (1.7014118346046923e38, 62258),
        (3.402823669209385e38, 46943),
        (2.5521177519070385e38, 41660),
        (6.80564733841877e38, 35196),
        (5.104235503814077e38, 31368),
        (1.4887353552791058e38, 27818),
        (1.0633823966279327e38, 27639),
        (1.361129467683754e39, 26292),
        (1.0208471007628154e39, 23299),
        (2.722258935367508e39, 19745),
        (2.0416942015256308e39, 17625),
        (5.444517870735016e39, 14949),
        (2.9774707105582116e38, 13963),
        (2.1267647932558654e38, 13727),
        (4.0833884030512616e39, 13082),
        (1.169720636290726e38, 12430),
        (1.3823971156163125e38, 12379),
        (1.595073594941899e38, 12359),
        (1.0889035741470031e40, 11070),
    ];
    let sample_median = (4.253529586511731e38, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(5.5180886046975516e45),
        standard_deviation: NiceFloat(1.9464007021815662e48),
        skewness: NiceFloat(500.41964668581875),
        excess_kurtosis: NiceFloat(285116.05539365485),
    };
    special_random_primitive_float_range_helper::<f64>(
        1.0e38,
        f64::INFINITY,
        129,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = -f64::MIN_POSITIVE_SUBNORMAL, b = 1.0e-323, mean abs of exponent = 1075, mean
    // precision = 2, mean zero P = 1/10
    let values = &[
        5.0e-324, 5.0e-324, 5.0e-324, -0.0, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, 5.0e-324,
        5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, 0.0, -5.0e-324, -5.0e-324, -5.0e-324, -0.0,
        5.0e-324, -5.0e-324, -5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, -5.0e-324, 5.0e-324,
        -5.0e-324, 5.0e-324, 5.0e-324, -0.0, -5.0e-324, -5.0e-324, 0.0, 5.0e-324, -5.0e-324,
        -5.0e-324, -5.0e-324, 5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, 5.0e-324, -5.0e-324,
        5.0e-324, -5.0e-324, 5.0e-324, 0.0, -5.0e-324, 5.0e-324, 5.0e-324,
    ];
    let common_values = &[(5.0e-324, 450531), (-5.0e-324, 449245), (-0.0, 50351), (0.0, 49873)];
    let sample_median = (0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(5.0e-324),
        standard_deviation: NiceFloat(0.0),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_range_helper::<f64>(
        -f64::MIN_POSITIVE_SUBNORMAL,
        1.0e-323,
        1075,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = -0.0, b = 1.0e-323, mean abs of exponent = 1075, mean precision = 2, mean zero P =
    // 1/10
    let values = &[
        5.0e-324, 5.0e-324, 5.0e-324, -0.0, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324,
        5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 0.0, 5.0e-324, 5.0e-324, 5.0e-324, -0.0, 5.0e-324,
        5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324,
        5.0e-324, -0.0, 5.0e-324, 5.0e-324, 0.0, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324,
        5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 5.0e-324, 0.0,
        5.0e-324, 5.0e-324, 5.0e-324,
    ];
    let common_values = &[(5.0e-324, 899776), (-0.0, 50351), (0.0, 49873)];
    let sample_median = (5.0e-324, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(5.0e-324),
        standard_deviation: NiceFloat(0.0),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_range_helper::<f64>(
        -0.0,
        1.0e-323,
        1075,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = 0.0, b = f64::MIN_POSITIVE_SUBNORMAL, mean abs of exponent = 1075, mean precision =
    // 2, mean zero P = 1/10
    let values = &[0.0; 50];
    let common_values = &[(0.0, 1000000)];
    let sample_median = (0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(0.0),
        standard_deviation: NiceFloat(0.0),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_range_helper::<f64>(
        0.0,
        f64::MIN_POSITIVE_SUBNORMAL,
        1075,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // -f64::MIN_POSITIVE_SUBNORMAL, b = -0.0, mean abs of exponent = 1075, mean precision = 2, mean
    // zero P = 1/10
    let values = &[-5.0e-324; 50];
    let common_values = &[(-5.0e-324, 1000000)];
    let sample_median = (-5.0e-324, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(-5.0e-324),
        standard_deviation: NiceFloat(0.0),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_range_helper::<f64>(
        -f64::MIN_POSITIVE_SUBNORMAL,
        -0.0,
        1075,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = -f64::MIN_POSITIVE_SUBNORMAL, b = f64::MIN_POSITIVE_SUBNORMAL, mean abs of exponent
    // = 1075, mean precision = 2, mean zero P = 1/10
    let values = &[
        -5.0e-324, -5.0e-324, -5.0e-324, -0.0, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324,
        -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, 0.0, -5.0e-324, -5.0e-324,
        -5.0e-324, -0.0, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324,
        -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -0.0, -5.0e-324, -5.0e-324, 0.0,
        -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324,
        -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, -5.0e-324, 0.0, -5.0e-324, -5.0e-324,
        -5.0e-324,
    ];
    let common_values = &[(-5.0e-324, 899776), (-0.0, 50351), (0.0, 49873)];
    let sample_median = (-5.0e-324, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(-5.0e-324),
        standard_deviation: NiceFloat(0.0),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_range_helper::<f64>(
        -f64::MIN_POSITIVE_SUBNORMAL,
        f64::MIN_POSITIVE_SUBNORMAL,
        1075,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = -Infinity, b = Infinity, mean abs of exponent = 1, mean precision = 2, mean zero P =
    // 1/10
    let values = &[
        12.0,
        1.5,
        0.75,
        0.0,
        -12.0,
        -1.5,
        -0.75,
        -4.0,
        4.0,
        2.0,
        -2.0,
        -1.2041015625,
        -2.0,
        f64::NEGATIVE_INFINITY,
        -0.75,
        -0.75,
        -3.0,
        0.0,
        1.2041015625,
        -40.0,
        -0.6875,
        2.0,
        0.75,
        0.75,
        -1.3125,
        3.0,
        -2.0,
        40.0,
        0.6875,
        f64::NEGATIVE_INFINITY,
        -0.5,
        -1.5,
        -0.0,
        1.3125,
        -0.125,
        -1.125,
        -0.046875,
        2.0,
        -1.5,
        -1.9375,
        -1.0,
        0.5,
        -0.75,
        1.5,
        -1.0,
        0.125,
        0.0,
        -1.875,
        1.125,
        0.046875,
    ];
    let common_values = &[
        (1.0, 49883),
        (-1.0, 49740),
        (1.5, 33699),
        (f64::NEGATIVE_INFINITY, 33640),
        (-1.5, 33608),
        (0.0, 33393),
        (-0.0, 33191),
        (0.5, 25188),
        (-0.5, 25120),
        (2.0, 24765),
        (-2.0, 24711),
        (3.0, 16988),
        (-3.0, 16939),
        (0.75, 16654),
        (-0.75, 16610),
        (4.0, 12510),
        (0.25, 12465),
        (-4.0, 12461),
        (-0.25, 12430),
        (1.75, 11176),
    ];
    let sample_median = (-0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(f64::NAN),
        standard_deviation: NiceFloat(f64::NAN),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_range_helper::<f64>(
        f64::NEGATIVE_INFINITY,
        f64::INFINITY,
        1,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );

    // f64, a = -0.0, b = 0.0, mean abs of exponent = 1, mean precision = 2, mean zero P = 1/10
    let values = &[-0.0; 50];
    let common_values = &[(-0.0, 1000000)];
    let sample_median = (-0.0, None);
    let sample_moment_stats = MomentStats {
        mean: NiceFloat(0.0),
        standard_deviation: NiceFloat(0.0),
        skewness: NiceFloat(f64::NAN),
        excess_kurtosis: NiceFloat(f64::NAN),
    };
    special_random_primitive_float_range_helper::<f64>(
        -0.0,
        0.0,
        1,
        1,
        2,
        1,
        1,
        10,
        values,
        common_values,
        sample_median,
        sample_moment_stats,
    );
}

fn special_random_primitive_float_range_fail_helper<T: PrimitiveFloat>() {
    assert_panic!(special_random_primitive_float_range::<T>(
        EXAMPLE_SEED,
        T::ONE,
        T::ZERO,
        1,
        1,
        2,
        1,
        1,
        10
    ));
    assert_panic!(special_random_primitive_float_range::<T>(
        EXAMPLE_SEED,
        T::ONE,
        T::ONE,
        1,
        1,
        2,
        1,
        1,
        10
    ));
    assert_panic!(special_random_primitive_float_range::<T>(
        EXAMPLE_SEED,
        T::ONE,
        T::NAN,
        1,
        1,
        2,
        1,
        1,
        10
    ));
    assert_panic!(special_random_primitive_float_range::<T>(
        EXAMPLE_SEED,
        T::ONE,
        T::TWO,
        0,
        1,
        2,
        1,
        1,
        10
    ));
    assert_panic!(special_random_primitive_float_range::<T>(
        EXAMPLE_SEED,
        T::ONE,
        T::TWO,
        1,
        0,
        2,
        1,
        1,
        10
    ));
    assert_panic!(special_random_primitive_float_range::<T>(
        EXAMPLE_SEED,
        T::from(100.0),
        T::from(101.0),
        6,
        1,
        2,
        1,
        1,
        10
    ));
    assert_panic!(special_random_primitive_float_range::<T>(
        EXAMPLE_SEED,
        T::ONE,
        T::TWO,
        1,
        1,
        1,
        2,
        1,
        10
    ));
    assert_panic!(special_random_primitive_float_range::<T>(
        EXAMPLE_SEED,
        T::ONE,
        T::TWO,
        1,
        1,
        1,
        0,
        1,
        10
    ));
    assert_panic!(special_random_primitive_float_range::<T>(
        EXAMPLE_SEED,
        T::ONE,
        T::TWO,
        1,
        1,
        2,
        1,
        2,
        1
    ));
    assert_panic!(special_random_primitive_float_range::<T>(
        EXAMPLE_SEED,
        T::ONE,
        T::TWO,
        1,
        1,
        2,
        1,
        1,
        0
    ));
}

#[test]
fn special_random_primitive_float_range_fail() {
    apply_fn_to_primitive_floats!(special_random_primitive_float_range_fail_helper);
}
