# Backporting Process

## Intro

While the _porting_ process of `rustybuzz` is complete, _backporting_ is a never-ending task.
And even if we would be able to port simple commits in 5min, multiply it by a thousand
and you would understand why we're so behind. This is why any help is welcome.

If you want to help, you don't have to know anything about text shaping, fonts or even
C++ and Rust. Most `harfbuzz` commits consist of minor tweaks and fixes, which can easily be ported
by anyone with even basic CS knowledge.

If you have any questions, we have a dedicated [discussion] for backporting.

## Working Environment

Ideally, you should use Linux or macOS. While working on Windows is potentially possible,
it would be pain. Mainly because of `harfbuzz` and not `rustybuzz`. `harfbuzz` relies on
many Unix tools.

For one, you would need at least: git, Python, Rust, clang/gcc and meson.
Potentially even [ragel](./ragel.md), but it has to be built from sources.

## General Algorithm

The porting algorithm is quite simple and applies to most projects.

1. Take the latest not ported `harfbuzz` commit.
  You can find the last ported in `rustybuzz` commit messages.
1. Analyze it to make sure it applies to `rustybuzz` or not.
1. Port it to `rustybuzz`.
1. Port `harfbuzz` tests (if needed).
1. Run `rustybuzz` tests (`cargo test`).
1. Run auto-format `cargo fmt`.
1. Commit.
1. Rinse and repeat.

Overall, if tests are passing - we're good. Everything else doesn't really matter.

## Analyzing Changes

The main difference between `harfbuzz` and `rustybuzz` is that `rustybuzz` is just a shaper,
while `harfbuzz` is collection of things. More specifically, `harfbuzz` includes:
shaping, subsetting, fonts parsing, Unicode tables, precompiled state-machines,
custom C++ containers (instead of std one) and bindings to 3rd-party libraries.

`rustybuzz` doesn't have subsetting. If you see changes in a file/function
with a name `subset` or a commit with the `[subset]` prefix, you can easily skip those.

Fonts parsing is done via [`ttf-parser`](https://github.com/RazrFalcon/ttf-parser)
and any changes to the fonts parsing in `harfbuzz` probably do not affect us,
unless new tests are failing. In which case better ask about in the [discussion].

Unicode tables and related routines are done differently in `rustybuzz`,
usually by using 3rd-party crates instead of a custom solution.
Again, better ask about such changes in the [discussion].

Precompiled state-machines are generated by [ragel](./ragel.md).
Follow the linked instruction if you see that any of `*.rl` files had changed.

We do not use any custom C++ containers `harfbuzz` implements and rely on Rust's std instead.
So changes to those can be ignored as well.

Changes to the `harfbuzz` build systems (it has 3... welcome to C++) should be completely ignored
as well.

Overall, if you see a change in a `hb-ot-*` file - this change probably affects `rustybuzz`.

## Updating Tests

Almost all of `harfbuzz` shaping tests (found in `test/shaping`) are as simple as
font + text = glyph ids + positions.
The difference is that `harfbuzz` uses Python to run tests, while we auto-generate
the default Rust tests. And to do so, we use `scripts/gen-shaping-tests.py`.

But to get the expected output, we first have to build `harfbuzz` at the commit we're porting.
You cannot use the latest `harfbuzz` version to do so. For example the one installed on your
system if you're using Linux.

The easies way to do so is to use `meson`:

```sh
git clone https://github.com/harfbuzz/harfbuzz
cd harfbuzz
git checkout HASH # put the required hash here
meson builddir --reconfigure # `--reconfigure` is needed only on the first run
ninja -Cbuilddir
```

Now you can run:

```sh
# Pass harfbuzz clone dir, not `harfbuzz/src` one.
cd rustybuzz/scripts
./gen-shaping-tests.py /path/to/harfbuzz
```

It will do everything for you. All you need is to run `cargo test` afterward to make sure
all tests are passing.

If not, then either your change was incorrect or there is a bug in `rustybuzz`.

## Commit Messages

Make sure each `rustybuzz` commit contains a link to the related `harfbuzz` commit.
[Example](https://github.com/RazrFalcon/rustybuzz/commit/5637691426b72dcac2c56a3d1fabe104438b5db7).
Commit message itself can be copy-pasted from `harfbuzz`.


[discussion]: https://github.com/RazrFalcon/rustybuzz/discussions/79
